﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/detail/hio/atk_HioSyncChannel.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <climits>
#include <nn/atk/viewer/detail/hio/atk_HioManager.h>
#include <nn/atk/fnd/basis/atkfnd_Time.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/nn_Windows.h>
#include <WinSock2.h>
#endif

#if !defined(NN_SDK_BUILD_RELEASE)
//#define NN_ATK_ENABLE_COM_DEBUG
#endif

namespace nn {
namespace atk {
namespace viewer {
namespace detail {

NN_DEFINE_STATIC_CONSTANT( const uint32_t HioSyncChannel::TimeoutInfinity );

//----------------------------------------------------------
HioResult
HioSyncChannel::Send(const HioPacket*& replyPacket, const HioPacket& message, uint32_t timeout /*= TimeoutInfinity*/) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        replyPacket = &HioPacket::Null();
        return HioResult(HioResultType_TargetNotInitialized);
    }

    // メッセージを送信します。
    HioResult result = GetStream().WritePacket(message);

#if defined(NN_ATK_ENABLE_COM_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[sndedit][HioSyncChannel]  send packet :  write : message=%s(0x%08x), Size=%d, BodySize=%d.\n",
        message.GetHeader().message.ToString(),
        uint32_t(message.GetHeader().message),
        sizeof(HioPacketHeader) + message.GetHeader().bodySize,
        static_cast<uint32_t>(message.GetHeader().bodySize));
#endif

    if(!result.IsTrue())
    {
        replyPacket = &HioPacket::Null();
        return HioResult(HioResultType_TargetSendError);
    }

    return Receive(replyPacket, timeout);
}

//----------------------------------------------------------
HioResult
HioSyncChannel::Receive(const HioPacket*& packet, uint32_t timeout /*= TimeoutInfinity*/) NN_NOEXCEPT
{
    if(GetHioManager() == NULL)
    {
        return HioResult(HioResultType_TargetNotInitialized);
    }

    HioResult result = HioResult(HioResultType_False);

    atk::detail::fnd::StopWatch stopWatch;
    stopWatch.Start();

    for(;;)
    {
        GetHioManager()->Update(false);

        result = GetStream().TryReadPacket();

        if(result.IsTrue())
        {
            packet = GetStream().GetReadPacket();
            break;
        }
        else if(!result.IsFalse())
        {
            break;
        }

        if(stopWatch.GetElapsedTime().ToMilliSeconds() >= timeout)
        {
            result = HioResult(HioResultType_TargetTimeout);
            break;
        }

#if defined(NN_BUILD_CONFIG_OS_WIN)
        ::Sleep(0);
#endif
    }

#if defined(NN_ATK_ENABLE_COM_DEBUG)
    if(result.IsTrue())
    {
        NN_DETAIL_ATK_INFO(
            "[sndedit][HioSyncChannel] receive packet : time=%d : message=%s(0x%08x), Size=%d, BodySize=%d.\n",
            stopWatch.GetElapsedTime().ToMilliSeconds(),
            packet->GetHeader().message.ToString(),
            uint32_t(packet->GetHeader().message),
            sizeof(HioPacketHeader) + packet->GetHeader().bodySize,
            static_cast<uint32_t>(packet->GetHeader().bodySize));
    }
#endif

    return result;
}

} // namespace nn::atk::viewer::detail
} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
