﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/detail/hio/atk_HioManager.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/viewer/detail/hio/atk_HioAsyncChannel.h>
#include <nn/atk/viewer/detail/hio/atk_HioProtocol.h>
#include <nn/atk/fnd/io/atkfnd_FileStreamImpl.h>

namespace nn {
namespace atk {
namespace viewer {
namespace detail {

//==========================================================
// class HioManager
//==========================================================
//----------------------------------------------------------
HioManager::HioManager() NN_NOEXCEPT
    : m_IsInitialized(false)
{
}

//----------------------------------------------------------
bool
HioManager::Initialize() NN_NOEXCEPT
{
    if (IsInitialized())
    {
        return true;
    }

    m_IsInitialized = true;
    return true;
}

//----------------------------------------------------------
void
HioManager::Finalize() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return;
    }

    m_IsInitialized = false;
}

//----------------------------------------------------------
void
HioManager::Update(bool withAsyncChannels/*= true*/) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return;
    }

    if (!Polling(withAsyncChannels))
    {
        return;
    }

    UpdateAsyncChannels();
}

//----------------------------------------------------------
void
HioManager::RegisterChannel(HioChannel& channel) NN_NOEXCEPT
{
#if !defined(NN_SDK_BUILD_RELEASE)
    // 重複登録がないことをチェックします。
    for(HioChannelRefList::const_iterator it = HioChannelRefList::const_iterator(m_ChannelRefs.begin()); it != HioChannelRefList::const_iterator(m_ChannelRefs.end()); ++it)
    {
        NN_SDK_ASSERT(it->value != &channel);
    }
#endif

    m_ChannelRefs.push_back(channel.GetReference());

    HioChannel::HioManagerLinker::SetHioManager(channel, this);
}

//----------------------------------------------------------
void
HioManager::UnregisterChannel(HioChannel& channel) NN_NOEXCEPT
{
    // HACK: 多重 erase しないようにするための保護処理
    //if(channel.GetReference().node.GetNext() == NULL &&
    //    channel.GetReference().node.GetPrev() == NULL)
    //{
    //    return;
    //}

    m_ChannelRefs.erase(m_ChannelRefs.iterator_to(channel.GetReference()));

    HioChannel::HioManagerLinker::SetHioManager(channel, NULL);
}

//----------------------------------------------------------
void
HioManager::UpdateAsyncChannels() NN_NOEXCEPT
{
    // チャンネル毎に更新します。
    for(HioChannelRefList::const_iterator it = HioChannelRefList::const_iterator(m_ChannelRefs.begin()); it != HioChannelRefList::const_iterator(m_ChannelRefs.end()); ++it)
    {
        NN_SDK_ASSERT_NOT_NULL(it->value);
        if(!it->value->IsAsync())
        {
            continue;
        }

        HioAsyncChannel& asyncChannel = *reinterpret_cast<HioAsyncChannel*>(it->value);
        asyncChannel.Update();
    }
}

//----------------------------------------------------------
size_t
HioManager::GetRequiredMemorySizeForOpenFile() NN_NOEXCEPT
{
    return sizeof(atk::detail::fnd::FileStreamImpl);
}

//----------------------------------------------------------
atk::detail::fnd::FileStream*
HioManager::OpenFile(void* buffer, size_t bufferLength, const char* filePath) NN_NOEXCEPT
{
    if(!IsInitialized() ||
        buffer == NULL ||
        bufferLength < GetRequiredMemorySizeForOpenFile() ||
        filePath == NULL ||
        *filePath == '\0')
    {
        return NULL;
    }

    atk::detail::fnd::FileStream* fileStream = new(buffer) atk::detail::fnd::FileStreamImpl();
    NN_SDK_ASSERT_NOT_NULL(fileStream);

    if(fileStream->Open(
        filePath,
        atk::detail::fnd::FileStream::AccessMode_Read
        ).IsFailed())
    {
        return NULL;
    }

    return fileStream;
}

//----------------------------------------------------------
bool
HioManager::Polling(bool withAsyncChannels) NN_NOEXCEPT
{
    // チャンネル毎にポーリング処理を行います。
    for(HioChannelRefList::const_iterator it = HioChannelRefList::const_iterator(m_ChannelRefs.begin()); it != HioChannelRefList::const_iterator(m_ChannelRefs.end()); ++it)
    {
        it->value->Polling();
    }

    if (!withAsyncChannels)
    {
        return false;
    }

    return true;
}


} // namespace nn::atk::viewer::detail
} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
