﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/detail/hio/atk_HioChannel.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/viewer/detail/hio/atk_HioManager.h>
#include <nn/atk/viewer/detail/hio/atk_HioProtocol.h>

namespace nn {
namespace atk {
namespace viewer {
namespace detail {

NN_DEFINE_STATIC_CONSTANT( const size_t HioChannel::DefaultWorkBufferLength );

//----------------------------------------------------------
HioChannel::HioChannel() NN_NOEXCEPT :
m_RecvStreamBufferLength(0),
m_RecvPacketBufferLength(0)
{
    m_Reference.value = this;
}

//----------------------------------------------------------
HioChannel::~HioChannel() NN_NOEXCEPT
{
    Finalize();
}

//----------------------------------------------------------
HioResult
HioChannel::Initialize(
    void* recvStreamBuffer,
    size_t recvStreamBufferLength,
    void* recvPacketBuffer,
    size_t recvPacketBufferLength,
    void* workBuffer,
    size_t workBufferLength
) NN_NOEXCEPT
{
    if(IsInitialized())
    {
        NN_SDK_ASSERT(false, "HioChannel is already initialized.\n");
        return HioResult(HioResultType_Failed);
    }

    if(recvStreamBuffer == NULL ||
        recvPacketBuffer == NULL ||
        recvStreamBufferLength == 0 ||
        recvPacketBufferLength == 0)
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        return HioResult(HioResultType_Failed);
    }

    m_RecvStreamBuffer = recvStreamBuffer;
    m_RecvPacketBuffer = reinterpret_cast<HioPacket*>(recvPacketBuffer);
    m_RecvStreamBufferLength = recvStreamBufferLength;
    m_RecvPacketBufferLength = recvPacketBufferLength;

    m_Stream.Initialize(workBuffer, workBufferLength);

    HioResult result = m_PacketStream.Initialize(m_Stream, m_RecvPacketBuffer, m_RecvPacketBufferLength);

    if(!result.IsTrue())
    {
        Finalize();
        return result;
    }

    return HioResult(HioResultType_True);
}

//----------------------------------------------------------
void
HioChannel::Finalize() NN_NOEXCEPT
{
    Close();

    m_PacketStream.Finalize();

    m_Stream.Finalize();

    m_RecvStreamBuffer = NULL;
    m_RecvPacketBuffer = NULL;
    m_RecvStreamBufferLength = 0;
    m_RecvPacketBufferLength = 0;
}

//----------------------------------------------------------
bool
HioChannel::Open(HioStream::ChannelType channel) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return false;
    }

    return m_Stream.Open(channel, m_RecvStreamBuffer, m_RecvStreamBufferLength);
}

//----------------------------------------------------------
void
HioChannel::Close() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return;
    }

    if(m_Stream.IsAvailable())
    {
        m_Stream.Close();
    }
}

//----------------------------------------------------------
void
HioChannel::ClearBuffer() NN_NOEXCEPT
{
    if(!IsInitialized() || !m_Stream.IsAvailable())
    {
        return;
    }

    m_Stream.Skip(m_Stream.GetReadableBytes());
    m_PacketStream.ClearBuffer();
}

//----------------------------------------------------------
bool
HioChannel::Polling() NN_NOEXCEPT
{
    if (!this->IsAsync())
    {
        // HioSyncChannelが切断を検知するタイミングがないので、バッファ読み込みを実行するようにしておく
        m_Stream.GetReadableBytes();
    }

    return m_Stream.Polling();
}

} // namespace nn::atk::viewer::detail
} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
