﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/detail/atk_SoundArchiveFileEditor.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/util/util_StringUtil.h>
#include <nn/atk/atk_Util.h>
#include <nn/atk/atk_SoundArchivePlayer.h>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/viewer/detail/atk_SoundArchiveEditController.h>
#include <nn/atk/viewer/detail/res/atk_ResItemInfo.h>

#include <nn/atk/fnd/string/atkfnd_String.h>
#include <nn/util/util_FormatString.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
namespace {
    void ReplaceFilePath(const nn::atk::viewer::detail::ResSoundInfo* soundInfo, const char* originalPath, char* replacedPath) NN_NOEXCEPT
    {
        const nn::atk::viewer::detail::ResStreamSoundInfo* streamSoundInfo =
            reinterpret_cast<const nn::atk::viewer::detail::ResStreamSoundInfo*>(soundInfo);
        NN_SDK_ASSERT_NOT_NULL(streamSoundInfo);
            nn::util::Strlcpy(replacedPath, originalPath, nn::atk::detail::FilePathMax);
        }
    }
#endif

namespace nn {
namespace atk {
namespace viewer {
namespace detail {

//----------------------------------------------------------
SoundArchiveFileEditor::SoundArchiveFileEditor() NN_NOEXCEPT :
m_EditController(NULL),
m_SoundArchivePlayer(NULL)
{
    m_SoundArchiveHook.Initialize(*this);
}

//----------------------------------------------------------
Result
SoundArchiveFileEditor::Initialize(
    SoundArchiveEditController* editController,
    SoundArchivePlayer* soundArchivePlayer) NN_NOEXCEPT
{
    if(editController == NULL ||
        soundArchivePlayer == NULL)
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        return Result(ResultType_Failed);
    }

    m_EditController = editController;
    m_SoundArchivePlayer = soundArchivePlayer;

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
SoundArchiveFileEditor::Finalize() NN_NOEXCEPT
{
    m_EditController = NULL;
    m_SoundArchivePlayer = NULL;
}

//----------------------------------------------------------
void
SoundArchiveFileEditor::Start() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundArchiveFileEditor is not initialized.");
    m_SoundArchivePlayer->detail_SetSoundArchiveFilesHook(&m_SoundArchiveHook);
}

//----------------------------------------------------------
void
SoundArchiveFileEditor::Stop() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundArchiveFileEditor is not initialized.");
    m_SoundArchivePlayer->detail_SetSoundArchiveFilesHook(NULL);
}

//----------------------------------------------------------
bool
SoundArchiveFileEditor::Hook::IsTargetItemImpl(const char* itemLabel) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(itemLabel);
    NN_SDK_ASSERT(*itemLabel != '\0');

    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    return m_Owner->m_EditController->GetItemCacheState(itemLabel) == CacheState_Cached;
}

//----------------------------------------------------------
void
SoundArchiveFileEditor::Hook::LockImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    return m_Owner->m_EditController->LockCacheStates();
}

//----------------------------------------------------------
void
SoundArchiveFileEditor::Hook::UnlockImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    return m_Owner->m_EditController->UnlockCacheStates();
}

//----------------------------------------------------------
atk::detail::fnd::FileStream*
SoundArchiveFileEditor::Hook::OpenFileImpl(void* buffer, size_t bufferLength, void* cacheBuffer, size_t cacheBufferLength, const char* itemLabel, const char* fileType) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT_NOT_NULL(itemLabel);
    NN_SDK_ASSERT(*itemLabel != '\0');
    NN_SDK_ASSERT_NOT_NULL(fileType);
    NN_SDK_ASSERT(StringToResDataType(fileType) == ResDataType_StreamSoundFile);

    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    ResDataType dataType = ResDataType_Unknown;

    // ファイルパスを参照するためにサウンド情報を取得します。
    const ResSoundInfo* soundInfo = reinterpret_cast<const ResSoundInfo*>(
        m_Owner->m_EditController->GetSoundInfo(
        itemLabel,
        &ResItemInfoUtility::IsSoundInfoDataType,
        &dataType));

    if(soundInfo == NULL)
    {
        return NULL;
    }

    // 現状はストリームサウンドのみ。
    if(dataType != ResDataType_StreamSound)
    {
        NN_ATK_WARNING("[sndedit] item '%s' is not StreamSound.", itemLabel);
        return NULL;
    }

    const ResName* pcFilePath = ResItemInfoUtility::GetFilePath(
        *soundInfo,
        ResDataType_StreamSound,
        ResDataType_StreamSoundFile);

    if(pcFilePath == NULL ||
        pcFilePath->GetLength() == 0)
    {
        return NULL;
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // PC版の時は、代わりのファイルを鳴らすようにするためにパスを書き換える
    char replacedPath[nn::atk::detail::FilePathMax];
    ReplaceFilePath(soundInfo, pcFilePath->GetName(), replacedPath);
#endif

    void* fileStreamBuffer = buffer;

    atk::detail::fnd::FileStream* fileStream = m_Owner->m_EditController->OpenFile(
        fileStreamBuffer,
        bufferLength,
#if defined(NN_BUILD_CONFIG_OS_WIN)
        replacedPath
#else
        pcFilePath->GetName()
#endif
        );

    if(fileStream == NULL)
    {
        return NULL;
    }

    if((cacheBuffer != NULL) && (cacheBufferLength > 0))
    {
        fileStream->EnableCache(cacheBuffer, cacheBufferLength);
    }

    return fileStream;
}

//----------------------------------------------------------
const void*
SoundArchiveFileEditor::Hook::GetFileAddressImpl(
    const char* itemLabel,
    const char* itemType,
    const char* fileType,
    uint32_t fileIndex/*= 0*/) NN_NOEXCEPT
{
    if(itemLabel == NULL || *itemLabel == '\0')
    {
        return NULL;
    }

    NN_SDK_ASSERT_NOT_NULL(itemType);
    NN_SDK_ASSERT_NOT_NULL(fileType);
    NN_SDK_ASSERT_NOT_NULL(m_Owner);

    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    ResDataType itemDataType = StringToResDataType(itemType);
    ResDataType fileDataType = StringToResDataType(fileType);

    const ResName* pcFilePath = NULL;

    // SEQ が参照するバンクファイルか波形アーカイブの場合
    if(itemDataType == ResDataType_SequenceSound &&
        fileDataType != ResDataType_SequenceFile)
    {
        pcFilePath = GetBankFilePath(itemLabel, fileDataType, fileIndex);
    }
    else
    {
        ResDataType cachedItemDataType = ResDataType_Unknown;

        // ファイルパスを参照するためにサウンド情報とバンク情報を取得します。
        const ResSoundInfo* soundInfo = reinterpret_cast<const ResSoundInfo*>(
            m_Owner->m_EditController->GetSoundInfo(
            itemLabel,
            &ResItemInfoUtility::IsSoundInfoDataType,
            &cachedItemDataType)
            );

        if(soundInfo == NULL || itemDataType != cachedItemDataType)
        {
            return NULL;
        }

        pcFilePath = ResItemInfoUtility::GetFilePath(
            *soundInfo,
            static_cast<ResDataType>(cachedItemDataType),
            fileDataType);
    }

    if(pcFilePath == NULL ||
        pcFilePath->GetLength() == 0)
    {
        return NULL;
    }

    return m_Owner->m_EditController->GetFile(pcFilePath->GetName(), fileDataType);
}

//----------------------------------------------------------
const ResName*
SoundArchiveFileEditor::Hook::GetBankFilePath(const char* itemLabel, ResDataType fileDataType, uint32_t bankIndex) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(itemLabel);
    NN_SDK_ASSERT(*itemLabel != '\0');
    NN_SDK_ASSERT_NOT_NULL(m_Owner);

    ResDataType itemInfoDataType = ResDataType_Unknown;

    // ファイルパスを参照するためにサウンド情報とバンク情報を取得します。
    const ResSoundInfo* soundInfo = reinterpret_cast<const ResSoundInfo*>(
        m_Owner->m_EditController->GetSoundInfo(
        itemLabel,
        &ResItemInfoUtility::IsSequenceSoundInfoDataType,
        &itemInfoDataType)
        );

    if(bankIndex >= SeqBankCount)
    {
        NN_SDK_ASSERT(false, "[sndedit] invalid bank index.\n");
        return NULL;
    }

    const char* bankName = NULL;

    if(soundInfo != NULL)
    {
        const ResSequenceSoundInfo* sequenceSoundInfo = reinterpret_cast<const ResSequenceSoundInfo*>(soundInfo);
        bankName = sequenceSoundInfo->sequenceSoundParam.bankNameOffsets[bankIndex].to_ptr<ResName>()->GetName();
    }
    else
    {
        const SoundArchive& soundArchive = m_Owner->m_SoundArchivePlayer->GetSoundArchive();

        SoundArchive::SequenceSoundInfo sequenceSoundInfo;
        if(!soundArchive.ReadSequenceSoundInfo(&sequenceSoundInfo, soundArchive.GetItemId(itemLabel)))
        {
            return NULL;
        }

        bankName = soundArchive.GetItemLabel(sequenceSoundInfo.bankIds[bankIndex]);
    }

    if(bankName == NULL || *bankName == '\0')
    {
        return NULL;
    }

    const ResBankInfo* bankInfo = m_Owner->m_EditController->GetBankInfo(bankName);

    if(bankInfo == NULL)
    {
        return NULL;
    }

    switch(fileDataType)
    {
    case ResDataType_BankFile:
    case ResDataType_WaveArchiveFile:
        break;

    default:
        NN_SDK_ASSERT(false, "[sndedit] invalid file data type.\n");
        return NULL;
    }

    return ResItemInfoUtility::GetFilePath(*bankInfo, fileDataType);
}

//----------------------------------------------------------
ResDataType
SoundArchiveFileEditor::Hook::StringToResDataType(const char* value) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(value);

    // 終端文字も含めて比較するために strlen() + 1 しています。
    if(std::strncmp(
        value,
        Hook::ItemTypeStreamSound,
        std::strlen(Hook::ItemTypeStreamSound) + 1) == 0)
    {
        return ResDataType_StreamSound;
    }

    if(std::strncmp(
        value,
        Hook::ItemTypeWaveSound,
        std::strlen(Hook::ItemTypeWaveSound) + 1) == 0)
    {
        return ResDataType_WaveSound;
    }

    if(std::strncmp(
        value,
        Hook::ItemTypeSequenceSound,
        std::strlen(Hook::ItemTypeSequenceSound) + 1) == 0)
    {
        return ResDataType_SequenceSound;
    }

    if(std::strncmp(
        value,
        Hook::FileTypeStreamBinary,
        std::strlen(Hook::FileTypeStreamBinary) + 1) == 0)
    {
        return ResDataType_StreamSoundFile;
    }

    if(std::strncmp(
        value,
        Hook::FileTypeWaveSoundBinary,
        std::strlen(Hook::FileTypeWaveSoundBinary) + 1) == 0)
    {
        return ResDataType_WaveSoundFile;
    }

    if(std::strncmp(
        value,
        Hook::FileTypeSequenceBinary,
        std::strlen(Hook::FileTypeSequenceBinary) + 1) == 0)
    {
        return ResDataType_SequenceFile;
    }

    if(std::strncmp(
        value,
        Hook::FileTypeBankBinary,
        std::strlen(Hook::FileTypeBankBinary) + 1) == 0)
    {
        return ResDataType_BankFile;
    }

    if(std::strncmp(
        value,
        Hook::FileTypeWaveArchiveBinary,
        std::strlen(Hook::FileTypeWaveArchiveBinary) + 1) == 0)
    {
        return ResDataType_WaveArchiveFile;
    }

    if(std::strncmp(
        value,
        Hook::FileTypeStreamPrefetchBinary,
        std::strlen(Hook::FileTypeStreamPrefetchBinary) + 1) == 0)
    {
        return ResDataType_StreamPreftchSoundFile;
    }

    return ResDataType_Unknown;
}

} // namespace nn::atk::viewer::detail
} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
