﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/detail/atk_SoundArchiveEditController.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/atk_BasicSound.h>
#include <nn/atk/atk_SoundHandle.h>
#include <nn/atk/atk_SoundArchivePlayer.h>
#include <nn/atk/atk_SoundPlayer.h>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/fnd/os/atkfnd_CriticalSection.h>
#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>
#include <nn/atk/fnd/io/atkfnd_FileStream.h>
#include <nn/atk/fnd/string/atkfnd_Path.h>
#include <nn/atk/fnd/basis/atkfnd_RuntimeTypeInfo.h>
#include <nn/atk/viewer/atk_Config.h>
#include <nn/atk/viewer/detail/atk_IErrorProvider.h>
#include <nn/atk/viewer/detail/atk_SoundEditConnection.h>
#include <nn/atk/viewer/detail/res/atk_ResItemInfo.h>
#include <nn/atk/viewer/detail/res/atk_ResourceManager.h>
#include <nn/atk/viewer/detail/protocol/atk_QueryItemInfoPacket.h>

#if !defined(NN_SDK_BUILD_RELEASE)
//#define NN_ATK_ENABLE_CACHE_DEBUG
#endif

namespace nn {
namespace atk {
namespace viewer {
namespace detail {

namespace {

//----------------------------------------------------------
//! @brief  キャッシュを更新する関数オブジェクトです。
class CacheUpdator
{
    NN_DISALLOW_COPY(CacheUpdator);

public:
    explicit CacheUpdator(SoundArchiveEditController& editController) NN_NOEXCEPT :
      m_EditController(editController) { }

public:
    void operator()(const char* name) NN_NOEXCEPT
    {
        m_EditController.MakeItemInfoEditable(name);
    }

private:
    SoundArchiveEditController& m_EditController;
};

}

//----------------------------------------------------------
Result
SoundArchiveEditController::Initialize(const InitializeArgs& args) NN_NOEXCEPT
{
    if(args.errorProvider == NULL ||
        args.buffer == NULL ||
        args.maxItemName == 0 ||
        args.resourceManager == NULL ||
        args.soundArchive == NULL ||
        args.soundArchivePlayer == NULL)
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        return Result(ResultType_Failed);
    }

    if(args.bufferLength < GetRequiredMemorySize(args.maxItemName))
    {
        NN_SDK_ASSERT(false, "out of memory.\n");
        return Result(ResultType_OutOfMemory);
    }

    atk::detail::fnd::FrameHeap frameHeap;
    frameHeap.Initialize(args.buffer, args.bufferLength);

    m_OpenFileBufferLength = SoundEditConnection::GetRequiredMemorySizeForOpenFile();
    m_OpenFileBuffer = frameHeap.Alloc(m_OpenFileBufferLength);

    if(m_OpenFileBuffer == NULL)
    {
        NN_SDK_ASSERT(false, "out of memory.\n");
        Finalize();
        return Result(ResultType_OutOfMemory);
    }

    m_ItemNameBufferLength = GetItemNameBufferRequiredMemorySize(args.maxItemName);
    m_ItemNameBuffer = reinterpret_cast<char*>(frameHeap.Alloc(m_ItemNameBufferLength));

    if(m_ItemNameBuffer == NULL)
    {
        NN_SDK_ASSERT(false, "out of memory.\n");
        Finalize();
        return Result(ResultType_OutOfMemory);
    }

    Result result = InitializeForPlatform(frameHeap);

    if(result.IsFailed())
    {
        NN_SDK_ASSERT(false, "failed to initialize.\n");
        Finalize();
        return result;
    }

    m_ErrorProvider = args.errorProvider;
    m_ResourceManager = args.resourceManager;
    m_SoundArchive = args.soundArchive;
    m_SoundArchivePlayer = args.soundArchivePlayer;

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
SoundArchiveEditController::Finalize() NN_NOEXCEPT
{
    if(IsStarted())
    {
        Stop();
    }

    FinalizeForPlatform();

    m_ErrorProvider = NULL;
    m_ResourceManager = NULL;
    m_SoundArchive = NULL;
    m_SoundArchivePlayer = NULL;
    m_OpenFileBuffer = NULL;
    m_OpenFileBufferLength = 0;
    m_ItemNameBuffer = NULL;
    m_ItemNameBufferLength = 0;
}

//----------------------------------------------------------
Result
SoundArchiveEditController::Start(SoundEditConnection* connection) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(connection);

    NN_SDK_ASSERT(IsInitialized(), "SoundArchiveEditController is not initialized.\n");

    m_Connection = connection;

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
SoundArchiveEditController::Stop() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundArchiveEditController is not initialized.\n");

    m_Connection = NULL;

    if(m_ResourceManager != NULL && m_ResourceManager->IsInitialized())
    {
        nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> unfreezeLock(m_UnfreezeCacheLockObject);
        nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> cacheLock(m_CacheLockObject);

        // アイテム情報は接続毎に破棄します。
        m_ResourceManager->RemoveAllItemInfos();

        // ファイルの参照カウントをクリアし、ガベージコレクション対象にします。
        m_ResourceManager->ClearAllItemFileReferenceCounts();
    }
}

//----------------------------------------------------------
size_t
SoundArchiveEditController::GetRequiredMemorySize(int maxItemName) const NN_NOEXCEPT
{
    return
        SoundEditConnection::GetRequiredMemorySizeForOpenFile() +
        GetItemNameBufferRequiredMemorySize(maxItemName) +
        GetRequiredMemorySizeForPlatform();
}

//----------------------------------------------------------
size_t
SoundArchiveEditController::GetTotalMemoryUsage() const NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return 0;
    }

    // SoundArchiveEditController 用のメモリ領域
    size_t result = m_OpenFileBufferLength + m_ItemNameBufferLength;

    // プラットフォーム固有のメモリ領域
    result += GetRequiredMemorySizeForPlatform();

    // ResourceManager 用のメモリ領域
    result += m_ResourceManager->GetMemoryUsage();

    return result;
}

//----------------------------------------------------------
void
SoundArchiveEditController::LockCacheStates() NN_NOEXCEPT
{
    ++m_LockCacheStatesCount;

    if(m_LockCacheStatesCount > 1)
    {
        return;
    }

    m_UnfreezeCacheLockObject.Lock();
}

//----------------------------------------------------------
void
SoundArchiveEditController::UnlockCacheStates() NN_NOEXCEPT
{
    if(m_LockCacheStatesCount == 0)
    {
        return;
    }

    --m_LockCacheStatesCount;

    if(m_LockCacheStatesCount == 0)
    {
        m_UnfreezeCacheLockObject.Unlock();
    }
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateCaches(ICanceller* canceller /*= NULL*/, void* cancellerParam /*= NULL*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditController is not started.\n");

    Result result = Result(ResultType_True);

    if(m_Connection == NULL || !m_Connection->IsOpened())
    {
        return Result(ResultType_False);
    }

    if(canceller != NULL)
    {
        canceller->Reset(cancellerParam);
    }

    if (m_SoundArchive == NULL || !m_SoundArchive->IsAvailable())
    {
        return Result(ResultType_False);
    }

    for(;;)
    {
        // IsCancel() が true なら終了する。
        if(canceller != NULL && canceller->IsCanceled(cancellerParam))
        {
            break;
        }

        {
            ResourceManager::ErrorItem* errorItem = NULL;
            {
                nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_ErrorItemLockObject);
                errorItem = m_ResourceManager->PopErrorItem();
            }
            if (errorItem != NULL)
            {
                m_Connection->EndUpdateItemCache(errorItem->GetResult(), errorItem->GetItemName());
                m_ResourceManager->DestroyErrorItem(errorItem);

                break;
            }
        }

        // すべてのアイテムを対象としている場合は、一括処理します。
        UpdateCachesForAllItems();

        ResourceManager::ResourceState itemState = ResourceManager::ResourceState_Unknown;
        const char* itemName = NULL;

        // 非フリーズアイテムを取得します。
        {
            nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_UnfreezeCacheLockObject);

            if(m_IsItemInfoInfoDictionaryInvalid)
            {
                m_ResourceManager->UpdateItemInfoDictionary();
                m_IsItemInfoInfoDictionaryInvalid = false;
            }

            itemName = m_ResourceManager->GetFirstUnfreezedItemName(&itemState);
        }

        if(itemName == NULL)
        {
            break;
        }

        switch(itemState)
        {
        case ResourceManager::ResourceState_Updating:
            result = UpdateCache(itemName);
            NotifyEditItemsChanged();
            break;

        case ResourceManager::ResourceState_Removing:
            result = RemoveCache(itemName);
            NotifyEditItemsChanged();
            break;

        default:
            NN_SDK_ASSERT(false, "[sndedit] invalid RESOURCE_STATE.\n");
            break;
        }

        // メモリ不足時は、その時のメモリ使用量を記憶しておき、
        // それを下回るまで追加のキャッシュ処理を行いません。
        // ファイル数オーバーフロー時は、
        // 解消されるまで追加のキャッシュ処理を行いません。
        switch(result.GetValue())
        {
        case ResultType_OutOfMemory:
            m_MemoryUsageAtOutOfMemory = GetTotalMemoryUsage();
            break;

        case ResultType_CacheOverFlow:
            m_IsFilesOverFlow = true;
            break;

        default:
            m_MemoryUsageAtOutOfMemory = 0;
            m_IsFilesOverFlow = false;
            break;
        }

        if(result.IsFailed())
        {
            m_ErrorProvider->SetLastError(result);
            break;
        }
    }

    return result;
}

//----------------------------------------------------------
CacheState
SoundArchiveEditController::GetItemCacheState(const char* name) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(name);
    NN_SDK_ASSERT(*name != '\0');

    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditController is not started.\n");

    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_CacheLockObject);

    return m_ResourceManager->GetItemCacheState(name);
}

//----------------------------------------------------------
const ResSoundInfo*
SoundArchiveEditController::GetSoundInfo(
     const char* name,
     FilterFunc filter /*= NULL*/,
     ResDataType* itemInfoDataType /*= NULL*/) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(name);
    NN_SDK_ASSERT(*name != '\0');

    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditController is not started.\n");

    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_CacheLockObject);

    ResDataType dataType = ResDataType_Unknown;
    const void* result = m_ResourceManager->GetItemInfo(name, &dataType);

    if(itemInfoDataType != NULL)
    {
        *itemInfoDataType = static_cast<ResDataType>(dataType);
    }

    if(result == NULL || !ResItemInfoUtility::IsSoundInfoDataType(dataType))
    {
        return NULL;
    }

    if(filter != NULL && !filter(dataType))
    {
        return NULL;
    }

    return reinterpret_cast<const ResSoundInfo*>(result);
}

//----------------------------------------------------------
const ResBankInfo*
SoundArchiveEditController::GetBankInfo(const char* name) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(name);
    NN_SDK_ASSERT(*name != '\0');

    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditController is not started.\n");

    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_CacheLockObject);

    // キャッシュがあれば、それを返します。
    ResDataType dataType = ResDataType_Unknown;
    const void* result = m_ResourceManager->GetItemInfo(name, &dataType);

    if(result == NULL || dataType != ResDataType_Bank)
    {
        return NULL;
    }

    return reinterpret_cast<const ResBankInfo*>(result);
}

//----------------------------------------------------------
const void*
    SoundArchiveEditController::GetFile(
    const char* filePath,
    ResDataType fileDataType,
    const Hash32** hashCode /*= NULL*/) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditController is not started.\n");

    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_CacheLockObject);
    return m_ResourceManager->GetFile(filePath, fileDataType, hashCode);
}

//----------------------------------------------------------
void
SoundArchiveEditController::InvokeUpdateAllItemInfos() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditController is not started.\n");

    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_UnfreezeCacheLockObject);

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO("[sndedit] invoke update all itemInfos.\n");
#endif

    m_ResourceManager->SetAllItemsState(ResourceManager::ResourceState_Updating);
}

//----------------------------------------------------------
void
SoundArchiveEditController::InvokeRemoveAllItemInfos() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditController is not started.\n");

    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_UnfreezeCacheLockObject);

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO("[sndedit] invoke remove all itemInfos.\n");
#endif

    m_ResourceManager->SetAllItemsState(ResourceManager::ResourceState_Removing);
}

//----------------------------------------------------------
Result
SoundArchiveEditController::InvokeRemoveItemInfo(const char* name) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditController is not started.\n");

    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_UnfreezeCacheLockObject);

    Result result = m_ResourceManager->UnfreezeItemInfo(name, ResourceManager::ResourceState_Removing);

    if(result.IsFailed())
    {
        m_ErrorProvider->SetLastError(result);
    }

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO("[sndedit] invoke remove itemInfo '%s'. : result=%s\n", name, result.ToString());
#endif

    // リソースから見つからないということは削除できているのでTRUE扱いにする
    if (result == ResultType_NameNotFound)
    {
        ResourceManager::ErrorItem* errorItem = m_ResourceManager->CreateErrorItem(name, Result(ResultType_True));
        if (errorItem != NULL)
        {
            nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> errorItemLock(m_ErrorItemLockObject);
            m_ResourceManager->PushErrorItem(errorItem);
        }
    }

    return result;
}

//----------------------------------------------------------
Result
SoundArchiveEditController::MakeItemInfoEditable(const char* name) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(name);
    NN_SDK_ASSERT(*name != '\0');

    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditController is not started.\n");

    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_UnfreezeCacheLockObject);

    // 既存のキャッシュが見つかった場合は、キャッシュのフリーズを解除し更新可能にします。
    // 既存のキャッシュが見つからない場合は、新しくエントリーを追加します。
    if(m_ResourceManager->ContainsItemInfo(name))
    {
        m_ResourceManager->UnfreezeItemInfo(name, ResourceManager::ResourceState_Updating);
        return Result(ResultType_True);
    }

    if (!m_SoundArchive->IsAvailable())
    {
        return Result(ResultType_False);
    }

    Result result = m_ResourceManager->NewItemInfoEntry(
        name,
        m_SoundArchive->GetItemId(name));

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO("[sndedit] make itemInfo editable '%s'. : result=%s\n", name, result.ToString());
#endif

    // キャッシュ上限に達した時は、
    // キャッシュが削除されるまで追加のキャッシュ処理を行いません。
    m_IsItemInfosOverFlow = result == ResultType_CacheOverFlow;

    if(result.IsFailed())
    {
        if(m_IsItemInfosOverFlow)
        {
            ResourceManager::ErrorItem* errorItem = m_ResourceManager->CreateErrorItem(name, result);
            if (errorItem != NULL)
            {
                nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> errorItemLock(m_ErrorItemLockObject);
                m_ResourceManager->PushErrorItem(errorItem);
            }
        }

        m_ErrorProvider->SetLastError(result);
        return result;
    }

    m_IsItemInfoInfoDictionaryInvalid = true;

    return result;
}

//----------------------------------------------------------
size_t
SoundArchiveEditController::GetItemNameBufferRequiredMemorySize(int maxItemName) const NN_NOEXCEPT
{
    int itemNameCountMax =
        nn::util::align_up( maxItemName + 1, atk::detail::fnd::MemoryTraits::DefaultAlignment );

    return static_cast<size_t>(itemNameCountMax);
}

//----------------------------------------------------------
void
SoundArchiveEditController::NotifyEditItemsChanged() NN_NOEXCEPT
{
    if(m_EditItemsChangedCallback != NULL)
    {
        m_EditItemsChangedCallback(m_EditItemsChangedCallbackParam);
    }
}

//----------------------------------------------------------
void
SoundArchiveEditController::UpdateCachesForAllItems() NN_NOEXCEPT
{
    ResourceManager::ResourceState resourceState = ResourceManager::ResourceState_Unknown;

    {
        nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_UnfreezeCacheLockObject);

        resourceState = m_ResourceManager->GetAllItemsState();

        // 状態をクリアします。
        m_ResourceManager->SetAllItemsState(ResourceManager::ResourceState_Unknown);
    }

    switch(resourceState)
    {
    case ResourceManager::ResourceState_Unknown:
        break;

    case ResourceManager::ResourceState_Updating:
        UpdateAllCaches();
        break;

    case ResourceManager::ResourceState_Removing:
        RemoveAllCaches();
        break;

    default:
        NN_SDK_ASSERT(false, "[sndedit] invalid RESOURCE_STATE.\n");
        break;
    }
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateCache(const char* itemName) NN_NOEXCEPT
{
    // メモリ不足中の場合は、メモリ使用量が下がるまで、
    // 追加のキャッシュ処理を行いません。
    if(m_MemoryUsageAtOutOfMemory > 0 &&
        m_MemoryUsageAtOutOfMemory <= GetTotalMemoryUsage())
    {
        return Result(ResultType_OutOfMemory);
    }

    // ファイル数がオーバーフロー中は、解消されるまで、
    // 追加のキャッシュ処理を行いません。
    if(m_IsFilesOverFlow)
    {
        return Result(ResultType_CacheOverFlow);
    }

    Result result = Result(ResultType_Failed);

    result = m_Connection->BeginUpdateItemCache(itemName);

    if(result.IsFailed())
    {
        return result;
    }

    // アイテム情報を取得します。
    ResDataType currentItemDataType = ResDataType_Unknown;
    const void* currentItemInfo = m_ResourceManager->GetItemInfo(itemName, &currentItemDataType, true);

    ResDataType newItemDataType = ResDataType_Unknown;
    const void* newItemInfo = NULL;

    result = QueryItemInfo(&newItemInfo, itemName, &newItemDataType);

    if(result.IsFailed())
    {
        return result;
    }

    NN_SDK_ASSERT_NOT_NULL(newItemInfo);

    if(currentItemInfo != NULL)
    {
        // 未キャッシュ状態の場合は、有効値が格納されていないので NULL ポインタにします。
        if(currentItemDataType == ResDataType_Unknown)
        {
            currentItemInfo = NULL;
        }
        // データ種別が違う場合は、ファイルを削除します。
        else if(currentItemDataType != newItemDataType)
        {
            m_ResourceManager->RemoveItemFiles(itemName);
            currentItemInfo = NULL;
            currentItemDataType = ResDataType_Unknown;
        }
    }

    // ファイルを更新します。
    switch(newItemDataType)
    {
    case ResDataType_StreamSound:
        // ストリームサウンドは、再生時にファイルを読み込むためキャッシュしませんが、
        // プリフェッチストリームデータについてキャッシュを行います。
        result = UpdateStreamSoundFile(
            reinterpret_cast<const ResStreamSoundInfo*>(currentItemInfo),
            reinterpret_cast<const ResStreamSoundInfo*>(newItemInfo)
            );
        break;

    case ResDataType_WaveSound:
        result = UpdateWaveSoundFile(
            reinterpret_cast<const ResWaveSoundInfo*>(currentItemInfo),
            reinterpret_cast<const ResWaveSoundInfo*>(newItemInfo)
            );
        break;

    case ResDataType_SequenceSound:
#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
        {
            const ResSequenceSoundInfo* inf = reinterpret_cast<const ResSequenceSoundInfo*>(newItemInfo);
            NN_DETAIL_ATK_INFO("[sndedit] Update SeqInfo : Name=%s, StartOffset=%u\n", itemName, static_cast<size_t>(inf->sequenceSoundParam.startOffset));
        }
#endif

        result = UpdateSequenceSoundFile(
            reinterpret_cast<const ResSequenceSoundInfo*>(currentItemInfo),
            reinterpret_cast<const ResSequenceSoundInfo*>(newItemInfo)
            );
        break;

    case ResDataType_Bank:
        result = UpdateBankFile(
            reinterpret_cast<const ResBankInfo*>(currentItemInfo),
            reinterpret_cast<const ResBankInfo*>(newItemInfo)
            );
        break;

    default:
        NN_SDK_ASSERT(false, "[sndedit] invalid cache data type.\n");
        break;
    }

    if(result.IsSucceeded())
    {
        nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_CacheLockObject);

        // アイテム情報を登録、フリーズします。
        result = m_ResourceManager->SetItemInfo(
            itemName,
            m_SoundArchive->GetItemId(itemName),
            newItemDataType,
            newItemInfo);

        if(result.IsFailed())
        {
            NN_SDK_ASSERT(false, "unexpected error.\n");
        }
        else
        {
            m_ResourceManager->UpdateItemInfoDictionary();

            result = m_ResourceManager->FreezeItemInfo(itemName);

            if(result.IsFailed())
            {
                NN_ATK_WARNING("[sndedit] item info cache state is invalid.");
            }
            else
            {
                if(ResItemInfoUtility::IsSoundInfoDataType(newItemDataType))
                {
                    UpdatePlayingSoundParameters(itemName);
                }
            }
        }
    }

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO("[sndedit] update itemInfo '%s'. : result=%s\n", itemName, result.ToString());
#endif

    Result notifyResult = m_Connection->EndUpdateItemCache(result, itemName);

    if(result.IsSucceeded() && notifyResult.IsFailed())
    {
        return notifyResult;
    }

    return result;
} // NOLINT(impl/function_size)

//----------------------------------------------------------
Result
SoundArchiveEditController::RemoveCache(const char* itemName) NN_NOEXCEPT
{
    Result result = Result(ResultType_Failed);

    m_ResourceManager->GetItemInfoDataType(itemName, &result);

    if(result.IsFailed())
    {
        NN_ATK_WARNING("[sndedit] item info cache state is invalid.");
        return result;
    }

    result = m_Connection->BeginRemoveItemCache(itemName);

    if(result.IsFailed())
    {
        return result;
    }

    // アイテム情報キャッシュを削除すると名前にアクセスできなくなるので、
    // バッファに退避します。
    size_t itemNameLength = std::strlen(itemName) + 1;
    NN_SDK_ASSERT(itemNameLength < m_ItemNameBufferLength);
    util::Strlcpy(m_ItemNameBuffer, itemName, static_cast<int>(std::min(m_ItemNameBufferLength, itemNameLength)));

    // アイテム情報とファイルのキャッシュを削除します。
    {
        nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_CacheLockObject);

        result = m_ResourceManager->RemoveItemData(itemName);

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
        NN_DETAIL_ATK_INFO("[sndedit] remove itemInfo '%s'. : result=%s\n", m_ItemNameBuffer, result.ToString());
#endif
    }

    if(result.IsTrue())
    {
        m_IsItemInfosOverFlow = false;
        m_IsFilesOverFlow = false;
    }

    Result notifyResult = m_Connection->EndRemoveItemCache(result, m_ItemNameBuffer);

    if(result.IsSucceeded() && notifyResult.IsFailed())
    {
        return notifyResult;
    }

    return result;
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateAllCaches() NN_NOEXCEPT
{
    // ファイルのキャッシュを削除してから全更新します。
    {
        nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_CacheLockObject);
        m_ResourceManager->RemoveAllItemFiles();
    }

    m_IsItemInfosOverFlow = false;
    m_IsFilesOverFlow = false;

    // 全アイテム更新対象とします。
    CacheUpdator cacheUpdator(*this);
    m_ResourceManager->ForEachItemInfo(cacheUpdator);

    return Result(ResultType_True);
}

//----------------------------------------------------------
Result
SoundArchiveEditController::RemoveAllCaches() NN_NOEXCEPT
{
    Result result = m_Connection->BeginRemoveAllItemCaches();

    if(result.IsFailed())
    {
        return result;
    }

    // アイテム情報とファイルのキャッシュを削除します。
    {
        nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_CacheLockObject);
        m_ResourceManager->RemoveAllItemInfos();
        m_ResourceManager->RemoveAllItemFiles();
    }

    m_IsItemInfosOverFlow = false;
    m_IsFilesOverFlow = false;

    result = m_Connection->EndRemoveAllItemCaches(Result(ResultType_True));

    NotifyEditItemsChanged();

    return result;
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateWaveSoundFile(
    const ResWaveSoundInfo* currentInfo,
    const ResWaveSoundInfo* newInfo) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(newInfo);

    // bxwsd ファイルを更新します。
    const ResName* newWsdFilePath = NULL;
    {
        const ResName* currentWsdFilePath = currentInfo == NULL ?
            NULL :
            ResItemInfoUtility::GetFilePath(*currentInfo, ResDataType_WaveSound, ResDataType_WaveSoundFile);

        newWsdFilePath =
            ResItemInfoUtility::GetFilePath(*newInfo, ResDataType_WaveSound, ResDataType_WaveSoundFile);

        int32_t fileSize = newInfo->waveSoundParam.fileSize;
        Hash32 wsdFileHashCode(newInfo->waveSoundParam.hashCode.value);

        Result result = UpdateFile(
            currentWsdFilePath,
            newWsdFilePath,
            ResDataType_WaveSoundFile,
            fileSize,
            &wsdFileHashCode);

        if(result.IsFailed())
        {
            return result;
        }
    }

    // bxwar ファイルを更新します。
    const ResName* newWarFilePath = NULL;
    {
        const ResName* currentWarFilePath = currentInfo == NULL ?
            NULL :
            ResItemInfoUtility::GetFilePath(*currentInfo, ResDataType_WaveSound, ResDataType_WaveArchiveFile);

        newWarFilePath =
            ResItemInfoUtility::GetFilePath(*newInfo, ResDataType_WaveSound, ResDataType_WaveArchiveFile);

        int32_t warFileSize = newInfo->waveSoundParam.waveArchiveFileSize;
        Hash32 warFileHashCode(newInfo->waveSoundParam.waveArchiveHashCode.value);

        Result result = UpdateFile(
            currentWarFilePath,
            newWarFilePath,
            ResDataType_WaveArchiveFile,
            warFileSize,
            &warFileHashCode);

        if(result.IsFailed())
        {
            return result;
        }
    }

    return Result(ResultType_True);
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateSequenceSoundFile(
    const ResSequenceSoundInfo* currentInfo,
    const ResSequenceSoundInfo* newInfo) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(newInfo);

    // bxseq ファイルを更新します。
    const ResName* currentSeqFilePath = currentInfo == NULL ?
        NULL :
        ResItemInfoUtility::GetFilePath(*currentInfo, ResDataType_SequenceSound, ResDataType_SequenceFile);

    const ResName* newSeqFilePath =
        ResItemInfoUtility::GetFilePath(*newInfo, ResDataType_SequenceSound, ResDataType_SequenceFile);

    int32_t fileSize = newInfo->sequenceSoundParam.fileSize;
    Hash32 seqFileHashCode(newInfo->sequenceSoundParam.hashCode.value);

    Result result = UpdateFile(
        currentSeqFilePath,
        newSeqFilePath,
        ResDataType_SequenceFile,
        fileSize,
        &seqFileHashCode);

    if(result.IsFailed())
    {
        return result;
    }

    return result;
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateStreamSoundFile(
    const ResStreamSoundInfo* currentInfo,
    const ResStreamSoundInfo* newInfo) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(newInfo);

    if (newInfo->streamSoundParam.streamFileType == 2)
    {
        return Result(ResultType_True);
    }

    if (newInfo->streamSoundParam.streamPrefetchFileSize <= 0)
    {
        return Result(ResultType_True);
    }

    // bxstp ファイルを更新します。
    const ResName* currentStrmPrefetchFilePath = currentInfo == NULL ?
        NULL :
        ResItemInfoUtility::GetFilePath(*currentInfo, ResDataType_StreamSound, ResDataType_StreamPreftchSoundFile);

    const ResName* newStrmPrefetchFilePath =
        ResItemInfoUtility::GetFilePath(*newInfo, ResDataType_StreamSound, ResDataType_StreamPreftchSoundFile);

    int32_t strmPrefetchFileSize = newInfo->streamSoundParam.streamPrefetchFileSize;
    Hash32 strmPrefetchFileHashCode(newInfo->streamSoundParam.streamPrefetchHashCode.value);

    Result result = UpdateFile(
        currentStrmPrefetchFilePath,
        newStrmPrefetchFilePath,
        ResDataType_StreamPreftchSoundFile,
        strmPrefetchFileSize,
        &strmPrefetchFileHashCode);

    if(result.IsFailed())
    {
        return result;
    }

    return result;
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateBankFile(
    const ResBankInfo* currentInfo,
    const ResBankInfo* newInfo) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(newInfo);

    // bxbnk ファイルを更新します。
    const ResName* newBnkFilePath = NULL;
    {
        const ResName* currentBnkFilePath = currentInfo == NULL ?
            NULL :
            ResItemInfoUtility::GetFilePath(*currentInfo, ResDataType_BankFile);

        newBnkFilePath =
            ResItemInfoUtility::GetFilePath(*newInfo, ResDataType_BankFile);

        int32_t fileSize = newInfo->bankParam.fileSize;
        Hash32 bnkFileHashCode(newInfo->bankParam.hashCode.value);

        Result result = UpdateFile(
            currentBnkFilePath,
            newBnkFilePath,
            ResDataType_BankFile,
            fileSize,
            &bnkFileHashCode);

        if(result.IsFailed())
        {
            return result;
        }
    }

    // bxwar ファイルを更新します。
    const ResName* newWarFilePath = NULL;
    {
        const ResName* currentWarFilePath = currentInfo == NULL ?
            NULL :
            ResItemInfoUtility::GetFilePath(*currentInfo, ResDataType_WaveArchiveFile);

        newWarFilePath =
            ResItemInfoUtility::GetFilePath(*newInfo, ResDataType_WaveArchiveFile);

        int32_t warFileSize = newInfo->bankParam.waveArchiveFileSize;
        Hash32 warFileHashCode(newInfo->bankParam.waveArchiveHashCode.value);

        Result result = UpdateFile(
            currentWarFilePath,
            newWarFilePath,
            ResDataType_WaveArchiveFile,
            warFileSize,
            &warFileHashCode);

        if(result.IsFailed())
        {
            return result;
        }
    }

    return Result(ResultType_True);
}

//----------------------------------------------------------
Result
SoundArchiveEditController::QueryItemInfo(
    const void** data,
    const char* itemName,
    ResDataType* itemDataType /*= NULL*/,
    FilterFunc filter /*= NULL*/) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(data);
    NN_SDK_ASSERT_NOT_NULL(itemName);
    NN_SDK_ASSERT(*itemName != '\0');

    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditController is not started.");

    if(m_Connection == NULL || !m_Connection->IsOpened())
    {
        *data = NULL;
        return Result(ResultType_Failed);
    }

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO("[sndedit] query itemInfo '%s'.\n", itemName);
#endif

    const HioPacket* replyPacket = NULL;
    Result result = m_Connection->QueryItemInfo(replyPacket, itemName);

    if(result.IsFailed())
    {
        return result;
    }

    NN_SDK_ASSERT_NOT_NULL(replyPacket);

    const QueryItemInfoReplyPacket* queryItemInfoReply =
        reinterpret_cast<const QueryItemInfoReplyPacket*>(replyPacket);

    NN_SDK_ASSERT_NOT_NULL(queryItemInfoReply);

    if(filter != NULL && !filter(queryItemInfoReply->GetItemType()))
    {
        return Result(ResultType_InvalidDataType);
    }

    if(itemDataType != NULL)
    {
        *itemDataType = queryItemInfoReply->GetItemType();
    }

    *data = queryItemInfoReply->GetItemInfo();
    return Result(ResultType_True);
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateFile(
    const ResName* currentFilePath,
    const ResName* newFilePath,
    ResDataType fileDataType,
    size_t fileSize,
    Hash32* hashCode /*= NULL*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(newFilePath);
    NN_SDK_ASSERT(*newFilePath->GetName() != '\0');

    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditController is not started.\n");

    if(currentFilePath != NULL)
    {
        // ファイルパスが異なる場合は、古いファイルを削除します。
        if(currentFilePath->GetLength() == newFilePath->GetLength() &&
            std::strncmp(currentFilePath->GetName(), newFilePath->GetName(), newFilePath->GetLength()) != 0)
        {
            m_ResourceManager->RemoveFile(currentFilePath->GetName());
        }
    }

    // 該当キャッシュがあれば、それを返します。
    // なければ、SoundEditConnection から読み込みます。
    const Hash32* fileDataHashCode = NULL;
    const void* fileData = m_ResourceManager->GetFile(newFilePath->GetName(), fileDataType, &fileDataHashCode);

    if(fileData != NULL)
    {
        if(hashCode == NULL || *hashCode == *fileDataHashCode)
        {
            // 有効なキャッシュが見つかったら、参照カウントをインクリメントします。
            if(m_ResourceManager->IncrementFileReferenceCount(newFilePath->GetName()).IsFailed())
            {
                NN_SDK_ASSERT(false, "unexpected error : file cache state is invalid. Path=%s\n", newFilePath->GetName());
                return Result(ResultType_Failed);
            }

            return Result(ResultType_True);
        }

        // ハッシュ値が異なる場合は、古いファイルを削除して再キャッシュします。
        if(currentFilePath != NULL && currentFilePath->GetLength() > 0)
        {
            m_ResourceManager->RemoveFile(currentFilePath->GetName());
        }
    }

    // キーの重複を防ぐために、事前に削除しておきます。
    m_ResourceManager->RemoveFile(newFilePath->GetName());

    Result result = ReadAndCacheFile(newFilePath->GetName(), fileDataType, fileSize, hashCode);

    if(result.IsSucceeded())
    {
        // 辞書を更新します。
        m_ResourceManager->UpdateFileDictionary();
    }

    return result;
}

//----------------------------------------------------------
Result
SoundArchiveEditController::ReadAndCacheFile(
    const char* filePath,
    ResDataType fileDataType,
    size_t fileSize,
    Hash32* hashCode /*= NULL*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(filePath);
    NN_SDK_ASSERT(*filePath != '\0');
    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditController is not started.\n");

    atk::detail::fnd::FileStream* fileStream =
        OpenFileForCache(m_OpenFileBuffer, m_OpenFileBufferLength, filePath);

    if(fileStream == NULL)
    {
        return Result(ResultType_Failed);
    }

    if(fileSize == 0)
    {
        fileStream->Close();
        return Result(ResultType_Failed);
    }

    size_t alignedFileSize = fileSize + Alignments::FileIoBuffer;

    // ファイルキャッシュ領域を作成します。
    void* fileBuffer = NULL;
    Result cacheResult = m_ResourceManager->NewFileEntry(filePath, alignedFileSize, fileDataType, &fileBuffer, hashCode);

    if(cacheResult.IsFailed())
    {
        fileStream->Close();

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
        NN_DETAIL_ATK_INFO(
            "[sndedit] cache allocation failed. : usage=%dkb, alignedFileSize=%dbyte, result=%s.\n",
            m_ResourceManager->GetMemoryUsage() / 1024,
            alignedFileSize,
            cacheResult.ToString()
            );
#endif

        return cacheResult;
    }

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[sndedit] cache allocation succeeded. : usage=%dkb, alignedFileSize=%dbyte.\n",
        m_ResourceManager->GetMemoryUsage() / 1024,
        alignedFileSize
        );
#endif

    NN_SDK_ASSERT_NOT_NULL(fileBuffer);

    // ファイル入出力用にアライメントして、データを読み込みます。
    void* alignedFileBuffer = util::BytePtr(fileBuffer).AlignUp(Alignments::FileIoBuffer).Get();

    nn::atk::detail::fnd::FndResult result;
    fileStream->Read(alignedFileBuffer, fileSize, &result);
    fileStream->Close();

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    if(fileSize > 32)
    {
        const uint8_t* data = reinterpret_cast<const uint8_t*>(alignedFileBuffer);
        NN_DETAIL_ATK_INFO(
            "[sndedit] ReadAndCacheFile : Data=%02x%02x%02x%02x%02x%02x%02x%02x %02x%02x%02x%02x%02x%02x%02x%02x %02x%02x%02x%02x%02x%02x%02x%02x %02x%02x%02x%02x%02x%02x%02x%02x, Path=%s\n",
            data[0],data[1],data[2],data[3],
            data[4],data[5],data[6],data[7],
            data[8],data[9],data[10],data[11],
            data[12],data[13],data[14],data[15],
            data[16],data[17],data[18],data[19],
            data[20],data[21],data[22],data[23],
            data[24],data[25],data[26],data[27],
            data[28],data[29],data[30],data[31],
            filePath);
    }
#endif

    if (result.IsFailed())
    {
        m_ResourceManager->RemoveFile(filePath);
        return Result(ResultType_Failed);
    }

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
SoundArchiveEditController::UpdatePlayingSoundParameters(const char* soundName) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditController is not started.\n");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return;
    }

    ResDataType soundInfoType = ResDataType_Unknown;
    const ResSoundInfo* soundInfo = GetSoundInfo(soundName, NULL, &soundInfoType);

    if(soundInfo == NULL)
    {
        return;
    }

    UpdatePlayingSoundParameters(soundName, soundInfo, soundInfoType);
}

//----------------------------------------------------------
void
SoundArchiveEditController::UpdatePlayingSoundParameters(
    const char* soundName,
    const ResSoundInfo* soundInfo,
    ResDataType soundInfoType) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(soundInfo);

    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditController is not started.\n");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return;
    }

    SoundArchive::ItemId soundId = m_SoundArchive->GetItemId(soundName);

    if(soundId == SoundArchive::InvalidId)
    {
        return;
    }

    // 再生中サウンドのパラメータに適用します。
    for(uint32_t playerIndex = 0; playerIndex < m_SoundArchive->GetPlayerCount(); ++playerIndex)
    {
        atk::SoundArchive::ItemId playerId = m_SoundArchive->GetPlayerIdFromIndex(playerIndex);

        m_SoundArchivePlayer->GetSoundPlayer(playerId).ForEachSound(
            ResSoundInfoApplier(soundId, soundInfo, soundInfoType));
    }
}

//----------------------------------------------------------
SoundArchiveEditController::ResSoundInfoApplier::ResSoundInfoApplier(
    SoundArchive::ItemId soundId,
    const ResSoundInfo* soundEditInfo,
    ResDataType soundInfoType) NN_NOEXCEPT :
m_SoundId(soundId),
m_SoundEditInfo(soundEditInfo),
m_SoundInfoType(soundInfoType)
{
    NN_SDK_ASSERT_NOT_NULL(soundEditInfo);
    NN_SDK_ASSERT(ResItemInfoUtility::IsSoundInfoDataType(soundInfoType));
}

//----------------------------------------------------------
void
SoundArchiveEditController::ResSoundInfoApplier::operator()(SoundHandle& soundHandle) NN_NOEXCEPT
{
    if(m_SoundId != soundHandle.GetId())
    {
        return;
    }

    atk::detail::BasicSound* sound = soundHandle.detail_GetAttachedSound();

    ApplySoundInitialVolume(sound);
    ApplyStreamSoundTrackInitialVolume(sound);
}

//----------------------------------------------------------
void
SoundArchiveEditController::ResSoundInfoApplier::ApplySoundInitialVolume(
    atk::detail::BasicSound* sound) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(sound);

    sound->SetInitialVolume(static_cast<float>(m_SoundEditInfo->soundBasicParam.volume) / 127.f);
}

//----------------------------------------------------------
void
SoundArchiveEditController::ResSoundInfoApplier::ApplyStreamSoundTrackInitialVolume(
    atk::detail::BasicSound* sound) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(sound);

    if(!ResItemInfoUtility::IsStreamSoundInfoDataType(m_SoundInfoType))
    {
        return;
    }

    atk::detail::StreamSound* streamSound = atk::detail::fnd::DynamicCast<atk::detail::StreamSound*>(sound);

    if(streamSound == NULL)
    {
        return;
    }

    const ResStreamSoundInfo* streamSoundEditInfo =
        reinterpret_cast<const ResStreamSoundInfo*>(m_SoundEditInfo);

    uint32_t trackFlags = 1;

    for(uint32_t index = 0; index < SoundArchive::StreamTrackCount; ++index)
    {
        streamSound->SetTrackInitialVolume(
            trackFlags,
            streamSoundEditInfo->streamSoundParam.trackParams[index].volume);

        trackFlags <<= 1;
    }
}

} // namespace nn::atk::viewer::detail
} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
