﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SoundSystem.h>
#include <nn/atk/atk_AudioRendererPerformanceReader.h>

namespace nn {
namespace atk {

// ----------------------------------------------------------------------------------
NN_DEFINE_STATIC_CONSTANT( const int AudioRendererPerformanceReader::PerformanceInfoCountMin );
AudioRendererPerformanceReader::AudioRendererPerformanceReader() NN_NOEXCEPT
: m_pPerformanceInfo(nullptr)
, m_PerformanceInfoCount(0)
, m_WriteIndex(0)
, m_ReadIndex(0)
, m_IsInitialized(false)
{
}

// ----------------------------------------------------------------------------------
size_t AudioRendererPerformanceReader::GetRequiredMemorySize(int performanceInfoCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( nn::atk::SoundSystem::IsInitialized() );
    NN_SDK_REQUIRES_GREATER_EQUAL( performanceInfoCount, PerformanceInfoCountMin );

    size_t size = 0;
    size += sizeof(PerformanceInfo) * performanceInfoCount;
    size += nn::atk::SoundSystem::GetPerformanceFrameBufferSize() * performanceInfoCount;

    return size;
}

// ----------------------------------------------------------------------------------
void AudioRendererPerformanceReader::Initialize(int performanceInfoCount, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( nn::atk::SoundSystem::IsInitialized() );
    NN_SDK_REQUIRES_GREATER_EQUAL( performanceInfoCount, PerformanceInfoCountMin );
    NN_SDK_REQUIRES_NOT_NULL( buffer );

    NN_SDK_REQUIRES_GREATER_EQUAL( bufferSize, GetRequiredMemorySize(performanceInfoCount) );
    NN_UNUSED( bufferSize );

    m_PerformanceInfoCount = performanceInfoCount;

    nn::util::BytePtr ptr( buffer );
    m_pPerformanceInfo = ptr.Get<PerformanceInfo>();
    ptr.Advance( sizeof(PerformanceInfo) * m_PerformanceInfoCount );

    const size_t performanceBufferSize = nn::atk::SoundSystem::GetPerformanceFrameBufferSize();
    for( int i = 0; i < m_PerformanceInfoCount; ++i )
    {
        m_pPerformanceInfo[i].performanceBuffer = ptr.Get();
        m_pPerformanceInfo[i].performanceBufferSize = performanceBufferSize;
        ptr.Advance( performanceBufferSize );
    }

    m_WriteIndex = 0;
    m_ReadIndex = m_PerformanceInfoCount - 1;
    m_IsInitialized = true;
}

// ----------------------------------------------------------------------------------
const AudioRendererPerformanceReader::PerformanceInfo* AudioRendererPerformanceReader::ReadPerformanceInfo() NN_NOEXCEPT
{
    int nextReadIndex = m_ReadIndex + 1;
    if( nextReadIndex >= m_PerformanceInfoCount )
    {
        nextReadIndex = 0;
    }

    if( nextReadIndex == m_WriteIndex )
    {
        return nullptr;
    }

    const int readIndex = nextReadIndex;
    NN_SDK_ASSERT_NOT_NULL( m_pPerformanceInfo );
    NN_SDK_ASSERT_NOT_NULL( m_pPerformanceInfo[readIndex].performanceBuffer );

    m_ReadIndex = readIndex;
    return &m_pPerformanceInfo[readIndex];
}

// ----------------------------------------------------------------------------------
void AudioRendererPerformanceReader::Record(const void* performanceFrameBuffer, size_t performanceFrameBufferSize, nn::os::Tick tick) NN_NOEXCEPT
{
    if( m_WriteIndex != m_ReadIndex )
    {
        const int writeIndex = m_WriteIndex;
        NN_SDK_ASSERT_NOT_NULL( m_pPerformanceInfo );
        NN_SDK_ASSERT_NOT_NULL( m_pPerformanceInfo[writeIndex].performanceBuffer );
        NN_SDK_ASSERT_LESS_EQUAL( performanceFrameBufferSize, m_pPerformanceInfo[writeIndex].performanceBufferSize );

        m_pPerformanceInfo[writeIndex].tick = tick;
        memcpy( m_pPerformanceInfo[writeIndex].performanceBuffer, performanceFrameBuffer, performanceFrameBufferSize );

        int nextWriteIndex = writeIndex + 1;
        if( nextWriteIndex >= m_PerformanceInfoCount )
        {
            nextWriteIndex = 0;
        }
        m_WriteIndex = nextWriteIndex;
    }
}

} // namespace nn::atk
} // namespace nn

