﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/atk/atk_SubMix.h>
#include <nn/atk/atk_FinalMix.h>
#include <nn/atk/atk_HardwareManager.h>
#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>

#include <nn/atk/fnd/basis/atkfnd_WorkBufferAllocator.h>


namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT( const int SubMix::SubMixAlignment );

SubMix::SubMixParam::SubMixParam() NN_NOEXCEPT
    : m_SrcBusCount(1)
    , m_SrcChannelCount(1)
    , m_DstBusCount(1)
    , m_DstChannelCount(1)
    , m_pOutputReceiver(nullptr)
    , m_IsEffectEnabled(true)
{
    for(int i = 0; i < OutputReceiver::BusCountMax; ++i)
    {
        m_IsSoundSendClampEnabledArray[i] = true;
    }
}

void SubMix::SubMixParam::SetSrcBusCount(int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX( count, 1, OutputReceiver::BusCountMax );
    m_SrcBusCount = count;
}

int SubMix::SubMixParam::GetSrcBusCount() const NN_NOEXCEPT
{
    return m_SrcBusCount;
}

void SubMix::SubMixParam::SetSrcChannelCount(int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX( count, 1, nn::audio::MixBufferCountMax );
    m_SrcChannelCount = count;
}

int SubMix::SubMixParam::GetSrcChannelCount() const NN_NOEXCEPT
{
    return m_SrcChannelCount;
}

void SubMix::SubMixParam::SetDstBusCount(int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX( count, 1, OutputReceiver::BusCountMax );
    m_DstBusCount = count;
}

int SubMix::SubMixParam::GetDstBusCount() const NN_NOEXCEPT
{
    return m_DstBusCount;
}

void SubMix::SubMixParam::SetDstChannelCount(int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX( count, 1, nn::audio::MixBufferCountMax );
    m_DstChannelCount = count;
}

int SubMix::SubMixParam::GetDstChannelCount() const NN_NOEXCEPT
{
    return m_DstChannelCount;
}

void SubMix::SubMixParam::SetOutputReceiver(const OutputReceiver* pOutputReceiver) NN_NOEXCEPT
{
    m_pOutputReceiver = pOutputReceiver;
}

const OutputReceiver* SubMix::SubMixParam::GetOutputReceiver() const NN_NOEXCEPT
{
    return m_pOutputReceiver;
}

void SubMix::SubMixParam::SetEffectEnabled(bool isEnabled) NN_NOEXCEPT
{
    m_IsEffectEnabled = isEnabled;
}

bool SubMix::SubMixParam::IsEffectEnabled() const NN_NOEXCEPT
{
    return m_IsEffectEnabled;
}

void SubMix::SubMixParam::SetSoundSendClampEnabled(int bus, bool isEnabled) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE( bus, 0, OutputReceiver::BusCountMax );
    m_IsSoundSendClampEnabledArray[bus] = isEnabled;
}

bool SubMix::SubMixParam::IsSoundSendClampEnabled(int bus) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE( bus, 0, OutputReceiver::BusCountMax );
    return m_IsSoundSendClampEnabledArray[bus];
}

SubMix::VolumeData::VolumeData() NN_NOEXCEPT
    : m_IsMute( false )
    , m_IsPrevMute( false )
    , m_IsDirtyFlag( false )
{
    m_Volume.InitValue( 1.0f );
}
bool SubMix::VolumeData::Update() NN_NOEXCEPT
{
    bool result = m_IsDirtyFlag;
    m_IsDirtyFlag = false;

    if( m_Volume.IsFinished() )
    {
        if( m_IsMute != m_IsPrevMute )
        {
            m_IsPrevMute = m_IsMute;
            result = true;
        }
    }
    else
    {
        m_Volume.Update();
        result = true;
    }
    return result;
}


SubMix::SubMix() NN_NOEXCEPT
    : m_ReferenceCount( 0 )
    , m_pReceiver( nullptr )
    , m_IsInitialized( false )
    , m_IsUnusedEffectChannelMuted( false )
    , m_IsAppliedOutputReceiver( false )
{
}
size_t SubMix::GetRequiredMemorySize(int srcBusCount, int srcChannelCount, int dstBusCount, int dstChannelCount) NN_NOEXCEPT
{
    return GetRequiredMemorySize( srcBusCount, srcChannelCount, dstBusCount, dstChannelCount, true, false );
}
size_t SubMix::GetRequiredMemorySize(int srcBusCount, int srcChannelCount, int dstBusCount, int dstChannelCount, bool isEffectEnabled, bool isInternalCall) NN_NOEXCEPT
{
    SubMixParam param;
    param.SetSrcBusCount( srcBusCount );
    param.SetSrcChannelCount( srcChannelCount );
    param.SetDstBusCount( dstBusCount );
    param.SetDstChannelCount( dstChannelCount );
    param.SetEffectEnabled( isEffectEnabled );
    return GetRequiredMemorySizeImpl( param, isInternalCall );
}
size_t SubMix::GetRequiredMemorySize(int srcBusCount, int srcChannelCount, const OutputReceiver* pDstReceiver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstReceiver );
    return GetRequiredMemorySize( srcBusCount, srcChannelCount, pDstReceiver, true );
}
size_t SubMix::GetRequiredMemorySize(int srcBusCount, int srcChannelCount, const OutputReceiver* pDstReceiver, bool isEffectEnabled) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstReceiver );
    SubMixParam param;
    param.SetSrcBusCount( srcBusCount );
    param.SetSrcChannelCount( srcChannelCount );
    param.SetOutputReceiver( pDstReceiver );
    param.SetEffectEnabled( isEffectEnabled );
    return GetRequiredMemorySizeImpl( param, false );
}

size_t SubMix::GetRequiredMemorySize(const SubMixParam & param) NN_NOEXCEPT
{
    return GetRequiredMemorySizeImpl( param, false );
}

bool SubMix::Initialize(int srcBusCount, int srcChannelCount, int dstBusCount, int dstChannelCount, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    return Initialize( srcBusCount, srcChannelCount, dstBusCount, dstChannelCount, true, false, buffer, bufferSize );
}
bool SubMix::Initialize(int srcBusCount, int srcChannelCount, int dstBusCount, int dstChannelCount, bool isEffectEnabled, bool isInternalCall, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    SubMixParam param;
    param.SetSrcBusCount( srcBusCount );
    param.SetSrcChannelCount( srcChannelCount );
    param.SetDstBusCount( dstBusCount );
    param.SetDstChannelCount( dstChannelCount );
    param.SetEffectEnabled( isEffectEnabled );

    return InitializeImpl(param, buffer, bufferSize, isInternalCall);
}
bool SubMix::Initialize(int srcBusCount, int srcChannelCount, const OutputReceiver* pDstReceiver, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstReceiver );
    return Initialize(srcBusCount, srcChannelCount, pDstReceiver->GetBusCount(), pDstReceiver->GetChannelCount(), true, false, buffer, bufferSize);
}
bool SubMix::Initialize(int srcBusCount, int srcChannelCount, const OutputReceiver* pDstReceiver, bool isEffectEnabled, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstReceiver );
    return Initialize(srcBusCount, srcChannelCount, pDstReceiver->GetBusCount(), pDstReceiver->GetChannelCount(), isEffectEnabled, false, buffer, bufferSize);
}

bool SubMix::Initialize(const SubMixParam& param, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    return InitializeImpl(param, buffer, bufferSize, false);
}

void SubMix::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsFinalizable() );

    if( m_IsInitialized )
    {
        nn::atk::detail::driver::HardwareManager::GetInstance().RemoveSubMix( this );
        m_IsInitialized = false;

        nn::atk::detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::AudioRendererConfig& config = nn::atk::detail::driver::HardwareManager::GetInstance().GetAudioRendererConfig();
        nn::audio::ReleaseSubMix( &config, &m_SubMix );
    }

    if( m_pReceiver != nullptr )
    {
        //  出力先ではなくなったため参照カウントを 1 つ減らします。
        m_pReceiver->AddReferenceCount( -1 );
    }

    m_pReceiver = nullptr;
    m_ppSendVolume = nullptr;
    m_pBusVolume = nullptr;
    m_pChannelVolume = nullptr;

    OutputMixer::Finalize();
}
bool SubMix::IsRemovable() const NN_NOEXCEPT
{
    return IsFinalizable();
}
bool SubMix::IsFinalizable() const NN_NOEXCEPT
{
    return m_ReferenceCount == 0u;
}
void SubMix::Update() NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsInitialized );

    for(int bus = 0; bus < m_BusCount; bus++)
    {
        if( m_pBusVolume[bus].Update() )
        {
            UpdateBusMixVolume( bus );
        }
    }

    for(int channel = 0; channel < m_ChannelCount; channel++)
    {
        if( m_pChannelVolume[channel].Update() )
        {
            UpdateChannelMixVolume( channel );
        }
    }

    if( m_SubMixVolume.Update() )
    {
        float volume;
        {
            detail::fnd::ScopedLock<detail::fnd::CriticalSection> volumeLock( m_VolumeLock );
            volume = m_SubMixVolume.GetVolume();
        }

        volume = nn::atk::detail::fnd::Clamp( volume, nn::audio::SubMixType::GetVolumeMin(), nn::audio::SubMixType::GetVolumeMax() );

        {
#if  defined( NN_ATK_CONFIG_ENABLE_VOICE_COMMAND )
            // FrameProcess() から呼ばれるため VoiceCommand 時はロックが必要
            nn::atk::detail::driver::HardwareManager::UpdateAudioRendererScopedLock audioLock;
#endif
            nn::audio::SetSubMixVolume( &m_SubMix, volume );
        }
    }
}
void SubMix::UpdateBusMixVolume(int bus) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pReceiver );
    NN_SDK_ASSERT_RANGE( bus, 0, m_BusCount );

    const int dstBusCount = m_pReceiver->GetBusCount();
    const int dstChannelCount = m_pReceiver->GetChannelCount();

    for(int channel = 0; channel < m_ChannelCount; channel++)
    {
        for(int dstBus = 0; dstBus < dstBusCount; dstBus++)
        {
            for(int dstChannel = 0; dstChannel < dstChannelCount; dstChannel++)
            {
                UpdateMixVolume( bus, channel, dstBus, dstChannel );
            }
        }
    }
}
void SubMix::UpdateChannelMixVolume(int channel) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pReceiver );
    NN_SDK_ASSERT_RANGE( channel, 0, m_ChannelCount );

    const int dstBusCount = m_pReceiver->GetBusCount();
    const int dstChannelCount = m_pReceiver->GetChannelCount();

    for(int bus = 0; bus < m_BusCount; bus++)
    {
        for(int dstBus = 0; dstBus < dstBusCount; dstBus++)
        {
            for(int dstChannel = 0; dstChannel < dstChannelCount; dstChannel++)
            {
                UpdateMixVolume( bus, channel, dstBus, dstChannel );
            }
        }
    }
}
void SubMix::UpdateMixVolume(int srcBus, int srcChannel, int dstBus, int dstChannel) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pReceiver );
    NN_SDK_ASSERT_RANGE( srcBus, 0, m_BusCount );
    NN_SDK_ASSERT_RANGE( srcChannel, 0, m_ChannelCount );
    NN_SDK_ASSERT_RANGE( dstBus, 0, m_pReceiver->GetBusCount() );
    NN_SDK_ASSERT_RANGE( dstChannel, 0, m_pReceiver->GetChannelCount() );

    const int srcIdx = srcBus * m_ChannelCount + srcChannel;
    const int dstIdx = dstBus * m_ReceiverChannelCountMax + dstChannel;
    float volume = 0;
    {
        detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_VolumeLock );
        volume = m_pBusVolume[srcBus].GetVolume() * m_pChannelVolume[srcChannel].GetVolume() * m_ppSendVolume[srcIdx][dstIdx];
    }
    volume = nn::atk::detail::fnd::Clamp( volume, nn::audio::SubMixType::GetVolumeMin(), nn::audio::SubMixType::GetVolumeMax() );

    {
        detail::fnd::ScopedLock<detail::fnd::CriticalSection> destinationLock( m_DestinationLock );
        if( m_IsAppliedOutputReceiver )
        {
#if  defined( NN_ATK_CONFIG_ENABLE_VOICE_COMMAND )
            // FrameProcess() から呼ばれるため VoiceCommand 時はロックが必要
            nn::atk::detail::driver::HardwareManager::UpdateAudioRendererScopedLock audioLock;
#endif
            switch( m_pReceiver->GetReceiverType() )
            {
                case ReceiverType::ReceiverType_SubMix:
                {
                    nn::audio::SetSubMixMixVolume( &m_SubMix, reinterpret_cast<SubMix*>( m_pReceiver )->GetAudioSubMixInstance(), volume, srcIdx, dstIdx );
                    break;
                }
                case ReceiverType::ReceiverType_FinalMix:
                {
                    nn::audio::SetSubMixMixVolume( &m_SubMix, reinterpret_cast<FinalMix*>( m_pReceiver )->GetAudioFinalMixInstance(), volume, srcIdx, dstIdx );
                    break;
                }
                default:
                    NN_UNEXPECTED_DEFAULT;
            }
        }
    }
}
void SubMix::SetDestination(OutputReceiver* pReceiver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_NOT_NULL( pReceiver );
    NN_SDK_ASSERT_GREATER_EQUAL( pReceiver->GetBusCount(), 1 );
    NN_SDK_ASSERT_GREATER_EQUAL( pReceiver->GetChannelCount(), 1 );

    {
        detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_DestinationLock );
        m_IsAppliedOutputReceiver = false;

        if( m_pReceiver != nullptr )
        {
            //  出力先ではなくなったため参照カウントを 1 つ減らします。
            m_pReceiver->AddReferenceCount( -1 );
        }

        m_pReceiver = pReceiver;

        //  this の出力先として設定されたため参照カウントを 1 増やします。
        m_pReceiver->AddReferenceCount( 1 );
    }

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandSubMixApplyDestination* command =
        cmdmgr.AllocCommand<detail::DriverCommandSubMixApplyDestination>();
    command->id = detail::DriverCommandId_SubMixApplyDestination;
    command->pReceiver = this;
    cmdmgr.PushCommand(command);

    //  DriverCommand に参照されるためカウンタを 1 増やします。
    AddReferenceCount( 1 );
}
void SubMix::ApplyDestination() NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsInitialized );
    NN_SDK_ASSERT_NOT_NULL( m_pReceiver );
    nn::audio::AudioRendererConfig& config = nn::atk::detail::driver::HardwareManager::GetInstance().GetAudioRendererConfig();

    {
        detail::fnd::ScopedLock<detail::fnd::CriticalSection> destinationLock( m_DestinationLock );
#if  defined( NN_ATK_CONFIG_ENABLE_VOICE_COMMAND )
        // FrameProcess() から呼ばれるため VoiceCommand 時はロックが必要
        nn::atk::detail::driver::HardwareManager::UpdateAudioRendererScopedLock audioLock;
#endif
        switch( m_pReceiver->GetReceiverType() )
        {
            case ReceiverType::ReceiverType_SubMix:
            {
                nn::audio::SetSubMixDestination( &config, &m_SubMix, reinterpret_cast<SubMix*>( m_pReceiver )->GetAudioSubMixInstance() );
                break;
            }
            case ReceiverType::ReceiverType_FinalMix:
            {
                nn::audio::SetSubMixDestination( &config, &m_SubMix, reinterpret_cast<FinalMix*>( m_pReceiver )->GetAudioFinalMixInstance() );
                break;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
        }
        m_IsAppliedOutputReceiver = true;
    }

    for(int bus = 0; bus < m_BusCount; bus++)
    {
        UpdateBusMixVolume( bus );
    }
}
float SubMix::GetSend(int srcBus, int dstBus) const NN_NOEXCEPT
{
    return GetSendImpl(srcBus, 0, dstBus, 0);
}
void SubMix::SetSend(int srcBus, int dstBus, float send) NN_NOEXCEPT
{
    // TODO: センド量のコードに関して、旧 SetSendVolume の設定に当たる機能を利用しないように内部実装の変更を行う
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_RANGE( srcBus, 0, m_BusCount );
    NN_SDK_REQUIRES_NOT_NULL( m_pReceiver );
    NN_SDK_REQUIRES_RANGE( dstBus, 0, m_pReceiver->GetBusCount() );

    const int dstChannelCount = m_pReceiver->GetChannelCount();
    for(int srcChannel = 0; srcChannel < m_ChannelCount; srcChannel++)
    {
        for(int dstChannel = 0; dstChannel < dstChannelCount; dstChannel++)
        {
            if( srcChannel == dstChannel )
            {
                SetSendImpl( srcBus, srcChannel, dstBus, dstChannel, send );
            }
            else
            {
                SetSendImpl( srcBus, srcChannel, dstBus, dstChannel, 0.0f );
            }
        }
    }
}
void SubMix::SetSendImpl(int srcBus, int srcChannel, int dstBus, int dstChannel, float volume) NN_NOEXCEPT
{
    // TODO: センド量のコードに関して、旧 SetSendVolume の設定に当たる機能を利用しないように内部実装の変更を行う
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_RANGE( srcBus, 0, m_BusCount );
    NN_SDK_REQUIRES_RANGE( srcChannel, 0, m_ChannelCount );
    NN_SDK_REQUIRES_NOT_NULL( m_pReceiver );
    NN_SDK_REQUIRES_RANGE( dstBus, 0, m_pReceiver->GetBusCount() );
    NN_SDK_REQUIRES_RANGE( dstChannel, 0, m_pReceiver->GetChannelCount() );

    {
        detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_VolumeLock );
        m_ppSendVolume[srcBus * m_ChannelCount + srcChannel][ dstBus * m_ReceiverChannelCountMax + dstChannel] = volume;
    }

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandSubMixUpdateMixVolume* command =
        cmdmgr.AllocCommand<detail::DriverCommandSubMixUpdateMixVolume>();
    command->id = detail::DriverCommandId_SubMixUpdateMixVolume;
    command->pReceiver = this;
    command->srcBus = srcBus;
    command->srcChannel = srcChannel;
    command->dstBus = dstBus;
    command->dstChannel = dstChannel;
    cmdmgr.PushCommand(command);

    //  DriverCommand に参照されるためカウンタを 1 増やします
    AddReferenceCount( 1 );
}

float SubMix::GetSendImpl(int srcBus, int srcChannel, int dstBus, int dstChannel) const NN_NOEXCEPT
{
    // TODO: センド量のコードに関して、旧 GetSendVolume の設定に当たる機能を利用しないように内部実装の変更を行う
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_RANGE( srcBus, 0, m_BusCount );
    NN_SDK_REQUIRES_RANGE( srcChannel, 0, m_ChannelCount );
    NN_SDK_REQUIRES_NOT_NULL( m_pReceiver );
    NN_SDK_REQUIRES_RANGE( dstBus, 0, m_pReceiver->GetBusCount() );
    NN_SDK_REQUIRES_RANGE( dstChannel, 0, m_pReceiver->GetChannelCount() );

    return m_ppSendVolume[srcBus * m_ChannelCount + srcChannel][ dstBus * m_ReceiverChannelCountMax + dstChannel];
}

void SubMix::SetBusVolume(int bus, float volume, int fadeFrame) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_RANGE( bus, 0, m_BusCount );
    NN_SDK_REQUIRES_GREATER_EQUAL( fadeFrame, 0 );

    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_VolumeLock );
    m_pBusVolume[bus].SetVolume( volume, fadeFrame );
}
float SubMix::GetBusVolume(int bus) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_RANGE( bus, 0, m_BusCount );
    return m_pBusVolume[bus].GetVolume();
}
void SubMix::SetBusMute(int bus, bool isMute) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_RANGE( bus, 0, m_BusCount );

    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_VolumeLock );
    m_pBusVolume[bus].SetMute( isMute );
}
bool SubMix::IsBusMuted(int bus) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_RANGE( bus, 0, m_BusCount );
    return m_pBusVolume[bus].IsMute();
}
bool SubMix::IsSoundSendClampEnabled(int bus) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_RANGE( bus, 0, m_BusCount );
    return m_pIsSoundSendClampEnabledArray[bus];
}
size_t SubMix::GetRequiredMemorySizeImpl(const SubMixParam& param, bool isInternalCall) NN_NOEXCEPT
{
    NN_UNUSED( isInternalCall );
    NN_SDK_REQUIRES( SoundSystem::IsInitialized() || isInternalCall );
    size_t result = 0;
    const int SrcBusCount = param.GetSrcBusCount();
    const int SrcChannelCount = param.GetSrcChannelCount();
    const int DstBusCount = param.GetOutputReceiver() == nullptr ? param.GetDstBusCount() : param.GetOutputReceiver()->GetBusCount();
    const int DstChannelCount = param.GetOutputReceiver() == nullptr ? param.GetDstChannelCount() : param.GetOutputReceiver()->GetChannelCount();

    NN_SDK_REQUIRES_MINMAX( SrcBusCount, 1, SoundSystem::GetSoundInstanceConfig().busCount );
    NN_SDK_REQUIRES_MINMAX( DstBusCount, 1, SoundSystem::GetSoundInstanceConfig().busCount );

    //  OutputMixer のバッファ
    result += OutputMixer::GetRequiredMemorySize( SrcBusCount, param.IsEffectEnabled() );

    //  m_ppSendVolume のバッファ
    result = nn::util::align_up(result, NN_ALIGNOF(float*));
    result += SrcChannelCount * SrcBusCount * sizeof(float*);
    result += SrcChannelCount * SrcBusCount * DstChannelCount * DstBusCount * sizeof(float);

    //  IsSoundSendClampDisabled のバッファ
    result = nn::util::align_up(result, NN_ALIGNOF(bool));
    result += sizeof(bool) * SrcBusCount;

    //  BusVolume のバッファ
    // (直前の IsSoundSendClampDisabled のバッファがアラインに沿わないバイト数を要求する場合があるため、
    //  BusVolume のアラインを揃えるためのバッファサイズも含めます。)
    result = nn::util::align_up(result, NN_ALIGNOF(VolumeData));
    result += sizeof(VolumeData) * SrcBusCount;

    //  ChannelVolume のバッファ
    result = nn::util::align_up(result, NN_ALIGNOF(VolumeData));
    result += sizeof(VolumeData) * SrcChannelCount;

    return result;
}
bool SubMix::InitializeImpl(const SubMixParam & param, void* buffer, size_t bufferSize, bool isInternalCall) NN_NOEXCEPT
{
    NN_UNUSED( isInternalCall );
    NN_SDK_REQUIRES( SoundSystem::IsInitialized() || isInternalCall );
    NN_SDK_REQUIRES( !m_IsInitialized );
    const int SrcBusCount = param.GetSrcBusCount();
    const int SrcChannelCount = param.GetSrcChannelCount();
    const bool IsOutputReceiverUsed = param.GetOutputReceiver() != nullptr;
    const int DstBusCount = IsOutputReceiverUsed ? param.GetOutputReceiver()->GetBusCount() : param.GetDstBusCount();
    const int DstChannelCount = IsOutputReceiverUsed ? param.GetOutputReceiver()->GetChannelCount() : param.GetDstChannelCount();
    const bool IsEffectEnabled = param.IsEffectEnabled();
    NN_SDK_REQUIRES_MINMAX( SrcBusCount, 1, SoundSystem::GetSoundInstanceConfig().busCount );
    NN_SDK_REQUIRES_MINMAX( DstBusCount, 1, SoundSystem::GetSoundInstanceConfig().busCount );
    NN_SDK_REQUIRES_LESS_EQUAL( SrcChannelCount * SrcBusCount, nn::audio::MixBufferCountMax );
    NN_SDK_REQUIRES_LESS_EQUAL( DstChannelCount * DstBusCount, nn::audio::MixBufferCountMax );
    NN_SDK_REQUIRES_NOT_NULL( buffer );
    NN_UNUSED( bufferSize );
    NN_SDK_REQUIRES_GREATER_EQUAL( bufferSize, GetRequiredMemorySizeImpl( param, isInternalCall ) );
    NN_SDK_REQUIRES_ALIGNED(buffer, SubMixAlignment);

    //  config の取得は Singleton 前提の実装です。
    nn::audio::AudioRendererConfig& config = nn::atk::detail::driver::HardwareManager::GetInstance().GetAudioRendererConfig();

    int sampleRate = nn::atk::detail::driver::HardwareManager::GetInstance().GetAudioRendererParameter().sampleRate;
    detail::fnd::WorkBufferAllocator allocator(buffer, bufferSize);

    m_ChannelCount = SrcChannelCount;
    m_BusCount = SrcBusCount;
    m_ReceiverChannelCountMax = DstChannelCount;
    m_IsAppliedOutputReceiver = false;

    //  OutputMixer の初期化
    const size_t requiredSize = OutputMixer::GetRequiredMemorySize( m_BusCount, IsEffectEnabled );
    OutputMixer::Initialize( m_BusCount, IsEffectEnabled, allocator.Allocate(requiredSize, OutputMixer::RequiredAlignment), requiredSize );

    //  m_ppSendVolume の割り当てと初期化
    const int mixBufferCount = m_BusCount * m_ChannelCount;
    m_ppSendVolume = allocator.Allocate<float*>(mixBufferCount);
    for(int srcIdx = 0 ; srcIdx < mixBufferCount; srcIdx++)
    {
        const int receiverMixBufferCount = m_ReceiverChannelCountMax * DstBusCount;
        m_ppSendVolume[srcIdx] = allocator.Allocate<float>(receiverMixBufferCount);

        for(int dstIdx = 0; dstIdx < receiverMixBufferCount; dstIdx++)
        {
            m_ppSendVolume[srcIdx][dstIdx] = 0.0f;
        }
    }

    const int rootingBusCount = std::min( m_BusCount, DstBusCount );
    const int rootingChannelCount = std::min( m_ChannelCount, m_ReceiverChannelCountMax );
    for(int bus = 0; bus < rootingBusCount; bus++)
    {
        for(int channel = 0; channel < rootingChannelCount; channel++)
        {
            m_ppSendVolume[bus * m_ChannelCount + channel][bus * m_ReceiverChannelCountMax + channel] = 1.0f;
        }
    }

    m_SubMixVolume.SetMute( false );
    m_SubMixVolume.SetVolume( 1.0f, 0 );

    // m_pIsSoundSendClampEnabledArray の割り当てと初期化
    m_pIsSoundSendClampEnabledArray = allocator.Allocate<bool>(m_BusCount);
    for(int i = 0; i < m_BusCount; i++)
    {
        m_pIsSoundSendClampEnabledArray[i] = param.IsSoundSendClampEnabled(i);
    }

    //  m_pBusVolume, m_pChannelVolume の割り当てと初期化
    // （直前の m_pIsSoundSendClampEnabledArray の割り当ての際に、アラインに沿わないバイト数分確保されることがあるので、
    //  VolumeData クラスのアラインメントに沿うように調整する）
    m_pBusVolume = allocator.Allocate<VolumeData>(m_BusCount);
    for(int i = 0; i < m_BusCount; i++)
    {
        new( &m_pBusVolume[i] ) VolumeData();
    }

    m_pChannelVolume = allocator.Allocate<VolumeData>(m_ChannelCount);
    for(int i = 0; i < m_ChannelCount; i++)
    {
        new( &m_pChannelVolume[i] ) VolumeData();
    }

    //  audio SubMix の確保
    bool isSuccess = false;
    {
        nn::atk::detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        isSuccess = nn::audio::AcquireSubMix( &config, &m_SubMix, sampleRate, mixBufferCount );
    }

    m_IsInitialized = isSuccess;
    if( isSuccess )
    {
        nn::atk::detail::driver::HardwareManager::GetInstance().AddSubMix( this );
    }

    return isSuccess;
}
void SubMix::SetChannelVolume(int channel, float volume, int fadeFrame) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_RANGE( channel, 0, m_ChannelCount );
    NN_SDK_REQUIRES_GREATER_EQUAL( fadeFrame, 0 );

    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_VolumeLock );
    m_pChannelVolume[channel].SetVolume( volume, fadeFrame );
}
float SubMix::GetChannelVolume(int channel) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_RANGE( channel, 0, m_ChannelCount );
    return m_pChannelVolume[channel].GetVolume();
}
void SubMix::SetChannelMute(int channel, bool isMute) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_RANGE( channel, 0, m_ChannelCount );

    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_VolumeLock );
    m_pChannelVolume[channel].SetMute( isMute );
}
bool SubMix::IsChannelMuted(int channel) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_RANGE( channel, 0, m_ChannelCount );
    return m_pChannelVolume[channel].IsMute();
}
void SubMix::SetSubMixVolume(float volume, int fadeFrame) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_GREATER_EQUAL( fadeFrame, 0 );

    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_VolumeLock );
    m_SubMixVolume.SetVolume( volume, fadeFrame );
}
float SubMix::GetSubMixVolume() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    return m_SubMixVolume.GetVolume();
}
void SubMix::SetSubMixMute(bool isMute) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );

    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_VolumeLock );
    m_SubMixVolume.SetMute( isMute );
}
bool SubMix::IsSubMixMuted() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    return m_SubMixVolume.IsMute();
}
void SubMix::SetMuteUnusedEffectChannel(bool isUnusedEffectChannelMuted) NN_NOEXCEPT
{
    m_IsUnusedEffectChannelMuted = isUnusedEffectChannelMuted;
}
bool SubMix::IsUnusedEffectChannelMuted() const NN_NOEXCEPT
{
    return m_IsUnusedEffectChannelMuted;
}
void SubMix::MuteUnusedEffectChannel(ChannelIndex* effectChannelIndex, int effectChannelCount, int bus) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsInitialized );
    NN_SDK_ASSERT_NOT_NULL( effectChannelIndex );
    NN_SDK_ASSERT_RANGE( bus, 0, m_BusCount );

    const int dstBusCount = m_pReceiver->GetBusCount();
    const int dstChannelCount = m_pReceiver->GetChannelCount();
    bool isUpdated = false;

    for ( int channelIndex = 0; channelIndex < m_ChannelCount; ++channelIndex )
    {
        bool isExist = false;
        for ( auto i = 0; i < effectChannelCount; ++i )
        {
            if ( channelIndex == effectChannelIndex[i] )
            {
                isExist = true;
            }
        }

        // 有効なインデックスに含まれていなかったらミュート
        if ( !isExist )
        {
            detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_VolumeLock );
            float* pSendVolume = m_ppSendVolume[ bus * m_ChannelCount + channelIndex ];

            for(int dstBus = 0; dstBus < dstBusCount; dstBus++)
            {
                for(int dstChannel = 0; dstChannel < dstChannelCount; dstChannel++)
                {
                    pSendVolume[ dstBus * m_ReceiverChannelCountMax + dstChannel ] = 0;
                }
            }

            isUpdated = true;
        }
    }

    if( isUpdated )
    {
        UpdateBusMixVolume( bus );
    }
}
bool SubMix::AppendEffect(EffectBase* pEffect, int bus, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsEffectEnabled() );
    return OutputMixer::AppendEffect( pEffect, bus, buffer, bufferSize );
}
bool SubMix::AppendEffect(EffectAux* pEffect, int bus, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsEffectEnabled() );
    return OutputMixer::AppendEffect( pEffect, bus, buffer, bufferSize );
}
bool SubMix::RemoveEffect(EffectBase* pEffect, int bus) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsEffectEnabled() );
    return OutputMixer::RemoveEffect( pEffect, bus );
}
bool SubMix::RemoveEffect(EffectAux* pEffect, int bus) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsEffectEnabled() );
    return OutputMixer::RemoveEffect( pEffect, bus );
}
bool SubMix::ClearEffect(int bus) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsEffectEnabled() );
    return OutputMixer::ClearEffect( bus );
}
bool SubMix::IsEffectEnabled() const NN_NOEXCEPT
{
    return OutputMixer::IsEffectEnabled();
}
void SubMix::AppendEffectImpl(EffectBase* pEffect, int bus, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsEffectEnabled() );
    OutputMixer::AppendEffectImpl( pEffect, bus, buffer, bufferSize );

    if( m_IsUnusedEffectChannelMuted )
    {
        ChannelIndex effectChannelIndices[ChannelIndex_Count];
        for ( int channelIndex = 0; channelIndex < ChannelIndex_Count; ++channelIndex )
        {
            effectChannelIndices[channelIndex] = nn::atk::ChannelIndex_Count;
        }
        pEffect->GetChannelIndex( effectChannelIndices, pEffect->GetChannelSettingCountMax() );
        MuteUnusedEffectChannel( effectChannelIndices, pEffect->GetChannelSettingCountMax(), bus );
    }
}
void SubMix::AppendEffectImpl(EffectAux* pEffect, int bus, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsEffectEnabled() );
    OutputMixer::AppendEffectImpl( pEffect, bus, buffer, bufferSize );

    if( m_IsUnusedEffectChannelMuted )
    {
        ChannelIndex effectChannelIndices[ChannelIndex_Count];
        for ( int channelIndex = 0; channelIndex < ChannelIndex_Count; ++channelIndex )
        {
            effectChannelIndices[channelIndex] = nn::atk::ChannelIndex_Count;
        }
        pEffect->GetChannelIndex( effectChannelIndices, pEffect->GetChannelSettingCountMax() );
        MuteUnusedEffectChannel( effectChannelIndices, pEffect->GetChannelSettingCountMax(), bus );
    }
}
OutputReceiver::ReceiverType SubMix::GetReceiverType() const NN_NOEXCEPT
{
    return ReceiverType::ReceiverType_SubMix;
}
int SubMix::GetChannelCount() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    return m_ChannelCount;
}
int SubMix::GetBusCount() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    return m_BusCount;
}
void SubMix::AddReferenceCount(int value) NN_NOEXCEPT
{
    m_ReferenceCount += value;
}


} // namespace nn::atk
} // namespace nn
