﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_OutputMixer.h>
#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>
#include <nn/atk/atk_HardwareManager.h>
#include <nn/atk/atk_DriverCommand.h>

#include <nn/atk/fnd/basis/atkfnd_WorkBufferAllocator.h>

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT( const int OutputMixer::RequiredAlignment );

OutputMixer::OutputMixer() NN_NOEXCEPT
    : m_pEffectList( nullptr )
    , m_pEffectAuxList( nullptr )
    , m_IsEffectEnabled( false )
{
}
size_t OutputMixer::GetRequiredMemorySize(int bus, bool isEffectEnabled) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL( bus, 0 );
    size_t result = 0;

    if( isEffectEnabled )
    {
        result += sizeof(EffectList) * bus;
        result += sizeof(EffectAuxList) * bus;
    }

    return result;
}
void OutputMixer::Initialize(int bus, bool isEffectEnabled, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL( bus, 0 );
    NN_SDK_ASSERT_GREATER_EQUAL( bufferSize, GetRequiredMemorySize( bus, isEffectEnabled ) );
    NN_UNUSED( bufferSize );
    NN_SDK_ASSERT_ALIGNED(buffer, RequiredAlignment);

    if( isEffectEnabled )
    {
        detail::fnd::WorkBufferAllocator allocator(buffer, bufferSize);
        m_pEffectList = allocator.Allocate<EffectList>(bus);
        for(int i = 0; i < bus; i++)
        {
            new( &m_pEffectList[i] ) EffectList();
        }

        m_pEffectAuxList = allocator.Allocate<EffectAuxList>(bus);
        for(int i = 0; i < bus; i++)
        {
            new( &m_pEffectAuxList[i] ) EffectAuxList();
        }
    }
    m_IsEffectEnabled = isEffectEnabled;
}
void OutputMixer::Finalize() NN_NOEXCEPT
{
    m_pEffectList = nullptr;
    m_pEffectAuxList = nullptr;
    m_IsEffectEnabled = false;
}
bool OutputMixer::HasEffect(int bus) const NN_NOEXCEPT
{
    if( !m_IsEffectEnabled )
    {
        return false;
    }

    NN_SDK_REQUIRES_RANGE( bus, 0, GetBusCount() );
    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_EffectListLock );

    if( m_pEffectList[bus].empty() && m_pEffectAuxList[bus].empty() )
    {
        return false;
    }
    else
    {
        return true;
    }
}
bool OutputMixer::AppendEffect(EffectBase* pEffect, int bus, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    // バッファの必要ないユーザーエフェクトが存在することを考慮し、 effectBuffer の nullptr チェックは行わない
    NN_SDK_ASSERT( m_IsEffectEnabled );
    NN_SDK_REQUIRES_NOT_NULL( pEffect );
    NN_SDK_REQUIRES_RANGE( bus, 0, GetBusCount() );
    NN_SDK_REQUIRES_NOT_NULL( buffer );
    NN_SDK_REQUIRES_ALIGNED( buffer, nn::audio::BufferAlignSize );
    NN_SDK_REQUIRES_GREATER_EQUAL( bufferSize, pEffect->GetRequiredMemSize() );

    int effectSampleRate;
    switch(pEffect->GetSampleRate())
    {
    case nn::atk::EffectBase::SampleRate_32000:
        effectSampleRate = 32000;
        break;
    case nn::atk::EffectBase::SampleRate_48000:
        effectSampleRate = 48000;
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    NN_SDK_REQUIRES_EQUAL( effectSampleRate, SoundSystem::GetRendererSampleRate() );

    // ドライバーコマンドの発行
    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandEffect* command =
        cmdmgr.AllocCommand<detail::DriverCommandEffect>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return false; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = detail::DriverCommandId_AppendEffect;
    command->bus = bus;
    command->effect = pEffect;
    command->effectBuffer = buffer;
    command->effectBufferSize = bufferSize;
    command->pOutputMixer = this;
    cmdmgr.PushCommand(command);

    //  Effect から参照されるようになったため 1 増やします
    AddReferenceCount( 1 );

    return true;
}
bool OutputMixer::AppendEffect(EffectAux* pEffect, int bus, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsEffectEnabled );
    NN_SDK_REQUIRES_NOT_NULL( pEffect );
    NN_SDK_REQUIRES_RANGE( bus, 0, GetBusCount() );
    NN_SDK_REQUIRES_NOT_NULL( buffer );
    NN_SDK_REQUIRES_ALIGNED( buffer, nn::audio::BufferAlignSize );
    NN_SDK_REQUIRES_GREATER_EQUAL( bufferSize, nn::atk::SoundSystem::GetRequiredEffectAuxBufferSize( pEffect ) );

    // ドライバーコマンドの発行
    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandEffectAux* command =
        cmdmgr.AllocCommand<detail::DriverCommandEffectAux>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return false; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = detail::DriverCommandId_AppendEffectAux;
    command->bus = bus;
    command->effect = pEffect;
    command->effectBuffer = buffer;
    command->effectBufferSize = bufferSize;
    command->pOutputMixer = this;
    cmdmgr.PushCommand(command);

    //  Effect から参照されるようになったため 1 増やします
    AddReferenceCount( 1 );

    return true;
}
bool OutputMixer::RemoveEffect(EffectBase* pEffect, int bus) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsEffectEnabled );
    NN_SDK_REQUIRES_NOT_NULL( pEffect );
    NN_SDK_REQUIRES_RANGE( bus, 0, GetBusCount() );

    // ドライバーコマンドの発行
    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandEffect* command =
        cmdmgr.AllocCommand<detail::DriverCommandEffect>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return false; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = detail::DriverCommandId_RemoveEffect;
    command->effect = pEffect;
    command->bus = bus;
    command->pOutputMixer = this;
    cmdmgr.PushCommand(command);

    uint32_t tag = cmdmgr.FlushCommand( true );
    cmdmgr.WaitCommandReply( tag );

    return true;
}
bool OutputMixer::RemoveEffect(EffectAux* pEffect, int bus) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsEffectEnabled );
    NN_SDK_REQUIRES_NOT_NULL( pEffect );
    NN_SDK_REQUIRES_RANGE( bus, 0, GetBusCount() );

    // ドライバーコマンドの発行
    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandEffectAux* command =
        cmdmgr.AllocCommand<detail::DriverCommandEffectAux>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return false; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = detail::DriverCommandId_RemoveEffectAux;
    command->effect = pEffect;
    command->bus = bus;
    command->pOutputMixer = this;
    cmdmgr.PushCommand(command);

    uint32_t tag = cmdmgr.FlushCommand( true );
    cmdmgr.WaitCommandReply( tag );

    return true;
}
bool OutputMixer::ClearEffect(int bus) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsEffectEnabled );
    NN_SDK_REQUIRES_RANGE( bus, 0, GetBusCount() );

    // ドライバーコマンドの発行
    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandEffect* command =
        cmdmgr.AllocCommand<detail::DriverCommandEffect>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return false; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = detail::DriverCommandId_ClearEffect;
    command->bus = bus;
    command->pOutputMixer = this;
    cmdmgr.PushCommand(command);

    uint32_t tag = cmdmgr.FlushCommand( true );
    cmdmgr.WaitCommandReply( tag );

    return true;
}
void OutputMixer::UpdateEffectAux() NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsEffectEnabled );
    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_EffectListLock );
    const int busCount = GetBusCount();

    for(int bus = 0; bus < busCount; bus++)
    {
        for(auto itr = m_pEffectAuxList[bus].begin(); itr != m_pEffectAuxList[bus].end(); ++itr)
        {
            itr->Update();
        }
    }
}
void OutputMixer::OnChangeOutputMode() NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsEffectEnabled );
    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_EffectListLock );
    const int busCount = GetBusCount();

    for(int bus = 0; bus < busCount; bus++)
    {
        for(auto itr = m_pEffectAuxList[bus].begin(); itr != m_pEffectAuxList[bus].end(); ++itr)
        {
            itr->OnChangeOutputMode();
        }
    }
}
void OutputMixer::AppendEffectImpl(EffectBase* pEffect, int bus, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsEffectEnabled );
    NN_SDK_ASSERT_RANGE( bus, 0, GetBusCount() );
    NN_SDK_ASSERT_NOT_NULL( pEffect );
    //NN_SDK_ASSERT_NOT_NULL( buffer );     buffer は nullptr であることがありえます。

    const auto& param = nn::atk::detail::driver::HardwareManager::GetInstance().GetAudioRendererParameter();
    if( !(param.sampleRate == 32000 && pEffect->GetSampleRate() == EffectBase::SampleRate_32000)
     && !(param.sampleRate == 48000 && pEffect->GetSampleRate() == EffectBase::SampleRate_48000))
    {
        NN_DETAIL_ATK_INFO("Effect samplerate is different from renderer samplerate\n");
        return;
    }

    pEffect->SetEffectBuffer( buffer, bufferSize );

    auto& config = nn::atk::detail::driver::HardwareManager::GetInstance().GetAudioRendererConfig();
    const bool result = pEffect->AddEffect( &config, this );
    if( result == false )
    {
        NN_DETAIL_ATK_INFO("Failed to AddEffect()\n");
        return;
    }

    const int channelCount = GetChannelCount();
    const int channelBase = bus * channelCount;
    int8_t inputOutputChannel[ChannelIndex_Count];
    for(int i = 0; i < channelCount; i++)
    {
        inputOutputChannel[i] = static_cast<int8_t>( i + channelBase );
    }
    pEffect->SetEffectInputOutput( inputOutputChannel, inputOutputChannel, channelCount, channelCount );

    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_EffectListLock );
    m_pEffectList[bus].push_back( *pEffect );
}
void OutputMixer::AppendEffectImpl(EffectAux* pEffect, int bus, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsEffectEnabled );
    NN_SDK_ASSERT_RANGE( bus, 0, GetBusCount() );
    NN_SDK_ASSERT_NOT_NULL( pEffect );
    //NN_SDK_ASSERT_NOT_NULL( buffer );     buffer は nullptr であることがありえます。

    if ( !pEffect->Initialize() )
    {
        NN_DETAIL_ATK_INFO("Failed to initialize effectAux\n");
        return;
    }

    pEffect->SetEffectBuffer( buffer, bufferSize );

    auto& config = nn::atk::detail::driver::HardwareManager::GetInstance().GetAudioRendererConfig();
    const auto& param = nn::atk::detail::driver::HardwareManager::GetInstance().GetAudioRendererParameter();
    const bool result = pEffect->AddEffect( &config, param, this );
    if( result == false )
    {
        NN_DETAIL_ATK_INFO("Failed to AddEffect()\n");
        return;
    }

    const int channelCount = GetChannelCount();
    const int channelBase = bus * channelCount;
    int8_t inputOutputChannel[ChannelIndex_Count];
    for(int i = 0; i < channelCount; i++)
    {
        inputOutputChannel[i] = static_cast<int8_t>( i + channelBase );
    }
    pEffect->SetEffectInputOutput( inputOutputChannel, inputOutputChannel, channelCount, channelCount );

    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_EffectListLock );
    m_pEffectAuxList[bus].push_back( *pEffect );
}
void OutputMixer::RemoveEffectImpl(EffectBase* pEffect, int bus) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsEffectEnabled );
    NN_SDK_ASSERT_RANGE( bus, 0, GetBusCount() );
    NN_SDK_ASSERT_NOT_NULL( pEffect );

    auto& config = nn::atk::detail::driver::HardwareManager::GetInstance().GetAudioRendererConfig();
    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_EffectListLock );
    for(auto itr = m_pEffectList[bus].begin(); itr != m_pEffectList[bus].end(); ++itr)
    {
        if( static_cast<nn::atk::EffectBase*>( &(*itr) ) == pEffect )
        {
            itr->RemoveEffect( &config, this );
            m_pEffectList[bus].erase( itr );

            //  Effect から参照されなくなったため 1 減らします
            AddReferenceCount( -1 );
            return;
        }
    }
}
void OutputMixer::RemoveEffectImpl(EffectAux* pEffect, int bus) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsEffectEnabled );
    NN_SDK_ASSERT_RANGE( bus, 0, GetBusCount() );
    NN_SDK_ASSERT_NOT_NULL( pEffect );

    auto& config = nn::atk::detail::driver::HardwareManager::GetInstance().GetAudioRendererConfig();
    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_EffectListLock );
    for(auto itr = m_pEffectAuxList[bus].begin(); itr != m_pEffectAuxList[bus].end(); ++itr)
    {
        if( static_cast<nn::atk::EffectAux*>( &(*itr) ) == pEffect )
        {
            itr->RemoveEffect( &config, this );
            itr->Finalize();
            m_pEffectAuxList[bus].erase( itr );

            //  Effect から参照されなくなったため 1 減らします
            AddReferenceCount( -1 );
            return;
        }
    }
}
void OutputMixer::ClearEffectImpl(int bus) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsEffectEnabled );
    NN_SDK_ASSERT_RANGE( bus, 0, GetBusCount() );

    auto& config = nn::atk::detail::driver::HardwareManager::GetInstance().GetAudioRendererConfig();
    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_EffectListLock );
    int count = 0;

    for(auto itr = m_pEffectList[bus].begin(); itr != m_pEffectList[bus].end(); ++itr)
    {
        itr->RemoveEffect( &config, this );
        count++;
    }
    m_pEffectList[bus].clear();

    for(auto itr = m_pEffectAuxList[bus].begin(); itr != m_pEffectAuxList[bus].end(); ++itr)
    {
        itr->RemoveEffect( &config, this );
        itr->Finalize();
        count++;
    }
    m_pEffectAuxList[bus].clear();

    //  count 分の Effect からの参照カウントを減らします
    AddReferenceCount( -count );
}


} // namespace nn::atk
} // namespace nn
