﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/fnd/string/atkfnd_String.h>

#include <algorithm>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BytePtr.h>

namespace nn {
namespace atk {
namespace detail {
namespace fnd {

NN_DEFINE_STATIC_CONSTANT( const int String::IndexNotFound );

//----------------------------------------------------------
void
String::Copy(char* dest, uint32_t destLength, const char* source, uint32_t sourceLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(dest);
    NN_SDK_ASSERT_NOT_NULL(source);
    NN_SDK_ASSERT(sourceLength <= destLength, "buffer over flow.\n");

    // コピーバッファ競合チェック
    NN_SDK_ASSERT(dest >= source || static_cast<uint32_t>(util::BytePtr(dest).Distance(source)) >= destLength);
    NN_SDK_ASSERT(dest <= source || static_cast<uint32_t>(util::ConstBytePtr(source).Distance(dest)) >= sourceLength);

    uint32_t length = std::min(destLength, sourceLength);

    for(uint32_t sourceIndex = 0; sourceIndex < length; ++sourceIndex)
    {
        dest[sourceIndex] = source[sourceIndex];
    }
}

//----------------------------------------------------------
void
String::SafeCopy(char* dest, uint32_t destLength, const char* source, uint32_t sourceLength) NN_NOEXCEPT
{
    Copy(dest, destLength, source, sourceLength);

    // 終端文字を付加する
    if(sourceLength > 0 && source[sourceLength - 1] != '\0')
    {
        NN_SDK_ASSERT(destLength > sourceLength, "buffer over flow.\n");
        dest[sourceLength] = '\0';
    }
}

//----------------------------------------------------------
int
String::IndexOf(const char* source, char targetChar, uint32_t maxLength) NN_NOEXCEPT
{
    const char* strPtr = fnd::strnchr(source, maxLength, targetChar);

    if (strPtr == NULL)
    {
        return IndexNotFound;
    }
    else
    {
        ptrdiff_t distance = strPtr - source;
        return static_cast<int>(distance);
    }
}

//----------------------------------------------------------
int
String::LastIndexOf(const char* source, char targetChar, int maxLength) NN_NOEXCEPT
{
    int index = 0;
    int lastIndex = IndexNotFound;
    const char* current = source;
    while (index != IndexNotFound)
    {
        const char* next = fnd::strnchr(current, maxLength, targetChar);
        if (next != NULL)
        {
            ptrdiff_t distance = next - source;
            index = static_cast<int>(distance);
            lastIndex = index;
        }
        else
        {
            index = IndexNotFound;
        }

        ++next;
        current = next;
    }

    return lastIndex;
}

} // namespace nn::atk::detail::fnd
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
