﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <climits>
#include <nn/atk/fnd/os/atkfnd_Thread.h>

namespace nn {
namespace atk {
namespace detail {
namespace fnd {

NN_DEFINE_STATIC_CONSTANT( const int64_t Thread::InvalidId );
NN_DEFINE_STATIC_CONSTANT( const int Thread::DefaultThreadPriority );
NN_DEFINE_STATIC_CONSTANT( const int Thread::MinThreadPriority );
NN_DEFINE_STATIC_CONSTANT( const int Thread::MaxThreadPriority );
NN_DEFINE_STATIC_CONSTANT( const int Thread::StackAlignment );
#if defined(NN_ATK_FND_CONFIG_ENABLE_THREAD_CORE_NUMBER_OBSERVATION)
NN_DEFINE_STATIC_CONSTANT( const int Thread::InvalidCoreNumber );
#endif

//---------------------------------------------------------------------------
Thread::RunArgs::RunArgs() NN_NOEXCEPT :
name(""),
stack(NULL),
stackSize(0),
idealCoreNumber(nn::os::IdealCoreDontCare),
affinityMask(Thread::AffinityMask_CoreDefault),
priority(Thread::DefaultThreadPriority),
fsPriority(Thread::FsPriority_Normal),
param(NULL),
handler(NULL)
{ }

//---------------------------------------------------------------------------
bool
Thread::RunArgs::IsValid() const NN_NOEXCEPT
{
    if(stack == NULL) { return false; }
    if(stackSize == 0) { return false; }
    if(priority > MaxThreadPriority) { return false; }
    if(handler == NULL) { return false; }
    return true;
}

//---------------------------------------------------------------------------
Thread::~Thread() NN_NOEXCEPT
{
}

//---------------------------------------------------------------------------
bool
Thread::Run(const RunArgs& args) NN_NOEXCEPT
{
    if(!args.IsValid())
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        return false;
    }

    m_Param = args.param;
    m_Handler = args.handler;

    if(!Create(m_Handle, m_Id, args))
    {
        return false;
    }

    SetName(args.name);

    if(args.affinityMask != AffinityMask_CoreDefault)
    {
        SetAffinityMask(args.idealCoreNumber, args.affinityMask);
    }

    m_Priority = args.priority;

    Resume();

    return true;
}

//---------------------------------------------------------------------------
void
Thread::WaitForExit() NN_NOEXCEPT
{
    Join();
}

//---------------------------------------------------------------------------
void
Thread::Release() NN_NOEXCEPT
{
    switch(m_State)
    {
    case State_Exited:
        break;

    case State_Running:
        NN_SDK_ASSERT(false, "must not Release() for running thread.\n");
        return;

    default:
        return;
    }

    Detach();
    SetState(State_Released);
}

//---------------------------------------------------------------------------
int32_t
Thread::GetPriority() const NN_NOEXCEPT
{
    return m_Priority;
}

//---------------------------------------------------------------------------
Thread::State
Thread::GetState() const NN_NOEXCEPT
{
    return static_cast<State>(m_State);
}

//---------------------------------------------------------------------------
void
Thread::SetState(State value) NN_NOEXCEPT
{
    m_State = value;
}

//---------------------------------------------------------------------------
void
Thread::OnRun() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_State == State_NotRun || m_State == State_Released);
    SetState(State_Running);
}

//---------------------------------------------------------------------------
void
Thread::OnExit() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_State == State_Running);
    SetState(State_Exited);
}

} // namespace nn::atk::detail::fnd
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
