﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/fnd/io/atkfnd_Stream.h>

#include <nn/util/util_BytePtr.h>
#include <nn/util/util_FormatString.h>

namespace nn {
namespace atk {
namespace detail {
namespace fnd {

namespace {
static const char NewLineCode[]      = "\r\n";
static const uint32_t  NewLineCodeLength = sizeof(char) * 2;
}

NN_DEFINE_STATIC_CONSTANT( const position_t Stream::InvalidPosition );
NN_DEFINE_STATIC_CONSTANT( const size_t Stream::InvalidSize );

//---------------------------------------------------------------------------
FndResult
StreamWriter::WriteLine(const char* text, int length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(text);
    NN_UNUSED(length);

    FndResult result(FndResultType_True);
    Write(text, static_cast<int>(std::strlen(text)), &result);

    if(result.IsFailed())
    {
        return result;
    }

    Write(NewLineCode, NewLineCodeLength, &result);

    return result;
}

//---------------------------------------------------------------------------
FndResult
StreamWriter::WriteFormat(const char* format, ...) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(format);

    std::va_list vargs;
    va_start(vargs, format);

    FndResult result = VWriteFormat(format, vargs);

    va_end(vargs);

    return result;
}

//---------------------------------------------------------------------------
FndResult
StreamWriter::VWriteFormat(const char* format, va_list vargs) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_WriteFormatBuffer);
    NN_SDK_ASSERT(m_WriteFormatBufferLength > 0);
    NN_SDK_ASSERT_NOT_NULL(format);

    int result = nn::util::VSNPrintf(
        m_WriteFormatBuffer,
        m_WriteFormatBufferLength,
        format,
        vargs);

    if(result < 0)
    {
        NN_SDK_ASSERT(0);
        return FndResult(FndResultType_Failed);
    }

    return Write(m_WriteFormatBuffer, result);
}

//---------------------------------------------------------------------------
void
StreamWriter::EnableWriteFormat(void* buffer, int length) NN_NOEXCEPT
{
    NN_SDK_ASSERT( (buffer == NULL) == (length == 0) );
    m_WriteFormatBuffer = reinterpret_cast<char*>(buffer);
    m_WriteFormatBufferLength = length;
}

//---------------------------------------------------------------------------
void
StreamWriter::DisableWriteFormat() NN_NOEXCEPT
{
    m_WriteFormatBuffer = NULL;
    m_WriteFormatBufferLength = 0;
}

//---------------------------------------------------------------------------
void
StreamAligner::Open(Stream* stream, int alignment, void* buf, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(stream);
    NN_SDK_ASSERT(alignment > 0);
    NN_SDK_ASSERT_NOT_NULL(buf);
    NN_SDK_ASSERT(length >= GetRequiredBufferLength(alignment));

    m_Stream = stream;
    m_Alignment = alignment;
    m_BufferForAlignment = util::BytePtr(buf).AlignUp(alignment).Get();
    m_BufferLengthForAlignment = length - util::BytePtr(buf).Distance(m_BufferForAlignment);
}

//---------------------------------------------------------------------------
void
StreamAligner::Close() NN_NOEXCEPT
{
    m_Stream = NULL;
    m_Alignment = 0;
    m_BufferForAlignment = NULL;
    m_BufferLengthForAlignment = 0;
}

//---------------------------------------------------------------------------
size_t
StreamAligner::Read(void* buf, size_t length, FndResult* result /*= NULL*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buf);
    NN_SDK_ASSERT(IsOpened(), "AlignedStreamReader is not opened.\n");

    void* alignedBuf = util::BytePtr(buf).AlignUp(m_Alignment).Get();

    if(alignedBuf == buf)
    {
        return m_Stream->Read(buf, length, result);
    }

    ptrdiff_t alignLength = util::BytePtr(buf).Distance(alignedBuf);
    size_t restLength = length - alignLength;

    FndResult readResult(FndResultType_True);

    // アライメント調整されていない場合は、２回に分けて読み込みます。
    if(alignLength > 0)
    {
        NN_SDK_ASSERT(alignLength < m_Alignment);

        size_t readLength = m_Stream->Read(m_BufferForAlignment, alignLength, &readResult);

        if(result != NULL)
        {
            *result = readResult;
        }

        if(readResult.IsFailed())
        {
            return 0;
        }

        if(readLength <= static_cast<size_t>(alignLength))
        {
            return readLength;
        }
    }

    // 残りを読み込みます。
    size_t restReadLength = m_Stream->Read(alignedBuf, restLength, &readResult);

    if(result != NULL)
    {
        *result = readResult;
    }

    if(readResult.IsFailed())
    {
        return 0;
    }

    return alignLength + restReadLength;
}

} // namespace nn::atk::detail::fnd
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
