﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/atk/fnd/io/atkfnd_HioStream.h>
#include <nn/atk/fnd/basis/atkfnd_Time.h>
#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>
#include <nn/ige/ige_HtcsHelper.h>


namespace {
const uint32_t ReadTimeout = 30 * 1000;   //!< 読み込みのタイムアウトです。
const int NnAtkFndInvalidSocket = -1;
const int NnAtkFndSocketError = -1;
}

namespace nn {
namespace atk {
namespace detail {
namespace fnd {

//---------------------------------------------------------------------------
HioStream::HioStream() NN_NOEXCEPT
: m_IsOpened( false )
, m_Channel(0)
, m_WorkBuffer( NULL )
, m_WorkBufferLength( 0 )
, m_ReadSize( 0 )
{
}

//---------------------------------------------------------------------------
HioStream::~HioStream() NN_NOEXCEPT
{
    this->Close();
}

//---------------------------------------------------------------------------
bool
HioStream::Open(ChannelType channel, void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);

    this->Close();

    m_RingBuffer.Init(buffer, length);

    m_Channel = channel;

    OpenImpl(channel);

    return IsOpened();
}


//---------------------------------------------------------------------------
void
HioStream::Close() NN_NOEXCEPT
{
    CloseImpl();

    m_Channel = 0;
}

//---------------------------------------------------------------------------
size_t
HioStream::Read(void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( buffer );

    atk::detail::fnd::StopWatch stopWatch;
    stopWatch.Start();
    while(GetReadableBytes() < length)
    {
        if (stopWatch.GetElapsedTime().ToMilliSeconds() >= ReadTimeout)
        {
            stopWatch.Stop();
            NN_DETAIL_ATK_INFO("HioStream::Read cannot read buffer\n");
            return 0;
        }
    }

    return m_RingBuffer.Read(buffer, length);
}

//---------------------------------------------------------------------------
size_t
HioStream::Peek( void* buffer, size_t length ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( buffer );

    atk::detail::fnd::StopWatch stopWatch;
    stopWatch.Start();
    while(GetReadableBytes() < length)
    {
        if (stopWatch.GetElapsedTime().ToMilliSeconds() >= ReadTimeout)
        {
            stopWatch.Stop();
            NN_DETAIL_ATK_INFO("HioStream::Read cannot peek buffer\n");
            return 0;
        }
    }

    return m_RingBuffer.Peek(buffer, length);
}

//---------------------------------------------------------------------------
size_t
HioStream::Skip(size_t length) NN_NOEXCEPT
{
    atk::detail::fnd::StopWatch stopWatch;
    stopWatch.Start();
    while(GetReadableBytes() < length)
    {
        if (stopWatch.GetElapsedTime().ToMilliSeconds() >= ReadTimeout)
        {
            stopWatch.Stop();
            NN_DETAIL_ATK_INFO("HioStream::Skip cannot skip buffer\n");
            return 0;
        }
    }

    return m_RingBuffer.Skip(length);
}

//---------------------------------------------------------------------------
size_t
HioStream::Write( const void* buffer, size_t length ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);

    size_t result = WriteImpl(buffer, length);

    return result;
}

//---------------------------------------------------------------------------
size_t
HioStream::GetReadableBytes() NN_NOEXCEPT
{
    if ( !IsAvailable() )
    {
        return 0;
    }

    {
        nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_ReadingSection );
        ReadBuffer();
    }

    return m_RingBuffer.GetReadableBytes();
}

//---------------------------------------------------------------------------
size_t
HioStream::GetWritableBytes() const NN_NOEXCEPT
{
    // 未実装（未使用のため）
    NN_SDK_ASSERT(false);
    if ( !IsAvailable() )
    {
        return 0;
    }

    return 0;
}

//---------------------------------------------------------------------------
void
HioStream::Initialize(void* workBuffer, size_t workBufferLength) NN_NOEXCEPT
{
    m_WorkBuffer = workBuffer;
    m_WorkBufferLength = workBufferLength;

    m_Socket = NnAtkFndInvalidSocket;
    m_ClientSocket = NnAtkFndInvalidSocket;
}

//---------------------------------------------------------------------------
void
HioStream::Finalize() NN_NOEXCEPT
{
    m_ClientSocket = NnAtkFndInvalidSocket;
    m_Socket = NnAtkFndInvalidSocket;

    m_WorkBuffer = NULL;
}

//---------------------------------------------------------------------------
void
HioStream::OpenImpl(ChannelType channel) NN_NOEXCEPT
{
    m_Socket = nn::ige::HtcsHelperSocket();

    if (m_Socket == NnAtkFndInvalidSocket)
    {
        return;
    }

    // ノンブロッキングモードに設定
    {
        int result = nn::ige::HtcsHelperFcntl(
                m_Socket, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK);
        if (result == NnAtkFndSocketError)
        {
            this->Close();
            NN_DETAIL_ATK_INFO("ioctlsocket() failed. (%d)\n", nn::ige::HtcsHelperGetLastError());
            return;
        }
    }

    // ポートにバインド
    {
        nn::htcs::SockAddrHtcs address;
        address.family = nn::htcs::HTCS_AF_HTCS;
        address.peerName = nn::ige::HtcsHelperGetPeerNameAny();
        std::strncpy(address.portName.name, channel, nn::htcs::PortNameBufferLength - 1);
        int result = nn::ige::HtcsHelperBind(m_Socket, &address);
        if (result == NnAtkFndSocketError)
        {
            this->Close();
            return;
        }
    }

    // リッスン
    {
        const int BacklogCount = 1;
        int result = nn::ige::HtcsHelperListen(m_Socket, BacklogCount);
        if (result == NnAtkFndSocketError)
        {
            this->Close();
            return;
        }
    }

    m_IsOpened = true;
}

//---------------------------------------------------------------------------
void
HioStream::CloseImpl() NN_NOEXCEPT
{
    m_IsOpened = false;

    Disconnect();

    if (m_Socket != NnAtkFndInvalidSocket)
    {
        nn::ige::HtcsHelperClose(m_Socket);
        m_Socket = NnAtkFndInvalidSocket;
    }
}

//---------------------------------------------------------------------------
size_t
HioStream::WriteImpl(const void* buffer, size_t length) NN_NOEXCEPT
{
    if(!IsAvailable())
    {
        return 0;
    }

    if(length == 0)
    {
        return 0;
    }

    nn::htcs::ssize_t sendLength = nn::ige::HtcsHelperSend(m_ClientSocket, buffer, length, 0);

    if (sendLength == NnAtkFndSocketError)
    {
        int result = nn::ige::HtcsHelperGetLastError();
        return result;
    }

    return sendLength;
}

//---------------------------------------------------------------------------
bool
HioStream::Connect() NN_NOEXCEPT
{
    if(!IsOpened())
    {
        NN_SDK_ASSERT(false, "HioStream is not opened.");
        return false;
    }

    if(IsAvailable())
    {
        //NN_DETAIL_ATK_INFO("[%d is Available] m_ClientSocket = %d\n", this->m_Channel, m_ClientSocket);
        return true;
    }

    // Accept
    {
        nn::htcs::SockAddrHtcs address;
        m_ClientSocket = nn::ige::HtcsHelperAccept(m_Socket, &address);
        //NN_DETAIL_ATK_INFO("[%d] m_ClientSocket = %d\n", this->m_Channel, m_ClientSocket);
    }

    if (m_ClientSocket == NnAtkFndInvalidSocket)
    {
        Disconnect();
        return false;
    }

    return true;
}

//---------------------------------------------------------------------------
void
HioStream::Disconnect() NN_NOEXCEPT
{
    if (m_ClientSocket != NnAtkFndInvalidSocket)
    {
        nn::ige::HtcsHelperClose(m_ClientSocket);
        m_ClientSocket = NnAtkFndInvalidSocket;
    }
}

//---------------------------------------------------------------------------
bool
HioStream::IsAvailable() const NN_NOEXCEPT
{
    return IsOpened() && m_ClientSocket != NnAtkFndInvalidSocket;
}

//---------------------------------------------------------------------------
void
HioStream::ReadBuffer() NN_NOEXCEPT
{
    if (!IsAvailable())
    {
        return;
    }

    nn::htcs::ssize_t result =
        nn::ige::HtcsHelperRecv(m_ClientSocket, m_WorkBuffer, m_WorkBufferLength, 0);

    if (result == NnAtkFndSocketError)
    {
        int errCode = nn::ige::HtcsHelperGetLastError();
        if (errCode != nn::htcs::HTCS_EWOULDBLOCK)
        {
            Disconnect();
        }

        return;
    }

    // 通信相手がコネクションを閉じると0を返すため、この場合は切断だと判断する
    if (result == 0)
    {
        Disconnect();
        return;
    }

    m_RingBuffer.Write(m_WorkBuffer, result);
}

} // namespace nn::atk::detail::fnd
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
