﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/fnd/io/atkfnd_FileStreamProxy.h>

#include <nn/atk/atk_Config.h>

namespace nn {
namespace atk {
namespace detail {
namespace fnd {

//---------------------------------------------------------------------------
FileStreamProxy::FileStreamProxy(FileStream* fileStream, position_t offset, size_t fileSize) NN_NOEXCEPT
: m_pFileStream(fileStream)
, m_Offset(offset)
, m_FileSize(fileSize)
{
}

FileStreamProxy::~FileStreamProxy() NN_NOEXCEPT
{
    m_FileSize = 0;
    m_Offset = 0;
    m_pFileStream = nullptr;
}

//---------------------------------------------------------------------------
FndResult FileStreamProxy::Open(const char* filePath, FileStream::AccessMode openMode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(false, "Should not reach here.");
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->Open(filePath, openMode);
}

//---------------------------------------------------------------------------
void FileStreamProxy::Close() NN_NOEXCEPT
{
    NN_SDK_ASSERT(false, "Should not reach here.");
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    m_pFileStream->Close();
}

//---------------------------------------------------------------------------
void FileStreamProxy::Flush() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    m_pFileStream->Flush();
}

//---------------------------------------------------------------------------
bool FileStreamProxy::IsOpened() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->IsOpened();
}

//---------------------------------------------------------------------------
bool FileStreamProxy::CanRead() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->CanRead();
}

//---------------------------------------------------------------------------
bool FileStreamProxy::CanWrite() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->CanWrite();
}

//---------------------------------------------------------------------------
bool FileStreamProxy::CanSeek() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->CanSeek();
}

//---------------------------------------------------------------------------
size_t FileStreamProxy::GetSize() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->GetSize();
}

//---------------------------------------------------------------------------
size_t FileStreamProxy::Read(void* buffer, size_t length, FndResult* result /*= NULL*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->Read(buffer, length, result);
}

//---------------------------------------------------------------------------
size_t FileStreamProxy::Write(const void* buffer, size_t length, FndResult* result /*= NULL*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->Write(buffer, length, result);
}

//---------------------------------------------------------------------------
FndResult FileStreamProxy::Seek(position_t offset, Stream::SeekOrigin origin) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    position_t fileSizeEndPosition = static_cast<position_t>(m_FileSize);

    switch( origin ) {
    case Stream::SeekOrigin::SeekOrigin_Begin:
        offset += m_Offset;
        break;
    case Stream::SeekOrigin::SeekOrigin_Current:
        offset += m_pFileStream->GetCurrentPosition();
        break;
    case Stream::SeekOrigin::SeekOrigin_End:
        offset = m_Offset + fileSizeEndPosition - offset;
        break;
    default:
        NN_SDK_ASSERT( false, "Unsupported Seek origin" );
        return FndResult(FndResultType_Failed);
    }

    if ( offset < m_Offset )
    {
        offset = m_Offset;
    }
    else if ( offset > m_Offset + fileSizeEndPosition )
    {
        offset = m_Offset + fileSizeEndPosition;
    }

    return m_pFileStream->Seek(offset, origin);
}

//---------------------------------------------------------------------------
position_t FileStreamProxy::GetCurrentPosition() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->GetCurrentPosition();
}

//---------------------------------------------------------------------------
void FileStreamProxy::EnableCache(void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    m_pFileStream->EnableCache(buffer, length);
}

//---------------------------------------------------------------------------
void FileStreamProxy::DisableCache() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    m_pFileStream->DisableCache();
}

//---------------------------------------------------------------------------
bool FileStreamProxy::IsCacheEnabled() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->IsCacheEnabled();
}

//---------------------------------------------------------------------------
int FileStreamProxy::GetIoBufferAlignment() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->GetIoBufferAlignment();
}

//---------------------------------------------------------------------------
bool FileStreamProxy::CanSetFsAccessLog() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->CanSetFsAccessLog();
}

//---------------------------------------------------------------------------
void* FileStreamProxy::SetFsAccessLog(FsAccessLog* pFsAccessLog) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->SetFsAccessLog(pFsAccessLog);
}

//---------------------------------------------------------------------------
position_t FileStreamProxy::GetCachePosition() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->GetCachePosition();
}

//---------------------------------------------------------------------------
size_t FileStreamProxy::GetCachedLength() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileStream);
    return m_pFileStream->GetCachedLength();
}

} // namespace nn::atk::detail::fnd
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
