﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/fnd/io/atkfnd_FileStreamImpl.h>

#include <nn/nn_Abort.h>

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
//#define NN_ATK_FND_STREAMIO_DEBUG
#endif

namespace nn {
namespace atk {
namespace detail {
namespace fnd {

namespace {

nn::fs::OpenMode
ConvertAccessMode(FileStream::AccessMode accessMode) NN_NOEXCEPT
{
    switch(accessMode)
    {
    case FileStream::AccessMode_Read:
        return nn::fs::OpenMode_Read;

    case FileStream::AccessMode_Write:
        return nn::fs::OpenMode_Write;

    case FileStream::AccessMode_Read | FileStream::AccessMode_Write:
        return static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write);

    case FileStream::AccessMode_AllowAppendAndWrite:
        return static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_AllowAppend | nn::fs::OpenMode_Write);

    default:
        break;
    }

    return nn::fs::OpenMode_Read;
}

//---------------------------------------------------------------------------
position_t
GetSeekPosition(FileStreamImpl& target, position_t offset, Stream::SeekOrigin origin) NN_NOEXCEPT
{
    position_t targetSizeEnd = static_cast<position_t>(target.GetSize());

    position_t result = 0;

    switch(origin)
    {
    case Stream::SeekOrigin_Begin:
        result = offset <= 0 ? 0 : offset;
        break;

    case Stream::SeekOrigin_End:
        result = offset >= 0 ? targetSizeEnd : targetSizeEnd - offset;
        break;

    case Stream::SeekOrigin_Current:
        if(offset > 0)
        {
            result = target.GetCurrentPosition() + offset;

            if(result > targetSizeEnd)
            {
                result = targetSizeEnd;
            }
        }
        else if(offset < 0)
        {
            if(target.GetCurrentPosition() > -offset)
            {
                result = target.GetCurrentPosition() - (-offset);
            }
            else
            {
                result = 0;
            }
        }
        else
        {
            result = target.GetCurrentPosition();
        }
        break;

    default:
        NN_SDK_ASSERT(false, "invalid seek origin.\n");
        break;
    }

    return result;
}

}

//---------------------------------------------------------------------------
FileStreamImpl::FileStreamImpl() NN_NOEXCEPT :
m_FileHandle(),
m_IsOpened(false),
m_CurrentPosition(0),
m_pAccessLog(nullptr)
{
    m_DirectStream.Initialize(*this);
}

//---------------------------------------------------------------------------
FndResult
FileStreamImpl::Open(const char* filePath, AccessMode accessMode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!IsOpened(), "FileStreamImpl is already opened.\n");

    m_CurrentPosition = 0;
    m_IsOpened = true;

    nn::fs::OpenMode openMode = ConvertAccessMode(accessMode);
    nn::Result result = nn::fs::OpenFile(&m_FileHandle, filePath, openMode);

    if(result.IsFailure())
    {
        if (nn::fs::ResultPathNotFound::Includes(result))
        {
            return FndResult(FndResultType_IoFileNotFound);
        }
        else if (nn::fs::ResultDataCorrupted::Includes(result))
        {
            return FndResult(FndResultType_IoInvalidAccess);
        }
        else if (nn::fs::ResultTargetLocked::Includes(result))
        {
            return FndResult(FndResultType_IoTargetLocked);
        }

        return FndResult(FndResultType_IoError);
    }

    return FndResult(FndResultType_True);
}

//---------------------------------------------------------------------------
void
FileStreamImpl::Close() NN_NOEXCEPT
{
    if(!IsOpened())
    {
        return;
    }

    m_CurrentPosition = 0;
    m_IsOpened = false;

    nn::fs::CloseFile(m_FileHandle);
}

void
FileStreamImpl::Flush() NN_NOEXCEPT
{
    if(!IsOpened())
    {
        return;
    }

    nn::fs::FlushFile(m_FileHandle);
}

//---------------------------------------------------------------------------
bool
FileStreamImpl::IsOpened() const NN_NOEXCEPT
{
    return m_IsOpened;
}

//---------------------------------------------------------------------------
size_t
FileStreamImpl::GetSize() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsOpened(), "FileStreamImpl is closed.\n");

    int64_t fileSize;
    nn::Result result = nn::fs::GetFileSize(&fileSize, m_FileHandle);
#if defined(NN_SDK_BUILD_RELEASE)
    NN_UNUSED( result );
#endif
    NN_ABORT_UNLESS( result.IsSuccess(), "FileStreamImpl::GetSize(): nn::fs::GetFileSize() is Failed");

    return static_cast<uint32_t>(fileSize);
}

//---------------------------------------------------------------------------
int
FileStreamImpl::GetIoBufferAlignment() const NN_NOEXCEPT
{
    return 1;
}

//---------------------------------------------------------------------------
size_t
FileStreamImpl::ReadDirect(void* buf, size_t length, FndResult* result /*= NULL*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buf);
    NN_SDK_ASSERT(IsOpened(), "FileStreamImpl is closed.\n");

    ValidateAlignment(buf);

    size_t readFileLength = 0;
    nn::Result nnResult = nn::fs::ReadFile(&readFileLength, m_FileHandle, m_CurrentPosition, buf, length);

    FndResult readResult(FndResultType_True);
    if(nnResult.IsFailure())
    {
        if (nn::fs::ResultOutOfRange::Includes(nnResult))
        {
            readResult = FndResult(FndResultType_IoInvalidAccess);
        }
        else
        {
            readResult = FndResult(FndResultType_IoError);
        }

        readResult = FndResult(FndResultType_IoError);
    }
    else
    {
        readResult = length == readFileLength ?
            FndResult(FndResultType_True) : FndResult(FndResultType_False);

        m_CurrentPosition += readFileLength;
    }

    if(result != NULL)
    {
        *result = readResult;
    }

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[%s][%08x] FSReadFile : curPos=%08x, buf=0x%08x, count=%d, result=%d.\n",
        __FUNCTION__, this, m_CurrentPosition, buf, length, readFileLength);
#endif

    return readFileLength;
}

//---------------------------------------------------------------------------
size_t
FileStreamImpl::WriteDirect(const void* buf, size_t length, FndResult* result /*= NULL*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buf);
    NN_SDK_ASSERT(IsOpened(), "FileStreamImpl is closed.\n");

    ValidateAlignment(buf);

    FndResult writeResult(FndResultType_True);

    nn::fs::WriteOption option;
    nn::Result nnResult = nn::fs::WriteFile(m_FileHandle, m_CurrentPosition, buf, length, option);

    if(nnResult.IsFailure())
    {
        writeResult = FndResult(FndResultType_IoError);
    }
    else
    {
        m_CurrentPosition += length;
    }

    if(result != NULL)
    {
        *result = writeResult;
    }

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[%s][%08x] FSWriteFile : curPos=%08x, buf=0x%08x, count=%d, result=%d.\n",
        __FUNCTION__, this, m_CurrentPosition, buf, length, length);
#endif

    return length;
}

//---------------------------------------------------------------------------
FndResult
FileStreamImpl::SeekDirect(position_t offset, Stream::SeekOrigin origin) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsOpened(), "FileStreamImpl is closed.\n");

    position_t seekPosition = GetSeekPosition(*this, offset, origin);

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[%s][%08x] try FSSetPosFile : curPos=0x%08x, newPos=0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, seekPosition);
#endif

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
    NN_SDK_ASSERT(
        seekPosition < GetSize(),
        "[%s][%08x] seek to out of range : curPos=0x%08x, size=0x%08x, newPos=0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, GetSize(), seekPosition);
#endif

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[%s][%08x] FSSetPosFile : curPos=%08x.\n",
        __FUNCTION__, this, m_CurrentPosition);
#endif

    m_CurrentPosition = seekPosition;

    return FndResult(FndResultType_True);
}

} // namespace nn::atk::detail::fnd
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
