﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/fnd/io/atkfnd_AiffStream.h>

#include <algorithm>
#include <nn/atk/fnd/basis/atkfnd_Endian.h>
#include <nn/atk/fnd/basis/atkfnd_Alignment.h>
#include <nn/atk/fnd/io/atkfnd_FileStream.h>
#include <nn/atk/fnd/binary/atkfnd_AiffBinary.h>

#if !defined(NN_SDK_BUILD_RELEASE)
//#define NN_ATK_FND_STREAMIO_DEBUG
#endif

namespace nn {
namespace atk {
namespace detail {
namespace fnd {

namespace {

template<typename TChunk>
bool ReadChunkBodyAiff(Stream* stream, TChunk& chunk) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(stream);

    uint32_t size = sizeof(TChunk) - sizeof(ChunkHeader);

    return stream->Read(
        util::BytePtr(&chunk, sizeof(ChunkHeader)).Get(),
        size) == size;
}

template<typename TSample>
void EndianSwapAiff(TSample* samples, int sampleCount) NN_NOEXCEPT
{
#if defined(NN_ATK_FND_IS_LITTLE_ENDIAN)
    // リトルエンディアンの場合、.aif のサンプル値（BE）をエンディアン変換する
    for( auto i = 0; i < sampleCount; ++i)
    {
        samples[i] = nn::atk::detail::fnd::endian::ByteSwap(samples[i]);
    }
#endif
}

}

//---------------------------------------------------------------------------
AiffStreamReader::AiffStreamReader() NN_NOEXCEPT :
m_Stream(NULL),
m_BlockAlignment(0),
m_DataOffset(0),
m_DataSize(0),
m_CurrentDataPosition(Stream::InvalidPosition)
{
}

AiffStreamReader::AiffStreamReader(Stream* stream) NN_NOEXCEPT :
m_Stream(NULL),
m_BlockAlignment(0),
m_DataOffset(0),
m_DataSize(0),
m_CurrentDataPosition(Stream::InvalidPosition)
{
    Open(stream);
}

//---------------------------------------------------------------------------
FndResult
AiffStreamReader::Open(Stream* stream) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(stream);
    NN_SDK_ASSERT(stream->IsOpened(), "stream must be opened.\n");

    if(!stream->CanSeek())
    {
        NN_SDK_ASSERT(false, "not seekable stream is not supported.\n");
        return FndResult(FndResultType_InvalidArgument);
    }

    m_Stream = stream;
    m_DataOffset = 0;
    m_DataSize = 0;
    m_CurrentDataPosition = Stream::InvalidPosition;

    return FndResult(FndResultType_True);
}

//---------------------------------------------------------------------------
void
AiffStreamReader::Close() NN_NOEXCEPT
{
    m_Stream = NULL;
}

//---------------------------------------------------------------------------
FndResult
AiffStreamReader::ReadHeader() NN_NOEXCEPT
{
    if(!IsOpened())
    {
        NN_SDK_ASSERT(false, "AiffStreamReader must be opened.\n");
        return FndResult(FndResultType_NotOpened);
    }

    NN_SDK_ASSERT(m_DataOffset == 0);
    NN_SDK_ASSERT(m_DataSize == 0);

    FndResult result = ReadFormChunk();

    if(!result.IsTrue())
    {
        return FndResult(FndResultType_NotSupported);
    }

    bool isCommonChunkFound = false;
    bool isSoundDataChunkFound = false;
    const size_t fileSize = m_Stream->GetSize();
    position_t currentPosition = static_cast<position_t>(sizeof(AiffFormChunkHeader));

    for ( ;; )
    {
        // CommonChunk と SoundDataChunk が見つかったら終了します。
        if(isCommonChunkFound && isSoundDataChunkFound)
        {
            break;
        }

        ChunkHeaderBE chunkHeader;
        m_Stream->Read(&chunkHeader, sizeof(ChunkHeaderBE), &result);

        if(!result.IsTrue())
        {
            return FndResult(FndResultType_NotSupported);
        }

        currentPosition += sizeof(ChunkHeaderBE);

        switch(chunkHeader.id)
        {
        case AiffCommChunk::ValidId:
            if(chunkHeader.size >= AiffCommChunk::GetValidBodySize())
            {
                AiffCommChunk commonChunk;
                if(!ReadChunkBodyAiff(m_Stream, commonChunk))
                {
                    return FndResult(FndResultType_NotSupported);
                }

                m_WaveFormat.channels = static_cast<uint8_t>(commonChunk.channelCount);
                m_WaveFormat.bitsPerSample = static_cast<uint8_t>(commonChunk.sampleSize);
                m_WaveFormat.samplesPerSec = static_cast<uint32_t>(static_cast<double>(commonChunk.sampleRate));
                m_BlockAlignment = static_cast<uint8_t>(commonChunk.channelCount) * static_cast<uint8_t>(commonChunk.sampleSize) / 8;

                isCommonChunkFound = true;

                // 想定するサイズより大きかった場合は、次のチャンクまでシークします。
                int32_t extendedSize = chunkHeader.size - AiffCommChunk::GetValidBodySize();

                if(extendedSize > 0)
                {
                    m_Stream->Seek(extendedSize, Stream::SeekOrigin_Current);
                }

                currentPosition += AiffCommChunk::GetValidBodySize();

                continue;
            }
            break;

        case AiffSoundDataChunk::ValidId:
            if(chunkHeader.size >= AiffSoundDataChunk::GetValidBodySize())
            {
                m_DataSize = chunkHeader.size - 4 - 4; // offset + blockSize 分を減算します。
                m_DataOffset = currentPosition + 4 + 4; // offset + blockSize 分を加算します。

                isSoundDataChunkFound = true;

                // 次のチャンクまでシークします。
                m_Stream->Seek(chunkHeader.size, Stream::SeekOrigin_Current);

                continue;
            }
            break;

        default:
            break;
        }

        // 次のチャンク開始位置がファイル終端なら終了します。
        if(static_cast<position_t>(fileSize) <= m_Stream->GetCurrentPosition() + chunkHeader.size)
        {
            break;
        }

        // 次のチャンクまでシークします。
        result = m_Stream->Seek(chunkHeader.size, Stream::SeekOrigin_Current);

        if(result.IsFailed())
        {
            return FndResult(FndResultType_NotSupported);
        }

        currentPosition += chunkHeader.size;
    }

    if(!isCommonChunkFound ||
        !isSoundDataChunkFound ||
        (m_DataOffset == 0 || static_cast<position_t>(fileSize) <= m_DataOffset) ||
        m_DataSize == 0)
    {
        return FndResult(FndResultType_NotSupported);
    }

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
    NN_DETAIL_ATK_INFO("[%s] Channels       : %d\n", __FUNCTION__, m_WaveFormat.channels);
    NN_DETAIL_ATK_INFO("[%s] SamplesPerSec  : %d\n", __FUNCTION__, m_WaveFormat.samplesPerSec);
    NN_DETAIL_ATK_INFO("[%s] BitsPerSample  : %d\n", __FUNCTION__, m_WaveFormat.bitsPerSample);
    NN_DETAIL_ATK_INFO("[%s] DataSize       : %d\n", __FUNCTION__, m_DataSize);
#endif

    return FndResult(FndResultType_True);
}

//---------------------------------------------------------------------------
size_t AiffStreamReader::ReadFrames(void* buffer, size_t frames, FndResult* result /*= NULL*/) NN_NOEXCEPT
{
    if(!IsOpened() || !IsHeaderRead())
    {
        NN_SDK_ASSERT(false, "WavStreamReader must be opened.\n");

        if(result != NULL)
        {
            *result = FndResult(FndResultType_NotOpened);
        }

        return 0;
    }

    // 初めて読み込む場合は、データ開始位置にシークします。
    if(m_CurrentDataPosition == Stream::InvalidPosition)
    {
        FndResult seekResult = m_Stream->Seek(m_DataOffset, Stream::SeekOrigin_Begin);

        if(seekResult.IsFailed())
        {
            if(result != NULL)
            {
                *result = seekResult;
            }

            return 0;
        }

        m_CurrentDataPosition = 0;
    }

    size_t tryReadBytes = std::min(m_DataSize - m_CurrentDataPosition, frames * m_BlockAlignment);

    if(tryReadBytes == 0)
    {
        if(result != NULL)
        {
            *result = FndResult(FndResultType_False);
        }

        return 0;
    }

    FndResult readResult(FndResultType_True);
    size_t readBytes = m_Stream->Read(buffer, tryReadBytes, &readResult);

    m_CurrentDataPosition += readBytes;

    if(result != NULL)
    {
        *result = readResult;
    }

    switch(m_WaveFormat.bitsPerSample)
    {
    case 16:
        EndianSwapAiff<int16_t>(static_cast<int16_t*>(buffer), static_cast<int>(readBytes / sizeof(int16_t)));
        break;

    case 32:
        EndianSwapAiff<int32_t>(static_cast<int32_t*>(buffer), static_cast<int>(readBytes / sizeof(int32_t)));
        break;

    default:
        break;
    }

    return readBytes / m_BlockAlignment;
}

//---------------------------------------------------------------------------
FndResult
AiffStreamReader::ReadFormChunk() NN_NOEXCEPT
{
    FndResult result(FndResultType_True);

    AiffFormChunkHeader formChunkHeader;
    m_Stream->Read(&formChunkHeader, sizeof(AiffFormChunkHeader), &result);

    if(!result.IsTrue())
    {
        return FndResult(FndResultType_NotSupported);
    }

    if(!formChunkHeader.IsValid())
    {
        return FndResult(FndResultType_NotSupported);
    }

    return result;
}

} // namespace nn::atk::detail::fnd
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
