﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/fnd/basis/atkfnd_WorkBufferSizeCalculator.h>
#include <nn/util/util_BitUtil.h>

namespace nn { namespace atk { namespace detail { namespace fnd {

void WorkBufferSizeCalculator::Add(size_t size) NN_NOEXCEPT
{
    AddImpl(size, 1);
}

void WorkBufferSizeCalculator::Add(size_t size, size_t alignment) NN_NOEXCEPT
{
    AddImpl(size, alignment);
}

void WorkBufferSizeCalculator::Add(size_t size, size_t alignment, int count) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER(count, 0);
    for(int i = 0; i < count; ++i)
    {
        AddImpl(size, alignment);
    }
}

size_t WorkBufferSizeCalculator::GetSize() const NN_NOEXCEPT
{
    return m_Size;
}

void WorkBufferSizeCalculator::AlignUp(size_t alignment) NN_NOEXCEPT
{
    const auto Increment = nn::util::align_up(m_Size, alignment) - m_Size;
    m_Size += Increment;
    m_Offset = (m_Offset + Increment) % m_Alignment;
}

void WorkBufferSizeCalculator::AddImpl(size_t size, size_t alignment) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER(alignment, 0u);
    NN_SDK_ASSERT(nn::util::ispow2(alignment));

    if(size == 0)
    {
        return;
    }

    // 初回 Add 時の処理
    if(m_Alignment == 0)
    {
        m_Alignment = alignment;
    }

    // 今までのアライメントよりも大きなアライメントに対する、バッファ要求量計算
    if(m_Alignment < alignment)
    {
        if(m_Offset == 0)
        {
            m_Size += alignment - m_Alignment;
        }
        else
        {
            m_Size += alignment - m_Offset;
        }

        m_Alignment = alignment;
        m_Size += size;
        m_Offset = size % alignment;
        return;
    }

    // 通常のバッファ要求量計算
    {
        // アライメントの調整
        if(m_Size % alignment != 0)
        {
            AlignUp(alignment);
        }

        // サイズの加算
        m_Size += size;
        m_Offset = (m_Offset + size) % m_Alignment;
    }
}

}}}}
