﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/fnd/basis/atkfnd_FrameHeapImpl.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>  // for GetOffsetFromPtr(), etc...
#include <new>

namespace nn { namespace atk { namespace detail { namespace fnd {

NN_DEFINE_STATIC_CONSTANT( const int FrameHeapImpl::FreeHeadMode );
NN_DEFINE_STATIC_CONSTANT( const int FrameHeapImpl::FreeTailMode );
NN_DEFINE_STATIC_CONSTANT( const int FrameHeapImpl::FreeAllMode );

//---------------------------------------------------------------------------
//! @brief        フレームヒープを作成します。
//!
//! @details       基本はスレッドセーフではない。
//!                スレッドセーフにする場合、ヒープの属性を指定する引数を追加するようにするか、
//!                あるいは、属性をセットする関数で制御してもらうか。
//!
//! @param[out]   startAddress  ヒープ領域の先頭アドレス。
//! @param[in]    size          ヒープ領域のサイズ。
//! @param[in]    optFlag       オプションフラグ。
//!                             MEM_HEAP_OPT_0_CLEAR    - メモリ確保時の0クリアフラグ
//!                             MEM_HEAP_OPT_DEBUG_FILL - デバッグフィルフラグ
//!
//! @return       関数が成功した場合、作成されたフレームヒープのハンドルが返ります。
//!               関数が失敗すると、MEM_INVALID_HEAP_HANDLE が返ります。
//---------------------------------------------------------------------------
FrameHeapImpl*
FrameHeapImpl::Create( void* startAddress, size_t size, uint16_t optFlag ) NN_NOEXCEPT
{
    void* endAddress;

    NN_SDK_ASSERT_NOT_NULL( startAddress );

    endAddress   = RoundDown( AddOffsetToPtr( startAddress, size ), MIN_ALIGNMENT );
    startAddress = RoundUp( startAddress, MIN_ALIGNMENT );

    // start/end がおかしかったり、FrameHeap サイズに足りない場合は NULL を返す
    if ( ComparePtr( startAddress, endAddress ) > 0
        || GetOffsetFromPtr( startAddress, endAddress ) < sizeof(FrameHeapImpl) )
    {
        return NULL;
    }

    // フレームヒープ初期化
    {
        FrameHeapImpl* pHeap = new( startAddress ) FrameHeapImpl;

        pHeap->Initialize(
            FrameHeapSignature,
            AddOffsetToPtr( pHeap, sizeof(FrameHeapImpl) ), // heapStart
            endAddress,                                 // heapEnd
            optFlag );

        pHeap->m_pHeadAllocator = pHeap->GetHeapStart();
        pHeap->m_pTailAllocator = pHeap->GetHeapEnd();
        pHeap->m_pState = NULL;

        return pHeap;
    }
}

//---------------------------------------------------------------------------
//! @brief        フレームヒープを破棄します。
//!
//! @return      破棄したヒープが占めていた領域へのポインタを返します。
//---------------------------------------------------------------------------
void*
FrameHeapImpl::Destroy() NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValid() );

    Finalize();
    return this;
}

//---------------------------------------------------------------------------
//! @brief       フレームヒープからメモリブロックを確保します。
//!              メモリブロックのアライメントを指定できます。
//!              アライメント値を負の値で指定すると、ヒープの空き領域を後方から探します。
//!
//! @param[in]   size       確保するメモリブロックのサイズ(バイト単位)。
//! @param[in]   alignment  確保するメモリブロックのアライメント。
//!                         ±4, ±8, ±16, ±32, ±64, ±128, ... の値が指定できます。
//!
//! @return      メモリブロックの確保が成功した場合、確保したメモリブロックへの
//!              ポインタが返ります。
//!              失敗した場合、NULLが返ります。
//---------------------------------------------------------------------------
void*
FrameHeapImpl::Alloc( size_t size, int alignment ) NN_NOEXCEPT
{
    void* memory = NULL;

    NN_SDK_ASSERT( IsValid() );

    // アライメントのチェック
    NN_SDK_ASSERT( alignment % MIN_ALIGNMENT == 0 );
    NN_SDK_ASSERT( (Abs(alignment) & (Abs(alignment) - 1)) == 0 );
    NN_SDK_ASSERT( MIN_ALIGNMENT <= Abs(alignment) );

    // size 補正
    if ( size == 0 )
    {
        size = 1;
    }
    size = RoundUp( size, MIN_ALIGNMENT );

    LockHeap();

    if ( alignment >= 0 )   // ヒープ先頭から確保
    {
        memory = AllocFromHead( size, alignment );
    }
    else                    // ヒープ後ろから確保
    {
        memory = AllocFromTail( size, -alignment );
    }

    UnlockHeap();

    return memory;
}

//---------------------------------------------------------------------------
//! @brief       ヒープの先頭からメモリブロックを確保します。
//!              アラインメントの指定があります。
//!
//! @param[in]   size     確保するメモリブロックのサイズ。
//! @param[in]   alignment   アライメント値。
//!
//! @return      メモリブロックの確保が成功した場合、確保したメモリブロックへの
//!              ポインタが返ります。
//!              失敗した場合、NULLが返ります。
//---------------------------------------------------------------------------
void*
FrameHeapImpl::AllocFromHead( size_t size, int alignment ) NN_NOEXCEPT
{
    void* newBlock = RoundUp( m_pHeadAllocator, static_cast<uint32_t>(alignment) );
    void* endAddress = AddOffsetToPtr( newBlock, size );

    if ( GetUintPtr( endAddress ) > GetUintPtr( m_pTailAllocator ) )
    {
        return NULL;
    }

    FillAllocMemory(
        m_pHeadAllocator,
        static_cast<uint32_t>( GetOffsetFromPtr( m_pHeadAllocator, endAddress ) )
    );

    m_pHeadAllocator = endAddress;

    return newBlock;
}

//---------------------------------------------------------------------------
//! @brief       ヒープの末尾からメモリブロックを確保します。
//!              アラインメントの指定があります。
//!
//! @param[in]   size        確保するメモリブロックのサイズ。
//! @param[in]   alignment   アライメント値。
//!
//! @return      メモリブロックの確保が成功した場合、確保したメモリブロックへの
//!              ポインタが返ります。
//!              失敗した場合、NULLが返ります。
//---------------------------------------------------------------------------
void*
FrameHeapImpl::AllocFromTail( size_t size, int alignment ) NN_NOEXCEPT
{
    void* newBlock = RoundDown(
        AddOffsetToPtr( m_pTailAllocator, -static_cast<int32_t>(size) ), static_cast<uint32_t>(alignment) );

    if ( GetUintPtr( newBlock ) < GetUintPtr( m_pHeadAllocator ) )
    {
        return NULL;
    }

    FillAllocMemory(
        newBlock,
        static_cast<uint32_t>( GetOffsetFromPtr( newBlock, m_pTailAllocator ) )
    );

    m_pTailAllocator = newBlock;

    return newBlock;
}

//---------------------------------------------------------------------------
//! @brief        フレームヒープから確保されたメモリブロックのサイズを変更します。
//!
//!               サイズを変更するメモリブロックは、ヒープの空き領域の前方から確保された
//!               末尾のメモリブロックである必要があります。
//!
//! @param[out]   memBlock  サイズを変更するメモリブロックへのポインタ。
//! @param[in]    newSize   新しく割り当てるサイズ(バイト単位)。
//!                         4未満の値を指定された場合は、4が指定されたものとして処理します。
//!
//! @return       関数が成功した場合、変更されたメモリブロックのサイズを返します(バイト単位)。
//!                関数が失敗した場合、0 が返ります。
//---------------------------------------------------------------------------
size_t
FrameHeapImpl::ResizeForMBlock( void* memBlock, size_t newSize ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValid() );

    // 最低限、最小アライメントの境界にあるかチェック
    NN_SDK_ASSERT( memBlock == RoundDown( memBlock, MIN_ALIGNMENT ) );

    // メモリブロックは前方に存在すること
    NN_SDK_ASSERT( ComparePtr( mHeapStart, memBlock ) <= 0
        && ComparePtr( m_pHeadAllocator, memBlock ) > 0 );

    // 状態保存がメモリブロックの後方に無いこと
    NN_SDK_ASSERT( m_pState == NULL || ComparePtr( m_pState, memBlock ) < 0 );

    // newSize == 0 は認めない。
    // 0 だと、memBlock が指すメモリブロックが存在しなくなるため。
    if ( newSize == 0 )
    {
        newSize = 1;
    }
    newSize = RoundUp( newSize, MIN_ALIGNMENT );

    LockHeap();

    {
        const ptrdiff_t oldSize = GetOffsetFromPtr( memBlock, m_pHeadAllocator );
        void* endAddress = AddOffsetToPtr( memBlock, newSize );

        // 同じ場合はなにもしない
        if ( static_cast<int>(newSize) != oldSize )
        {
            // 拡大するとき
            if ( newSize > static_cast<uint32_t>(oldSize) )
            {
                // サイズが足りない
                if ( ComparePtr( endAddress, m_pTailAllocator ) > 0 )
                {
                    newSize = 0;
                }
                else
                {
                    FillAllocMemory( m_pHeadAllocator, newSize - oldSize );
                }
            }
            // 縮小するとき
            else
            {
                FillFreeMemory( endAddress, oldSize - newSize );
            }
            m_pHeadAllocator = endAddress;
        }
    }

    UnlockHeap();

    return newSize;
}

//---------------------------------------------------------------------------
//! @brief       フレームヒープ内の割り当て可能な最大サイズを取得します。
//!              メモリブロックのアライメントを指定できます。
//!
//! @param[in]   alignment  確保するメモリブロックのアライメント。
//!                     ±4, ±8, ±16, ±32, ±64, ±128, ... の値が指定できます。
//!
//! @return      フレームヒープ内の割り当て可能な最大サイズを返します(バイト単位)。
//---------------------------------------------------------------------------
size_t
FrameHeapImpl::GetAllocatableSize( int alignment ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValid() );

    NN_SDK_ASSERT( alignment % MIN_ALIGNMENT == 0 );
    NN_SDK_ASSERT( (Abs( alignment ) & (Abs( alignment ) - 1)) == 0 );
    NN_SDK_ASSERT( MIN_ALIGNMENT <= Abs( alignment ) );

    alignment = Abs( alignment );

    {

        size_t retVal;
        const void* block = RoundUp( m_pHeadAllocator, static_cast<uint32_t>(alignment) );

        if ( GetUintPtr( block ) > GetUintPtr( m_pTailAllocator ) )
        {
            retVal = 0;
        }
        else
        {
            retVal = static_cast<size_t>( GetOffsetFromPtr( block, m_pTailAllocator ) );
        }

        return retVal;
    }
}

//---------------------------------------------------------------------------
//! @brief       フレームヒープへメモリブロックを返却します。
//!
//! @param[in]   mode  メモリブロックの返却方法。
//---------------------------------------------------------------------------
void
FrameHeapImpl::Free( int mode ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValid() );

    LockHeap();

    if ( mode & FreeHeadMode )
    {
        FreeHead();
    }

    if ( mode & FreeTailMode )
    {
        FreeTail();
    }

    UnlockHeap();
}

//---------------------------------------------------------------------------
//! @brief       ヒープ領域の先頭から確保したメモリブロックを一括して開放します。
//---------------------------------------------------------------------------
void
FrameHeapImpl::FreeHead() NN_NOEXCEPT
{
    NN_SDK_ASSERT( ComparePtr( m_pHeadAllocator, mHeapStart ) >= 0 );

    FillFreeMemory( mHeapStart, static_cast<uint32_t>( GetOffsetFromPtr( mHeapStart, m_pHeadAllocator ) ) );
    m_pHeadAllocator = mHeapStart;
    m_pState = NULL;
}

//---------------------------------------------------------------------------
//! @brief       ヒープ領域の末尾から確保したメモリブロックを一括して開放します。
//---------------------------------------------------------------------------
void
FrameHeapImpl::FreeTail() NN_NOEXCEPT
{
    NN_SDK_ASSERT( ComparePtr( mHeapEnd, m_pTailAllocator ) >= 0 );

    FillFreeMemory( m_pTailAllocator, static_cast<uint32_t>( GetOffsetFromPtr( m_pTailAllocator, mHeapEnd ) ) );

    // ヒープの割り当て状態の復帰により、解放したメモリブロックが復活してしまわ
    // ないように、保存状態の交尾割り当てポインタを再設定しておく
    for ( HeapState* pState = m_pState; pState; pState = pState->pPrevState )
    {
        pState->tailAllocator = mHeapEnd;
    }
    m_pTailAllocator = mHeapEnd;
}

//---------------------------------------------------------------------------
//! @brief       フレームヒープの現在のメモリ使用状態を記録します。
//!              後で記録したメモリ使用状況に戻すことができます。
//!              状態を記録するのに20バイト使用します。
//!
//! @param[in]   tagName   状態記録に付けるタグ名。
//!
//! @return      関数が成功した場合、true が返ります。
//!              失敗したら、false が返ります。
//---------------------------------------------------------------------------
bool
FrameHeapImpl::RecordState( uint32_t tagName ) NN_NOEXCEPT
{
    bool retVal;
    NN_SDK_ASSERT( IsValid() );

    LockHeap();

    {
        void* oldHeadAllocator = m_pHeadAllocator;
        void* stateHeap = AllocFromHead( sizeof(HeapState), MIN_ALIGNMENT );

        if ( stateHeap == NULL )
        {
            retVal = false;
        }
        else
        {
            HeapState* pState = new( stateHeap ) HeapState;
            if ( pState == NULL )
            {
                retVal = false;
            }
            else
            {
                pState->tagName         = tagName;
                pState->headAllocator   = oldHeadAllocator;
                pState->tailAllocator   = m_pTailAllocator;
                pState->pPrevState      = m_pState;

                m_pState = pState;
                retVal = true;
            }
        }
    }

    UnlockHeap();

    return retVal;
}

//---------------------------------------------------------------------------
//! @brief        フレームヒープのメモリブロックを記録された状態に従って返却します。
//!
//! @details      指定したタグ名を用いてRecordStateForFrmHeap()を呼び出す直前
//!               のメモリの使用状況に戻ります。
//!               タグ名に0を指定すると最後にRecordStateForFrmHeap()を呼び出す
//!               直前の状態になります。
//!
//!               タグ名を指定して返却した場合、それ以後に呼び出された
//!               RecordStateForFrmHeap()によって記録された情報は
//!               無くなります。
//!
//! @param[in]    tagName 状態記録に付けるタグ名。
//!
//! @return       関数が成功した場合、true が返ります。
//!               失敗したら、false が返ります。
//---------------------------------------------------------------------------
bool
FrameHeapImpl::FreeByState( uint32_t tagName ) NN_NOEXCEPT
{
    bool retVal;
    NN_SDK_ASSERT( IsValid() );

    LockHeap();

    {
        HeapState* pState = m_pState;

        // タグ検索
        if ( tagName != 0 )
        {
            for ( ; pState; pState = pState->pPrevState )
            {
                if ( pState->tagName == tagName )
                {
                    break;
                }
            }
        }

        if ( pState == NULL )
        {
            retVal = false;
        }
        else
        {
            void* oldHeadAllocator = m_pHeadAllocator;
            void* oldTailAllocator = m_pTailAllocator;

            m_pHeadAllocator = pState->headAllocator;
            m_pTailAllocator = pState->tailAllocator;
            m_pState = pState->pPrevState;

            FillFreeMemory( m_pHeadAllocator,
                    static_cast<uint32_t>( GetOffsetFromPtr( m_pHeadAllocator, oldHeadAllocator ) ) );
            FillFreeMemory( oldTailAllocator,
                    static_cast<uint32_t>( GetOffsetFromPtr( oldTailAllocator, m_pTailAllocator ) ) );

            retVal = true;
        }
    }

    UnlockHeap();

    return retVal;
}

//---------------------------------------------------------------------------
//! @brief       フレームヒープの空き領域をヒープ領域から開放し、その分ヒープ領域を
//!              縮小します。
//!              ヒープメモリの後ろから確保されたメモリブロックが存在していては
//!              いけません。
//!
//! @return      関数が成功した場合、縮小後のフレームヒープのサイズを返します
//!              (バイト単位)。
//!              失敗した場合、0を返します。
//---------------------------------------------------------------------------
uint32_t
FrameHeapImpl::Adjust() NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValid() );

    uint32_t retVal;

    LockHeap();

    // 後方から確保されたメモリブロックがある場合は失敗
    if ( 0 < GetOffsetFromPtr( m_pTailAllocator, mHeapEnd ) )
    {
        retVal = 0;
    }
    else
    {
        m_pTailAllocator = m_pHeadAllocator;
        mHeapEnd = m_pHeadAllocator;
        retVal = static_cast<uint32_t>( GetOffsetFromPtr( this, mHeapEnd ) );
    }

    UnlockHeap();

    return retVal;
}

}}}} // namespace nn::atk::detail::fnd
