﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_EffectReverb.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>
#include <nn/atk/atk_HardwareManager.h>
#include <nn/atk/detail/atk_Macro.h>

namespace
{
nn::audio::ReverbType::EarlyMode ConvertAtkEarlyModeToAudioEarlyMode(nn::atk::EffectReverb::EarlyMode earlyMode) NN_NOEXCEPT
{
    switch(earlyMode)
    {
    case nn::atk::EffectReverb::EarlyMode_SmallRoom:
        return nn::audio::ReverbType::EarlyMode_SmallRoom;
    case nn::atk::EffectReverb::EarlyMode_LargeRoom:
        return nn::audio::ReverbType::EarlyMode_LargeRoom;
    case nn::atk::EffectReverb::EarlyMode_Hall:
        return nn::audio::ReverbType::EarlyMode_Hall;
    case nn::atk::EffectReverb::EarlyMode_Cavern:
        return nn::audio::ReverbType::EarlyMode_Cavern;
    case nn::atk::EffectReverb::EarlyMode_NoEarlyReflection:
        return nn::audio::ReverbType::EarlyMode_NoEarlyReflection;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

nn::audio::ReverbType::LateMode ConvertAtkLateModeToAudioLateMode(nn::atk::EffectReverb::LateMode lateMode) NN_NOEXCEPT
{
    switch(lateMode)
    {
    case nn::atk::EffectReverb::LateMode_Room:
        return nn::audio::ReverbType::LateMode_Room;
    case nn::atk::EffectReverb::LateMode_Hall:
        return nn::audio::ReverbType::LateMode_Hall;
    case nn::atk::EffectReverb::LateMode_MetalCorridor:
        return nn::audio::ReverbType::LateMode_MetalCorridor;
    case nn::atk::EffectReverb::LateMode_Cavern:
        return nn::audio::ReverbType::LateMode_Cavern;
    case nn::atk::EffectReverb::LateMode_MaximumDelay:
        return nn::audio::ReverbType::LateMode_MaximumDelay;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

int ConvertReverbChannelModeToInt(nn::atk::EffectReverb::ChannelMode channelMode) NN_NOEXCEPT
{
    switch(channelMode)
    {
    case nn::atk::EffectBase::ChannelMode_1Ch:
        return 1;
    case nn::atk::EffectBase::ChannelMode_2Ch:
        return 2;
    case nn::atk::EffectBase::ChannelMode_4Ch:
        return 4;
    case nn::atk::EffectBase::ChannelMode_6Ch:
        return 6;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool IsChannelModeSupported(nn::atk::EffectBase::ChannelMode channelMode) NN_NOEXCEPT
{
    return channelMode == nn::atk::EffectBase::ChannelMode_1Ch || channelMode == nn::atk::EffectBase::ChannelMode_2Ch
        || channelMode == nn::atk::EffectBase::ChannelMode_4Ch || channelMode == nn::atk::EffectBase::ChannelMode_6Ch;
}

int ConvertReverbSampleRateToInt(nn::atk::EffectReverb::SampleRate sampleRate) NN_NOEXCEPT
{
    switch(sampleRate)
    {
    case nn::atk::EffectBase::SampleRate_32000:
        return 32000;
    case nn::atk::EffectBase::SampleRate_48000:
        return 48000;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}
}

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT(const int EffectReverb::ChannelSettingCountMax);

// 最大値・最小値の定義
const float EffectReverb::EarlyGainMin = 0.0f;
const float EffectReverb::EarlyGainMax = 1.0f;
const int64_t EffectReverb::PredelayTimeMilliSecondsMin = 0;
const int64_t EffectReverb::PredelayTimeMilliSecondsMax = 300;
const float EffectReverb::LateGainMin = 0.0f;
const float EffectReverb::LateGainMax = 1.0f;
const int64_t EffectReverb::DecayTimeMilliSecondsMin = 100;
const int64_t EffectReverb::DecayTimeMilliSecondsMax = 20000;
const float EffectReverb::HighFrequencyDecayRatioMin = 0.1f;
const float EffectReverb::HighFrequencyDecayRatioMax = 1.0f;
const float EffectReverb::ColorationMin = 0.0f;
const float EffectReverb::ColorationMax = 1.0f;
const float EffectReverb::ReverbGainMin = 0.0f;
const float EffectReverb::ReverbGainMax = 1.0f;
const float EffectReverb::OutGainMin = 0.0f;
const float EffectReverb::OutGainMax = 1.0f;
const float EffectReverb::DryGainMin = 0.0f;
const float EffectReverb::DryGainMax = 1.0f;

// デフォルト値の定義
const float EffectReverb::DefaultEarlyGain = 0.7f;
const int64_t EffectReverb::DefaultPredelayTimeMilliSeconds = 15;
const float EffectReverb::DefaultLateGain = 0.7f;
const int64_t EffectReverb::DefaultDecayTimeMilliSeconds = 1500;
const float EffectReverb::DefaultHighFrequencyDecayRatio = 0.8f;
const float EffectReverb::DefaultColoration = 0.7f;
const float EffectReverb::DefaultReverbGain = 0.2f;
const float EffectReverb::DefaultOutGain = 1.0f;
const float EffectReverb::DefaultDryGain = 0.7f;

EffectReverb::EffectReverb() NN_NOEXCEPT
    : EffectBase()
    , m_AudioRendererUpdateCountWhenAddedReverb(0)
    , m_ChannelMode(ChannelMode_4Ch)
{
    ResetChannelIndex();
}

EffectReverb::~EffectReverb() NN_NOEXCEPT
{
}

size_t EffectReverb::GetRequiredMemSize() const NN_NOEXCEPT
{
    int maxChannelCount = ConvertReverbChannelModeToInt(m_ChannelMode);
    int sampleRate = ConvertReverbSampleRateToInt(m_SampleRate);
    return nn::audio::GetRequiredBufferSizeForReverb(sampleRate, maxChannelCount);
}

nn::TimeSpan EffectReverb::GetPredelayTimeMin() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(PredelayTimeMilliSecondsMin);
}

nn::TimeSpan EffectReverb::GetPredelayTimeMax() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(PredelayTimeMilliSecondsMax);
}

nn::TimeSpan EffectReverb::GetDefaultPredelayTime() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(DefaultPredelayTimeMilliSeconds);
}

nn::TimeSpan EffectReverb::GetDecayTimeMin() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(DecayTimeMilliSecondsMin);
}

nn::TimeSpan EffectReverb::GetDecayTimeMax() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(DecayTimeMilliSecondsMax);
}

nn::TimeSpan EffectReverb::GetDefaultDecayTime() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(DefaultDecayTimeMilliSeconds);
}

bool EffectReverb::AddEffect(nn::audio::AudioRendererConfig* pConfig, nn::atk::OutputMixer* pOutputMixer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConfig);
    NN_SDK_ASSERT_NOT_NULL(pOutputMixer);
    if(m_IsActive)
    {
        return false;
    }

    int channelCount = ConvertReverbChannelModeToInt(m_ChannelMode);
    int sampleRate = ConvertReverbSampleRateToInt(m_SampleRate);
    NN_SDK_ASSERT_NOT_NULL(m_EffectBuffer);
    NN_SDK_REQUIRES_ALIGNED( m_EffectBuffer, nn::audio::BufferAlignSize );
    NN_SDK_ASSERT_GREATER_EQUAL(m_EffectBufferSize, nn::audio::GetRequiredBufferSizeForReverb(sampleRate, channelCount));
    NN_UNUSED(sampleRate);

    {
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
#endif
        m_AudioRendererUpdateCountWhenAddedReverb = detail::driver::HardwareManager::GetInstance().GetAudioRendererUpdateCount();

        nn::Result result;
        switch( pOutputMixer->GetReceiverType() )
        {
            case OutputReceiver::ReceiverType::ReceiverType_SubMix:
            {
                result = nn::audio::AddReverb( pConfig, &m_ReverbType, m_EffectBuffer, m_EffectBufferSize, reinterpret_cast<SubMix*>( pOutputMixer )->GetAudioSubMixInstance(), channelCount );
                break;
            }
            case OutputReceiver::ReceiverType::ReceiverType_FinalMix:
            {
                result = nn::audio::AddReverb( pConfig, &m_ReverbType, m_EffectBuffer, m_EffectBufferSize, reinterpret_cast<FinalMix*>( pOutputMixer )->GetAudioFinalMixInstance(), channelCount );
                break;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
        }

        if( result.IsFailure() )
        {
            return false;
        }
    }

    m_IsActive = true;

    // 初期化後に ReverbType へパラメータを反映させる
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetReverbEarlyMode(&m_ReverbType, ConvertAtkEarlyModeToAudioEarlyMode(m_Param.earlyMode));
        nn::audio::SetReverbEarlyGain(&m_ReverbType, m_Param.earlyGain);
        nn::audio::SetReverbPredelayTime(&m_ReverbType, m_Param.predelayTime);
        nn::audio::SetReverbLateMode(&m_ReverbType, ConvertAtkLateModeToAudioLateMode(m_Param.lateMode));
        nn::audio::SetReverbLateGain(&m_ReverbType, m_Param.lateGain);
        nn::audio::SetReverbDecayTime(&m_ReverbType, m_Param.decayTime);
        nn::audio::SetReverbHighFrequencyDecayRatio(&m_ReverbType, m_Param.highFrequencyDecayRatio);
        nn::audio::SetReverbColoration(&m_ReverbType, m_Param.coloration);
        nn::audio::SetReverbReverbGain(&m_ReverbType, m_Param.reverbGain);
        nn::audio::SetReverbOutGain(&m_ReverbType, m_Param.outGain);
        nn::audio::SetReverbDryGain(&m_ReverbType, m_Param.dryGain);
        nn::audio::SetReverbEnabled(&m_ReverbType, m_Param.isEnabled);
    }

    return true;
}

void EffectReverb::SetEffectInputOutput(const int8_t* input, const int8_t* output, int inputCount, int outputCount) NN_NOEXCEPT
{
    // reverb は入出力のチャンネル数が同じ
    NN_SDK_ASSERT_EQUAL(inputCount, outputCount);
    NN_SDK_ASSERT_NOT_NULL(input);
    NN_SDK_ASSERT_NOT_NULL(output);
    // reverb が 2 ch, 4 ch のみの適用であるため 2 ch 以上であることを確認する
    NN_SDK_ASSERT_GREATER_EQUAL(inputCount, ConvertReverbChannelModeToInt(EffectReverb::ChannelMode_2Ch));

    int channelCount = ConvertReverbChannelModeToInt(m_ChannelMode);
    // 設定した reverb のチャンネル数が与えられたバッファ数よりも少ないことを確認する
    NN_SDK_ASSERT_LESS_EQUAL(channelCount, inputCount);

    NN_UNUSED(inputCount);
    NN_UNUSED(outputCount);
    NN_UNUSED(output);

    // リバーブのチャンネル設定を適用する
    NN_SDK_ASSERT_LESS_EQUAL(channelCount, ChannelSettingCountMax);
    int8_t channelBus[ChannelSettingCountMax];
    for (int i = 0; i < channelCount; i++)
    {
        // m_ChannelSetting[i] が input の範囲外の配列を指していないか確認する
        NN_SDK_ASSERT_RANGE(m_ChannelSetting[i], 0, inputCount);
        channelBus[i] = input[m_ChannelSetting[i]];
    }

    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetReverbInputOutput(&m_ReverbType, channelBus, channelBus, channelCount);
    }
}

void EffectReverb::RemoveEffect(nn::audio::AudioRendererConfig* pConfig, nn::atk::OutputMixer* pOutputMixer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConfig);
    NN_SDK_ASSERT_NOT_NULL(pOutputMixer);
    if(!m_IsActive)
    {
        return;
    }

    {
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
#endif
        NN_SDK_ASSERT(nn::audio::IsReverbRemovable(&m_ReverbType));

        void* returnedBuffer = nullptr;
        switch( pOutputMixer->GetReceiverType() )
        {
            case OutputReceiver::ReceiverType::ReceiverType_SubMix:
            {
                returnedBuffer = nn::audio::RemoveReverb( pConfig, &m_ReverbType, reinterpret_cast<SubMix*>( pOutputMixer )->GetAudioSubMixInstance() );
                break;
            }
            case OutputReceiver::ReceiverType::ReceiverType_FinalMix:
            {
                returnedBuffer = nn::audio::RemoveReverb( pConfig, &m_ReverbType, reinterpret_cast<FinalMix*>( pOutputMixer )->GetAudioFinalMixInstance() );
                break;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
        }

        NN_SDK_ASSERT_EQUAL(returnedBuffer, m_EffectBuffer);
        NN_UNUSED(returnedBuffer);
    }
    m_IsActive = false;
}

bool EffectReverb::IsRemovable() const NN_NOEXCEPT
{
    if (!m_IsActive)
    {
        return false;
    }

    //  AddReverb 直後は IsReverbRemovable が true を返すため、
    //  1 回以上 RequestUpdateAudioRenderer が呼ばれてから IsReverbRemovable で確認するようにします

    if( !( m_AudioRendererUpdateCountWhenAddedReverb < detail::driver::HardwareManager::GetInstance().GetAudioRendererUpdateCount() ) )
    {
        return false;
    }

    detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
    if( !nn::audio::IsReverbRemovable(&m_ReverbType) )
    {
        return false;
    }

    //  RequestUpdateAudioRenderer の呼び出し回数を 2 度比較している理由はコミットログを参照
    return m_AudioRendererUpdateCountWhenAddedReverb < detail::driver::HardwareManager::GetInstance().GetAudioRendererUpdateCount();
}

bool EffectReverb::IsClearable() NN_NOEXCEPT
{
    return IsRemovable();
}

EffectReverb::ChannelMode EffectReverb::GetChannelMode() const NN_NOEXCEPT
{
    return m_ChannelMode;
}

bool EffectReverb::SetChannelMode(ChannelMode mode) NN_NOEXCEPT
{
    ChannelIndex defaultChannelSetting[ChannelModeCountMax];
    for (int i = 0; i < ChannelModeCountMax; i++)
    {
        defaultChannelSetting[i] = static_cast<ChannelIndex>(i);
    }

    return SetChannelIndex(defaultChannelSetting, mode);
}

void EffectReverb::GetChannelIndex(ChannelIndex* pChannel, int channelCount) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pChannel);
    int effectChannelCount = ConvertReverbChannelModeToInt(m_ChannelMode);
    NN_SDK_REQUIRES_GREATER_EQUAL(channelCount, effectChannelCount);
    NN_UNUSED(channelCount);
    for (int i = 0; i < effectChannelCount; i++)
    {
        pChannel[i] = m_ChannelSetting[i];
    }
}

bool EffectReverb::SetChannelIndex(const ChannelIndex* pChannel, ChannelMode channelMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pChannel);

    int channelCount = ConvertChannelModeToInt(channelMode);
    NN_SDK_ASSERT_LESS_EQUAL(channelCount, ChannelModeCountMax);
    for (int i = 0; i < channelCount; i++)
    {
        NN_SDK_REQUIRES_RANGE(pChannel[i], 0, nn::atk::ChannelIndex_Count);
        for (int j = i + 1; j < channelCount; j++)
        {
            NN_SDK_REQUIRES_NOT_EQUAL(pChannel[i], pChannel[j]);
        }
    }

    if (m_IsActive)
    {
        return false;
    }

    if (IsChannelModeSupported(channelMode))
    {
        m_ChannelMode = channelMode;
    }
    else
    {
        NN_ATK_WARNING("unsupported reverb channel mode.");
        return false;
    }

    NN_SDK_ASSERT_LESS_EQUAL(channelCount, ChannelSettingCountMax);
    for (int i = 0; i < channelCount; i++)
    {
        m_ChannelSetting[i] = pChannel[i];
    }

    return true;
}

EffectReverb::EarlyMode EffectReverb::GetEarlyMode() const NN_NOEXCEPT
{
    return m_Param.earlyMode;
}

void EffectReverb::SetEarlyMode( const EarlyMode earlyMode ) NN_NOEXCEPT
{
    m_Param.earlyMode = earlyMode;
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetReverbEarlyMode(&m_ReverbType, ConvertAtkEarlyModeToAudioEarlyMode(m_Param.earlyMode));
    }
}

float EffectReverb::GetEarlyGain() const NN_NOEXCEPT
{
    return m_Param.earlyGain;
}

void EffectReverb::SetEarlyGain( const float earlyGain ) NN_NOEXCEPT
{
    m_Param.earlyGain = nn::atk::detail::fnd::Clamp(earlyGain, EarlyGainMin, EarlyGainMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetReverbEarlyGain(&m_ReverbType, m_Param.earlyGain);
    }
}

nn::TimeSpan EffectReverb::GetPredelayTime() const NN_NOEXCEPT
{
    return m_Param.predelayTime;
}

void EffectReverb::SetPredelayTime( const nn::TimeSpan predelayTime ) NN_NOEXCEPT
{
    m_Param.predelayTime = nn::atk::detail::fnd::Clamp(predelayTime, GetPredelayTimeMin(), GetPredelayTimeMax());
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetReverbPredelayTime(&m_ReverbType, m_Param.predelayTime);
    }
}

EffectReverb::LateMode EffectReverb::GetLateMode() const NN_NOEXCEPT
{
    return m_Param.lateMode;
}

void EffectReverb::SetLateMode( const LateMode lateMode ) NN_NOEXCEPT
{
    m_Param.lateMode = lateMode;
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetReverbLateMode(&m_ReverbType, ConvertAtkLateModeToAudioLateMode(m_Param.lateMode));
    }
}

float EffectReverb::GetLateGain() const NN_NOEXCEPT
{
    return m_Param.lateGain;
}

void EffectReverb::SetLateGain( const float lateGain ) NN_NOEXCEPT
{
    m_Param.lateGain = nn::atk::detail::fnd::Clamp(lateGain, LateGainMin, LateGainMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetReverbLateGain(&m_ReverbType, m_Param.lateGain);
    }
}

nn::TimeSpan EffectReverb::GetDecayTime() const NN_NOEXCEPT
{
    return m_Param.decayTime;
}

void EffectReverb::SetDecayTime( const nn::TimeSpan decayTime ) NN_NOEXCEPT
{
    m_Param.decayTime = nn::atk::detail::fnd::Clamp(decayTime, GetDecayTimeMin(), GetDecayTimeMax());
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetReverbDecayTime(&m_ReverbType, m_Param.decayTime);
    }
}

float EffectReverb::GetHighFrequencyDecayRatio() const NN_NOEXCEPT
{
    return m_Param.highFrequencyDecayRatio;
}

void EffectReverb::SetHighFrequencyDecayRatio( const float highFrequencyDecayRatio ) NN_NOEXCEPT
{
    m_Param.highFrequencyDecayRatio = nn::atk::detail::fnd::Clamp(highFrequencyDecayRatio, HighFrequencyDecayRatioMin, HighFrequencyDecayRatioMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetReverbHighFrequencyDecayRatio(&m_ReverbType, m_Param.highFrequencyDecayRatio);
    }
}

float EffectReverb::GetColoration() const NN_NOEXCEPT
{
    return m_Param.coloration;
}

void EffectReverb::SetColoration( const float coloration ) NN_NOEXCEPT
{
    m_Param.coloration = nn::atk::detail::fnd::Clamp(coloration, ColorationMin, ColorationMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetReverbColoration(&m_ReverbType, m_Param.coloration);
    }
}

float EffectReverb::GetReverbGain() const NN_NOEXCEPT
{
    return m_Param.reverbGain;
}

void EffectReverb::SetReverbGain( const float reverbGain ) NN_NOEXCEPT
{
    m_Param.reverbGain = nn::atk::detail::fnd::Clamp(reverbGain, ReverbGainMin, ReverbGainMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetReverbReverbGain(&m_ReverbType, m_Param.reverbGain);
    }
}

float EffectReverb::GetOutGain() const NN_NOEXCEPT
{
    return m_Param.outGain;
}

void EffectReverb::SetOutGain( const float outGain ) NN_NOEXCEPT
{
    m_Param.outGain = nn::atk::detail::fnd::Clamp(outGain, OutGainMin, OutGainMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetReverbOutGain(&m_ReverbType, m_Param.outGain);
    }
}

float EffectReverb::GetDryGain() const NN_NOEXCEPT
{
    return m_Param.dryGain;
}

void EffectReverb::SetDryGain( const float dryGain ) NN_NOEXCEPT
{
    m_Param.dryGain = nn::atk::detail::fnd::Clamp(dryGain, DryGainMin, DryGainMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetReverbDryGain(&m_ReverbType, m_Param.dryGain);
    }
}

bool EffectReverb::IsEnabled() const NN_NOEXCEPT
{
    return m_Param.isEnabled;
}

void EffectReverb::SetEnabled( bool isEnabled ) NN_NOEXCEPT
{
    m_Param.isEnabled = isEnabled;
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetReverbEnabled(&m_ReverbType, m_Param.isEnabled);
    }
}

EffectReverb::ReverbParam::ReverbParam() NN_NOEXCEPT
    : earlyMode(EarlyMode_Default)
    , earlyGain(DefaultEarlyGain)
    , predelayTime(GetDefaultPredelayTime())
    , lateMode(LateMode_Default)
    , lateGain(DefaultLateGain)
    , decayTime(GetDefaultDecayTime())
    , highFrequencyDecayRatio(DefaultHighFrequencyDecayRatio)
    , coloration(DefaultColoration)
    , reverbGain(DefaultReverbGain)
    , outGain(DefaultOutGain)
    , dryGain(DefaultDryGain)
    , isEnabled(false)
{
}

void EffectReverb::ResetChannelIndex() NN_NOEXCEPT
{
    for(int i = 0; i < ChannelSettingCountMax; i++)
    {
        m_ChannelSetting[i] = static_cast<ChannelIndex>(i);
    }
}

}}
