﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_EffectAux.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>
#include <nn/util/util_BytePtr.h>
#include <nn/atk/atk_SoundSystem.h>
#include <nn/atk/atk_HardwareManager.h>

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT(const int EffectAux::ChannelCountMax);

EffectAux::EffectAux() NN_NOEXCEPT
    : m_AudioRendererUpdateCountWhenAddedAux(0)
    , m_AudioFrameCount(4)
    , m_ChannelCount(4)
    , m_IsActive(false)
    , m_IsEnabled(false)
    , m_EffectBuffer(nullptr)
    , m_EffectBufferSize(0)
    , m_AuxReadBuffer(nullptr)
{
    ResetChannelIndex();
}

EffectAux::~EffectAux() NN_NOEXCEPT
{
}

bool EffectAux::Initialize() NN_NOEXCEPT
{
    return true;
}

void EffectAux::Finalize() NN_NOEXCEPT
{
}

void EffectAux::OnChangeOutputMode() NN_NOEXCEPT
{
}

size_t EffectAux::GetRequiredMemSize(const nn::audio::AudioRendererParameter& parameter) const NN_NOEXCEPT
{
    size_t bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&parameter, m_AudioFrameCount, m_ChannelCount);
    NN_SDK_ASSERT_EQUAL(bufferSize % nn::audio::BufferAlignSize, 0u);

    return bufferSize * 3; // SendBuffer + ReturnBuffer + ReadBuffer
}

bool EffectAux::AddEffect(nn::audio::AudioRendererConfig* pConfig, const nn::audio::AudioRendererParameter& parameter, nn::atk::OutputMixer* pOutputMixer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConfig);
    NN_SDK_ASSERT_NOT_NULL(pOutputMixer);
    if(m_IsActive)
    {
        return false;
    }

    NN_SDK_ASSERT_NOT_NULL(m_EffectBuffer);
    NN_SDK_REQUIRES_ALIGNED( m_EffectBuffer, nn::audio::BufferAlignSize );
    NN_SDK_ASSERT_GREATER_EQUAL(m_EffectBufferSize, GetRequiredMemSize(parameter));

    BufferSet bufferSet;
    size_t sendReturnBufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&parameter, m_AudioFrameCount, m_ChannelCount);

    SplitEffectBuffer(&bufferSet, m_EffectBuffer, sendReturnBufferSize);

    NN_SDK_ASSERT_NOT_NULL(bufferSet.sendBuffer);
    NN_SDK_ASSERT_NOT_NULL(bufferSet.returnBuffer);
    NN_SDK_ASSERT_NOT_NULL(bufferSet.readBuffer);

    m_AuxReadBuffer = reinterpret_cast<int32_t*>(bufferSet.readBuffer);
    NN_SDK_ASSERT_NOT_NULL(m_AuxReadBuffer);

    {
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
#endif
        m_AudioRendererUpdateCountWhenAddedAux = detail::driver::HardwareManager::GetInstance().GetAudioRendererUpdateCount();

        nn::Result result;
        switch( pOutputMixer->GetReceiverType() )
        {
            case OutputReceiver::ReceiverType::ReceiverType_SubMix:
            {
                result = nn::audio::AddAux( pConfig, &m_AuxType, reinterpret_cast<SubMix*>( pOutputMixer )->GetAudioSubMixInstance(), bufferSet.sendBuffer, bufferSet.returnBuffer, sendReturnBufferSize );
                break;
            }
            case OutputReceiver::ReceiverType::ReceiverType_FinalMix:
            {
                result = nn::audio::AddAux( pConfig, &m_AuxType, reinterpret_cast<FinalMix*>( pOutputMixer )->GetAudioFinalMixInstance(), bufferSet.sendBuffer, bufferSet.returnBuffer, sendReturnBufferSize );
                break;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
        }

        if( result.IsFailure() )
        {
            return false;
        }
    }

    nn::audio::SetAuxEnabled(&m_AuxType, m_IsEnabled);
    m_IsActive = true;

    return true;
}

void EffectAux::SetEffectInputOutput(const int8_t* input, const int8_t* output, int inputCount, int outputCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(inputCount, outputCount);
    NN_SDK_ASSERT_NOT_NULL(input);
    NN_SDK_ASSERT_NOT_NULL(output);
    NN_SDK_ASSERT_GREATER(inputCount, 0);
    NN_SDK_ASSERT_GREATER(outputCount, 0);

    NN_SDK_ASSERT_LESS_EQUAL(m_ChannelCount, inputCount);

    NN_UNUSED(inputCount);
    NN_UNUSED(outputCount);
    NN_UNUSED(output);

    // Aux のチャンネル設定を適用する
    NN_SDK_ASSERT_LESS_EQUAL(m_ChannelCount, ChannelCountMax);
    int8_t channelBus[ChannelCountMax];
    for (int i = 0; i < m_ChannelCount; i++)
    {
        // m_ChannelSetting[i] が input の範囲外の配列を指していないか確認する
        NN_SDK_ASSERT_RANGE(m_ChannelSetting[i], 0, inputCount);
        channelBus[i] = input[m_ChannelSetting[i]];
    }

    nn::audio::SetAuxInputOutput(&m_AuxType, channelBus, channelBus, m_ChannelCount);
}

void EffectAux::RemoveEffect(nn::audio::AudioRendererConfig* pConfig, nn::atk::OutputMixer* pOutputMixer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConfig);
    NN_SDK_ASSERT_NOT_NULL(pOutputMixer);
    if(!m_IsActive)
    {
        return;
    }
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
#endif
    NN_SDK_ASSERT(nn::audio::IsAuxRemovable(&m_AuxType));

    switch( pOutputMixer->GetReceiverType() )
    {
        case OutputReceiver::ReceiverType::ReceiverType_SubMix:
        {
            nn::audio::RemoveAux( pConfig, &m_AuxType, reinterpret_cast<SubMix*>( pOutputMixer )->GetAudioSubMixInstance() );
            break;
        }
        case OutputReceiver::ReceiverType::ReceiverType_FinalMix:
        {
            nn::audio::RemoveAux( pConfig, &m_AuxType, reinterpret_cast<FinalMix*>( pOutputMixer )->GetAudioFinalMixInstance() );
            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    m_IsActive = false;
}

bool EffectAux::SetChannelCount(int channelCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(channelCount, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(channelCount, ChannelCountMax);

    ResetChannelIndex();

    return SetChannelIndex(m_ChannelSetting, channelCount);
}

int EffectAux::GetChannelCount() const NN_NOEXCEPT
{
    return m_ChannelCount;
}

bool EffectAux::SetChannelIndex(const ChannelIndex* pChannel, int channelCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pChannel);
    NN_SDK_REQUIRES_GREATER(channelCount, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(channelCount, ChannelCountMax);

    for (int i = 0; i < channelCount; i++)
    {
        NN_SDK_REQUIRES_RANGE(pChannel[i], 0, nn::atk::ChannelIndex_Count);
        for (int j = i + 1; j < channelCount; j++)
        {
            NN_SDK_REQUIRES_NOT_EQUAL(pChannel[i], pChannel[j]);
        }
    }

    if (m_IsActive)
    {
        return false;
    }
    m_ChannelCount = channelCount;

    for (int i = 0; i < channelCount; i++)
    {
        m_ChannelSetting[i] = pChannel[i];
    }

    return true;
}

void EffectAux::GetChannelIndex(ChannelIndex* pChannel, int channelCount) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pChannel);
    NN_SDK_ASSERT_GREATER_EQUAL(channelCount, m_ChannelCount);
    NN_UNUSED(channelCount);
    for(int i = 0; i < m_ChannelCount; i++)
    {
        pChannel[i] = m_ChannelSetting[i];
    }
}

bool EffectAux::SetAudioFrameCount(int audioFrameCount) NN_NOEXCEPT
{
    // エフェクト追加後には変更できない
    if (m_IsActive)
    {
        return false;
    }

    m_AudioFrameCount = audioFrameCount;

    return true;
}

int EffectAux::GetAudioFrameCount() const NN_NOEXCEPT
{
    return m_AudioFrameCount;
}

bool EffectAux::IsRemovable() const NN_NOEXCEPT
{
    if (!m_IsActive)
    {
        return false;
    }

    //  AddAux 直後は IsAuxRemovable が true を返すため、
    //  1 回以上 RequestUpdateAudioRenderer が呼ばれてから IsAuxRemovable で確認するようにします

    if( !( m_AudioRendererUpdateCountWhenAddedAux < detail::driver::HardwareManager::GetInstance().GetAudioRendererUpdateCount() ) )
    {
        return false;
    }

    detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
    if( !nn::audio::IsAuxRemovable(&m_AuxType) )
    {
        return false;
    }

    //  RequestUpdateAudioRenderer の呼び出し回数を 2 度比較している理由はコミットログを参照
    return m_AudioRendererUpdateCountWhenAddedAux < detail::driver::HardwareManager::GetInstance().GetAudioRendererUpdateCount();
}

bool EffectAux::IsClearable() NN_NOEXCEPT
{
    return IsRemovable();
}

bool EffectAux::IsEnabled() const NN_NOEXCEPT
{
    return m_IsEnabled;
}

void EffectAux::SetEnabled( bool isEnabled ) NN_NOEXCEPT
{
    m_IsEnabled = isEnabled;

    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetAuxEnabled(&m_AuxType, isEnabled);
    }
}

void EffectAux::SetEffectBuffer( void* effectBuffer, size_t effectBufferSize ) NN_NOEXCEPT
{
    m_EffectBuffer = effectBuffer;
    m_EffectBufferSize = effectBufferSize;
}

void EffectAux::Update() NN_NOEXCEPT
{
    if ( m_IsActive )
    {
        NN_SDK_ASSERT_NOT_NULL(m_AuxReadBuffer);
        int sampleCount = nn::audio::GetAuxSampleCount(&m_AuxType);

        UpdateSamplesArg arg;
        arg.sampleCountPerAudioFrame = sampleCount / (m_AudioFrameCount * m_ChannelCount);
        arg.sampleRate = nn::audio::GetAuxSampleRate(&m_AuxType);
        arg.audioFrameCount = m_AudioFrameCount;
        arg.channelCount = m_ChannelCount;
        arg.outputMode = SoundSystem::GetOutputMode(OutputDevice_Main);

        int readSampleCount = nn::audio::ReadAuxSendBuffer(&m_AuxType, m_AuxReadBuffer, sampleCount);
        arg.readSampleCount = readSampleCount;
        UpdateSamples(m_AuxReadBuffer, arg);
        nn::audio::WriteAuxReturnBuffer(&m_AuxType, m_AuxReadBuffer, readSampleCount);
    }
}

void EffectAux::SplitEffectBuffer(BufferSet* pOutBufferSet, void* effectBuffer, size_t sendReturnBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutBufferSet);

    nn::util::BytePtr bytePtr(effectBuffer);

    bytePtr.AlignUp(nn::audio::BufferAlignSize);
    pOutBufferSet->sendBuffer = bytePtr.Get();

    bytePtr.Advance(sendReturnBufferSize);
    bytePtr.AlignUp(nn::audio::BufferAlignSize);
    pOutBufferSet->returnBuffer = bytePtr.Get();

    bytePtr.Advance(sendReturnBufferSize);
    bytePtr.AlignUp(nn::audio::BufferAlignSize);
    pOutBufferSet->readBuffer = bytePtr.Get();
}


void EffectAux::ResetChannelIndex() NN_NOEXCEPT
{
    for(int i = 0; i < ChannelCountMax; i++)
    {
        m_ChannelSetting[i] = static_cast<ChannelIndex>(i);
    }
}

}}
