﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_Voice.h>
#include <nn/atk/atk_Config.h>
#include <nn/atk/atk_Util.h>
#include <nn/atk/atk_HardwareManager.h>
#include <nn/atk/atk_VoiceCommand.h>
#include <nn/atk/detail/atk_Macro.h>

namespace nn {
namespace atk {
namespace detail {

namespace
{

const BiquadFilterCoefficients BiquadFilterCoefZero = { 0 };
const OutputMix DefaultTvMix =
{
    { 1.0f, 1.0f, 0.0f, 0.0f, 0.0f, 0.0f,  // Default: mainBus
      0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f,  // Default: auxA
      0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f,  // Default: auxB
      0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f } // Default: auxC
};
#if 0 // 参考用 : NW4F 実装
const OutputMix DefaultDrcMix =
{
    { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f },      // mainBus
    {
        { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f },  // auxA
        { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f },  // auxB
        { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f }   // auxC
    }
};
#endif

}

void VoiceParam::Initialize() NN_NOEXCEPT
{
    m_Volume = 1.0f;
    m_Pitch = 1.0f;
    m_TvMix = DefaultTvMix;

    m_MonoFilterFlag = false;
    m_BiquadFilterFlag = false;

    m_BiquadFilterCoefficients = BiquadFilterCoefZero;
    m_MonoFilterCutoff = 0;
    m_InterpolationType = 0;
}

//==================================================================
//
// Voice
//
//==================================================================

NN_DEFINE_STATIC_CONSTANT( const uint32_t Voice::PriorityMin );
NN_DEFINE_STATIC_CONSTANT( const uint32_t Voice::PriorityMax );
NN_DEFINE_STATIC_CONSTANT( const uint32_t Voice::PriorityNoDrop );

Voice::Voice() NN_NOEXCEPT
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
: m_VoiceId( VirtualVoiceManager::InvalidVoiceId )
#else
    : m_pLowLevelVoice( NULL )
#endif
{
}

Voice::~Voice() NN_NOEXCEPT
{
}

void Voice::Initialize( uint32_t priority ) NN_NOEXCEPT
{
    m_State = VoiceState_Stop;
    m_Priority = priority;

    m_VoiceParam.Initialize();

    m_SampleRate = 32000;
    m_SampleFormat = SampleFormat_PcmS16;
    m_pOutputReceiver = nullptr;
    m_PlayPosition = 0;

#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    m_WaveBufferListBegin = NULL;
    m_WaveBufferListEnd = NULL;

    m_VoiceInfoEnableFlag = false;
#endif

}

bool Voice::IsAvailable() const NN_NOEXCEPT
{
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if ( m_VoiceId != VirtualVoiceManager::InvalidVoiceId )
    {
        LowLevelVoice* pVoice = VirtualVoiceManager::GetInstance().GetLowLevelVoice(m_VoiceId);
        if (pVoice != NULL)
        {
            return pVoice->IsAvailable();
        }

        return true;
    }
#else
    if ( m_pLowLevelVoice != NULL )
    {
        return m_pLowLevelVoice->IsAvailable();
    }
#endif

    return false;
}

bool Voice::AllocVoice( uint32_t priority ) NN_NOEXCEPT
{
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    m_VoiceId = VirtualVoiceManager::GetInstance().AllocVirtualVoice();
    if ( m_VoiceId == VirtualVoiceManager::InvalidVoiceId ) return false;

    LowLevelVoiceCommand& cmdmgr = LowLevelVoiceCommand::GetInstance();
    VoiceCommandAlloc* command = cmdmgr.AllocCommand<VoiceCommandAlloc>( false );
    //@@@ NN_DETAIL_ATK_INFO("__ALLOC__\n");
    if ( command == NULL )
    {
        NN_ATK_WARNING("[%s:%d] cannot alloc command", __FUNCTION__, __LINE__);
        return false;
    }
    command->id = VoiceCommandId_AllocVoice;
    command->voiceId = m_VoiceId;
    command->priority = priority;
    command->userId = this;
    m_CommandTag = cmdmgr.PushCommand(command);

    Initialize( priority );

#else // NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    LowLevelVoice* lowLevelVoice = driver::HardwareManager::GetInstance().GetLowLevelVoiceAllocator().AllocVoice();
    if ( lowLevelVoice == NULL )
    {
        return false;
    }

    lowLevelVoice->SetPriority( priority );

    lowLevelVoice->SetVoice(this);

    m_pLowLevelVoice = lowLevelVoice;

    Initialize( priority );

#endif // NN_ATK_CONFIG_ENABLE_VOICE_COMMAND

    return true;
}

void Voice::Free() NN_NOEXCEPT
{
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    LowLevelVoiceCommand& cmdmgr = LowLevelVoiceCommand::GetInstance();
    VoiceCommandFree* command = cmdmgr.AllocCommand<VoiceCommandFree>( false );
    //@@@ NN_DETAIL_ATK_INFO("__FREE__\n");
    if ( command == NULL )
    {
        NN_ATK_WARNING("[%s:%d] cannot alloc command", __FUNCTION__, __LINE__);
        FreeAllWaveBuffer();
        return;
    }
    command->id = VoiceCommandId_FreeVoice;
    command->voiceId = m_VoiceId;
    cmdmgr.PushCommand(command);

    FreeAllWaveBuffer();

    if ( m_VoiceId != VirtualVoiceManager::InvalidVoiceId )
    {
        VirtualVoiceManager::GetInstance().FreeVirtualVoice( m_VoiceId );
        m_VoiceId = VirtualVoiceManager::InvalidVoiceId;
    }
    m_PlayPosition = 0;
#else
    m_State = VoiceState_Stop;

    if ( m_pLowLevelVoice != NULL )
    {
        driver::HardwareManager::GetInstance().GetLowLevelVoiceAllocator().FreeVoice(m_pLowLevelVoice);
        m_pLowLevelVoice = NULL;
    }
#endif
}

void Voice::SetPriority(uint32_t priority) NN_NOEXCEPT
{
    m_Priority = priority;

#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    LowLevelVoiceCommand& cmdmgr = LowLevelVoiceCommand::GetInstance();
    VoiceCommandPriority* command = cmdmgr.AllocCommand<VoiceCommandPriority>( false );
    //@@@ NN_DETAIL_ATK_INFO("__PRIO__\n");
    if ( command == NULL )
    {
        NN_ATK_WARNING("[%s:%d] cannot alloc command", __FUNCTION__, __LINE__);
        return;
    }
    command->id = VoiceCommandId_SetPriority;
    command->voiceId = m_VoiceId;
    command->priority = priority;
    cmdmgr.PushCommand(command);
#else
    if ( m_pLowLevelVoice != NULL )
    {
        m_pLowLevelVoice->SetPriority( priority );
    }
#endif
}

void Voice::SetState( VoiceState state ) NN_NOEXCEPT
{
    m_State = state;

#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if ( state == VoiceState_Play )
    {
        LowLevelVoiceCommand& cmdmgr = LowLevelVoiceCommand::GetInstance();
        VoiceCommandPlay* command = cmdmgr.AllocCommand<VoiceCommandPlay>( false );
        if ( command == NULL )
        {
            NN_ATK_WARNING("[%s:%d] cannot alloc command", __FUNCTION__, __LINE__);
            return;
        }
        command->id = VoiceCommandId_Play;
        command->voiceId = m_VoiceId;
        command->sampleFormat = m_SampleFormat;
        command->sampleRate = m_SampleRate;
        command->adpcmParam = m_AdpcmParam;
        command->pOutputReceiver = m_pOutputReceiver;
        cmdmgr.PushCommand(command);
    }
    else if ( state == VoiceState_Pause )
    {
        LowLevelVoiceCommand& cmdmgr = LowLevelVoiceCommand::GetInstance();
        VoiceCommandPause* command = cmdmgr.AllocCommand<VoiceCommandPause>( false );
        //@@@ NN_DETAIL_ATK_INFO("__PAUSE__\n");
        if ( command == NULL )
        {
            NN_ATK_WARNING("[%s:%d] cannot alloc command", __FUNCTION__, __LINE__);
            return;
        }
        command->id = VoiceCommandId_Pause;
        command->voiceId = m_VoiceId;
        cmdmgr.PushCommand(command);
    }
    else if ( state == VoiceState_Stop )
    {
        LowLevelVoiceCommand& cmdmgr = LowLevelVoiceCommand::GetInstance();
        VoiceCommandFree* command = cmdmgr.AllocCommand<VoiceCommandFree>( false );
        //@@@ NN_DETAIL_ATK_INFO("__FREE__\n");
        if ( command == NULL )
        {
            NN_ATK_WARNING("[%s:%d] cannot alloc command", __FUNCTION__, __LINE__);
            FreeAllWaveBuffer();
            return;
        }
        command->id = VoiceCommandId_Stop;
        command->voiceId = m_VoiceId;
        cmdmgr.PushCommand(command);

        FreeAllWaveBuffer();
    }
#else
    if ( m_pLowLevelVoice != NULL )
    {
        if ( state == VoiceState_Play )
        {
            m_pLowLevelVoice->SetSampleRate( m_SampleRate );
            m_pLowLevelVoice->SetSampleFormat( m_SampleFormat );
            m_pLowLevelVoice->SetAdpcmParam( m_AdpcmParam );
            m_pLowLevelVoice->SetOutputReceiver( m_pOutputReceiver );
        }
        else if ( state == VoiceState_Stop )
        {
            m_pLowLevelVoice->FreeAllWaveBuffer();
        }

        m_pLowLevelVoice->SetState( state );
    }

#endif
} // NOLINT(impl/function_size)

void Voice::AppendWaveBuffer( WaveBuffer* waveBuffer ) NN_NOEXCEPT
{
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    LowLevelVoiceCommand& cmdmgr = LowLevelVoiceCommand::GetInstance();
    VoiceCommandAppendWaveBuffer* command = cmdmgr.AllocCommand<VoiceCommandAppendWaveBuffer>( false );
    //@@@ NN_DETAIL_ATK_INFO("__APPEND__\n");
    if ( command == NULL )
    {
        NN_ATK_WARNING("[%s:%d] cannot alloc command", __FUNCTION__, __LINE__);
        return;
    }
    command->id = VoiceCommandId_AppendWaveBuffer;
    command->voiceId = m_VoiceId;
    command->tag = waveBuffer;
    command->bufferAddress = waveBuffer->bufferAddress;
    command->bufferSize = waveBuffer->bufferSize;
    command->sampleLength = waveBuffer->sampleLength;
    command->sampleOffset = waveBuffer->sampleOffset;
    if ( waveBuffer->pAdpcmContext != NULL )
    {
        command->adpcmContext = *(waveBuffer->pAdpcmContext);
        command->adpcmContextEnable = true;
    }
    else
    {
        command->adpcmContextEnable = false;
    }
    command->loopFlag = waveBuffer->loopFlag;
    cmdmgr.PushCommand(command);

    waveBuffer->next = NULL;
    waveBuffer->status = WaveBuffer::Status_Wait;

    if ( m_WaveBufferListEnd == NULL )
    {
        m_WaveBufferListBegin = waveBuffer;
    }
    else
    {
        m_WaveBufferListEnd->next = waveBuffer;
    }
    m_WaveBufferListEnd = waveBuffer;
#else
    if ( m_pLowLevelVoice != NULL )
    {
        m_pLowLevelVoice->AppendWaveBuffer(waveBuffer);
    }
#endif
}

void Voice::FreeAllWaveBuffer() NN_NOEXCEPT
{
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    WaveBuffer* waveBuffer = m_WaveBufferListBegin;
    while( waveBuffer != NULL ) {
        waveBuffer->status = WaveBuffer::Status_Done;
        waveBuffer = waveBuffer->next;
    }
    m_WaveBufferListBegin = NULL;
    m_WaveBufferListEnd = NULL;
#endif
}

void Voice::UpdateParam() NN_NOEXCEPT
{
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    LowLevelVoiceCommand& cmdmgr = LowLevelVoiceCommand::GetInstance();
    VoiceCommandParam* command = cmdmgr.AllocCommand<VoiceCommandParam>( false );
    //@@@ NN_DETAIL_ATK_INFO("__PARAM__\n");
    if ( command == NULL )
    {
        NN_ATK_WARNING("[%s:%d] cannot alloc command", __FUNCTION__, __LINE__);
        return;
    }
    command->id = VoiceCommandId_UpdateParam;
    command->voiceId = m_VoiceId;
    command->voiceParam = m_VoiceParam;
    cmdmgr.PushCommand(command);
#else
    if ( m_pLowLevelVoice != NULL )
    {
        m_pLowLevelVoice->SetVoiceParam( m_VoiceParam );
    }
#endif
}

position_t Voice::GetPlayPosition() const NN_NOEXCEPT
{
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    return m_PlayPosition;
#else
    if ( m_pLowLevelVoice != NULL )
    {
        return m_pLowLevelVoice->GetPlayPosition();
    }
    return 0;
#endif
}

void Voice::SetMonoFilter( bool enable, uint16_t cutoff ) NN_NOEXCEPT
{
    m_VoiceParam.m_MonoFilterFlag = enable;

    if ( enable )
    {
        // 呼び出し側の MultiVoice::CalcLpf で 16000 未満であることが保証されている
        NN_SDK_ASSERT( cutoff < 16000 );

        m_VoiceParam.m_MonoFilterCutoff = cutoff;
    }
}

void Voice::SetBiquadFilter( bool enable, const BiquadFilterCoefficients* coef ) NN_NOEXCEPT
{
    m_VoiceParam.m_BiquadFilterFlag = enable;

    if ( enable )
    {
        // 呼び出し側の MultiVoice::CalcBiquadFilter で非 NULL であることが保証されている
        NN_SDK_ASSERT_NOT_NULL( coef );

        m_VoiceParam.m_BiquadFilterCoefficients = *coef;
    }
}

#if defined(NN_ATK_CONFIG_ENABLE_VOICE_COMMAND)

void Voice::UpdateVoiceStatus() NN_NOEXCEPT
{
    if ( m_VoiceId == VirtualVoiceManager::InvalidVoiceId ) return;

    if ( ! m_VoiceInfoEnableFlag )
    {
        LowLevelVoiceCommand& cmdmgr = LowLevelVoiceCommand::GetInstance();
        if ( ! cmdmgr.IsFinishCommand( m_CommandTag ) ) return;

        m_VoiceInfoEnableFlag = true;
    }

    const VoiceInfo* voiceInfo = VirtualVoiceManager::GetInstance().GetVoiceInfo( m_VoiceId );
    if ( voiceInfo == NULL ) return;

    if ( voiceInfo->userId != this ) return;

    m_State = voiceInfo->voiceState;

    if ( voiceInfo->voiceState == VoiceState_Play )
    {
        if ( voiceInfo->waveBufferStatus == WaveBuffer::Status_Free )
        {
            FreeAllWaveBuffer();
        }
        else
        {
            WaveBuffer* waveBuffer = m_WaveBufferListBegin;
            if (waveBuffer != NULL)
            {
                while( waveBuffer != NULL && waveBuffer != voiceInfo->waveBufferTag )
                {
                    waveBuffer->status = WaveBuffer::Status_Done;
                    waveBuffer = waveBuffer->next;
                }

                if(waveBuffer != NULL)
                {
                    waveBuffer->status = voiceInfo->waveBufferStatus;
                    m_WaveBufferListBegin = waveBuffer;
                }
                else
                {
                    FreeAllWaveBuffer();
                }
            }
        }

        m_PlayPosition = voiceInfo->playPosition;
    }
}

#endif

//===========================================================================
//
// VirtualVoiceManager
//
//===========================================================================

NN_DEFINE_STATIC_CONSTANT( const uint32_t VirtualVoiceManager::InvalidVoiceId );
NN_DEFINE_STATIC_CONSTANT( const int VirtualVoiceManager::VirtualVoiceCount );
NN_DEFINE_STATIC_CONSTANT( const int VirtualVoiceManager::VirtualVoiceElementCount );

VirtualVoiceManager& VirtualVoiceManager::GetInstance() NN_NOEXCEPT
{
    static VirtualVoiceManager instance;
    return instance;
}

void VirtualVoiceManager::Initialize() NN_NOEXCEPT
{
    m_VoiceInfoTableRead = 0;
    m_VoiceInfoDirtyTable[0].Set();
    m_VoiceInfoDirtyTable[1].Set();

    for(uint32_t i = 0 ; i < VirtualVoiceElementCount; i++ )
    {
        m_VirtualVoiceAllocationTable[i] = 0xffffffff;
    }
    for(uint32_t i = 0 ; i < VirtualVoiceCount; i++ )
    {
        m_LowLevelVoiceTable[i] = NULL;

        m_VoiceInfoTable[0][i].voiceState = VoiceState_Stop;
        m_VoiceInfoTable[1][i].voiceState = VoiceState_Stop;
    }
}


uint32_t VirtualVoiceManager::AllocVirtualVoice() NN_NOEXCEPT
{
    uint32_t disableBit = 0u;
    uint32_t tableIndex = 0u;

    while( tableIndex < VirtualVoiceElementCount )
    {
        const uint32_t x = m_VirtualVoiceAllocationTable[tableIndex] ^ disableBit;
        if( x == 0u )
        {
            //  tableIndex からは確保できないため、次のテーブルに進みます。
            disableBit = 0u;
            tableIndex++;
            continue;
        }

        const uint32_t allocatableBit = nn::util::isols1b( x );
        const uint32_t shift = nn::util::popcount( allocatableBit - 1 );

        //  i * 32 + shift
        //  voiceId は Voice クラスで管理され、返却時に FreeVirtualVoice の引数に与えられます
        const uint32_t voiceId= ( tableIndex << 5 ) | shift;

        if( m_LowLevelVoiceTable[voiceId] != nullptr )
        {
            //  m_LowLevelVoiceTable[voiceId] はまだ解放されていないため、
            //  別のボイスを確保するようにします。
            disableBit |= allocatableBit;
            continue;
        }

        m_VirtualVoiceAllocationTable[tableIndex] ^= allocatableBit;
        return voiceId;
    }
    return InvalidVoiceId;
}

void VirtualVoiceManager::FreeVirtualVoice(uint32_t voiceId) NN_NOEXCEPT
{
    uint32_t i = (voiceId >> 5); // i = voiceId / 32;
    uint32_t shift = (voiceId & 0x1f);
    m_VirtualVoiceAllocationTable[i] |= (1<<shift);
}

void VirtualVoiceManager::UpdateVoiceInfo() NN_NOEXCEPT
{
    uint32_t writeIndex = m_VoiceInfoTableRead ? 0 : 1;

    for(uint32_t i = 0 ; i < VirtualVoiceCount; i++ )
    {
        VoiceInfo* voiceInfo = &m_VoiceInfoTable[writeIndex][i];

        if (m_LowLevelVoiceTable[i] != nullptr)
        {
            m_LowLevelVoiceTable[i]->UpdateVoiceInfo( voiceInfo );
            m_VoiceInfoDirtyTable[writeIndex].Set(i);
        }
        else
        {
            if (m_VoiceInfoDirtyTable[writeIndex].Test(i))
            {
                voiceInfo->voiceState = VoiceState_Stop;
                voiceInfo->waveBufferStatus = WaveBuffer::Status_Free;
                voiceInfo->userId = nullptr;
                m_VoiceInfoDirtyTable[writeIndex].Reset(i);
            }
        }
    }

    m_VoiceInfoTableRead = writeIndex;
}

int VirtualVoiceManager::GetAllocatedVirtualVoiceCount() const NN_NOEXCEPT
{
    int allocatedCount = 0;

    for (int i = 0; i < VirtualVoiceElementCount; i++)
    {
        // AllocationTable は未確保ボイスのビットがオンになっているため反転する
        uint32_t allocatedFlag = ~m_VirtualVoiceAllocationTable[i];

        allocatedCount += nn::util::popcount( allocatedFlag );
    }

    return allocatedCount;
}

int VirtualVoiceManager::GetUnreleasedLowLevelVoiceCount() const NN_NOEXCEPT
{
    int count = 0;
    for (int i = 0; i < VirtualVoiceCount; i++)
    {
        if (m_LowLevelVoiceTable[i] != nullptr)
        {
            count++;
        }
    }

    return count;
}

}}}
