﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/atk_MultiVoiceManager.h>
#include <nn/atk/atk_BasicSoundPlayer.h>
#include <nn/atk/atk_SequenceSoundPlayer.h>
#include <nn/atk/atk_StreamSoundPlayer.h>
#include <nn/atk/atk_WaveSoundPlayer.h>
#include <nn/atk/atk_SequenceTrack.h>
#include <nn/atk/atk_DisposeCallbackManager.h>
#include <nn/atk/atk_MultiVoice.h>
#include <nn/atk/atk_TaskManager.h>
#include <nn/atk/atk_SoundThread.h>



namespace nn {
namespace atk {
namespace detail {

void DriverCommand::ProcessCommandList( Command* commandList ) NN_NOEXCEPT
{
    Command* command = commandList;

    while ( command != NULL )
    {
        switch( command->id )
        {
        case DriverCommandId_Dummy:
            // do nothing
            break;

        case DriverCommandId_Reply:
        {
            DriverCommandReply* c = reinterpret_cast<DriverCommandReply*>(command);
            *(c->ptr) = true;
            break;
        }

        case DriverCommandId_PlayerInit:
        {
            DriverCommandPlayerInit* c =
                reinterpret_cast<DriverCommandPlayerInit*>(command);
            c->player->Initialize( c->pOutputReceiver );
            *(c->availableFlagPtr) = true;
            break;
        }

        case DriverCommandId_PlayerPanmode:
        {
            DriverCommandPlayerPanParam* c =
                reinterpret_cast<DriverCommandPlayerPanParam*>(command);
            c->player->SetPanMode( static_cast<PanMode>(c->panMode) );
            break;
        }
        case DriverCommandId_PlayerPancurve:
        {
            DriverCommandPlayerPanParam* c =
                reinterpret_cast<DriverCommandPlayerPanParam*>(command);
            c->player->SetPanCurve( static_cast<PanCurve>(c->panCurve) );
            break;
        }

        case DriverCommandId_PlayerFinalize:
        {
            DriverCommandPlayer* c = reinterpret_cast<DriverCommandPlayer*>(command);
            c->player->Finalize();
            break;
        }

        case DriverCommandId_PlayerStart:
        {
            DriverCommandPlayer* c = reinterpret_cast<DriverCommandPlayer*>(command);
            c->player->Start();
            break;
        }

        case DriverCommandId_PlayerStop:
        {
            DriverCommandPlayer* c = reinterpret_cast<DriverCommandPlayer*>(command);
            if ( c->flag ) {
                c->player->SetVolume( 0.0f );
            }
            c->player->Stop();
            break;
        }

        case DriverCommandId_PlayerPause:
        {
            DriverCommandPlayer* c = reinterpret_cast<DriverCommandPlayer*>(command);
            c->player->Pause( c->flag );
            break;
        }

        case DriverCommandId_PlayerParam:
        {
            DriverCommandPlayerParam* c = reinterpret_cast<DriverCommandPlayerParam*>(command);
            c->player->SetVolume( c->volume );
            c->player->SetPitch( c->pitch );
            c->player->SetLpfFreq( c->lpfFreq );
            c->player->SetBiquadFilter( c->biquadFilterType, c->biquadFilterValue );
            c->player->SetOutputLine( c->outputLineFlag );
            c->player->SetTvParam( c->tvParam );
            break;
        }

        case DriverCommandId_PlayerAdditionalSend:
        {
            DriverCommandPlayerAdditionalSend* c = reinterpret_cast<DriverCommandPlayerAdditionalSend*>(command);
            c->player->SetTvAdditionalSend( c->bus, c->send );
            break;
        }

        case DriverCommandId_PlayerBusMixVolumeUsed:
        {
            DriverCommandPlayerBusMixVolumeUsed* c = reinterpret_cast<DriverCommandPlayerBusMixVolumeUsed*>(command);
            c->player->SetTvBusMixVolumeUsed( c->isUsed );
            break;
        }

        case DriverCommandId_PlayerBusMixVolume:
        {
            DriverCommandPlayerBusMixVolume* c = reinterpret_cast<DriverCommandPlayerBusMixVolume*>(command);
            c->player->SetTvBusMixVolume( c->tvBusMixVolume );
            break;
        }

        case DriverCommandId_PlayerBusMixVolumeEnabled:
        {
            DriverCommandPlayerBusMixVolumeEnabled* c = reinterpret_cast<DriverCommandPlayerBusMixVolumeEnabled*>(command);
            c->player->SetTvBusMixVolumeEnabled( c->bus, c->isEnabled );
            break;
        }

        case DriverCommandId_PlayerBinaryVolume:
        {
            DriverCommandPlayerBinaryVolume* c = reinterpret_cast<DriverCommandPlayerBinaryVolume*>(command);
            c->player->SetTvBinaryVolume( c->volume );
            break;
        }

        case DriverCommandId_PlayerVolumeThroughModeUsed:
        {
            DriverCommandPlayerVolumeThroughModeUsed* c = reinterpret_cast<DriverCommandPlayerVolumeThroughModeUsed*>(command);
            c->player->SetTvVolumeThroughModeUsed( c->isVolumeThroughModeUsed );
            break;
        }

        case DriverCommandId_PlayerVolumeThroughMode:
        {
            DriverCommandPlayerVolumeThroughMode* c = reinterpret_cast<DriverCommandPlayerVolumeThroughMode*>(command);
            c->player->SetTvVolumeThroughMode( c->bus, c->volumeThroughMode );
            break;
        }

        case DriverCommandId_PlayerClearResourceFlag:
        {
            DriverCommandPlayerClearResourceFlag* c =
                reinterpret_cast<DriverCommandPlayerClearResourceFlag*>(command);
            c->player->ClearIsFinalizedForCannotAllocatedResourceFlag();
            break;
        }

        case DriverCommandId_SeqSetup:
        {
            DriverCommandSequenceSoundSetup* c =
                reinterpret_cast<DriverCommandSequenceSoundSetup*>(command);
            c->player->Setup(c->arg);
            c->player->SetChannelPriority(c->channelPriority);
            c->player->SetReleasePriorityFix(c->isReleasePriorityFix);
            c->player->SetSequenceUserprocCallback(
                reinterpret_cast<SequenceUserProcCallback>(c->userproc),
                c->userprocArg
            );
            break;
        }

        case DriverCommandId_SeqLoad:
        {
            DriverCommandSequenceSoundLoad* c =
                reinterpret_cast<DriverCommandSequenceSoundLoad*>(command);
            c->player->RequestLoad(c->startInfo, c->arg);
            break;
        }

        case DriverCommandId_SeqPrepare:
        {
            DriverCommandSequenceSoundPrepare* c =
                reinterpret_cast<DriverCommandSequenceSoundPrepare*>(command);
            c->player->Prepare(c->arg);
            break;
        }

        case DriverCommandId_SeqSkip:
        {
            DriverCommandSequenceSoundSkip* c =
                reinterpret_cast<DriverCommandSequenceSoundSkip*>(command);
            c->player->Skip(
                driver::SequenceSoundPlayer::StartOffsetType(c->offsetType),
                c->offset
            );
            break;
        }

        case DriverCommandId_SeqTempoRatio:
        {
            DriverCommandSequenceSoundTempoRatio* c =
                reinterpret_cast<DriverCommandSequenceSoundTempoRatio*>(command);
            c->player->SetTempoRatio(
                c->tempoRatio
            );
            break;
        }

        case DriverCommandId_SeqChannelPrio:
        {
            DriverCommandSequenceSoundChannelPrio* c =
                reinterpret_cast<DriverCommandSequenceSoundChannelPrio*>(command);
            c->player->SetChannelPriority(
                c->priority
            );
            break;
        }

        case DriverCommandId_SeqSetVar:
        {
            DriverCommandSequenceSoundSetVar* c =
                reinterpret_cast<DriverCommandSequenceSoundSetVar*>(command);
            c->player->SetLocalVariable(
                c->varNo,
                static_cast<int16_t>(c->var)
            );
            break;
        }

        case DriverCommandId_SeqSetGvar:
        {
            DriverCommandSequenceSoundSetVar* c =
                reinterpret_cast<DriverCommandSequenceSoundSetVar*>(command);
            driver::SequenceSoundPlayer::SetGlobalVariable(
                c->varNo,
                static_cast<int16_t>(c->var)
            );
            break;
        }

        case DriverCommandId_SeqSetTvar:
        {
            DriverCommandSequenceSoundSetVar* c =
                reinterpret_cast<DriverCommandSequenceSoundSetVar*>(command);
            driver::SequenceTrack* track = c->player->GetPlayerTrack( c->trackNo );
            if ( track != NULL ) {
                track->SetTrackVariable( c->varNo, static_cast<int16_t>(c->var) );
            }
            break;
        }

        case DriverCommandId_SeqTrackMute:
        {
            DriverCommandSequenceSoundTrackMute* c = reinterpret_cast<DriverCommandSequenceSoundTrackMute*>(command);
            c->player->SetTrackMute(
                c->trackBitFlag,
                SequenceMute(c->mute)
            );
            break;
        }

        case DriverCommandId_SeqTrackSilence:
        {
            DriverCommandSequenceSoundTrackSilence* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackSilence*>(command);
            c->player->SetTrackSilence(
                c->trackBitFlag,
                c->silenceFlag,
                c->fadeFrames
            );
            break;
        }

        case DriverCommandId_SeqTrackVolume:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackVolume(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DriverCommandId_SeqTrackPitch:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackPitch(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DriverCommandId_SeqTrackLpf:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackLpfFreq(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DriverCommandId_SeqTrackBiquad:
        {
            DriverCommandSequenceSoundTrackBiquad* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackBiquad*>(command);
            c->player->SetTrackBiquadFilter(
                c->trackBitFlag,
                c->type,
                c->value
            );
            break;
        }

        case DriverCommandId_SeqTrackBankIndex:
        {
            DriverCommandSequenceSoundTrackBankIndex* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackBankIndex*>(command);
            c->player->SetTrackBankIndex(
                c->trackBitFlag,
                c->bankIndex
            );
            break;
        }

        case DriverCommandId_SeqTrackTranspose:
        {
            DriverCommandSequenceSoundTrackTranspose* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackTranspose*>(command);
            c->player->SetTrackTranspose(
                    c->trackBitFlag,
                    c->transpose
            );
            break;
        }

        case DriverCommandId_SeqTrackVelocityRange:
        {
            DriverCommandSequenceSoundTrackVelocityRange* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackVelocityRange*>(command);
            c->player->SetTrackVelocityRange(
                    c->trackBitFlag,
                    c->range
            );
            break;
        }

        case DriverCommandId_SeqTrackOutputline:
        {
            DriverCommandSequenceSoundTrackOutputLine* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackOutputLine*>(command);
            c->player->SetTrackOutputLine(
                    c->trackBitFlag,
                    c->outputLine
            );
            break;
        }

        case DriverCommandId_SeqTrackOutputlineReset:
        {
            DriverCommandSequenceSoundTrackOutputLine* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackOutputLine*>(command);
            c->player->ResetTrackOutputLine(
                    c->trackBitFlag
            );
            break;
        }

        case DriverCommandId_SeqTrackTvVolume:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackTvVolume(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DriverCommandId_SeqTrackTvMixParameter:
        {
            DriverCommandSequenceSoundTrackMixParameter* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackMixParameter*>(command);
            c->player->SetTrackChannelTvMixParameter(
                c->trackBitFlag,
                c->srcChNo,
                c->param
                );
            break;
        }

        case DriverCommandId_SeqTrackTvPan:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackTvPan(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DriverCommandId_SeqTrackTvSpan:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackTvSurroundPan(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DriverCommandId_SeqTrackTvMainSend:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackTvMainSend(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DriverCommandId_SeqTrackTvFxSend:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackTvFxSend(
                c->trackBitFlag,
                static_cast<AuxBus>(c->uint32Value),
                c->value
            );
            break;
        }

        case DriverCommandId_WsdPrepare:
        {
            DriverCommandWaveSoundPrepare* c =
                reinterpret_cast<DriverCommandWaveSoundPrepare*>(command);
            c->player->Prepare(c->startInfo, c->arg);
            break;
        }

        case DriverCommandId_WsdLoad:
        {
            DriverCommandWaveSoundLoad* c =
                reinterpret_cast<DriverCommandWaveSoundLoad*>(command);
            c->player->RequestLoad(c->startInfo, c->arg);
            break;
        }

        case DriverCommandId_WsdChannelPrio:
        {
            DriverCommandWaveSoundChannelPrio* c =
                reinterpret_cast<DriverCommandWaveSoundChannelPrio*>(command);
            c->player->SetChannelPriority(c->priority);
            break;
        }

        case DriverCommandId_WsdChannelParam:
        {
            DriverCommandWaveSoundChannelParam* c =
                reinterpret_cast<DriverCommandWaveSoundChannelParam*>(command);
            c->player->SetChannelPriority(c->priority);
            c->player->SetReleasePriorityFix(c->isReleasePriorityFix);
            break;
        }

        case DriverCommandId_AwsdPrepare:
        {
            DriverCommandAdvancedWaveSoundPrepare* c =
                reinterpret_cast<DriverCommandAdvancedWaveSoundPrepare*>(command);
            c->player->Prepare(c->parameter);
            break;
        }

        case DriverCommandId_StrmSetup:
        {
            DriverCommandStreamSoundSetup* c =
                reinterpret_cast<DriverCommandStreamSoundSetup*>(command);
            c->player->Setup(c->arg);
            break;
        }

        case DriverCommandId_StrmPrepare:
        {
            DriverCommandStreamSoundPrepare* c =
                reinterpret_cast<DriverCommandStreamSoundPrepare*>(command);
            c->player->Prepare(c->arg);
            break;
        }

        case DriverCommandId_StrmPreparePrefetch:
        {
            DriverCommandStreamSoundPreparePrefetch* c =
                reinterpret_cast<DriverCommandStreamSoundPreparePrefetch*>(command);
            c->player->PreparePrefetch(c->arg);
            break;
        }

        case DriverCommandId_StrmLoadHeader:
        {
            DriverCommandStreamSoundLoadHeader* c = reinterpret_cast<DriverCommandStreamSoundLoadHeader*>(command);
            c->player->LoadHeader(
                c->result,
                c->adpcmParam,
                c->assignNumber
            );
            break;
        }

        case DriverCommandId_StrmLoadData:
        {
            DriverCommandStreamSoundLoadData* c = reinterpret_cast<DriverCommandStreamSoundLoadData*>(command);
            c->player->LoadStreamData(
                c->result,
                c->loadDataParam,
                c->assignNumber
            );
            break;
        }

        case DriverCommandId_StrmForceFinish:
        {
            DriverCommandStreamSoundForceFinish* c = reinterpret_cast<DriverCommandStreamSoundForceFinish*>(command);
            c->player->ForceFinish();
            break;
        }


        case DriverCommandId_StrmTrackVolume:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackVolume(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DriverCommandId_StrmTrackInitialVolume:
        {
            DriverCommandStreamSoundTrackInitialVolume* c =
                reinterpret_cast<DriverCommandStreamSoundTrackInitialVolume*>(command);
            c->player->SetTrackInitialVolume(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DriverCommandId_StrmTrackOutputline:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackOutputLine(
                c->trackBitFlag,
                c->uint32Value
            );
            break;
        }

        case DriverCommandId_StrmTrackOutputlineReset:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->ResetTrackOutputLine(
                c->trackBitFlag
            );
            break;
        }

        case DriverCommandId_StrmTrackTvVolume:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackTvVolume(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DriverCommandId_StrmTrackTvMixParameter:
        {
            DriverCommandStreamSoundTrackMixParameter* c =
                reinterpret_cast<DriverCommandStreamSoundTrackMixParameter*>(command);
            c->player->SetTrackChannelTvMixParameter(
                c->trackBitFlag,
                c->srcChNo,
                c->param
                );
            break;
        }

        case DriverCommandId_StrmTrackTvPan:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackTvPan(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DriverCommandId_StrmTrackTvSpan:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackTvSurroundPan(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DriverCommandId_StrmTrackTvMainSend:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackTvMainSend(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DriverCommandId_StrmTrackTvFxSend:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackTvFxSend(
                c->trackBitFlag,
                static_cast<AuxBus>(c->uint32Value),
                c->value
            );
            break;
        }

        case DriverCommandId_InvalidateData:
        {
            DriverCommandInvalidateData* c =
                reinterpret_cast<DriverCommandInvalidateData*>(command);
            driver::DisposeCallbackManager::GetInstance().Dispose(
                c->mem,
                c->size
            );
            break;
        }

        case DriverCommandId_RegistDisposeCallback:
        {
            DriverCommandDisposeCallback* c =
                reinterpret_cast<DriverCommandDisposeCallback*>(command);
            driver::DisposeCallbackManager::GetInstance().RegisterDisposeCallback(
                c->callback
            );
            break;
        }

        case DriverCommandId_UnregistDisposeCallback:
        {
            DriverCommandDisposeCallback* c =
                reinterpret_cast<DriverCommandDisposeCallback*>(command);
            driver::DisposeCallbackManager::GetInstance().UnregisterDisposeCallback(
                c->callback
            );
            break;
        }

        case DriverCommandId_AppendEffect:
        {
            DriverCommandEffect* c = reinterpret_cast<DriverCommandEffect*>(command);
            c->pOutputMixer->AppendEffectImpl( c->effect, c->bus, c->effectBuffer, c->effectBufferSize );
            break;
        }

        case DriverCommandId_AppendEffectAux:
        {
            DriverCommandEffectAux* c = reinterpret_cast<DriverCommandEffectAux*>(command);
            c->pOutputMixer->AppendEffectImpl( c->effect, c->bus, c->effectBuffer, c->effectBufferSize );
            break;
        }

        case DriverCommandId_RemoveEffect:
        {
            DriverCommandEffect* c = reinterpret_cast<DriverCommandEffect*>(command);
            c->pOutputMixer->RemoveEffectImpl( c->effect, c->bus );
            break;
        }

        case DriverCommandId_RemoveEffectAux:
        {
            DriverCommandEffectAux* c = reinterpret_cast<DriverCommandEffectAux*>(command);
            c->pOutputMixer->RemoveEffectImpl( c->effect, c->bus );
            break;
        }

        case DriverCommandId_ClearEffect:
        {
            DriverCommandEffect* c = reinterpret_cast<DriverCommandEffect*>(command);
            c->pOutputMixer->ClearEffectImpl( c->bus );
            break;
        }

        case DriverCommandId_SubMixApplyDestination:
        {
            DriverCommandSubMixApplyDestination* c = reinterpret_cast<DriverCommandSubMixApplyDestination*>(command);
            switch( c->pReceiver->GetReceiverType() )
            {
                case OutputReceiver::ReceiverType::ReceiverType_SubMix:
                {
                    SubMix* pSubMix = reinterpret_cast<SubMix*>( c->pReceiver );
                    pSubMix->ApplyDestination();
                    pSubMix->AddReferenceCount( -1 );
                    break;
                }
                default:
                    NN_UNEXPECTED_DEFAULT;
            }
            break;
        }

        case DriverCommandId_SubMixUpdateMixVolume:
        {
            DriverCommandSubMixUpdateMixVolume* c = reinterpret_cast<DriverCommandSubMixUpdateMixVolume*>(command);
            switch( c->pReceiver->GetReceiverType() )
            {
                case OutputReceiver::ReceiverType::ReceiverType_SubMix:
                {
                    SubMix* pSubMix = reinterpret_cast<SubMix*>( c->pReceiver );
                    pSubMix->UpdateMixVolume( c->srcBus, c->srcChannel, c->dstBus, c->dstChannel );
                    pSubMix->AddReferenceCount( -1 );
                    break;
                }
                default:
                    NN_UNEXPECTED_DEFAULT;
            }
            break;
        }

        case DriverCommandId_AuxBusVolume:
        {
            DriverCommandAuxBusVolume* c = reinterpret_cast<DriverCommandAuxBusVolume*>(command);
            detail::driver::HardwareManager::GetInstance().SetAuxBusVolume(
                c->bus,
                c->volume,
                c->fadeFrames,
                c->subMixIndex
            );
            break;
        }

        case DriverCommandId_Debug:
        {
        #if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP) // Dump 用途
            DriverCommandDebug* c = reinterpret_cast<DriverCommandDebug*>(command);
            NN_DETAIL_ATK_INFO("[CMD] %s\n", c->str );
        #endif // NN_SDK_BUILD_DEBUG || NN_SDK_BUILD_DEVELOP
            break;
        }

        case DriverCommandId_AllVoicesSync:
        {
            DriverCommandAllVoicesSync* c =
                reinterpret_cast<DriverCommandAllVoicesSync*>(command);
            driver::MultiVoiceManager::GetInstance().UpdateAllVoicesSync( c->syncFlag );
            break;
        }

        case DriverCommandId_VoicePlay:
        {
            DriverCommandVoicePlay* c = reinterpret_cast<DriverCommandVoicePlay*>(command);
            switch ( c->state )
            {
            case DriverCommandVoicePlay::State_Start:
                c->voice->Start();
                break;
            case DriverCommandVoicePlay::State_Stop:
                c->voice->Stop();
                c->voice->Free();
                break;
            case DriverCommandVoicePlay::State_PauseOn:
                c->voice->Pause( true );
                break;
            case DriverCommandVoicePlay::State_PauseOff:
                c->voice->Pause( false );
                break;
            default:
                break;
            }
            break;
        }
        case DriverCommandId_VoiceWaveInfo:
        {
            DriverCommandVoiceWaveInfo* c =
                reinterpret_cast<DriverCommandVoiceWaveInfo*>(command);
            driver::MultiVoice* voice = c->voice;
            voice->SetSampleFormat( c->format );
            voice->SetSampleRate( c->sampleRate );
            voice->SetInterpolationType( static_cast<uint8_t>(c->interpolationType) );
            break;
        }
        case DriverCommandId_VoiceAdpcmParam:
        {
            DriverCommandVoiceAdpcmParam* c =
                reinterpret_cast<DriverCommandVoiceAdpcmParam*>(command);
            c->voice->SetAdpcmParam( c->channel, *c->param );
            break;
        }
#if 0
        case DriverCommandVoiceAppendWavebuffer:
        {
            DriverCommandVoiceAppendWaveBuffer* c =
                reinterpret_cast<DriverCommandVoiceAppendWaveBuffer*>(command);
            c->voice->AppendWaveBuffer(
                c->channel,
                c->buffer,
                c->lastFlag );
            break;
        }
#endif

#if 0
        case DriverCommandId_VoicePriority:
        {
            DriverCommandVoicePriority* c =
                reinterpret_cast<DriverCommandVoicePriority*>(command);
            c->voice->SetPriority( c->priority );
            break;
        }
        case DriverCommandId_VoiceVolume:
        {
            DriverCommandVoiceVolume* c = reinterpret_cast<DriverCommandVoiceVolume*>(command);
            c->voice->SetVolume( c->volume );
            break;
        }
        case DriverCommandVoicePitch:
        {
            DriverCommandVoicePitch* c = reinterpret_cast<DriverCommandVoicePitch*>(command);
            c->voice->SetPitch( c->pitch );
            break;
        }
        case DriverCommandVoicePanMode:
        {
            DriverCommandVoicePanMode* c =
                reinterpret_cast<DriverCommandVoicePanMode*>(command);
            c->voice->SetPanMode( c->mode );
            break;
        }
        case DriverCommandVoicePanCurve:
        {
            DriverCommandVoicePanCurve* c =
                reinterpret_cast<DriverCommandVoicePanCurve*>(command);
            c->voice->SetPanCurve( c->curve );
            break;
        }
        case DriverCommandVoicePan:
        {
            DriverCommandVoicePan* c = reinterpret_cast<DriverCommandVoicePan*>(command);
            c->voice->SetPan( c->pan );
            break;
        }
        case DriverCommandVoiceSpan:
        {
            DriverCommandVoiceSurroundPan* c =
                reinterpret_cast<DriverCommandVoiceSurroundPan*>(command);
            c->voice->SetSurroundPan( c->span );
            break;
        }
        case DriverCommandVoiceLpf:
        {
            DriverCommandVoiceLpfFreq* c =
                reinterpret_cast<DriverCommandVoiceLpfFreq*>(command);
            c->voice->SetLpfFreq( c->lpfFreq );
            break;
        }
        case DriverCommandVoiceBiquad:
        {
            DriverCommandVoiceBiquadFilter* c =
                reinterpret_cast<DriverCommandVoiceBiquadFilter*>(command);
            c->voice->SetBiquadFilter( c->type, c->value );
            break;
        }
        case DriverCommandVoiceOutputLine:
        {
            DriverCommandVoiceOutputLine* c =
                reinterpret_cast<DriverCommandVoiceOutputLine*>(command);
            c->voice->SetOutputLine( c->lineFlag );
            break;
        }
        case DriverCommandVoiceMainOutVolume:
        {
            DriverCommandVoiceMainOutVolume* c =
                reinterpret_cast<DriverCommandVoiceMainOutVolume*>(command);
            c->voice->SetMainOutVolume( c->volume );
            break;
        }
        case DriverCommandVoiceMainSend:
        {
            DriverCommandVoiceMainSend* c =
                reinterpret_cast<DriverCommandVoiceMainSend*>(command);
            c->voice->SetMainSend( c->send );
            break;
        }
        case DriverCommandVoiceFxSend:
        {
            DriverCommandVoiceFxSend* c = reinterpret_cast<DriverCommandVoiceFxSend*>(command);
            c->voice->SetFxSend( c->bus, c->send );
            break;
        }
#endif
        default:
            break;

        }   // switch

        command = command->next;

    } // while
} // NOLINT(impl/function_size)

DriverCommand& DriverCommand::GetInstance() NN_NOEXCEPT
{
    static DriverCommand instance;
    return instance;
}

DriverCommand& DriverCommand::GetInstanceForTaskThread() NN_NOEXCEPT
{
    static DriverCommand instance;
    return instance;
}

DriverCommand::DriverCommand() NN_NOEXCEPT
{
}

size_t DriverCommand::GetRequiredMemSize(size_t commandBufferSize, int queueCount) NN_NOEXCEPT
{
    return CommandManager::GetRequiredMemSize(commandBufferSize, queueCount);
}

void DriverCommand::Initialize( void* buffer, size_t bufferSize, size_t commandBufferSize, int queueCount ) NN_NOEXCEPT
{
    CommandManager::Initialize( buffer, bufferSize, commandBufferSize, queueCount, ProcessCommandList );
    CommandManager::SetRequestProcessCommandFunc( RequestProcessCommand );
}

void DriverCommand::RequestProcessCommand() NN_NOEXCEPT
{
    driver::SoundThread::GetInstance().ForceWakeup();
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
