﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_StreamSoundRuntime.h>

#include <nn/atk/atk_SoundArchivePlayer.h>
#include <nn/atk/atk_SoundDataManager.h>
#include <nn/atk/atk_SoundRuntimeUtility.h>
#include <nn/atk/atk_StartInfoReader.h>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/fnd/string/atkfnd_String.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>

namespace
{
    const int MemAlignSize = nn::audio::BufferAlignSize;
}

namespace nn { namespace atk { namespace detail {

    NN_DEFINE_STATIC_CONSTANT( const int StreamSoundRuntime::DefaultStreamBlockCount );

    StreamSoundRuntime::StreamSoundRuntime() NN_NOEXCEPT
    : m_pSoundArchiveFilesHook(NULL),
      m_StreamBlockCount(DefaultStreamBlockCount)
    {
    }

    StreamSoundRuntime::~StreamSoundRuntime() NN_NOEXCEPT
    {
    }

    bool StreamSoundRuntime::Initialize(
        int soundCount,
        void** pOutAllocatedAddr,
        const void* endAddr,
        void* streamInstanceBuffer,
        size_t streamInstanceBufferSize
    ) NN_NOEXCEPT
    {
        NN_UNUSED( streamInstanceBufferSize );
        NN_SDK_ASSERT_NOT_NULL( streamInstanceBuffer );

        // nn::atk::detail::StreamSound が NN_AUDIO_ALIGNAS_BUFFER_ALIGN のメンバ m_PlayerInstance を持つためアラインが必要です
        NN_SDK_ASSERT( util::is_aligned(reinterpret_cast<uintptr_t>(streamInstanceBuffer), nn::audio::BufferAlignSize) );
        if ( !util::is_aligned(reinterpret_cast<uintptr_t>(streamInstanceBuffer), nn::audio::BufferAlignSize) )
        {
            return false;
        }

        // StreamSoundInstanceManager
        {
            const size_t requireSize = GetRequiredStreamInstanceSize( soundCount );
            NN_SDK_ASSERT_GREATER_EQUAL( streamInstanceBufferSize, requireSize );

            const uint32_t createdSoundCount = m_StreamSoundInstanceManager.Create( streamInstanceBuffer, requireSize, nn::atk::SoundSystem::GetSoundInstanceConfig() );
#if defined(NN_SDK_BUILD_RELEASE)
            NN_UNUSED( createdSoundCount );
#endif
            NN_SDK_ASSERT( static_cast<int32_t>(createdSoundCount) == soundCount );
        }

        NN_SDK_ASSERT( util::is_aligned(reinterpret_cast<uintptr_t>(*pOutAllocatedAddr), MemAlignSize) );
        if ( !util::is_aligned(reinterpret_cast<uintptr_t>(*pOutAllocatedAddr), MemAlignSize) )
        {
            return false;
        }

        // StreamSoundLoaderManager
        {
            size_t requireSize = nn::util::align_up(
                detail::driver::StreamSoundLoaderManager::GetRequiredMemSize(soundCount), MemAlignSize );

            void* estimateEndAddr = util::BytePtr( *pOutAllocatedAddr, requireSize ).Get();
            if ( util::ConstBytePtr(endAddr).Distance(estimateEndAddr) > 0 )
            {
                return false;
            }

            uint32_t createdSoundLoaderCount = m_StreamSoundLoaderManager.Create( *pOutAllocatedAddr, requireSize );
#if defined(NN_SDK_BUILD_RELEASE)
            NN_UNUSED( createdSoundLoaderCount );
#endif
            NN_SDK_ASSERT( static_cast<int32_t>(createdSoundLoaderCount) == soundCount,
                    "StreamSoundLoaderManager::Create createdSoundLoaderCount(%d) but soundCount(%d)",
                    createdSoundLoaderCount, soundCount);
            *pOutAllocatedAddr = estimateEndAddr;
        }

        // 各 StreamSound に StreamSoundLoaderManager をセット
        detail::StreamSoundInstanceManager::PriorityList& list =
            m_StreamSoundInstanceManager.GetFreeList();
        for (detail::StreamSoundInstanceManager::PriorityList::iterator itr = list.begin();
                itr != list.end();
                ++itr)
        {
            itr->SetLoaderManager(m_StreamSoundLoaderManager);
        }
        detail::driver::SoundThread::GetInstance().RegisterSoundFrameCallback(&m_StreamSoundLoaderManager);

        return true;
    }

    bool StreamSoundRuntime::SetupStreamBuffer(
        const SoundArchive* pSoundArchive,
        void* strmBuffer,
        size_t strmBufferSize
    ) NN_NOEXCEPT
    {
        return SetupStreamBuffer( pSoundArchive, strmBuffer, strmBufferSize, &m_StreamBufferPool );
    }

    bool StreamSoundRuntime::SetupStreamBuffer(
        const SoundArchive* pSoundArchive,
        void* strmBuffer,
        size_t strmBufferSize,
        detail::driver::StreamBufferPool* pStreamBufferPool
    ) NN_NOEXCEPT
    {

        if ( strmBufferSize > 0 )
        {
            NN_SDK_ASSERT_NOT_NULL( strmBuffer );
        }
        NN_SDK_ASSERT_GREATER_EQUAL( strmBufferSize, GetRequiredStreamBufferSize( pSoundArchive ) );
        if (strmBufferSize < GetRequiredStreamBufferSize( pSoundArchive ) * GetRequiredStreamBufferTimes( pSoundArchive ))
        {
            NN_ATK_WARNING("StreamBuffer is not enough for prefetch data, Please figure in StreamBufferTimes.");
        }

        // ストリームバッファは nn::atk::SoundArchivePlayer::StreamBufferTimesMax 倍まで
        NN_SDK_ASSERT_LESS_EQUAL( strmBufferSize, GetRequiredStreamBufferSize( pSoundArchive ) * nn::atk::SoundArchivePlayer::StreamBufferTimesMax );

        if ( strmBufferSize < GetRequiredStreamBufferSize( pSoundArchive ) )
        {
            return false;
        }

        int strmChannelCount = 0;
        SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
        if ( pSoundArchive->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
        {
            strmChannelCount = soundArchivePlayerInfo.streamChannelCount;
        }

        pStreamBufferPool->Initialize( strmBuffer, strmBufferSize, strmChannelCount );

        return true;
    }

    bool StreamSoundRuntime::SetupStreamCacheBuffer(
        const SoundArchive* pSoundArchive,
        void* streamCacheBuffer,
        size_t streamCacheSize
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_ALIGNED( streamCacheBuffer, detail::fnd::FileStream::BufferAlignSize);

        // サウンドアーカイブから同時再生可能なストリームサウンドの数を取得
        int strmSoundCount = 0;
        SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
        if ( pSoundArchive->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
        {
            strmSoundCount = soundArchivePlayerInfo.streamSoundCount;
        }
        else
        {
            return false;
        }

        // すべての StreamSound にキャッシュをセット
        size_t cacheSize = streamCacheSize / strmSoundCount;
        NN_SDK_ASSERT_ALIGNED( cacheSize, detail::fnd::FileStream::BufferAlignSize );

        void* cache = streamCacheBuffer;

        detail::StreamSoundInstanceManager::PriorityList& list =
            m_StreamSoundInstanceManager.GetFreeList();
        for (detail::StreamSoundInstanceManager::PriorityList::iterator itr = list.begin();
             itr != list.end();
             ++itr)
        {
            NN_SDK_ASSERT_ALIGNED( cache, detail::fnd::FileStream::BufferAlignSize );
            itr->SetCacheBuffer(cache, cacheSize);

            cache = util::BytePtr(cache, cacheSize).Get();
        }

        return true;
    }

    void StreamSoundRuntime::Finalize() NN_NOEXCEPT
    {
        m_StreamBufferPool.Finalize();
        m_StreamSoundInstanceManager.Destroy();
        m_StreamSoundLoaderManager.Destroy();
        detail::driver::SoundThread::GetInstance().UnregisterSoundFrameCallback(&m_StreamSoundLoaderManager);
    }

    size_t StreamSoundRuntime::GetRequiredMemorySize(
        const SoundArchive::SoundArchivePlayerInfo& soundArchivePlayerInfo,
        int alignment
    ) NN_NOEXCEPT
    {
        size_t size = 0;

        size += nn::util::align_up(
            StreamSoundInstanceManager::GetRequiredMemSize(soundArchivePlayerInfo.streamSoundCount, nn::atk::SoundSystem::GetSoundInstanceConfig()),
            nn::audio::MemoryPoolType::SizeGranularity );
        // NN_ATK_INIT_LOG("[STRM] instance:max(%2d) ALIGN(%d) => %8d\n",
        //         soundArchivePlayerInfo.streamSoundCount, nn::audio::MemoryPoolType::SizeGranularity, size);

        size += nn::util::align_up(
            detail::driver::StreamSoundLoaderManager::GetRequiredMemSize(soundArchivePlayerInfo.streamSoundCount),
            alignment );
        // NN_ATK_INIT_LOG("[STRM] loader:   max(%2d) ALIGN(%d) => %8d\n",
        //         soundArchivePlayerInfo.streamSoundCount, alignment, size);

        return size;
    }

    size_t StreamSoundRuntime::GetRequiredStreamBufferSize(
        const SoundArchive* soundArchive
    ) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( soundArchive );

        int strmChannelCount = 0;

        SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
        if ( soundArchive->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
        {
            strmChannelCount = soundArchivePlayerInfo.streamChannelCount;
        }

        size_t memSize = static_cast<size_t>(
            detail::driver::StreamSoundLoader::DataBlockSizeMax
            * m_StreamBlockCount
            * strmChannelCount
        );
        return memSize;
    }

    int32_t StreamSoundRuntime::GetRequiredStreamBufferTimes(
        const SoundArchive* soundArchive
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( soundArchive );

        int strmBufferTimes  = 1;

        SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
        if ( soundArchive->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
        {
            strmBufferTimes  = soundArchivePlayerInfo.streamBufferTimes;
        }

        return strmBufferTimes;
    }

    size_t StreamSoundRuntime::GetRequiredStreamCacheSize(
        const SoundArchive* soundArchive,
        size_t cacheSizePerSound
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(soundArchive);

        int strmSoundCount = 0;

        SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
        if ( soundArchive->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
        {
            strmSoundCount = soundArchivePlayerInfo.streamSoundCount;
        }

        size_t memSize = strmSoundCount * cacheSizePerSound;
        return memSize;
    }

    size_t StreamSoundRuntime::GetRequiredStreamInstanceSize(
        int soundCount
    ) NN_NOEXCEPT
    {
        return nn::util::align_up( StreamSoundInstanceManager::GetRequiredMemSize(soundCount, nn::atk::SoundSystem::GetSoundInstanceConfig()), nn::audio::BufferAlignSize );
    }

    int StreamSoundRuntime::GetActiveCount() const NN_NOEXCEPT
    {
        return m_StreamSoundInstanceManager.GetActiveCount();
    }

    int StreamSoundRuntime::GetActiveChannelCount() const NN_NOEXCEPT
    {
        int count = 0;

        auto& activeList = m_StreamSoundInstanceManager.GetSoundList();
        for ( auto iterator = activeList.begin(); iterator != activeList.end(); ++iterator )
        {
            count += iterator->GetActiveChannelCount();
        }

        return count;
    }

    int StreamSoundRuntime::GetActiveTrackCount() const NN_NOEXCEPT
    {
        int count = 0;

        auto& activeList = m_StreamSoundInstanceManager.GetSoundList();
        for ( auto iterator = activeList.begin(); iterator != activeList.end(); ++iterator )
        {
            count += iterator->GetActiveTrackCount();
        }

        return count;
    }

    int StreamSoundRuntime::GetFreeCount() const NN_NOEXCEPT
    {
        return m_StreamSoundInstanceManager.GetFreeCount();
    }

    void StreamSoundRuntime::SetupUserParam(void** pOutAllocatedAddr, size_t adjustSize) NN_NOEXCEPT
    {
        void* curAddr = *pOutAllocatedAddr;

        detail::StreamSoundInstanceManager::PriorityList& list = m_StreamSoundInstanceManager.GetFreeList();
        for (detail::StreamSoundInstanceManager::PriorityList::iterator itr = list.begin();
                itr != list.end();
                ++itr)
        {
            itr->SetUserParamBuffer(curAddr, adjustSize);

            curAddr = util::BytePtr(curAddr, adjustSize).Get();
        }

        *pOutAllocatedAddr = curAddr;
    }

    void StreamSoundRuntime::Update() NN_NOEXCEPT
    {
        m_StreamSoundInstanceManager.SortPriorityList();
    }

    StreamSound* StreamSoundRuntime::AllocSound(
        SoundArchive::ItemId soundId,
        int priority,
        int ambientPriority,
        detail::BasicSound::AmbientInfo* ambientArgInfo,
        OutputReceiver* pOutputReceiver
    ) NN_NOEXCEPT
    {
        return SoundRuntimeUtility::AllocSound<StreamSound>(
            &m_StreamSoundInstanceManager,
            soundId,
            priority,
            ambientPriority,
            ambientArgInfo,
            pOutputReceiver
        );
    }

    SoundStartable::StartResult StreamSoundRuntime::PrepareImpl(
        const SoundArchive* pSoundArchive,
        const SoundDataManager* pSoundDataManager,
        SoundArchive::ItemId soundId,
        detail::StreamSound* sound,
        const SoundArchive::SoundInfo* commonInfo,
        const StartInfoReader& startInfoReader
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( sound );
        NN_SDK_ASSERT_NOT_NULL( commonInfo );
        NN_SDK_ASSERT_NOT_NULL( pSoundArchive );
        NN_SDK_ASSERT_NOT_NULL( pSoundDataManager );

        const SoundArchive::StreamSoundInfo* pStrmMetaInfo = startInfoReader.GetStrmMetaInfo();
        const SoundArchive::StreamSoundInfo2* pStrmMetaInfo2 = startInfoReader.GetStrmMetaInfo2();

        // ストリームサウンド情報の取得
        SoundArchive::StreamSoundInfo info;
        if ( pStrmMetaInfo != NULL )
        {
            info = *pStrmMetaInfo;
        }
        else
        {
            if ( ! pSoundArchive->ReadStreamSoundInfo( &info, soundId ) ) {
                SoundStartable::StartResult
                    result( SoundStartable::StartResult::ResultCode_ErrorInvalidSoundId );
                return result;
            }
        }

        const SoundArchive::StreamSoundInfo2* pInfo2 = nullptr;
        SoundArchive::StreamSoundInfo2 info2;
        if ( pStrmMetaInfo2 != nullptr )
        {
            info2 = *pStrmMetaInfo2;
            pInfo2 = &info2;
        }
        else
        {
            if (info.streamFileType == SoundArchive::StreamFileType_Opus)
            {
                if ( ! pSoundArchive->detail_ReadStreamSoundInfo2( soundId, &info2 ) )
                {
                    SoundStartable::StartResult
                        result( SoundStartable::StartResult::ResultCode_ErrorInvalidSoundId );
                    return result;
                }
                pInfo2 = &info2;
            }
        }


        // プレイヤーのセットアップ
        {
            detail::driver::StreamSoundPlayer::SetupArg arg;

            // 再生失敗時のケア無し
            switch(info.streamFileType){
            case SoundArchive::StreamFileType_NwStreamBinary:
                arg.fileType = detail::StreamFileType_Bfstm;
                break;
            case SoundArchive::StreamFileType_Opus:
                arg.fileType = detail::StreamFileType_Opus;
                break;
            default:
                SoundStartable::StartResult result(
                    SoundStartable::StartResult::ResultCode_ErrorUnknown );
                return result;
            }

            switch (info.decodeMode) {
            case SoundArchive::DecodeMode_Default:
                arg.decodeMode = detail::DecodeMode_Default;
                break;
            case SoundArchive::DecodeMode_Cpu:
                arg.decodeMode = detail::DecodeMode_Cpu;
                break;
            case SoundArchive::DecodeMode_Accelerator:
                arg.decodeMode = detail::DecodeMode_Accelerator;
                break;
            default:
                SoundStartable::StartResult result(
                    SoundStartable::StartResult::ResultCode_ErrorUnknown);
                return result;
            }

            for (int i = 0; i < detail::StreamTrackCount; i++)
            {
                arg.trackInfos.track[i].volume = info.trackInfo[i].volume;
                arg.trackInfos.track[i].pan = info.trackInfo[i].pan;
                arg.trackInfos.track[i].span = info.trackInfo[i].surroundPan;
                arg.trackInfos.track[i].flags = info.trackInfo[i].flags;
                arg.trackInfos.track[i].mainSend = info.trackInfo[i].mainSend;
                for (int j = 0; j < AuxBus_Count; j++)
                {
                    arg.trackInfos.track[i].fxSend[j] = info.trackInfo[i].fxSend[j];
                }
                arg.trackInfos.track[i].lpfFreq = info.trackInfo[i].lowPassFilterFrequency;
                arg.trackInfos.track[i].biquadType = info.trackInfo[i].biquadType;
                arg.trackInfos.track[i].biquadValue = info.trackInfo[i].biquadValue;
                arg.trackInfos.track[i].channelCount = info.trackInfo[i].channelCount;
                for (int j = 0; j < WaveChannelMax; j++)
                {
                    arg.trackInfos.track[i].channelIndex[j] =
                        info.trackInfo[i].globalChannelIndex[j];
                }
            }

            const SoundStartable::StartInfo::StreamSoundInfo* externalStrmInfo = startInfoReader.GetStrmInfo();
            if ( externalStrmInfo == NULL || externalStrmInfo->pStreamBufferPool == NULL )
            {
                arg.pBufferPool = &m_StreamBufferPool;
            }
            else
            {
                // 外部ストリームバッファプールが指定されている場合はそちらを使用
                arg.pBufferPool = externalStrmInfo->pStreamBufferPool;
            }

            arg.allocChannelCount = info.allocateChannelCount;
            arg.allocTrackFlag = info.allocateTrackFlags;

            // ループ設定
            arg.loopFlag = false;
            arg.loopFlagEnabled = false;
            arg.loopStart = 0;
            arg.loopEnd = 0xffffffff;
            if (pInfo2 != nullptr)
            {
                arg.loopFlag = pInfo2->isLoop;
                arg.loopFlagEnabled = true;
                if (arg.loopFlag)
                {
                    arg.loopStart = pInfo2->loopStartFrame;
                    arg.loopEnd = pInfo2->loopEndFrame;
                }
            }
            const auto pLoopInfo = startInfoReader.GetLoopInfo();
            if (pLoopInfo != nullptr)
            {
                if (pLoopInfo->enableParameterFlag & SoundStartable::StartInfo::LoopInfo::EnableParameterFlagBit_LoopEnabled)
                {
                    arg.loopFlag = pLoopInfo->isLoopEnabled;
                    arg.loopFlagEnabled = true;
                }
            }

            arg.pitch = info.pitch;
            arg.mainSend = info.mainSend;
            for (int i = 0; i < AuxBus_Count; i++)
            {
                arg.fxSend[i] = info.fxSend[i];
            }

            sound->Setup(arg);
        }

        // プリペア
        {
            // オフセット
            int startOffset = startInfoReader.GetStartOffset();
            detail::driver::StreamSoundPlayer::StartOffsetType strmStartOffsetType;
            switch ( startInfoReader.GetStartOffsetType() )
            {
            case SoundStartable::StartInfo::StartOffsetType_MilliSeconds:
                strmStartOffsetType = detail::driver::StreamSoundPlayer::StartOffsetType_Millisec;
                break;
            case SoundStartable::StartInfo::StartOffsetType_Tick:
                strmStartOffsetType = detail::driver::StreamSoundPlayer::StartOffsetType_Sample;
                startOffset = 0;
                break;
            case SoundStartable::StartInfo::StartOffsetType_Sample:
                strmStartOffsetType = detail::driver::StreamSoundPlayer::StartOffsetType_Sample;
                break;
            default:
                strmStartOffsetType = detail::driver::StreamSoundPlayer::StartOffsetType_Sample;
                startOffset = 0;
                break;
            }

            const SoundStartable::StartInfo::StreamSoundInfo* externalStrmInfo = startInfoReader.GetStrmInfo();

            detail::driver::StreamSoundPlayer::PrepareBaseArg arg;
            {
                // ストリームファイルのフックパラメータの保持
                // TODO : ★フックへのアクセスを遅延させると、Lock() が正しく動作しないので、必要な情報はここでかき集めたいところ・・・。
                if (m_pSoundArchiveFilesHook != NULL)
                {
                    arg.fileStreamHookParam.pSoundArchiveFilesHook = m_pSoundArchiveFilesHook;
                    arg.fileStreamHookParam.itemLabel = pSoundArchive->GetItemLabel(sound->GetId());
                }

                arg.startOffsetType = strmStartOffsetType;
                arg.offset = startOffset;
                arg.delayTime = startInfoReader.GetDelayTime();
                arg.delayCount = startInfoReader.GetDelayCount();
                arg.regionCallback = externalStrmInfo ? externalStrmInfo->regionCallback : NULL;
                arg.regionCallbackArg = externalStrmInfo ? externalStrmInfo->regionCallbackArg : NULL;
                arg.updateType = startInfoReader.GetUpdateType();

                bool externalPathEnable = false;
                if (externalStrmInfo != NULL)
                {
                    if (externalStrmInfo->externalPath != NULL)
                    {
                        externalPathEnable = true;
                    }
                    else if (externalStrmInfo->pExternalData != nullptr)
                    {
                        arg.pExternalData = externalStrmInfo->pExternalData;
                        arg.externalDataSize = externalStrmInfo->externalDataSize;
                    }
                }

                if (externalPathEnable)
                {
                    util::Strlcpy(arg.filePath, externalStrmInfo->externalPath, detail::FilePathMax);
                }
                else if(m_pSoundArchiveFilesHook == NULL || !m_pSoundArchiveFilesHook->GetIsEnable())
                {
                    // ファイル情報
                    SoundArchive::FileInfo fileInfo;
                    if (pSoundArchive->detail_ReadFileInfo(commonInfo->fileId, &fileInfo) == false)
                    {
                        SoundStartable::StartResult result(
                            SoundStartable::StartResult::ResultCode_ErrorInvalidStreamFileId );
                        return result;
                    }

                    if (fileInfo.externalFilePath == NULL)
                    {
                        SoundStartable::StartResult result(
                            SoundStartable::StartResult::ResultCode_ErrorInvalidStreamFilePath );
                        return result;
                    }

                    char buf[detail::FilePathMax];
                    pSoundArchive->detail_GetExternalFileFullPath(
                            fileInfo.externalFilePath, buf, detail::FilePathMax);
                    util::Strlcpy(arg.filePath, buf, detail::FilePathMax);
                }
            }

            const void* prefetchFile = NULL;
            if (externalStrmInfo != NULL && externalStrmInfo->prefetchData != NULL)
            {
                prefetchFile = externalStrmInfo->prefetchData;
            }
            else
            {
                // prefetchFileId が InvalidId でない場合、プリフェッチデータも一緒にコンバートされている
                if (info.prefetchFileId != SoundArchive::InvalidId)
                {
                    if (m_pSoundArchiveFilesHook != NULL)
                    {
                        const char* soundLabel = pSoundArchive->GetItemLabel(sound->GetId());

                        if (soundLabel != NULL)
                        {
                            prefetchFile = m_pSoundArchiveFilesHook->GetFileAddress(
                                soundLabel,
                                detail::SoundArchiveFilesHook::ItemTypeStreamSound,
                                detail::SoundArchiveFilesHook::FileTypeStreamPrefetchBinary);
                        }
                    }

                    if (prefetchFile == NULL)
                    {
                        prefetchFile = pSoundDataManager->detail_GetFileAddress( info.prefetchFileId );

                        if (prefetchFile == NULL)
                        {
                            NN_ATK_WARNING("[ID:%08x] Prefetch data is not loaded, so play without prefetch data.", sound->GetId());
                        }
                    }
                }
            }

            // startOffset 使用時はプリフェッチデータを使用しない
            if ( prefetchFile != NULL && arg.offset == 0 )
            {
                sound->PreparePrefetch( prefetchFile, arg );
            }

            sound->Prepare(arg);
        }

        SoundStartable::StartResult startResult(
                SoundStartable::StartResult::ResultCode_Success );
        return startResult;
    } // NOLINT(impl/function_size)

    void StreamSoundRuntime::DumpMemory(const SoundArchive* pSoundArchive) const NN_NOEXCEPT
    {
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP) // Dump 用途


        SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
        if ( pSoundArchive->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
        {
            size_t instanceManagerSize = m_StreamSoundInstanceManager.GetRequiredMemSize( soundArchivePlayerInfo.streamSoundCount, nn::atk::SoundSystem::GetSoundInstanceConfig() );
            size_t loaderManagerSize = detail::driver::StreamSoundLoaderManager::GetRequiredMemSize( soundArchivePlayerInfo.streamSoundCount );
            NN_DETAIL_ATK_INFO("    streamSound\n"
                       "      instanceManager        %8zu bytes\n"
                       "      loaderManager          %8zu bytes\n"
                       , instanceManagerSize, loaderManagerSize );
        }
#else
        NN_UNUSED(pSoundArchive);
#endif
    }

}}} // namespace nn::atk::detail
