﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_Config.h>
#include <nn/atk/atk_StreamSoundPlayer.h>
#include <nn/atk/atk_StreamBufferPool.h>
#include <nn/atk/atk_TaskManager.h>
#include <nn/atk/atk_MultiVoiceManager.h>
#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/atk_SoundSystem.h>
#include <nn/atk/atk_WaveFileReader.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>
#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>

// #define DEBUG_STRM
// #define ENABLE_PRINT_STREAM_INFO
// #define NN_ATK_DEBUG_ENABLE_DUMP_FILE

#if defined(NN_ATK_DEBUG_ENABLE_DUMP_FILE)
#include <nn/fs.h>
#include <nn/atk/fnd/io/atkfnd_FileStreamImpl.h>
#endif

namespace
{
    const uint8_t OpusFileType = static_cast<uint8_t>(nn::atk::detail::StreamFileType_Opus);
    const float OpusPitchMax = 4.0f;

#ifdef DEBUG_STRM
    FILE* g_pFile;
    const char* DumpFile = "_StreamSoundPlayer.dump";
#endif

    const uint32_t LoopRegionSizeMin = nn::atk::detail::DataBlockSizeMarginSamples;

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    const uint32_t Pcm8SamplesPerBlock = 8 * 1024;
    const uint32_t Pcm16SamplesPerBlock = 4 * 1024;
    const uint32_t AdpcmSamplesPerBlock = 14336;
    void CheckSampleLength(size_t length, const nn::atk::detail::StreamDataInfoDetail& info, uint8_t fileType, const bool isStartOffsetOfLastBlockApplied) NN_NOEXCEPT
    {
        bool doCheck = true;

        // TODO: Opus 再生時のサンプル数チェック対応
        if ( fileType == OpusFileType || isStartOffsetOfLastBlockApplied)
        {
            return;
        }

        // 下記の条件の時は、バッファジャンプが起こらないのでチェック不要
        if (info.loopFlag == false)
        {
            size_t sampleCountPerBlock = 0;
            switch (info.sampleFormat)
            {
            case nn::atk::SampleFormat_PcmS8:
                sampleCountPerBlock = Pcm8SamplesPerBlock;
                break;
            case nn::atk::SampleFormat_PcmS16:
                sampleCountPerBlock = Pcm16SamplesPerBlock;
                break;
            case nn::atk::SampleFormat_DspAdpcm:
                sampleCountPerBlock = AdpcmSamplesPerBlock;
                break;
            default:
                NN_SDK_ASSERT(false, "SampleFormat(%d) is invalid", info.sampleFormat);
                break;
            }
            if (length <= sampleCountPerBlock)
            {
                doCheck = false;
            }
        }

        if (doCheck)
        {
            size_t dataBlockSizeMarginSamples = nn::atk::detail::DataBlockSizeMarginSamples;
            NN_SDK_ASSERT(length >= dataBlockSizeMarginSamples);
        }
    }
#endif

#if defined(NN_ATK_DEBUG_ENABLE_DUMP_FILE)
    nn::atk::detail::fnd::FileStreamImpl g_DebugDumpFileStream;
    const char* DebugDumpFilePath = "C:\\temp\\dump_stream_file.dat";
    int16_t* g_DebugDumpBuffer;

    void InitializeRecording() NN_NOEXCEPT
    {

        nn::fs::DirectoryEntryType type;
        nn::Result result = nn::fs::GetEntryType(&type, DebugDumpFilePath);
        if (result.IsFailure())
        {
            nn::fs::CreateFile(DebugDumpFilePath, 1);
        }

        g_DebugDumpFileStream.Open(DebugDumpFilePath, nn::atk::detail::fnd::FileStream::AccessMode_AllowAppendAndWrite);

        g_DebugDumpBuffer = new int16_t[128 * 1024];
    }

    void FinalizeRecording() NN_NOEXCEPT
    {
        delete[] g_DebugDumpBuffer;

        g_DebugDumpFileStream.Close();
    }

    void Record(const int16_t* streamDataArray[], size_t blockSamples, int channelCount) NN_NOEXCEPT
    {
        int silentCount = 0;

        for (auto bufferIndex = 0; bufferIndex < blockSamples + silentCount; ++bufferIndex)
        {
            if (bufferIndex < blockSamples)
            {
                for (auto channelIndex = 0; channelIndex < channelCount; ++channelIndex)
                {
                    g_DebugDumpBuffer[bufferIndex * channelCount + channelIndex] = streamDataArray[channelIndex][bufferIndex];
                }
            }
            else
            {
                for (auto channelIndex = 0; channelIndex < channelCount; ++channelIndex)
                {
                    g_DebugDumpBuffer[bufferIndex * channelCount + channelIndex] = 0;
                }
            }
        }

        g_DebugDumpFileStream.Write(g_DebugDumpBuffer, (blockSamples + silentCount) * sizeof(int16_t) * channelCount);
        g_DebugDumpFileStream.Flush();
    }
#endif
} // anonymous namespace


namespace nn {
namespace atk {
namespace detail {
namespace driver {


uint16_t StreamSoundPlayer::g_AssignNumberCount = 0;

StreamSoundPlayer::StreamSoundPlayer() NN_NOEXCEPT
: m_IsInitialized(false)
, m_IsRegisterPlayerCallback(false)
, m_pLoaderManager(nullptr)
, m_pLoader(nullptr)
, m_pStreamPrefetchFile(nullptr)
, m_ChannelCount(0)
, m_TrackCount(0)
, m_IsSucceedPrepare(false)
, m_IsPreparePrefetchRequested(false)
, m_UpdateCriticalSection()
{
}

StreamSoundPlayer::~StreamSoundPlayer() NN_NOEXCEPT
{
    Finalize();
}

void StreamSoundPlayer::Initialize(OutputReceiver* pOutputReceiver) NN_NOEXCEPT
{
    fnd::ScopedLock<fnd::CriticalSection> lock(m_UpdateCriticalSection);
    BasicSoundPlayer::Initialize( pOutputReceiver );

    m_IsInitialized = false;

    m_IsPrepared          = false;
    m_PauseStatus         = false;
    m_LoadWaitFlag        = false;
    m_LoadFinishFlag      = false;
    m_IsFinalizing        = false;
    m_IsPreparedPrefetch  = false;
    m_IsSucceedPrepare    = false;
    m_IsPreparePrefetchRequested = false;

    m_LoopCounter         = 0;
    m_PlayingBlockLoopCounter = 0;
    m_PrefetchOffset      = 0;

    m_PlaySamplePosition          = 0;
    m_OriginalPlaySamplePosition  = 0;

    m_IsPrefetchRevisionCheckEnabled = false;
    m_PrefetchRevisionValue = 0;

    m_DelayCount          = 0;
    m_UseDelayCount       = false;

    if (TryAllocLoader() == true)
    {
        m_pLoader->Initialize();
    }

    // アイテム（トラックの親）データパラメータの初期化
    m_ItemData.pitch = 1.0f;
    m_ItemData.mainSend = 1.0f;
    for ( int i = 0; i < AuxBus_Count; i++ )
    {
        m_ItemData.fxSend[ i ] = 0.0f;
    }

    // トラックの初期化
    for ( int trackIndex = 0; trackIndex < StreamTrackCount; trackIndex++ )
    {
        StreamTrack& track  = m_Tracks[ trackIndex ];
        track.m_ActiveFlag  = false;
        track.m_Volume      = 1.0f;
        track.m_OutputLine  = -1;

        track.m_TvParam.Initialize();
    }

    // チャンネルの初期化
    for ( int channelIndex = 0; channelIndex < StreamChannelCount; channelIndex++ )
    {
        StreamChannel& channel = m_Channels[ channelIndex ];
        channel.m_pBufferAddress = nullptr;
        channel.m_pVoice = nullptr;
    }

#if defined(NN_ATK_DEBUG_ENABLE_DUMP_FILE)
    InitializeRecording();
#endif
}

/*--------------------------------------------------------------------------------*
  Name:         Finalize

  Description:  ストリームプレイヤーのシャットダウン

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void StreamSoundPlayer::Finalize() NN_NOEXCEPT
{
    fnd::ScopedLock<fnd::CriticalSection> lock(m_UpdateCriticalSection);
#if defined(NN_ATK_DEBUG_ENABLE_DUMP_FILE)
    FinalizeRecording();
#endif

    FinishPlayer();

    if ( m_IsInitialized )
    {
        m_IsFinalizing = true;

        // ボイスとチャンネルの解放
        FreeStreamBuffers();
        FreeVoices();

        FreeLoader();

        m_pBufferPool = nullptr;

    #ifdef DEBUG_STRM
        std::fclose( g_pFile );
    #endif

        BasicSoundPlayer::Finalize();
        SetActiveFlag(false);
        m_IsInitialized = false;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         Setup

  Description:  ストリームプレイヤーのセットアップ

  Arguments:    pBufferPool - ストリームバッファのメモリプール

  Returns:      セットアップに成功したかどうか
 *--------------------------------------------------------------------------------*/
void StreamSoundPlayer::Setup(const SetupArg& arg) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( arg.pBufferPool );
    fnd::ScopedLock<fnd::CriticalSection> lock(m_UpdateCriticalSection);

    if (m_pLoader == nullptr)
    {
        m_SetupArg = arg;
        return;
    }

    m_FileType = arg.fileType;
    m_DecodeMode = arg.decodeMode;
    m_LoopFlag = arg.loopFlag;
    m_IsLoopFlagEnabled = arg.loopFlagEnabled;
    m_LoopStart = arg.loopStart;
    m_LoopEnd = arg.loopEnd;

    m_AssignNumber = g_AssignNumberCount++;
    m_pLoader->SetAssignNumber(m_AssignNumber);

    // bXsar にあるストリームサウンドの情報を設定
    m_ItemData.Set( arg );

    // bXsar にあるストリームサウンドトラックの情報を StreamDataInfoDetail に設定
    if ( !SetupTrack( arg ) )
    {
        return;
    }

    m_pBufferPool = arg.pBufferPool;
    // NN_DETAIL_ATK_INFO("m_pBufferPool %08x\n",m_pBufferPool);

    m_IsInitialized = true;

#ifdef DEBUG_STRM
    g_pFile = std::fopen( DumpFile, "wb" );
#endif
}

void StreamSoundPlayer::Prepare(const PrepareArg& arg) NN_NOEXCEPT
{
    // プレイヤー登録
    if ( !m_IsRegisterPlayerCallback )
    {
        SoundThread::GetInstance().RegisterPlayerCallback( this );
        m_IsRegisterPlayerCallback = true;
    }

    if ( m_pLoader == nullptr )
    {
        m_PrepareArg = arg;
        m_IsSucceedPrepare = false;
        return;
    }

    if ( ! m_IsInitialized )
    {
        return;
    }

    if ( !m_IsPreparedPrefetch )
    {
        SetPrepareBaseArg(arg.baseArg);
    }

    m_ReportLoadingDelayFlag = false;
    m_IsStoppedByLoadingDelay = false;
    m_IsSucceedPrepare = true;

    RequestLoadHeader(arg);
}

void StreamSoundPlayer::PreparePrefetch(const PreparePrefetchArg& arg) NN_NOEXCEPT
{
    fnd::ScopedLock<fnd::CriticalSection> lock(m_UpdateCriticalSection);
    if ( m_pLoader == nullptr )
    {
        m_PreparePrefetchArg = arg;
        m_IsPreparePrefetchRequested = true;
        return;
    }

    if ( ! m_IsInitialized )
    {
        return;
    }

    // プリフェッチデータ読み込み
    m_pStreamPrefetchFile = arg.strmPrefetchFile;
    StreamSoundPrefetchFileReader reader;
    reader.Initialize( m_pStreamPrefetchFile );
    if ( !ReadPrefetchFile(reader) )
    {
        return;
    }

    SetPrepareBaseArg(arg.baseArg);

    if ( reader.IsIncludeRegionInfo() )
    {
        m_StreamDataInfo.isRegionIndexCheckEnabled = reader.IsRegionIndexCheckAvailable();
        // ローダーが割り当たってない時は失敗扱い
        if ( m_pLoader == nullptr || !m_pLoader->GetRegionManager().InitializeRegion( &reader, &m_StreamDataInfo ) )
        {
            return;
        }

        // 先頭以外のリージョンに移動した場合は、プリフェッチ処理対象外
        if ( !m_pLoader->GetRegionManager().IsInFirstRegion() )
        {
            SetActiveFlag( false );
            return;
        }
    }

    if ( !ApplyStreamDataInfo(m_StreamDataInfo) )
    {
        return;
    }

    // ストリームプレイヤーのセットアップ
    if ( ! SetupPlayer() )
    {
        return;
    }

    // ボイス確保
    if ( ! AllocVoices() )
    {
        FreeStreamBuffers();
        return;
    }

    m_IsPreparedPrefetch = true;

    // waveBuffer の追加
    if ( !LoadPrefetchBlocks( reader ) )
    {
        return;
    }
}

void StreamSoundPlayer::Start() NN_NOEXCEPT
{
    if ( !m_UseDelayCount && !IsStarted() )
    {
        StartPlayer();
    }
}

/*--------------------------------------------------------------------------------*
  Name:         Stop

  Description:  ストリームの再生停止

  Arguments:    None.

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void StreamSoundPlayer::Stop() NN_NOEXCEPT
{
    FinishPlayer();
}

/*--------------------------------------------------------------------------------*
  Name:         Pause

  Description:  ストリームの一時停止／再開

  Arguments:    flag - true で停止、false で再開

  Returns:      None.
 *--------------------------------------------------------------------------------*/

void StreamSoundPlayer::Pause( bool flag ) NN_NOEXCEPT
{
    SetPauseFlag(flag);
    if ( flag ) m_LoadWaitFlag = true;

    UpdatePauseStatus();
}

bool StreamSoundPlayer::IsLoadingDelayState() const NN_NOEXCEPT
{
    // m_IsPreparedPrefetch は StartSound 後に変化しないためチェックしない
    if ( !m_IsPrepared )
    {
        return false;
    }

    return (m_pLoader->IsBusy() && IsBufferEmpty()) || m_IsStoppedByLoadingDelay;
}

/*--------------------------------------------------------------------------------*
  Name:         ReadStreamSoundDataInfo

  Description:  ストリームデータの情報取得

  Arguments:    info - ストリームデータ情報構造体

  Returns:      現在の再生しているストリームデータの情報を取得する
 *--------------------------------------------------------------------------------*/
bool StreamSoundPlayer::ReadStreamSoundDataInfo( nn::atk::StreamSoundDataInfo* info ) const NN_NOEXCEPT
{
    fnd::ScopedLock<fnd::CriticalSection> lock(m_UpdateCriticalSection);

    if ( !IsPrepared() )
    {
        NN_ATK_WARNING("ReadStreamSoundDataInfo failed. Please use this API after playback preparing finished.");
        return false;
    }

    info->loopFlag              = m_StreamDataInfo.loopFlag;
    info->sampleRate            = m_StreamDataInfo.sampleRate;
    info->loopStart             = static_cast<int64_t>(m_StreamDataInfo.originalLoopStart);
    info->loopEnd               = static_cast<int64_t>(m_StreamDataInfo.originalLoopEnd);
    info->compatibleLoopStart   = static_cast<int64_t>(m_StreamDataInfo.loopStart);
    info->compatibleLoopEnd     = static_cast<int64_t>(m_StreamDataInfo.sampleCount);
    info->channelCount          = std::min( m_StreamDataInfo.channelCount, StreamChannelCount );
    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         GetPlaySamplePosition

  Description:  現在の再生位置の取得

  Arguments:    なし

  Returns:      現在の再生位置をサンプル単位で返す
                再生していない場合は、負の値を返す
 *--------------------------------------------------------------------------------*/
position_t StreamSoundPlayer::GetPlaySamplePosition(bool isOriginalSamplePosition) const NN_NOEXCEPT
{
    fnd::ScopedLock<fnd::CriticalSection> lock(m_UpdateCriticalSection);

    if ( !IsActive() )
    {
        return -1;
    }
    if ( ! m_Tracks[ 0 ].m_ActiveFlag )
    {
        return -1;
    }
    if ( ! m_IsPrepared )
    {
        return 0;
    }

    if ( isOriginalSamplePosition )
    {
        return m_OriginalPlaySamplePosition;
    }
    else
    {
        return m_PlaySamplePosition;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         GetFilledBufferPercentage

  Description:  ストリームバッファ中でデータがロードされている割合を取得する

  Arguments:    なし

  Returns:      データがロードされている割合を百分率で返す
 *--------------------------------------------------------------------------------*/
float StreamSoundPlayer::GetFilledBufferPercentage() const NN_NOEXCEPT
{
    fnd::ScopedLock<fnd::CriticalSection> lock(m_UpdateCriticalSection);

    if ( !IsActive() || ! m_Tracks[ 0 ].m_ActiveFlag )
    {
        return 0.0f;
    }
    if ( m_LoadFinishFlag )
    {
        return 1.0f;
    }
    if ( ! m_IsPrepared )
    {
        if ( m_BufferBlockCount == 0 )
        {
            return 0.0f;
        }
        return static_cast<float>( m_BufferBlockCount - m_PrepareCounter )
            * 100.f / m_BufferBlockCount;
    }

    NN_SDK_ASSERT_GREATER(m_BufferBlockCount, 0);

    size_t restSamples = 0;        // まだ再生していないサンプル数
    size_t entireSamples = 0;      // 全バッファのサンプル数
    for ( auto i = 0; i < m_BufferBlockCount; i++ )
    {
        size_t bufferSamples = m_Channels[0].m_WaveBuffer[i].sampleLength;
        switch ( m_Channels[0].m_WaveBuffer[i].status )
        {
        case nn::atk::WaveBuffer::Status_Play:
            restSamples += (bufferSamples - m_Channels[0].m_pVoice->GetCurrentPlayingSample());
            entireSamples += bufferSamples;
            break;
        case nn::atk::WaveBuffer::Status_Wait:
            restSamples += bufferSamples;
            entireSamples += bufferSamples;
            break;
        default:
            // NOTE: 波形末尾の半端なブロックは考慮できていないが、これはあきらめる
            entireSamples += m_StreamDataInfo.blockSampleCount;
            break;
        }
    }

    NN_SDK_ASSERT_GREATER(entireSamples, 0u);

    float percentage = 100.f * restSamples / entireSamples;
    return percentage;
}

int StreamSoundPlayer::GetBufferBlockCount(nn::atk::WaveBuffer::Status status) const NN_NOEXCEPT
{
    int count = 0;

    for ( auto channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        for ( auto bufferIndex = 0; bufferIndex < m_BufferBlockCount; bufferIndex++ )
        {
            if ( m_Channels[channelIndex].m_WaveBuffer[bufferIndex].status == status )
            {
                count++;
            }
        }
    }

    return count;
}

int StreamSoundPlayer::GetTotalBufferBlockCount() const NN_NOEXCEPT
{
    return m_ChannelCount * m_BufferBlockCount;
}

bool StreamSoundPlayer::LoadHeader(
    bool result,
    AdpcmParam* adpcmParam[],
    uint16_t assignNumber
) NN_NOEXCEPT
{
    if ( ! m_IsInitialized )
    {
        // 終了処理後に、コマンドバッファにたまっていたコマンドが処理され
        // ここに来る場合がある
        return false;
    }

    if ( assignNumber != m_AssignNumber )
    {
        // ロードヘッダリクエストしたときとは別のサウンドとして割り当てられた
        return false;
    }
    if ( ! result )
    {
        NN_ATK_WARNING("Task error is occured.");
        SetFinishFlag(true);
        Stop();
        return false;
    }

    m_ChannelCount = std::min( m_StreamDataInfo.channelCount, StreamChannelCount );

    if ( m_IsPreparedPrefetch )
    {
        // 短いプリフェッチデータを再生した場合、既に再生完了しボイスが解放済みの場合があるため null チェック
        for ( auto ch = 0; ch < m_ChannelCount; ch++ )
        {
            if ( m_Channels[ ch ].m_pVoice == nullptr )
            {
                return false;
            }
        }

        // リビジョン情報のチェック
        if ( !CheckPrefetchRevision(m_StreamDataInfo) )
        {
            return false;
        }
    }

    // ストリームバッファ確保
    if ( ! AllocStreamBuffers() )
    {
        NN_ATK_WARNING("Failed to start stream sound for not enough stream channel instance.");
        Finalize();
        SetFinalizedForCannotAllocateResourceFlag(true);
        return false;
    }

    if ( !ApplyStreamDataInfo(m_StreamDataInfo) )
    {
        return false;
    }

    if ( !m_IsPreparedPrefetch )
    {
        // ストリームプレイヤーのセットアップ
        if ( ! SetupPlayer() )
        {
            return false;
        }

        // ボイス確保
        if ( ! AllocVoices() )
        {
            FreeStreamBuffers();
            return false;
        }

        // セットアップコールバック呼び出し
        m_PrepareCounter = 0;
        for ( auto i = 0; i < m_BufferBlockCount; i++ )
        {
            UpdateLoadingBlockIndex();
            m_PrepareCounter++;
        }
    }
    else
    {
        m_IsPrepared = true;
    }

    for ( auto ch = 0; ch < m_ChannelCount; ch++ )
    {
        AdpcmParam* pAdpcmParam = adpcmParam[ch];
        if ( pAdpcmParam != nullptr )
        {
            m_Channels[ ch ].m_pVoice->SetAdpcmParam( 0, *pAdpcmParam );
        }
    }

    return true;
}

bool StreamSoundPlayer::LoadStreamData(
    bool result,
    const LoadDataParam& loadDataParam,
    uint16_t assignNumber
) NN_NOEXCEPT
{
    return LoadStreamData(
        result,
        loadDataParam,
        assignNumber,
        false,
        0,
        0
    );
}

bool StreamSoundPlayer::LoadStreamData(
    bool result,
    const LoadDataParam& loadDataParam,
    uint16_t assignNumber,
    bool usePrefetchFlag,
    uint32_t currentPrefetchBlockIndex,
    size_t currentPrefetchBlockBytes
) NN_NOEXCEPT
{
    if ( ! m_IsInitialized )
    {
        // 終了処理後に、コマンドバッファにたまっていたコマンドが処理され
        // ここに来る場合がある
        return false;
    }
    if ( ! result )
    {
        NN_ATK_WARNING("Task error is occured.");
        SetFinishFlag(true);
        Stop();
        return false;
    }

    if ( assignNumber != m_AssignNumber )
    {
        // ロードデータリクエストしたときとは別のサウンドとして割り当てられた
        return false;
    }
    // -------------------------
    // ロードが追いついているかをチェック
    if ( !m_LoadWaitFlag && IsStoppedByLoadingDelay() )
    {
        // ロードが追いついてない
        m_ReportLoadingDelayFlag = true;
        m_IsStoppedByLoadingDelay = true;
        m_LoadWaitFlag = true;
        UpdatePauseStatus();
    }

    if ( loadDataParam.samples > 0 )
    {
#if defined(NN_ATK_DEBUG_ENABLE_DUMP_FILE)
        const int16_t* streamDataArray[StreamChannelCount];
#endif
        for ( auto ch = 0; ch < m_ChannelCount ; ch++ )
        {
            const void* bufferAddress;

            if ( ! usePrefetchFlag )
            {
                NN_UNUSED(currentPrefetchBlockIndex); NN_UNUSED(currentPrefetchBlockBytes);
                bufferAddress = util::BytePtr(
                    m_Channels[ ch ].m_pBufferAddress,
                    ( m_StreamDataInfo.blockSize + StreamSoundLoader::DataBlockSizeMargin ) * loadDataParam.blockIndex ).Get();
            }
            else
            {
                bufferAddress = util::ConstBytePtr(
                    m_PrefetchDataInfo.dataAddress,
                    m_StreamDataInfo.blockSize * m_ChannelCount * currentPrefetchBlockIndex + currentPrefetchBlockBytes * ch ).Get();
            }
#if defined(NN_ATK_DEBUG_ENABLE_DUMP_FILE)
            streamDataArray[ch] = reinterpret_cast<const int16_t*>(bufferAddress);
#endif

            // WaveBuffer への割り付け
            WaveBuffer* waveBuffer =
                &m_Channels[ch].m_WaveBuffer[loadDataParam.blockIndex];
            AdpcmContext* pAdpcmContext = nullptr;
            if ( loadDataParam.adpcmContextEnable )
            {
                pAdpcmContext = &m_Channels[ch].m_AdpcmContext[loadDataParam.blockIndex];
                (*pAdpcmContext).audioAdpcmContext = loadDataParam.adpcmContext[ch].audioAdpcmContext;
            #ifdef ENABLE_PRINT_STREAM_INFO
                NN_DETAIL_ATK_INFO("[%s] predScale(%04x) yn1(%6d) yn2(%6d)\n", __FUNCTION__,
                    pAdpcmContext->audioAdpcmContext.predScale,
                    pAdpcmContext->audioAdpcmContext.history[0],
                    pAdpcmContext->audioAdpcmContext.history[1]);
            #endif
            }
            waveBuffer->Initialize();
            waveBuffer->bufferAddress = bufferAddress;
            waveBuffer->bufferSize = usePrefetchFlag ? m_StreamDataInfo.blockSize : loadDataParam.sampleBytes;
            waveBuffer->sampleLength = loadDataParam.samples;
            waveBuffer->sampleOffset = loadDataParam.sampleOffset;
            waveBuffer->pAdpcmContext = pAdpcmContext;

            // StreamChannel へのアペンド
        #ifdef ENABLE_PRINT_STREAM_INFO
            NN_DETAIL_ATK_INFO("AppendWaveBuffer ch(%d) bufferAddr(%p) length(%d) ofs(%d) blockIdx(%d)\n",
                   ch, bufferAddress, loadDataParam.samples, loadDataParam.sampleOffset, loadDataParam.blockIndex);
        #endif
        #if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
            if ( !m_IsLoopFlagEnabled )
            {
                CheckSampleLength(waveBuffer->sampleLength, m_StreamDataInfo, m_FileType, loadDataParam.isStartOffsetOfLastBlockApplied);
            }
        #endif
            m_Channels[ch].AppendWaveBuffer( waveBuffer, loadDataParam.lastBlockFlag );
        }

        WaveBufferInfo* waveBufferInfo = &m_WaveBufferInfo[loadDataParam.blockIndex];
        waveBufferInfo->sampleBegin = loadDataParam.sampleBegin;
        waveBufferInfo->sampleLength = loadDataParam.samples;
        waveBufferInfo->loopCount = loadDataParam.loopCount;

#if defined(NN_ATK_DEBUG_ENABLE_DUMP_FILE)
        Record(streamDataArray, loadDataParam.samples, m_ChannelCount);
#endif
    }

    if ( loadDataParam.lastBlockFlag )
    {
        m_LoadFinishFlag = true;
    }

    // バッファ書き換え完了
    if ( ! m_IsPrepared )
    {
        if ( ! usePrefetchFlag )
        {
            m_PrepareCounter--;
            if ( m_PrepareCounter == 0 || m_LoadFinishFlag )
            {
                m_IsPrepared = true;
            }
        }
    }

    return true;
}

void StreamSoundPlayer::StartPlayer() NN_NOEXCEPT
{
    for ( auto trackIndex = 0; trackIndex < m_TrackCount; trackIndex++ )
    {
        StreamTrack& track = m_Tracks[ trackIndex ];
        if ( ! track.m_ActiveFlag )
        {
            continue;
        }

        // ボイスセットアップ
        for ( int ch = 0; ch < track.channelCount; ch++ )
        {
            StreamChannel* channel = track.m_pChannels[ch];
            if ( channel == nullptr )
            {
                continue;
            }

            MultiVoice* voice = channel->m_pVoice;
            if ( voice != nullptr )
            {
                // blockOffset は使われない。かわりに LoadStreamData 内で設定される。
                voice->SetSampleFormat( m_StreamDataInfo.sampleFormat );
                voice->SetSampleRate( m_StreamDataInfo.sampleRate );
                voice->SetUpdateType( m_UpdateType );
                voice->SetOutputReceiver( GetOutputReceiver() );
                voice->Start();
            }
        }
    }

    UpdatePauseStatus();

    SetStartedFlag(true);
}

void StreamSoundPlayer::FinishPlayer() NN_NOEXCEPT
{
    if (m_pLoader != nullptr)
    {
        m_pLoader->CancelRequest();
    }

    // ボイスの停止
    for ( auto ch = 0; ch < m_ChannelCount; ch++ )
    {
        MultiVoice* voice = m_Channels[ch].m_pVoice;
        if ( voice != nullptr )
        {
            voice->Stop();
        }
    }

    // プレイヤー登録の解除
    if ( m_IsRegisterPlayerCallback )
    {
        SoundThread::GetInstance().UnregisterPlayerCallback( this );
        m_IsRegisterPlayerCallback = false;
    }

    if ( IsStarted() )
    {
        SetStartedFlag(false);
    }
}

/*--------------------------------------------------------------------------------*
  Name:         SetupPlayer

  Description:  ストリームのセットアップ

  Arguments:    header - ストリームファイルヘッダ

  Returns:      成功したかどうか
 *--------------------------------------------------------------------------------*/
bool StreamSoundPlayer::SetupPlayer() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pBufferPool );
    const size_t strmBufferSize = m_pBufferPool->GetBlockSize();    // 1ch 分のバッファサイズ

    // ブロックサイズの取得
    if ( m_StreamDataInfo.blockSize > StreamSoundLoader::DataBlockSizeBase )
    {
        NN_ATK_WARNING("Too large stream data block size.");
        return false;
    }
    m_BufferBlockCount = static_cast<int>( strmBufferSize / ( m_StreamDataInfo.blockSize + StreamSoundLoader::DataBlockSizeMargin ) );
    if ( m_BufferBlockCount < StreamBlockCountMin )
    {
        NN_ATK_WARNING("Too small stream buffer size.");
        return false;
    }
    if ( m_BufferBlockCount > StreamDataLoadTaskMax )
    {
        // 非同期ロードコマンド発行のため、上限を設定
        m_BufferBlockCount = StreamDataLoadTaskMax;
    }

    m_LoadingBufferBlockIndex = 0;
    m_PlayingBufferBlockIndex = 0;
    m_LastPlayFinishBufferBlockIndex = 0;

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         AllocStreamBuffers

  Description:  全チャンネルのストリームバッファを確保

  Arguments:    なし

  Returns:      成功したらtrue
 *--------------------------------------------------------------------------------*/
bool StreamSoundPlayer::AllocStreamBuffers() NN_NOEXCEPT
{
    // 全チャンネル分のストリームバッファ確保
    for( auto index = 0; index < m_ChannelCount; index++ )
    {
        // 1 ブロック分 (1ch 分) のバッファを確保
        void* strmBuffer = m_pBufferPool->Alloc();
        if ( strmBuffer == nullptr )
        {
            // ひとつでも失敗したら、すべてのバッファを返却し false を返す
            for( auto i = 0 ; i < index ; i++ )
            {
                m_pBufferPool->Free( m_Channels[i].m_pBufferAddress );
                m_Channels[i].m_pBufferAddress = nullptr;
            }
            return false;
        }

        m_Channels[ index ].m_pBufferAddress = strmBuffer;
        m_Channels[ index ].m_UpdateType = m_UpdateType;
    }
    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         FreeStreamBuffers

  Description:  全チャンネルのストリームバッファを開放

  Arguments:    なし

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void StreamSoundPlayer::FreeStreamBuffers() NN_NOEXCEPT
{
    // ストリームバッファ開放
    for( auto index = 0; index < m_ChannelCount; index++ )
    {
        if ( m_Channels[ index ].m_pBufferAddress == nullptr )
        {
            continue;
        }

        m_pBufferPool->Free( m_Channels[index].m_pBufferAddress );
        m_Channels[ index ].m_pBufferAddress = nullptr;
    }
}

// 全 StreamTrack に対し、Voice を割り当てる
bool StreamSoundPlayer::AllocVoices() NN_NOEXCEPT
{
    for ( auto channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        StreamChannel& channel = m_Channels[ channelIndex ];

        // MultiVoice 確保
        MultiVoice* voice = MultiVoiceManager::GetInstance().AllocVoice(
            1,      // channelCount
            MultiVoice::PriorityNoDrop,
            VoiceCallbackFunc,
            &channel
        );
        if ( voice == nullptr )
        {
            for ( auto i = 0; i < channelIndex; i++ )
            {
                StreamChannel& c = m_Channels[ i ];
                if ( c.m_pVoice != nullptr )
                {
                    c.m_pVoice->Free();
                    c.m_pVoice = nullptr;
                }
            }
            NN_ATK_WARNING("voice allocation failed.");

            return false;
        }
        channel.m_pVoice = voice;
    }
    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         FreeAllTrackVoice

  Description:  ストリーム用チャンネルの解放

  Arguments:    player - ストリームオブジェクト

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void StreamSoundPlayer::FreeVoices() NN_NOEXCEPT
{
    for ( auto ch = 0; ch < m_ChannelCount; ch++ )
    {
        StreamChannel& channel = m_Channels[ ch ];
        if ( channel.m_pVoice != nullptr )
        {
            channel.m_pVoice->Free();
            channel.m_pVoice = nullptr;
        }
    }
}

bool StreamSoundPlayer::TryAllocLoader() NN_NOEXCEPT
{
    if (m_pLoader != nullptr)
    {
        return true;    // 割り当て済み
    }

    if (m_pLoaderManager == nullptr)
    {
        return false;
    }

    StreamSoundLoader* loader = m_pLoaderManager->Alloc();
    if (loader == nullptr)
    {
        return false;
    }

    m_pLoader = loader;
    return true;
}

void StreamSoundPlayer::FreeLoader() NN_NOEXCEPT
{
    if (m_pLoader == nullptr)
    {
        return;
    }

    if (m_pLoaderManager == nullptr)
    {
        return;
    }

    m_pLoaderManager->Free(m_pLoader);
    m_pLoader = nullptr;
}

/*--------------------------------------------------------------------------------*
  Name:         Update

  Description:  パラメータ更新
                （サウンドスレッドから呼びだされる）

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void StreamSoundPlayer::Update() NN_NOEXCEPT
{
    fnd::ScopedLock<fnd::CriticalSection> lock(m_UpdateCriticalSection);
    if (TryAllocLoader() == true)
    {
        if (!m_IsSucceedPrepare)
        {
            m_pLoader->Initialize();
            Setup(m_SetupArg);
            if (m_IsPreparePrefetchRequested)
            {
                PreparePrefetch(m_PreparePrefetchArg);
                m_IsPreparePrefetchRequested = false;
            }
            Prepare(m_PrepareArg);
        }
    }
    if (m_pLoader == nullptr)
    {
        return;
    }

    m_pLoader->Update();

    if (m_DelayCount > 0)
    {
        --m_DelayCount;
        return;
    }

    // フラグチェック
    if (m_UseDelayCount)
    {
        if ( m_IsPrepared && !IsStarted() )
        {
            StartPlayer();
        }
    }

    UpdateBuffer();

    // ストリームチャンネルの停止チェック
    if ( IsStarted() )
    {
        for ( auto ch = 0; ch < m_ChannelCount; ch++ )
        {
            StreamChannel& channel = m_Channels[ ch ];
            if ( channel.m_pVoice == nullptr )
            {
                Stop();
                SetFinishFlag(true);
                return;
            }
        }

        // パラメータのアップデート
        for ( auto trackIndex = 0; trackIndex < m_TrackCount; trackIndex++ )
        {
            UpdateVoiceParams( &m_Tracks[ trackIndex ] );
        }
    }

    // ロード待ちが解消されたかをチェック
    if ( m_LoadWaitFlag )
    {
        if ( ! m_pLoader->IsBusy() && ! CheckDiskDriveError() )
        {
            m_LoadWaitFlag = false;
            m_IsStoppedByLoadingDelay = false;
            UpdatePauseStatus();
        }
    }

    // ロード遅延メッセージ出力
    if ( m_ReportLoadingDelayFlag )
    {
        NN_ATK_WARNING("Pause stream because of loading delay.");
        m_ReportLoadingDelayFlag = false;
    }
}

void StreamSoundPlayer::UpdateVoiceParams( StreamTrack* track ) NN_NOEXCEPT
{
    if ( ! track->m_ActiveFlag )
    {
        return;
    }

    TrackData trackData;
    trackData.Set(track);

    // volume
    float volume = 1.0f;
    volume *= GetVolume();
    volume *= trackData.volume;
    volume *= track->m_Volume;

    // pitch
    float pitchRatio = 1.0f;
    pitchRatio *= GetPitch();
    pitchRatio *= m_ItemData.pitch;

    if (m_FileType == OpusFileType)
    {
        if (pitchRatio > OpusPitchMax)
        {
            NN_ATK_WARNING("If file type is opus, pitch must be less than %lf.", OpusPitchMax);
            pitchRatio = OpusPitchMax;
        }
    }

    // lpf freq
    float lpfFreq = trackData.lpfFreq;
    lpfFreq += GetLpfFreq();

    // biquad filter
    int biquadType = trackData.biquadType;
    float biquadValue = trackData.biquadValue;
    int handleBiquadType = GetBiquadFilterType();
    if ( handleBiquadType != static_cast<int>(BiquadFilterType_Inherit) )
    {
        biquadType  = handleBiquadType;
        biquadValue = GetBiquadFilterValue();
    }

    // outputLine
    uint32_t outputLine = 0;
    if ( track->m_OutputLine != -1 )
    {
        outputLine = track->m_OutputLine;
    }
    else
    {
        outputLine = GetOutputLine();
    }

    // TV 出力向けパラメータ
    OutputParam tvParam;
    {
        tvParam = GetTvParam();
        SetOutputParam( &tvParam, track->m_TvParam, trackData );
    }

    for ( int ch = 0; ch < track->channelCount; ch++ )
    {
        MultiVoice* voice = track->m_pChannels[ch]->m_pVoice;

        if ( voice != nullptr )
        {
            voice->SetVolume( volume );
            voice->SetPitch( pitchRatio );
            voice->SetLpfFreq( lpfFreq );
            voice->SetBiquadFilter( biquadType, biquadValue );
            voice->SetOutputLine( outputLine );

            if ( track->channelCount == 1 )
            {
                voice->SetTvParam( tvParam );
                if(GetTvAdditionalParamAddr() != nullptr)
                {
                    voice->SetTvAdditionalParam( *GetTvAdditionalParamAddr() );
                }
            }
            else if ( track->channelCount == 2 )
            {
                ApplyTvOutputParamForMultiChannel( tvParam, GetTvAdditionalParamAddr(), voice, ch, static_cast<MixMode>(tvParam.mixMode) );
            }

        }
    }
}

void StreamSoundPlayer::SetOutputParam(OutputParam* pOutOutputParam, const OutputParam& trackParam, const TrackData& trackData) NN_NOEXCEPT
{
    pOutOutputParam->volume *= trackParam.volume;

    for ( int i = 0; i < WaveChannelMax; i++ )
    {
        for ( int j = 0; j < ChannelIndex_Count; j++ )
        {
            pOutOutputParam->mixParameter[i].ch[j] *= trackParam.mixParameter[i].ch[j];
        }
    }
    pOutOutputParam->pan += trackData.pan + trackParam.pan;
    pOutOutputParam->span += trackData.span + trackParam.span;

    for ( int i = 0; i < nn::atk::detail::DefaultBusCount; i++)
    {
        pOutOutputParam->send[i] += trackParam.send[i];
    }

    pOutOutputParam->send[Util::GetSubMixBusFromMainBus()] += trackData.mainSend + m_ItemData.mainSend;
    for ( int i = 0; i < AuxBus_Count; i++ )
    {
        pOutOutputParam->send[Util::GetSubMixBus(i)] += trackData.fxSend[i] + m_ItemData.fxSend[i];
    }
}

void StreamSoundPlayer::MixSettingForOutputParam(OutputParam* pOutOutputParam, OutputBusMixVolume* pOutOutputBusMixVolume, int32_t channelIndex, MixMode mixMode) NN_NOEXCEPT
{
    if ( mixMode == MixMode_Pan )
    {
        // パン補正
#if 0
        // 1trk == 2ch の場合は、パンモード別のパン計算を行う
        switch ( GetPanMode() )
        {
        case nn::atk::PanMode_Balance:
            if ( channelIndex == 0 ) { pan = -1.0f; } // L に振る
            if ( channelIndex == 1 ) { pan =  1.0f; } // R に振る
            break;
        case nn::atk::PanMode_Dual:
            if ( channelIndex == 0 ) { pan -= 1.0f; } // L に寄せる
            if ( channelIndex == 1 ) { pan += 1.0f; } // R に寄せる
            break;
        default:
            break;
        }
#else
        // TODO: DualPan, BalancePan に対応するには、
        //       StreamTrack に Voiceを持たせる形にする必要がある。
        //       上記の #if 0 の部分は、実装イメージ。
        //       BalancePan で pan を -1.0f や 1.0f に固定してしまうと、
        //       GetPan() やトラックごとのデータパン・API パンが反映されなく鳴る。
        if ( channelIndex == 0 )
        {
            pOutOutputParam->pan -= 1.0f;
        }
        else if ( channelIndex == 1 )
        {
            pOutOutputParam->pan += 1.0f;
        }
#endif
    }
    else
    {
        // ミックスモード補正

        // ストリームサウンドでは、波形のチャンネル毎にボイスを分けるため
        // サラウンドボリュームを 0 ch 側に設定し直す。
        if ( channelIndex == 1 )
        {
            for ( uint32_t i = 0; i < ChannelIndex_Count; i++ )
            {
                pOutOutputParam->mixParameter[0].ch[i] = pOutOutputParam->mixParameter[channelIndex].ch[i];
                pOutOutputParam->mixParameter[channelIndex].ch[i] = 0.0f;
            }

            if(pOutOutputBusMixVolume == nullptr)
            {
                return;
            }

            for ( int i = 0; i < OutputMix::ChannelCountMax; i++ )
            {
                pOutOutputBusMixVolume->volume[0][i]= pOutOutputBusMixVolume->volume[channelIndex][i];
                pOutOutputBusMixVolume->volume[channelIndex][i] = 0.0f;
            }
        }
    }
}

void StreamSoundPlayer::ApplyTvOutputParamForMultiChannel(const OutputParam& outputParam, const OutputAdditionalParam* const pOutputAdditionalParam, MultiVoice* pVoice, int32_t channelIndex, MixMode mixMode) NN_NOEXCEPT
{
    // outputParam, pOutputBusMixVolume の中身の値は利用しますが、
    // その値を書き換えてほしいわけではないため、一時変数を生成し、
    // その変数のポインタを MixSettingForOutputParam() に渡し、計算結果を pVoice に代入します。
    OutputParam param = outputParam;

    // pOutputBusMixVolume が nullptr の時に、余計な一時変数の生成とコピーを行わないための条件分岐です。
    if(pOutputAdditionalParam != nullptr)
    {
        if(pOutputAdditionalParam->GetBusMixVolumePacketAddr() != nullptr)
        {
            OutputBusMixVolume busMixVolume = pOutputAdditionalParam->GetBusMixVolume();
            MixSettingForOutputParam(&param, &busMixVolume, channelIndex, mixMode);
            pVoice->SetTvParam( param );
            pVoice->SetTvAdditionalParam( pOutputAdditionalParam->GetAdditionalSendAddr(), pOutputAdditionalParam->GetBusMixVolumePacketAddr(), &busMixVolume, pOutputAdditionalParam->GetVolumeThroughModePacketAddr() );
            return;
        }

        MixSettingForOutputParam(&param, nullptr, channelIndex, mixMode);
        pVoice->SetTvParam( param );
        pVoice->SetTvAdditionalParam( pOutputAdditionalParam->GetAdditionalSendAddr(), pOutputAdditionalParam->GetBusMixVolumePacketAddr(), nullptr, pOutputAdditionalParam->GetVolumeThroughModePacketAddr() );
    }
    else
    {
        MixSettingForOutputParam(&param, nullptr, channelIndex, mixMode);
        pVoice->SetTvParam( param );
    }
}

void StreamSoundPlayer::ItemData::Set( const SetupArg& arg ) NN_NOEXCEPT
{
    // pitch
    pitch = arg.pitch;

    // send
    mainSend = static_cast<float>( arg.mainSend ) / 127.0f - 1.0f;
    for ( int i = 0; i < AuxBus_Count; i++ )
    {
        fxSend[ i ] = static_cast<float>( arg.fxSend[i] ) / 127.0f;
    }
}

void StreamSoundPlayer::TrackData::Set( const StreamTrack* track ) NN_NOEXCEPT
{
    // volume
    volume = static_cast<float>( track->volume ) / 127.0f;

    // lpf freq
    lpfFreq = static_cast<float>( track->lpfFreq ) / 64.0f;

    // biquad filter;
    biquadType = static_cast<int>( track->biquadType );
    biquadValue = static_cast<float>( track->biquadValue ) / 127.0f;

    // pan
    if ( track->pan <= 1 )
    {
        pan = static_cast<float>( static_cast<int>( track->pan ) - 63 ) / 63.0f;
    }
    else
    {
        pan = static_cast<float>( static_cast<int>( track->pan ) - 64 ) / 63.0f;
    }

    // span
    if ( track->span <= 63 )
    {
        span = static_cast<float>( track->span ) / 63.0f;
    }
    else
    {
        span = static_cast<float>( track->span + 1 ) / 64.0f;
    }

    // send
    mainSend = static_cast<float>( track->mainSend ) / 127.0f - 1.0f;
    for ( int i = 0; i < AuxBus_Count; i++ )
    {
        fxSend[i] = static_cast<float>( track->fxSend[i] ) / 127.0f;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         StreamSoundPlayer::CheckDiskDriveError

  Description:  ディスクドライブのエラーが発生しているかどうかチェックします。

  Arguments:    None.

  Returns:      エラーが発生しているかどうかを返します。
                DVD ストリームでない場合は、false を返します。
  *--------------------------------------------------------------------------------*/
bool StreamSoundPlayer::CheckDiskDriveError() const NN_NOEXCEPT
{
    // TODO: 関数名と挙動があっていない
    return nn::atk::SoundSystem::detail_IsStreamLoadWait();
}

/*--------------------------------------------------------------------------------*
  Name:         UpdateBuffer

  Description:  再生バッファの更新
                （AXコールバックから呼びだされる）

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void StreamSoundPlayer::UpdateBuffer() NN_NOEXCEPT
{
    if ( !IsStarted() )
    {
        return;
    }
    if ( ! m_Tracks[ 0 ].m_ActiveFlag )
    {
        return;
    }

    if ( m_IsPreparedPrefetch && !m_IsPrepared )
    {
        return;
    }

    if ( CheckDiskDriveError() )
    {
        m_LoadWaitFlag = true;
        UpdatePauseStatus();
    }

    // -------------------------
    // バッファのアップデート
    // バッファブロックの境界を越えたかチェックし、ファイルロードを発行する
    for ( ;; )
    {
        bool isAllDone = true;
        for ( auto ch = 0; ch < m_ChannelCount; ch++ )
        {
            if ( m_Channels[ch].m_WaveBuffer[m_PlayingBufferBlockIndex].status != WaveBuffer::Status_Done )
            {
                isAllDone = false;
                break;
            }
        }
        if ( isAllDone == false )
        {
            break;
        }

        // すべてのチャンネルの WaveBuffer が Status_Done になってから更新する
        for ( auto ch = 0; ch < m_ChannelCount; ch++ )
        {
            m_Channels[ch].m_WaveBuffer[m_PlayingBufferBlockIndex].Initialize();
        }

        // 再生ブロックの更新
        const WaveBufferInfo* waveBufferInfo = &m_WaveBufferInfo[m_PlayingBufferBlockIndex];
        m_LoopCounter += waveBufferInfo->loopCount;

        // ロードブロックの更新
        UpdateLoadingBlockIndex();

        // バッファブロック番号更新
        m_LastPlayFinishBufferBlockIndex = m_PlayingBufferBlockIndex;
        m_PlayingBufferBlockIndex++;
        if ( m_PlayingBufferBlockIndex >= static_cast<uint32_t>(m_BufferBlockCount) )
        {
            m_PlayingBufferBlockIndex = 0;
        }
    }

    NN_SDK_ASSERT(m_pLoader != nullptr);

    // -------------------------
    // ロードが追いついているかをチェック
    if ( ! m_LoadWaitFlag && IsLoadingDelayState() )
    {
        // ロードが追いついてない
        m_ReportLoadingDelayFlag = true;
        m_IsStoppedByLoadingDelay = true;
        m_LoadWaitFlag = true;
        UpdatePauseStatus();
    }

    // サンプル位置の更新
    // 計算途中でスレッドが切り替わって、計算前の値と後の値が混在しないようにするために、値を一時変数に保持します
    position_t playSamplePosition = 0;
    if ( IsLoadingDelayState() || (IsBufferEmpty() && m_LoadFinishFlag) )
    {
        // ロード遅延状態および再生完了時は、一個前のバッファのパラメーターを利用する
        int previousBufferBlockIndex = static_cast<int>(m_PlayingBufferBlockIndex) - 1;
        if ( previousBufferBlockIndex < 0 )
        {
            previousBufferBlockIndex = m_BufferBlockCount - 1;
        }
        playSamplePosition = m_WaveBufferInfo[previousBufferBlockIndex].sampleBegin + m_WaveBufferInfo[previousBufferBlockIndex].sampleLength;
    }
    else
    {
        // それ以外の場合は現在再生中のバッファのパラメーターを利用する
        playSamplePosition = 0;
        MultiVoice* pMultiVoice = m_Channels[0].m_pVoice;
        if ( pMultiVoice != nullptr )
        {
            playSamplePosition = pMultiVoice->GetCurrentPlayingSample();
        }
        playSamplePosition += m_WaveBufferInfo[m_PlayingBufferBlockIndex].sampleBegin;
    }
    m_PlaySamplePosition = playSamplePosition;

    // ループ補正前のサンプル位置、ループ回数を計算
    int loopCount = 0;
    if ( m_StreamDataInfo.loopFlag )
    {
        loopCount = GetOriginalLoopCount( playSamplePosition, m_StreamDataInfo );
        playSamplePosition = GetOriginalPlaySamplePosition( playSamplePosition, m_StreamDataInfo );
    }
    m_OriginalPlaySamplePosition = playSamplePosition;
    m_PlayingBlockLoopCounter = loopCount;
}

bool StreamSoundPlayer::IsBufferEmpty() const NN_NOEXCEPT
{
    for ( auto i = 0; i < m_BufferBlockCount; i++ )
    {
        switch ( m_Channels[0].m_WaveBuffer[i].status )
        {
        case nn::atk::WaveBuffer::Status_Wait:
        case nn::atk::WaveBuffer::Status_Play:
            return false;
        default:
            break;
        }
    }
    return true;
}

bool StreamSoundPlayer::IsStoppedByLoadingDelay() const NN_NOEXCEPT
{
    // プリペア前はチェックの必要なし
    if(!m_IsPrepared)
    {
        return false;
    }

    bool isStatusDoneExisted = false;
    for (auto i = 0; i < m_BufferBlockCount; i++)
    {
        switch (m_Channels[0].m_WaveBuffer[i].status)
        {
        case nn::atk::WaveBuffer::Status_Free:
            break;
        case nn::atk::WaveBuffer::Status_Done:
            isStatusDoneExisted = true;
            break;
        default:
            return false;
        }
    }
    return isStatusDoneExisted;
}

// SetupTrack (サウンドアーカイブバイナリ → StreamDataInfoDetail)
// ApplyTrackDataInfo (StreamDataInfoDetail → StreamTrack)
bool StreamSoundPlayer::SetupTrack( const SetupArg& arg ) NN_NOEXCEPT
{
    // 有効なトラックを順次アクティブ化
    uint32_t bitMask = arg.allocTrackFlag;
    uint32_t trackIndex = 0;
    while ( bitMask != 0 )
    {
        if ( bitMask & 0x01 )
        {
            if ( trackIndex >= StreamTrackCount )
            {
                NN_ATK_WARNING(
                    "Too large track index (%d). Max track index is %d.",
                    trackIndex, StreamTrackCount - 1
                );
                break;
            }

            m_Tracks[ trackIndex ].m_ActiveFlag = true;
        }

        bitMask >>= 1;
        trackIndex++;
    }

    m_TrackCount = std::min( trackIndex, StreamTrackCount );
    if ( m_TrackCount == 0 )
    {
        Finalize();
        return false;
    }

    // サウンドアーカイブ (*.bXsar) 埋め込みのトラック情報反映
    for (int i = 0; i < StreamTrackCount; i++)
    {
        TrackDataInfo& data = m_StreamDataInfo.trackInfo[i];
        data = arg.trackInfos.track[i];
    }

    return true;
}

void StreamSoundPlayer::SetPrepareBaseArg(const PrepareBaseArg& baseArg) NN_NOEXCEPT
{
    m_DelayCount      = baseArg.delayCount != 0 ? baseArg.delayCount : ToDelayCount(baseArg.delayTime);
    m_UseDelayCount   = m_DelayCount > 0;
    m_StartOffsetType = baseArg.startOffsetType;
    m_StartOffset     = baseArg.offset;
    m_UpdateType      = baseArg.updateType;

    // プリフェッチの場合でも使えるようにこのタイミングで設定する
    NN_SDK_ASSERT_NOT_NULL(m_pLoader);
    m_pLoader->SetRegionCallback(baseArg.regionCallback, baseArg.regionCallbackArg);
    m_pLoader->SetStreamSoundPlayer(this);
    m_pLoader->SetStreamDataInfo(&m_StreamDataInfo);
    m_pLoader->SetFileType(static_cast<StreamFileType>(m_FileType));
    m_pLoader->SetDecodeMode(m_DecodeMode);

    SetActiveFlag(true);
}

void StreamSoundPlayer::RequestLoadHeader(const PrepareArg& arg) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pLoader != nullptr);

    m_pLoader->SetLoopParameter(m_LoopFlag, m_IsLoopFlagEnabled, m_LoopStart, m_LoopEnd);
    m_pLoader->SetFilePath(arg.baseArg.filePath, FilePathMax);
    m_pLoader->SetFileStreamHookParam(arg.baseArg.fileStreamHookParam);
    m_pLoader->SetExternalData(arg.baseArg.pExternalData, arg.baseArg.externalDataSize);
    m_pLoader->SetCacheBuffer(arg.cacheBuffer, arg.cacheSize);

    bool isStreamOpenFailureHalt = true;
    isStreamOpenFailureHalt = nn::atk::SoundSystem::detail_IsStreamOpenFailureHaltEnabled();

    m_pLoader->InitializeFileStream(isStreamOpenFailureHalt);
    m_pLoader->RequestLoadHeader();
}

bool StreamSoundPlayer::ReadPrefetchFile(StreamSoundPrefetchFileReader& reader) NN_NOEXCEPT
{
    // プリフェッチ情報の読み込み
    StreamSoundPrefetchFileReader::PrefetchDataInfo prefetchInfo;
    if ( !reader.ReadPrefetchDataInfo(&prefetchInfo, 0) ) // 現状プリフェッチデータは１つのみ
    {
        return false;
    }

    m_PrefetchDataInfo.startFrame   = prefetchInfo.startFrame;
    m_PrefetchDataInfo.prefetchSize = prefetchInfo.prefetchSize;
    m_PrefetchDataInfo.dataAddress  = prefetchInfo.dataAddress;

    // ストリームサウンド情報の読み込み
    StreamSoundFile::StreamSoundInfo info;
    reader.ReadStreamSoundInfo(&info);

    m_StreamDataInfo.channelCount   = reader.GetChannelCount();
    m_StreamDataInfo.SetStreamSoundInfo(info, reader.IsCrc32CheckAvailable());

    if (reader.IsCrc32CheckAvailable())
    {
        m_IsPrefetchRevisionCheckEnabled = true;
        m_PrefetchRevisionValue = info.crc32Value;
    }
    else
    {
        m_IsPrefetchRevisionCheckEnabled = false;
        m_PrefetchRevisionValue = 0;
    }

    m_ChannelCount = std::min( m_StreamDataInfo.channelCount, StreamChannelCount );

    return true;
}

bool StreamSoundPlayer::ApplyStreamDataInfo(const StreamDataInfoDetail& streamDataInfo) NN_NOEXCEPT
{
    if ( streamDataInfo.loopFlag && !m_IsLoopFlagEnabled)
    {
        NN_SDK_ASSERT( streamDataInfo.sampleCount - streamDataInfo.loopStart >= DataBlockSizeMarginSamples );
    }
    NN_SDK_ASSERT( util::is_aligned(streamDataInfo.blockSampleCount, 32) );

    // ループなし＋波形終端以降のオフセット再生の場合は失敗
    if ( !IsValidStartOffset(streamDataInfo) )
    {
        SetFinishFlag(true);
        Stop();
        return false;
    }

#ifdef ENABLE_PRINT_STREAM_INFO
    streamDataInfo.Dump(true);
#endif

    // トラックの情報を読み取る
    ApplyTrackDataInfo(streamDataInfo);

    return true;
}

position_t StreamSoundPlayer::GetOriginalPlaySamplePosition( position_t playSamplePosition, const StreamDataInfoDetail& streamDataInfo ) const NN_NOEXCEPT
{
    if ( playSamplePosition > streamDataInfo.originalLoopEnd )
    {
        position_t loopRegionSize = streamDataInfo.originalLoopEnd - streamDataInfo.originalLoopStart;

        if ( loopRegionSize < LoopRegionSizeMin )
        {
            return streamDataInfo.originalLoopStart + ( playSamplePosition - streamDataInfo.originalLoopStart ) % loopRegionSize;
        }
        else
        {
            return streamDataInfo.originalLoopStart + ( playSamplePosition - streamDataInfo.originalLoopEnd );
        }
    }

    return playSamplePosition;
}

int StreamSoundPlayer::GetOriginalLoopCount( position_t playSamplePosition, const StreamDataInfoDetail& streamDataInfo ) const NN_NOEXCEPT
{
    if ( playSamplePosition > streamDataInfo.originalLoopEnd )
    {
        position_t loopRegionSize = streamDataInfo.originalLoopEnd - streamDataInfo.originalLoopStart;

        if ( loopRegionSize < LoopRegionSizeMin )
        {
            return static_cast<int>(( playSamplePosition - streamDataInfo.originalLoopStart ) / loopRegionSize);
        }
        else
        {
            return 1;
        }
    }

    return 0;
}

position_t StreamSoundPlayer::GetStartOffsetSamples(const StreamDataInfoDetail& streamDataInfo) NN_NOEXCEPT
{
    position_t startOffsetSamples = 0;
    if ( m_StartOffsetType == StartOffsetType_Sample )
    {
        startOffsetSamples = m_StartOffset;
    }
    else if ( m_StartOffsetType == StartOffsetType_Millisec )
    {
        startOffsetSamples =
            static_cast<position_t>(nn::atk::detail::fnd::Clamp( static_cast<uint64_t>(m_StartOffset) * streamDataInfo.sampleRate / 1000,
                        static_cast<uint64_t>(0), static_cast<uint64_t>(UINT_MAX) ));
    }

    return startOffsetSamples;
}

bool StreamSoundPlayer::IsValidStartOffset(const StreamDataInfoDetail& streamDataInfo) NN_NOEXCEPT
{
    if (streamDataInfo.loopFlag == false)
    {
        if (static_cast<size_t>(GetStartOffsetSamples(streamDataInfo)) >= streamDataInfo.sampleCount)
        {
            return false;
        }
    }

    return true;
}

void StreamSoundPlayer::ApplyTrackDataInfo(const StreamDataInfoDetail& streamDataInfo) NN_NOEXCEPT
{
    for ( auto i = 0; i < m_TrackCount; i++ )
    {
        const TrackDataInfo& trackInfo = streamDataInfo.trackInfo[i];
        m_Tracks[i].volume = trackInfo.volume;
        m_Tracks[i].pan = trackInfo.pan;
        m_Tracks[i].span = trackInfo.span;
        m_Tracks[i].mainSend = trackInfo.mainSend;
        for ( auto j = 0; j < AuxBus_Count; j++ )
        {
            m_Tracks[i].fxSend[j] = trackInfo.fxSend[j];
        }
        m_Tracks[i].lpfFreq = trackInfo.lpfFreq;
        m_Tracks[i].biquadType = trackInfo.biquadType;
        m_Tracks[i].biquadValue = trackInfo.biquadValue;
        m_Tracks[i].flags = trackInfo.flags;
        m_Tracks[i].channelCount = trackInfo.channelCount;

        // StreamChannel を StreamTrack に割り当て
        for ( auto ch = 0; ch < trackInfo.channelCount; ch++ )
        {
            m_Tracks[i].m_pChannels[ch] = &m_Channels[ trackInfo.channelIndex[ch] ];
        }
    }
}

bool StreamSoundPlayer::CheckPrefetchRevision(const StreamDataInfoDetail& streamDataInfo) const NN_NOEXCEPT
{
    // リビジョン情報がともに存在している場合のみチェック
    if ( m_IsPrefetchRevisionCheckEnabled && streamDataInfo.isRevisionCheckEnabled )
    {
        bool result = streamDataInfo.revisionValue == m_PrefetchRevisionValue;
        NN_SDK_ASSERT(result,
            "Revision mismatch detected between bfstp and bfstm file.\n Please reconvert stream data and make sure prefetch data uses same waveform as stream data.");
        return result;
    }
    return true;
}

bool StreamSoundPlayer::LoadPrefetchBlocks(StreamSoundPrefetchFileReader& reader) NN_NOEXCEPT
{
    PrefetchIndexInfo indexInfo;
    indexInfo.Initialize( m_StreamDataInfo );
    position_t sampleBeginPosition = 0;
    size_t usedPrefetchMaxSize = 0;

    for ( auto blockIndex = 0; blockIndex < m_BufferBlockCount; blockIndex++ )
    {
        PrefetchLoadDataParam loadDataParam;
        loadDataParam.Initialize();

        loadDataParam.prefetchBlockIndex = 0;
        loadDataParam.prefetchBlockBytes = 0;

        loadDataParam.sampleBegin = sampleBeginPosition;

        uint32_t  blockOffsetFromLoopEnd = 0;
        if ( indexInfo.IsOverLastBlock( blockIndex ) )
        {
            blockOffsetFromLoopEnd = indexInfo.GetBlockOffsetFromLoopEnd( blockIndex );
        }

        if ( indexInfo.IsLastBlock( blockIndex, blockOffsetFromLoopEnd ) )
        {
            PreparePrefetchOnLastBlock( &loadDataParam, indexInfo );
        }
        else if ( m_StreamDataInfo.loopFlag && indexInfo.IsOverLastBlock( blockIndex ) )
        {
            if ( indexInfo.IsLoopStartBlock( blockOffsetFromLoopEnd ) )
            {
                // ループ開始ブロックの場合はサンプル開始位置をリセットする。loadDataParam.sampleBegin も上書きされるため注意
                sampleBeginPosition = 0;
                if ( !PreparePrefetchOnLoopStartBlock( &loadDataParam, indexInfo, reader ) )
                {
                    return false;
                }
            }
            else
            {
                PreparePrefetchOnLoopBlock( &loadDataParam, indexInfo, blockOffsetFromLoopEnd );
            }
        }
        else
        {
            if ( !PreparePrefetchOnNormalBlock( &loadDataParam, blockIndex, reader ) )
            {
                return false;
            }
        }

        // TODO: startOffset 対応時には、regionInfo を加味した adpcmContext の生成が必要
        loadDataParam.blockIndex = m_LoadingBufferBlockIndex;
        loadDataParam.sampleOffset = 0;
        sampleBeginPosition += loadDataParam.samples;

        // 使用するプリフェッチデータの最大サイズを計算します。
        usedPrefetchMaxSize = std::max( usedPrefetchMaxSize, m_StreamDataInfo.blockSize * m_ChannelCount * loadDataParam.prefetchBlockIndex + loadDataParam.prefetchBlockBytes * m_ChannelCount );

        LoadStreamData(
            true,
            loadDataParam,
            m_AssignNumber,
            true,
            loadDataParam.prefetchBlockIndex,
            loadDataParam.prefetchBlockBytes
            );

        // ロードバッファブロックインデックスの更新
        // プリフェッチデータ使用時も更新するのは、waveBuffer の管理インデックスとしても使用されているため。
        m_LoadingBufferBlockIndex++;
        if ( m_LoadingBufferBlockIndex >= static_cast<uint32_t>(m_BufferBlockCount) )
        {
            // ロードバッファループ
            m_LoadingBufferBlockIndex = 0;
        }

        if ( loadDataParam.lastBlockFlag )
        {
            break;
        }
    }

    // プリフェッチデータのサイズに食い違いがないことを確認します。食い違いがある場合は以下のケースが考えられます。
    // ・プリフェッチの外部再生において、間違った bfstp のバイナリを指定してしまっている。
    // ・SoundArchivePlayer のストリームバッファの大きさやブロック数の設定と SoundMaker のストリームバッファのサイズの設定に食い違いがある。
    NN_SDK_ASSERT( usedPrefetchMaxSize <= m_PrefetchDataInfo.prefetchSize, "Used prefetch data size (%zu) > given prefetch data size (%u)", usedPrefetchMaxSize, m_PrefetchDataInfo.prefetchSize );
    if( usedPrefetchMaxSize < m_PrefetchDataInfo.prefetchSize )
    {
        NN_ATK_WARNING( "Used prefetch data size (%zu) < given prefetch data size (%u)", usedPrefetchMaxSize, m_PrefetchDataInfo.prefetchSize );
    }

    return true;
}

void StreamSoundPlayer::PreparePrefetchOnLastBlock(PrefetchLoadDataParam* param, const PrefetchIndexInfo& indexInfo) NN_NOEXCEPT
{
    // 終端ブロック処理
    param->samples            = m_StreamDataInfo.lastBlockSampleCount;
    param->prefetchBlockBytes = m_StreamDataInfo.lastBlockSize;
    param->prefetchBlockIndex = indexInfo.lastBlockIndex;

    if ( !m_StreamDataInfo.loopFlag )
    {
        param->lastBlockFlag = true;

        m_PrefetchOffset     = 0;
    }
    else
    {
        param->loopCount++;

        m_PrefetchOffset     = m_StreamDataInfo.loopStart;
    }
}

bool StreamSoundPlayer::PreparePrefetchOnLoopStartBlock(PrefetchLoadDataParam* param, const PrefetchIndexInfo& indexInfo, StreamSoundPrefetchFileReader& reader) NN_NOEXCEPT
{
    param->samples            = m_StreamDataInfo.blockSampleCount;
    param->prefetchBlockBytes = m_StreamDataInfo.blockSize;
    param->prefetchBlockIndex = indexInfo.loopStartBlockIndex;
    param->sampleBegin        = indexInfo.loopStartInBlock;
    m_PrefetchOffset          = m_StreamDataInfo.blockSampleCount * ( indexInfo.loopStartBlockIndex + 1 );

    if ( m_StreamDataInfo.sampleFormat == SampleFormat_DspAdpcm )
    {
        // ループ先頭部分の ADPCM 情報を読み取る
        if ( !SetAdpcmLoopInfo(reader, m_StreamDataInfo, m_PrefetchAdpcmParam, param->adpcmContext) )
        {
            return false;
        }

        param->adpcmContextEnable = true;
    }

    return true;
}

void StreamSoundPlayer::PreparePrefetchOnLoopBlock(PrefetchLoadDataParam* param, const PrefetchIndexInfo& indexInfo, uint32_t blockOffsetFromLoopEnd) NN_NOEXCEPT
{
    param->samples             = m_StreamDataInfo.blockSampleCount;
    param->prefetchBlockBytes  = m_StreamDataInfo.blockSize;
    param->prefetchBlockIndex  = indexInfo.loopStartBlockIndex + blockOffsetFromLoopEnd - 1;
    m_PrefetchOffset          += m_StreamDataInfo.blockSampleCount;
}

bool StreamSoundPlayer::PreparePrefetchOnNormalBlock(PrefetchLoadDataParam* param, uint32_t blockIndex, StreamSoundPrefetchFileReader& reader) NN_NOEXCEPT
{
    param->samples             = m_StreamDataInfo.blockSampleCount;
    param->prefetchBlockBytes  = m_StreamDataInfo.blockSize;
    param->prefetchBlockIndex  = blockIndex;
    m_PrefetchOffset          += m_StreamDataInfo.blockSampleCount;

    if ( m_StreamDataInfo.sampleFormat == SampleFormat_DspAdpcm && param->prefetchBlockIndex == 0 )
    {
        // 先頭サンプル部分の ADPCM 情報を読み取る
        if ( !SetAdpcmInfo(reader, m_StreamDataInfo, m_PrefetchAdpcmParam, param->adpcmContext) )
        {
            return false;
        }

        param->adpcmContextEnable = true;
    }

    return true;
}

void StreamSoundPlayer::PrefetchIndexInfo::Initialize( const StreamDataInfoDetail& streamDataInfo ) NN_NOEXCEPT
{
    lastBlockIndex      = streamDataInfo.GetLastBlockIndex();
    loopStartInBlock    = streamDataInfo.GetLoopStartInBlock();
    loopStartBlockIndex = streamDataInfo.GetLoopStartBlockIndex(loopStartInBlock);
    loopBlockCount      = lastBlockIndex - loopStartBlockIndex + 1;
}

bool StreamSoundPlayer::SetAdpcmInfo(
    StreamSoundPrefetchFileReader& reader,
    const StreamDataInfoDetail& streamDataInfo,
    AdpcmParam* adpcmParam,
    AdpcmContextNotAligned* adpcmContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT( streamDataInfo.channelCount <= StreamChannelCount );

    for ( auto ch = 0; ch < streamDataInfo.channelCount; ch++ )
    {
        DspAdpcmParam dspAdpcmParam;
        DspAdpcmLoopParam dspAdpcmLoopParam;

        if ( ! reader.ReadDspAdpcmChannelInfo(
            &dspAdpcmParam,
            &dspAdpcmLoopParam,
            ch ) )
        {
            return false;
        }

        for ( auto i = 0; i < 8; ++i )
        {
            for ( auto j = 0; j < 2; ++j )
            {
                adpcmParam[ch].coefficients[i * 2 + j] = dspAdpcmParam.coef[i][j];
            }
        }

        m_Channels[ ch ].m_pVoice->SetAdpcmParam( 0, adpcmParam[ch] );

        adpcmContext[ch].audioAdpcmContext.predScale  = dspAdpcmParam.predScale;
        adpcmContext[ch].audioAdpcmContext.history[0] = dspAdpcmParam.yn1;
        adpcmContext[ch].audioAdpcmContext.history[1] = dspAdpcmParam.yn2;
    }

    return true;
}

bool StreamSoundPlayer::SetAdpcmLoopInfo(
    StreamSoundPrefetchFileReader& reader,
    const StreamDataInfoDetail& streamDataInfo,
    AdpcmParam* adpcmParam,
    AdpcmContextNotAligned* adpcmContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT( streamDataInfo.channelCount <= StreamChannelCount );

    for ( auto ch = 0; ch < streamDataInfo.channelCount; ch++ )
    {
        DspAdpcmParam dspAdpcmParam;
        DspAdpcmLoopParam dspAdpcmLoopParam;

        if ( ! reader.ReadDspAdpcmChannelInfo(
            &dspAdpcmParam,
            &dspAdpcmLoopParam,
            ch ) )
        {
            return false;
        }

        for ( auto i = 0; i < 8; ++i )
        {
            for ( auto j = 0; j < 2; ++j )
            {
                adpcmParam[ch].coefficients[i * 2 + j] = dspAdpcmParam.coef[i][j];
            }
        }

        m_Channels[ ch ].m_pVoice->SetAdpcmParam( 0, adpcmParam[ch] );

        adpcmContext[ch].audioAdpcmContext.predScale  = dspAdpcmLoopParam.loopPredScale;
        adpcmContext[ch].audioAdpcmContext.history[0] = dspAdpcmLoopParam.loopYn1;
        adpcmContext[ch].audioAdpcmContext.history[1] = dspAdpcmLoopParam.loopYn2;
    }

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         UpdateLoadingBlockIndex

  Description:  ロードブロックインデックスの更新

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void StreamSoundPlayer::UpdateLoadingBlockIndex() NN_NOEXCEPT
{
    void* bufferAddress[ StreamChannelCount ];
    for ( auto ch = 0; ch < m_ChannelCount; ch++ )
    {
        NN_SDK_ASSERT(ch < StreamChannelCount);
        bufferAddress[ ch ] = util::BytePtr(
            m_Channels[ ch ].m_pBufferAddress,
            (m_StreamDataInfo.blockSize + StreamSoundLoader::DataBlockSizeMargin) * m_LoadingBufferBlockIndex).Get();
    }

    position_t startOffsetSamples = 0;
    if ( m_StartOffsetType == StartOffsetType_Sample )
    {
        startOffsetSamples = m_StartOffset;
    }
    else if ( m_StartOffsetType == StartOffsetType_Millisec )
    {
        startOffsetSamples =
            static_cast<position_t>(nn::atk::detail::fnd::Clamp( static_cast<uint64_t>(m_StartOffset) * m_StreamDataInfo.sampleRate / 1000,
                    static_cast<uint64_t>(0), static_cast<uint64_t>(UINT_MAX) ));
        // NN_DETAIL_ATK_INFO("UINT_MAX(%d) (%X)\n", UINT_MAX, UINT_MAX);
    }
    position_t prefetchOffsetSamples = m_PrefetchOffset;

    m_StartOffset = 0;
    m_PrefetchOffset = 0;

    NN_SDK_ASSERT(m_pLoader != nullptr);

    m_pLoader->RequestLoadData(
        bufferAddress,
        m_LoadingBufferBlockIndex,
        startOffsetSamples,
        prefetchOffsetSamples,
        IsStarted() ? detail::TaskManager::TaskPriority_High : detail::TaskManager::TaskPriority_Middle
    );

    // ロードバッファブロックインデックスの更新
    m_LoadingBufferBlockIndex++;
    if ( m_LoadingBufferBlockIndex >= static_cast<uint32_t>(m_BufferBlockCount) )
    {
        // ロードバッファループ
        m_LoadingBufferBlockIndex = 0;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         UpdatePauseStatus

  Description:  ボイスのポーズステータスを更新

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void StreamSoundPlayer::UpdatePauseStatus() NN_NOEXCEPT
{
    bool pauseStatus = false;
    if ( IsPause() )
    {
        pauseStatus = true;
    }
    if ( m_LoadWaitFlag )
    {
        pauseStatus = true;
    }

    if ( pauseStatus != m_PauseStatus )
    {
        for ( auto ch = 0; ch < m_ChannelCount; ch++ )
        {
            MultiVoice* voice = m_Channels[ ch ].m_pVoice;
            if ( voice != nullptr )
            {
                voice->Pause( pauseStatus );
            }
        }
        m_PauseStatus = pauseStatus;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         VoiceCallbackFunc

  Description:  ボイスから呼びだされるコールバック関数

  Arguments:    voice - ボイス
                status - コールバックステータス
                arg - ユーザー引数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void StreamSoundPlayer::VoiceCallbackFunc(
    MultiVoice* voice,
    MultiVoice::VoiceCallbackStatus status,
    void* arg
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( arg );
    StreamChannel* channel = reinterpret_cast<StreamChannel*>( arg );

    NN_SDK_ASSERT( channel->m_pVoice == voice );

    switch ( status )
    {
    case MultiVoice::VoiceCallbackStatus_FinishWave:
    case MultiVoice::VoiceCallbackStatus_Cancel:
        voice->Free();
        channel->m_pVoice = nullptr;
        break;
    case MultiVoice::VoiceCallbackStatus_DropVoice:
    case MultiVoice::VoiceCallbackStatus_DropDsp:
        channel->m_pVoice = nullptr;
        break;
    default:
        NN_SDK_ASSERT( false, "Unknown Voice callback status %d", status );
        return;
    }
}

void StreamSoundPlayer::SetTrackVolume( uint32_t trackBitFlag, float volume ) NN_NOEXCEPT
{
    for( auto trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_Volume = volume;
    }
}

void StreamSoundPlayer::SetTrackInitialVolume( uint32_t trackBitFlag, uint32_t volume ) NN_NOEXCEPT
{
    for( auto trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].volume = static_cast<uint8_t>(volume);
    }
}

void StreamSoundPlayer::SetTrackOutputLine( uint32_t trackBitFlag, uint32_t outputLine ) NN_NOEXCEPT
{
    for( auto trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_OutputLine = outputLine;
    }
}

void StreamSoundPlayer::ResetTrackOutputLine( uint32_t trackBitFlag ) NN_NOEXCEPT
{
    for( auto trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_OutputLine = -1;
    }
}




void StreamSoundPlayer::SetTrackTvVolume( uint32_t trackBitFlag, float volume ) NN_NOEXCEPT
{
    for( auto trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_TvParam.volume = volume;
    }
}
void StreamSoundPlayer::SetTrackChannelTvMixParameter( uint32_t trackBitFlag, uint32_t srcChNo, const MixParameter& param ) NN_NOEXCEPT
{
    for( auto trackNo = 0;
        trackNo < m_TrackCount && trackBitFlag != 0;
        trackNo++, trackBitFlag >>= 1
        )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        for ( auto i = 0; i < ChannelIndex_Count; i++ )
        {
            m_Tracks[ trackNo ].m_TvParam.mixParameter[ srcChNo ].ch[i] = param.ch[i];
        }
    }
}
void StreamSoundPlayer::SetTrackTvPan( uint32_t trackBitFlag, float pan ) NN_NOEXCEPT
{
    for( auto trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_TvParam.pan = pan;
    }
}
void StreamSoundPlayer::SetTrackTvSurroundPan( uint32_t trackBitFlag, float span ) NN_NOEXCEPT
{
    for( auto trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_TvParam.span = span;
    }
}
void StreamSoundPlayer::SetTrackTvMainSend( uint32_t trackBitFlag, float send ) NN_NOEXCEPT
{
    for( auto trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_TvParam.send[Util::GetSubMixBusFromMainBus()] = send;
    }
}
void StreamSoundPlayer::SetTrackTvFxSend( uint32_t trackBitFlag, AuxBus bus, float send ) NN_NOEXCEPT
{
    for( auto trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_TvParam.send[Util::GetSubMixBus(bus)] = send;
    }
}

StreamTrack* StreamSoundPlayer::GetPlayerTrack( int trackNo ) NN_NOEXCEPT
{
    if ( trackNo > StreamTrackCount - 1 )
    {
        return nullptr;
    }

    return &m_Tracks[ trackNo ];
}

const StreamTrack* StreamSoundPlayer::GetPlayerTrack( int trackNo ) const NN_NOEXCEPT
{
    if ( trackNo > StreamTrackCount - 1 )
    {
        return nullptr;
    }

    return &m_Tracks[ trackNo ];
}

} // namespace nn::atk::detail::driver
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

