﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_StreamSoundFileReader.h>
#include <nn/atk/atk_WaveFile.h>                // EncodeMethod
#include <nn/atk/fnd/binary/atkfnd_Binary.h>
#include <nn/util/util_BinTypes.h>

namespace nn {
namespace atk {
namespace detail {

namespace {

const uint32_t SignatureFileStm = NN_UTIL_CREATE_SIGNATURE_4( 'F', 'S', 'T', 'M' );
const uint32_t SignatureInfoBlockStm = NN_UTIL_CREATE_SIGNATURE_4( 'I', 'N', 'F', 'O' );
// const uint32_t SignatureSeekBlockStm = NN_UTIL_CREATE_SIGNATURE_4( 'S', 'E', 'E', 'K' );
// const uint32_t SignatureDataBlockStm = NN_UTIL_CREATE_SIGNATURE_4( 'D', 'A', 'T', 'A' );

const uint32_t SupportedFileVersionStm           = NN_ATK_MAKE_VERSION( 0, 1, 0, 0 );
const uint32_t CurrentFileVersionStm             = NN_ATK_MAKE_VERSION( 0, 6, 3, 0 );
const uint32_t IncludeTrackinfoVersionStm        = NN_ATK_MAKE_VERSION( 0, 2, 0, 0 );
const uint32_t IncludeOriginalloopVersionStm     = NN_ATK_MAKE_VERSION( 0, 4, 0, 0 );
const uint32_t IncludeCrc32CheckVersionStm       = NN_ATK_MAKE_VERSION( 0, 5, 0, 0 );
const uint32_t IncludeRegionIndexCheckVersionStm = NN_ATK_MAKE_VERSION( 0, 6, 1, 0 );

} // anonymous namespace

StreamSoundFileReader::StreamSoundFileReader() NN_NOEXCEPT
: m_pHeader( NULL )
, m_pInfoBlockBody( NULL )
{
}

void StreamSoundFileReader::Initialize( const void* streamSoundFile ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( streamSoundFile );

    if ( ! IsValidFileHeader( streamSoundFile ) ) return;

    m_pHeader =
        reinterpret_cast<const StreamSoundFile::FileHeader*>( streamSoundFile );
    const StreamSoundFile::InfoBlock* infoBlock = m_pHeader->GetInfoBlock();
    NN_SDK_ASSERT( infoBlock->header.kind == SignatureInfoBlockStm );
    if ( infoBlock->header.kind != SignatureInfoBlockStm )
    {
        return;
    }


    m_pInfoBlockBody = &infoBlock->body;
    NN_SDK_ASSERT( m_pInfoBlockBody->GetStreamSoundInfo()->oneBlockBytes % 32 == 0 );
}

void StreamSoundFileReader::Finalize() NN_NOEXCEPT
{
    m_pHeader = NULL;
    m_pInfoBlockBody = NULL;
}

bool StreamSoundFileReader::IsTrackInfoAvailable() const NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( m_pHeader );
    if (header.version <= IncludeTrackinfoVersionStm)
    {
        return true;
    }
    return false;
}

bool StreamSoundFileReader::IsOriginalLoopAvailable() const NN_NOEXCEPT
{
    return IsOriginalLoopAvailableImpl( m_pHeader );
}

bool StreamSoundFileReader::IsOriginalLoopAvailableImpl(const StreamSoundFile::FileHeader* pHeader) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pHeader );
    if (pHeader->version >= IncludeOriginalloopVersionStm)
    {
        return true;
    }
    return false;
}

bool StreamSoundFileReader::IsCrc32CheckAvailable() const NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>(m_pHeader);
    if (header.version >= IncludeCrc32CheckVersionStm)
    {
        return true;
    }
    return false;
}

bool StreamSoundFileReader::IsRegionIndexCheckAvailable() const NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>(m_pHeader);
    if ( header.version >= IncludeRegionIndexCheckVersionStm )
    {
        return true;
    }
    return false;
}

bool StreamSoundFileReader::ReadStreamSoundInfo(
        StreamSoundFile::StreamSoundInfo* strmInfo ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    const StreamSoundFile::StreamSoundInfo* info =
        m_pInfoBlockBody->GetStreamSoundInfo();

    NN_SDK_ASSERT( util::is_aligned(info->oneBlockBytes, 32) );
    NN_SDK_ASSERT( util::is_aligned(info->lastBlockPaddedBytes, 32) );

    strmInfo->encodeMethod = info->encodeMethod;
    strmInfo->isLoop = info->isLoop;
    strmInfo->channelCount = info->channelCount;
    strmInfo->regionCount = info->regionCount;
    strmInfo->sampleRate = info->sampleRate;
    strmInfo->loopStart = info->loopStart;
    strmInfo->frameCount = info->frameCount;
    strmInfo->blockCount = info->blockCount;

    strmInfo->oneBlockBytes = info->oneBlockBytes;
    strmInfo->oneBlockSamples = info->oneBlockSamples;

    strmInfo->lastBlockBytes = info->lastBlockBytes;
    strmInfo->lastBlockSamples = info->lastBlockSamples;
    strmInfo->lastBlockPaddedBytes = info->lastBlockPaddedBytes;

    strmInfo->sizeofSeekInfoAtom = info->sizeofSeekInfoAtom;
    strmInfo->seekInfoIntervalSamples = info->seekInfoIntervalSamples;

    strmInfo->sampleDataOffset = info->sampleDataOffset;

    strmInfo->regionInfoBytes = info->regionInfoBytes;
    strmInfo->regionDataOffset = info->regionDataOffset;

    if ( IsOriginalLoopAvailable() )
    {
        strmInfo->originalLoopStart = info->originalLoopStart;
        strmInfo->originalLoopEnd = info->originalLoopEnd;
    }
    else
    {
        strmInfo->originalLoopStart = info->loopStart;
        strmInfo->originalLoopEnd = info->frameCount;
    }

    if ( IsCrc32CheckAvailable() )
    {
        strmInfo->crc32Value = info->crc32Value;
    }
    else
    {
        strmInfo->crc32Value = 0;
    }

    return true;
}

bool StreamSoundFileReader::ReadStreamTrackInfo(
        TrackInfo* pTrackInfo, int trackIndex ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( pTrackInfo );
    const StreamSoundFile::TrackInfoTable* table =
        m_pInfoBlockBody->GetTrackInfoTable();
    if ( table == NULL )
    {
        return false;
    }

    if ( trackIndex >= static_cast<int>( table->GetTrackCount() ) )
    {
        return false;
    }

    const StreamSoundFile::TrackInfo* src = table->GetTrackInfo( trackIndex );
    pTrackInfo->volume = src->volume;
    pTrackInfo->pan = src->pan;
    pTrackInfo->span = src->span;
    pTrackInfo->flags = src->flags;
    pTrackInfo->channelCount = static_cast<uint8_t>( src->GetTrackChannelCount() ); // TODO: バイナリデータも 1 バイトでいいかも (NW4C-0.1.0)

    uint32_t count = std::min( static_cast<uint32_t>( pTrackInfo->channelCount ), WaveChannelMax );
    for ( uint32_t i = 0; i < count; i++ )
    {
        pTrackInfo->globalChannelIndex[ i ] = src->GetGlobalChannelIndex( i );
    }

    return true;
}

bool StreamSoundFileReader::ReadDspAdpcmChannelInfo(
        DspAdpcmParam* pParam,
        DspAdpcmLoopParam* pLoopParam,
        int channelIndex ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( pParam );
    NN_SDK_ASSERT_NOT_NULL( pLoopParam );

    const StreamSoundFile::DspAdpcmChannelInfo* src =
        m_pInfoBlockBody->
        GetChannelInfoTable()->
        GetChannelInfo( channelIndex )->
        GetDspAdpcmChannelInfo();
    if ( src == NULL ) return false;

    *pParam = src->param;
    *pLoopParam = src->loopParam;
    return true;
}

bool StreamSoundFileReader::IsValidFileHeader( const void* streamSoundFile ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( streamSoundFile );

    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( streamSoundFile );

    // シグニチャ確認
    NN_SDK_ASSERT(
            header.signature == SignatureFileStm,
            "invalid file signature. stream data is not available." );
    if ( header.signature != SignatureFileStm ) return false;

    // バイトオーダーマーク確認
    NN_SDK_ASSERT( header.byteOrder == BinaryFileHeader::ValidByteOrderMark,
            "invalid file byte order mark. [expected:0x%04x][this:0x%04x]", BinaryFileHeader::ValidByteOrderMark, header.byteOrder );
    if ( header.byteOrder != BinaryFileHeader::ValidByteOrderMark )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SupportedFileVersionStm <= header.version) &&
                                   (header.version <= CurrentFileVersionStm) )
    {
        isSupportedVersion = true;

    }
    NN_SDK_ASSERT( isSupportedVersion,
            "bfstm file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SupportedFileVersionStm, CurrentFileVersionStm, header.version
    );
    return isSupportedVersion;
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

