﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SequenceSoundRuntime.h>

#include <nn/atk/atk_Bank.h>
#include <nn/atk/atk_SequenceSoundFile.h>
#include <nn/atk/atk_SequenceSoundFileReader.h>
#include <nn/atk/atk_SoundArchiveParametersHook.h>
#include <nn/atk/atk_SoundPlayer.h>
#include <nn/atk/atk_SoundRuntimeUtility.h>
#include <nn/atk/atk_StartInfoReader.h>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/detail/atk_SoundArchiveManager.h>

namespace
{
    const int MemAlignSize = nn::audio::BufferAlignSize;
}

namespace nn { namespace atk { namespace detail {

    SequenceSoundRuntime::SequenceSoundRuntime() NN_NOEXCEPT
    : m_pSequenceTrackAllocator( NULL )
    , m_MmlSequenceTrackAllocator( &m_MmlParser )
    , m_SequenceUserProcCallback( NULL )
    , m_pSequenceUserProcCallbackArg( NULL )
    , m_pSoundArchiveManager(nullptr)
    , m_pSoundArchiveFilesHook(NULL)
    {
        m_SequenceCallback.Initialize(*this);
    }

    SequenceSoundRuntime::~SequenceSoundRuntime() NN_NOEXCEPT
    {
        m_pSoundArchiveManager = nullptr;
    }

    bool SequenceSoundRuntime::Initialize(
        int soundCount,
        void** pOutAllocatedAddr,
        const void* endAddr
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT( util::is_aligned(reinterpret_cast<uintptr_t>(*pOutAllocatedAddr), MemAlignSize) );
        if ( reinterpret_cast<uint64_t>(*pOutAllocatedAddr) % MemAlignSize != 0 )
        {
            return false;
        }

        // SequenceSoundInstanceManager
        {
            const detail::SoundInstanceConfig config = nn::atk::SoundSystem::GetSoundInstanceConfig();
            size_t requireSize = nn::util::align_up(
                m_SequenceSoundInstanceManager.GetRequiredMemSize(soundCount, config), MemAlignSize);

            void* estimateEndAddr = util::BytePtr( *pOutAllocatedAddr, requireSize ).Get();
            if ( util::ConstBytePtr(endAddr).Distance(estimateEndAddr) > 0 )
            {
                return false;
            }

            unsigned long createdSoundCount = m_SequenceSoundInstanceManager.Create( *pOutAllocatedAddr, requireSize, config );
            NN_SDK_ASSERT( createdSoundCount == static_cast<unsigned long>(soundCount) );
#if defined(NN_SDK_BUILD_RELEASE)
            NN_UNUSED( createdSoundCount );
#endif
            *pOutAllocatedAddr = estimateEndAddr;
        }

        // SequenceSoundLoaderManager
        {
            size_t requireSize = nn::util::align_up(
                detail::driver::SequenceSoundLoaderManager::GetRequiredMemSize(soundCount), MemAlignSize);

            void* estimateEndAddr = util::BytePtr( *pOutAllocatedAddr, requireSize ).Get();
            if ( util::ConstBytePtr(endAddr).Distance(estimateEndAddr) > 0 )
            {
                return false;
            }

            uint32_t createdSoundLoaderCount = m_SequenceSoundLoaderManager.Create( *pOutAllocatedAddr, requireSize );
            NN_SDK_ASSERT( static_cast<int32_t>(createdSoundLoaderCount) == soundCount,
                    "SequenceSoundLoaderManager::Create createdSoundLoaderCount(%d) but soundCount(%d)",
                    createdSoundLoaderCount, soundCount);
#if defined(NN_SDK_BUILD_RELEASE)
            NN_UNUSED( createdSoundLoaderCount );
#endif
            *pOutAllocatedAddr = estimateEndAddr;
        }

        // 各 SequenceSound に SequenceSoundLoaderManager をセット
        detail::SequenceSoundInstanceManager::PriorityList& list =
            m_SequenceSoundInstanceManager.GetFreeList();
        for (detail::SequenceSoundInstanceManager::PriorityList::iterator itr = list.begin();
                itr != list.end();
                ++itr)
        {
            itr->SetLoaderManager(m_SequenceSoundLoaderManager);
        }
        detail::driver::SoundThread::GetInstance().RegisterSoundFrameCallback(&m_SequenceSoundLoaderManager);

        m_pSequenceTrackAllocator = &m_MmlSequenceTrackAllocator;

        return true;
    }

    void SequenceSoundRuntime::Finalize() NN_NOEXCEPT
    {
        m_pSequenceTrackAllocator = NULL;

        m_SequenceSoundInstanceManager.Destroy();
        m_SequenceSoundLoaderManager.Destroy();
        detail::driver::SoundThread::GetInstance().UnregisterSoundFrameCallback(&m_SequenceSoundLoaderManager);
        m_MmlSequenceTrackAllocator.Destroy();
    }

    bool SequenceSoundRuntime::SetupSequenceTrack(
        int trackCount,
        void** pOutAllocatedAddr,
        const void* endAddr
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT( util::is_aligned(reinterpret_cast<uintptr_t>(*pOutAllocatedAddr), MemAlignSize) );
        if ( reinterpret_cast<uint64_t>(*pOutAllocatedAddr) % MemAlignSize != 0 )
        {
            return false;
        }

        size_t requireSize = nn::util::align_up(
            trackCount * sizeof( detail::driver::MmlSequenceTrack ), MemAlignSize );

        void* estimateEndAddr = util::BytePtr( *pOutAllocatedAddr, requireSize ).Get();
        if ( util::ConstBytePtr(endAddr).Distance(estimateEndAddr) > 0 )
        {
            return false;
        }

        int createdTrackCount = m_MmlSequenceTrackAllocator.Create( *pOutAllocatedAddr, requireSize );
#if defined(NN_SDK_BUILD_RELEASE)
        NN_UNUSED( createdTrackCount );
#endif
        NN_SDK_ASSERT( createdTrackCount == trackCount );
        *pOutAllocatedAddr = estimateEndAddr;

        return true;
    }

    void SequenceSoundRuntime::SetupUserParam(void** pOutAllocatedAddr, size_t adjustSize) NN_NOEXCEPT
    {
        void* curAddr = *pOutAllocatedAddr;

        detail::SequenceSoundInstanceManager::PriorityList& list = m_SequenceSoundInstanceManager.GetFreeList();
        for (detail::SequenceSoundInstanceManager::PriorityList::iterator itr = list.begin();
                itr != list.end();
                ++itr)
        {
            itr->SetUserParamBuffer(curAddr, adjustSize);

            curAddr = util::BytePtr(curAddr, adjustSize).Get();
        }

        *pOutAllocatedAddr = curAddr;
    }

    size_t SequenceSoundRuntime::GetRequiredMemorySize(
        const SoundArchive::SoundArchivePlayerInfo& soundArchivePlayerInfo,
        int alignment
    ) NN_NOEXCEPT
    {
        size_t size = 0;

        size += nn::util::align_up(
            SequenceSoundInstanceManager::GetRequiredMemSize(soundArchivePlayerInfo.sequenceSoundCount, nn::atk::SoundSystem::GetSoundInstanceConfig()),
            alignment );
        // NN_ATK_INIT_LOG("[SEQ] instance: max(%2d) ALIGN(%d) => %8d\n",
        //         soundArchivePlayerInfo.sequenceSoundCount, alignment, size);
        size += nn::util::align_up(
            detail::driver::SequenceSoundLoaderManager::GetRequiredMemSize(soundArchivePlayerInfo.sequenceSoundCount),
            alignment );
        // NN_ATK_INIT_LOG("[SEQ] loader:   max(%2d) ALIGN(%d) => %8d\n",
        //         soundArchivePlayerInfo.sequenceSoundCount, alignment, size);

        return size;
    }

    size_t SequenceSoundRuntime::GetRequiredSequenceTrackMemorySize(
        const SoundArchive::SoundArchivePlayerInfo& soundArchivePlayerInfo,
        int alignment
    ) NN_NOEXCEPT
    {
        size_t size = 0;

        size += nn::util::align_up(
                soundArchivePlayerInfo.sequenceTrackCount * sizeof( detail::driver::MmlSequenceTrack ),
                alignment );
        // NN_ATK_INIT_LOG("[SEQ] track:    max(%2d) x trackSize(%d) ALIGN(%d) => %8d\n",
        //         soundArchivePlayerInfo.sequenceTrackMax,
        //         sizeof(detail::driver::MmlSequenceTrack), alignment, size);

        return size;
    }

    bool SequenceSoundRuntime::IsSoundArchiveAvailable() const NN_NOEXCEPT
    {
        if ( m_pSoundArchiveManager == nullptr )
        {
            return false;
        }
        if ( !m_pSoundArchiveManager->IsAvailable() )
        {
            return false;
        }

        return true;
    }

    int SequenceSoundRuntime::GetActiveCount() const NN_NOEXCEPT
    {
        return m_SequenceSoundInstanceManager.GetActiveCount();
    }

    int SequenceSoundRuntime::GetFreeCount() const NN_NOEXCEPT
    {
        return m_SequenceSoundInstanceManager.GetFreeCount();
    }

    void SequenceSoundRuntime::SetSequenceSkipIntervalTick( int intervalTick ) NN_NOEXCEPT
    {
        detail::driver::SequenceSoundPlayer::SetSkipIntervalTick( intervalTick );
    }

    int SequenceSoundRuntime::GetSequenceSkipIntervalTick() NN_NOEXCEPT
    {
        return detail::driver::SequenceSoundPlayer::GetSkipIntervalTick();
    }

    void SequenceSoundRuntime::Update() NN_NOEXCEPT
    {
        m_SequenceSoundInstanceManager.SortPriorityList();
    }

    SequenceSound* SequenceSoundRuntime::AllocSound(
        SoundArchive::ItemId soundId,
        int priority,
        int ambientPriority,
        detail::BasicSound::AmbientInfo* ambientArgInfo,
        OutputReceiver* pOutputReceiver
    ) NN_NOEXCEPT
    {
        return SoundRuntimeUtility::AllocSound<SequenceSound>(
            &m_SequenceSoundInstanceManager,
            soundId,
            priority,
            ambientPriority,
            ambientArgInfo,
            pOutputReceiver
        );
    }

    SoundStartable::StartResult SequenceSoundRuntime::PrepareImpl(
        const SoundArchiveManager::SnapShot& snapShot,
        SoundArchive::ItemId soundId,
        detail::SequenceSound* sound,
        const SoundArchive::SoundInfo* commonInfo,
        const StartInfoReader& startInfoReader
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( sound );
        NN_SDK_ASSERT_NOT_NULL( commonInfo );

        // シーケンスサウンド情報の取得
        SoundArchive::SequenceSoundInfo info;
        if ( !SetupSequenceSoundInfo(&info, soundId, snapShot.GetCurrentSoundArchive(), startInfoReader.GetSeqInfo()) )
        {
            return SoundStartable::StartResult(SoundStartable::StartResult::ResultCode_ErrorInvalidSoundId);
        }

        // コンテキストの初期化、初期値を設定
        PrepareContext prepareContext;
        prepareContext.Initialize();
        prepareContext.canUsePlayerHeap = sound->GetSoundPlayer()->detail_CanUsePlayerHeap();
        prepareContext.sequenceOffset = info.startOffset;
        prepareContext.allocateTrackFlags = info.allocateTrackFlags;

        // シーケンスサウンドファイルの情報の取得
        {
            SoundStartable::StartResult result =
                SetupSequenceSoundFile(
                    &prepareContext,
                    *sound,
                    snapShot.GetCurrentSoundArchive(),
                    snapShot.GetCurrentSoundDataManager(),
                    *commonInfo,
                    startInfoReader.GetSeqInfo());

            if (!result.IsSuccess())
            {
                return result;
            }
        }

        // バンクファイルと波形アーカイブの情報を取得
        {
            SoundStartable::StartResult result =
                SetupBankFileAndWaveArchiveFile(
                    &prepareContext,
                    *sound,
                    info,
                    snapShot,
                    startInfoReader.GetSeqInfo());

            if (!result.IsSuccess())
            {
                return result;
            }
        }

        // コマンド化のため、トラックが足りないときのケアは無し
        sound->Setup(
                m_pSequenceTrackAllocator,
                prepareContext.allocateTrackFlags,
                &m_SequenceCallback,
                info.channelPriority,
                info.isReleasePriorityFix,
                m_SequenceUserProcCallback, m_pSequenceUserProcCallbackArg );

        detail::driver::SequenceSoundPlayer::StartInfo startInfo;
        SetupSequenceSoundPlayerStartInfo(&startInfo, prepareContext.sequenceOffset, startInfoReader);

        if ( prepareContext.isRegisterDataLoadTaskNeeded )
        {
            // 必要なデータが欠けている場合は、プレイヤーヒープへのロードタスクを投げる
            // (プリペアやバンクの設定は、タスク完了後のコールバック内で処理される)
            detail::driver::SequenceSoundLoader::LoadInfo loadInfo(
                &snapShot.GetCurrentSoundArchive(),
                &snapShot.GetCurrentSoundDataManager(),
                &prepareContext.loadTargetSequenceInfo,
                prepareContext.loadTargetBankInfos,
                sound->GetSoundPlayer()
            );
            sound->RegisterDataLoadTask( loadInfo, startInfo );
        }
        else
        {
            detail::SequenceSound::Resource res;
            {
                res.seq = prepareContext.pSequenceSoundFile;
                for ( int i = 0; i < SoundArchive::SequenceBankMax; i++ )
                {
                    res.banks[i] = prepareContext.loadTargetBankInfos[i].address;
                    res.warcs[i] = prepareContext.loadTargetWaveArchiveInfos[i].address;
                    res.warcIsIndividuals[i] = prepareContext.isLoadIndividuals[i];
                }
            };

            sound->Prepare( res, startInfo );
        }

        return SoundStartable::StartResult(SoundStartable::StartResult::ResultCode_Success);
    }

    /*--------------------------------------------------------------------------------*
      Name:         NoteOn

      Description:  シーケンスサウンドのノートオン処理

      Arguments:

      Returns:      None.
     *--------------------------------------------------------------------------------*/
    detail::driver::Channel* SequenceSoundRuntime::SequenceNoteOnCallback::NoteOn(
        detail::driver::SequenceSoundPlayer* seqPlayer,
        uint8_t bankIndex,
        const detail::driver::NoteOnInfo& noteOnInfo
    ) NN_NOEXCEPT
    {
        if ( ! m_pSequenceSoundRuntime->IsSoundArchiveAvailable() )
        {
            return NULL;
        }

        const detail::BankFileReader& bankReader = seqPlayer->GetBankFileReader( bankIndex );
        if (!bankReader.IsInitialized())
        {
            NN_ATK_WARNING("invalid bank index(%d) is used in SequenceSound.", bankIndex);
        }

        // 当該ノートに該当する波形アーカイブ ID・波形アーカイブ内インデックス取得
        detail::driver::Bank bank;
        detail::driver::Channel* channel = bank.NoteOn(
            bankReader,
            seqPlayer->GetWaveArchiveFileReader( bankIndex ),
            noteOnInfo
        );

        return channel;
    }

    void SequenceSoundRuntime::DumpMemory(const SoundArchive* pSoundArchive) const NN_NOEXCEPT
    {
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP) // Dump 用途
        NN_SDK_ASSERT_NOT_NULL(pSoundArchive);
        SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
        if ( pSoundArchive->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
        {
            size_t instanceManagerSize = m_SequenceSoundInstanceManager.GetRequiredMemSize( soundArchivePlayerInfo.sequenceSoundCount, nn::atk::SoundSystem::GetSoundInstanceConfig() );
            size_t loaderManagerSize = detail::driver::SequenceSoundLoaderManager::GetRequiredMemSize( soundArchivePlayerInfo.sequenceSoundCount );
            size_t trackInstanceSize = soundArchivePlayerInfo.sequenceTrackCount * sizeof( detail::driver::MmlSequenceTrack );
            NN_DETAIL_ATK_INFO("    sequenceSound\n"
                       "      instanceManager        %8zu bytes\n"
                       "      loaderManager          %8zu bytes\n"
                       "      trackInstance          %8zu bytes\n"
                       , instanceManagerSize, loaderManagerSize, trackInstanceSize );
        }
#else
        NN_UNUSED(pSoundArchive);
#endif
    }

    bool SequenceSoundRuntime::SetupSequenceSoundInfo(
        SoundArchive::SequenceSoundInfo* pOutInfo,
        SoundArchive::ItemId soundId,
        const SoundArchive& soundArchive,
        const SoundStartable::StartInfo::SequenceSoundInfo* pExternalSequenceSoundInfo) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( pOutInfo );

        // SoundArchive から情報を読み取る
        if ( ! soundArchive.ReadSequenceSoundInfo( pOutInfo, soundId ) )
        {
            return false;
        }

        // StartInfo に設定されている場合は、その情報で上書き
        if ( pExternalSequenceSoundInfo != nullptr )
        {
            for ( int i = 0; i < SoundArchive::SequenceBankMax; i++ )
            {
                SoundArchive::ItemId bankId = pExternalSequenceSoundInfo->bankIds[ i ];
                if ( bankId != SoundArchive::InvalidId )
                {
                    pOutInfo->bankIds[ i ] = pExternalSequenceSoundInfo->bankIds[ i ];
                }
            }
        }

        return true;
    }

    SoundStartable::StartResult SequenceSoundRuntime::SetupSequenceSoundFile(
        PrepareContext* pOutContext,
        const detail::SequenceSound& sound,
        const SoundArchive& soundArchive,
        const SoundDataManager& soundDataManager,
        const SoundArchive::SoundInfo& soundInfo,
        const SoundStartable::StartInfo::SequenceSoundInfo* pExternalSequenceSoundInfo) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutContext);

        // シーケンスファイルアクセスのフック
        if ( m_pSoundArchiveFilesHook != nullptr )
        {
            // フックされていないときに余計なコストがかからないように、if 中でラベルを取得する
            const char* soundLabel = soundArchive.GetItemLabel(sound.GetId());

            if(soundLabel != nullptr)
            {
                pOutContext->pSequenceSoundFile = reinterpret_cast<const detail::SequenceSoundFile*>(
                    m_pSoundArchiveFilesHook->GetFileAddress(
                        soundLabel,
                        detail::SoundArchiveFilesHook::ItemTypeSequenceSound,
                        detail::SoundArchiveFilesHook::FileTypeSequenceBinary));
            }
            else
            {
                pOutContext->pSequenceSoundFile = nullptr;
            }
        }

        if ( pOutContext->pSequenceSoundFile == nullptr )
        {
            pOutContext->pSequenceSoundFile = reinterpret_cast<const detail::SequenceSoundFile*>(
                soundDataManager.detail_GetFileAddress( soundInfo.fileId ) );

            // シーケンスデータの確定 (外部ファイルかどうか)
            if ( pExternalSequenceSoundInfo != nullptr )
            {
                const void* pExternalSequenceSoundFile = pExternalSequenceSoundInfo->sequenceDataAddress;
                if ( pExternalSequenceSoundFile != nullptr )
                {
                    pOutContext->pSequenceSoundFile = reinterpret_cast<const detail::SequenceSoundFile*>( pExternalSequenceSoundFile );
                }

                detail::SequenceSoundFileReader fileReader( pOutContext->pSequenceSoundFile );
                const char* startLabel = pExternalSequenceSoundInfo->startLocationLabel;
                if ( startLabel != nullptr )
                {
                    if ( ! fileReader.GetOffsetByLabel( startLabel, &pOutContext->sequenceOffset ) )
                    {
                        return SoundStartable::StartResult(SoundStartable::StartResult::ResultCode_ErrorInvalidSequenceStartLocationLabel);
                    }
                    pOutContext->sequenceOffset = detail::driver::MmlParser::ParseAllocTrack(
                        fileReader.GetSequenceData(),
                        pOutContext->sequenceOffset,
                        &pOutContext->allocateTrackFlags );
                }
            }
        }

        // シーケンスデータがない場合は、プレイヤーヒープへのロードフラグを立てる
        pOutContext->loadTargetSequenceInfo.itemId = sound.GetId();
        pOutContext->loadTargetSequenceInfo.address = pOutContext->pSequenceSoundFile;
        if ( pOutContext->pSequenceSoundFile == NULL )
        {
            if ( !pOutContext->canUsePlayerHeap )
            {
                return SoundStartable::StartResult(SoundStartable::StartResult::ResultCode_ErrorNotSequenceLoaded);
            }
            pOutContext->isRegisterDataLoadTaskNeeded = true;
        }

        return SoundStartable::StartResult(SoundStartable::StartResult::ResultCode_Success);
    }

    SoundStartable::StartResult SequenceSoundRuntime::SetupBankFileAndWaveArchiveFile(
        PrepareContext* pOutContext,
        const detail::SequenceSound& sound,
        const SoundArchive::SequenceSoundInfo& sequenceSoundInfo,
        const SoundArchiveManager::SnapShot& snapShot,
        const SoundStartable::StartInfo::SequenceSoundInfo* pExternalSequenceSoundInfo) NN_NOEXCEPT
    {
        // バンク、波形アーカイブファイルアクセスのフック
        if(m_pSoundArchiveFilesHook != NULL)
        {
            SetupBankFileAndWaveArchiveFileFromHook(pOutContext, sound, snapShot.GetCurrentSoundArchive());
        }

        for ( int i = 0; i < SoundArchive::SequenceBankMax; i++ )
        {
            // フックによりロード済みの場合はスキップ
            if ( m_pSoundArchiveFilesHook != nullptr &&
                 pOutContext->loadTargetBankInfos[i].address != nullptr &&
                 pOutContext->loadTargetWaveArchiveInfos[i].address != nullptr )
            {
                continue;
            }

            // 使用するバンクのIDを取得
            SoundArchive::ItemId bankId;
            if ( pExternalSequenceSoundInfo != nullptr )
            {
                bankId = pExternalSequenceSoundInfo->bankIds[i];
            }
            else
            {
                bankId = sequenceSoundInfo.bankIds[i];
            }

            const SoundArchive* pSoundArchive = &snapShot.GetCurrentSoundArchive();
            const SoundDataManager* pSoundDataManager = &snapShot.GetCurrentSoundDataManager();

            if ( bankId != SoundArchive::InvalidId )
            {
                if ( snapShot.GetCurrentSoundArchive().IsAddon() )
                {
                    const char* bankName = pSoundArchive->GetItemLabel(bankId);
                    NN_SDK_ASSERT_NOT_NULL(bankName);

                    // メインアーカイブ にあるかどうかをチェック
                    SoundArchive::ItemId mainArchiveBankId = snapShot.GetMainSoundArchive().GetItemId(bankName);
                    if (mainArchiveBankId != SoundArchive::InvalidId)
                    {
                        bankId = mainArchiveBankId;
                        pSoundArchive = &snapShot.GetMainSoundArchive();
                        pSoundDataManager = &snapShot.GetMainSoundDataManager();
                    }
                }
            }

            SoundArchive::BankInfo bankInfo;
            if ( pSoundArchive->ReadBankInfo( &bankInfo, bankId ) )
            {
                const void* bankFile =
                    pSoundDataManager->detail_GetFileAddress( bankInfo.fileId );
                pOutContext->loadTargetBankInfos[i].address = bankFile;

                // バンクデータがない場合は、プレイヤーヒープへのロードフラグを立てる
                if ( bankFile == nullptr )
                {
                    if (pOutContext->canUsePlayerHeap == false)
                    {
                        // バンクデータは無いが再生処理を進める
                        continue;
                    }
                    else
                    {
                        pOutContext->loadTargetBankInfos[i].itemId = bankId;
                        pOutContext->isRegisterDataLoadTaskNeeded = true;
                        break;
                    }
                }
                // バンクデータがある場合は、波形アーカイブデータがあるかチェック
                else
                {
                    detail::Util::WaveArchiveLoadStatus status =
                        detail::Util::GetWaveArchiveOfBank(
                            pOutContext->loadTargetWaveArchiveInfos[i],
                            pOutContext->isLoadIndividuals[i],
                            bankFile,
                            *pSoundArchive,
                            *pSoundDataManager );
                    switch ( status )
                    {
                    case detail::Util::WaveArchiveLoadStatus_Ok:
                    case detail::Util::WaveArchiveLoadStatus_Noneed:
                        // なにもしない (プリペア処理に移る)
                        break;
                    case detail::Util::WaveArchiveLoadStatus_Partly:
                        if ( pOutContext->canUsePlayerHeap )
                        {
                            // プレイヤーヒープがある場合は、そちらへのロードを試みる
                            // (無い場合は、そのままプリペア処理に移る)
                            pOutContext->isRegisterDataLoadTaskNeeded = true;
                        }
                        break;
                    case detail::Util::WaveArchiveLoadStatus_NotYet:
                        if ( pOutContext->canUsePlayerHeap )
                        {
                            // プレイヤーヒープへのロード
                            pOutContext->isRegisterDataLoadTaskNeeded = true;
                        }
                        else
                        {
                            // 再生失敗
                            return SoundStartable::StartResult(SoundStartable::StartResult::ResultCode_ErrorNotWarcLoaded);
                        }
                        break;
                    case detail::Util::WaveArchiveLoadStatus_Error:
                        {
                            // 再生失敗
                            return SoundStartable::StartResult(SoundStartable::StartResult::ResultCode_ErrorNotBankLoaded);
                        }
                    default:
                        NN_SDK_ASSERT( false ); // ここに来ることは無いはず
                        {
                            // 再生失敗
                            return SoundStartable::StartResult(SoundStartable::StartResult::ResultCode_ErrorUnknown);
                        }
                    }
                }
            }
        }

        return SoundStartable::StartResult(SoundStartable::StartResult::ResultCode_Success);
    } // NOLINT(impl/function_size)

    void SequenceSoundRuntime::SetupBankFileAndWaveArchiveFileFromHook(
        PrepareContext* pOutContext,
        const detail::SequenceSound& sound,
        const SoundArchive& soundArchive) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutContext);

        // フックされていないときに余計なコストがかからないように、if 中でラベルを取得する
        const char* soundLabel = soundArchive.GetItemLabel(sound.GetId());

        if(soundLabel != nullptr)
        {
            for(int i = 0; i < SoundArchive::SequenceBankMax; ++i)
            {
                // SEQ はフックを無効化していても、バンクのためにフックを一時的に有効化する
                bool isSeqHookDisabled = m_pSoundArchiveFilesHook->GetIsEnable();

                if(!isSeqHookDisabled)
                {
                    NN_SDK_ASSERT_NOT_NULL(soundArchive.detail_GetParametersHook());

                    soundArchive.detail_GetParametersHook()->SetIsEnable(true);
                    m_pSoundArchiveFilesHook->SetIsEnable(true);
                }

                const void* bankFile = m_pSoundArchiveFilesHook->GetFileAddress(
                    soundLabel,
                    detail::SoundArchiveFilesHook::ItemTypeSequenceSound,
                    detail::SoundArchiveFilesHook::FileTypeBankBinary, i);

                const void* warcFile = m_pSoundArchiveFilesHook->GetFileAddress(
                    soundLabel,
                    detail::SoundArchiveFilesHook::ItemTypeSequenceSound,
                    detail::SoundArchiveFilesHook::FileTypeWaveArchiveBinary, i);

                // 一時的に有効化したフックを無効化する
                if(!isSeqHookDisabled)
                {
                    soundArchive.detail_GetParametersHook()->SetIsEnable(false);
                    m_pSoundArchiveFilesHook->SetIsEnable(false);
                }

                if(bankFile != nullptr && warcFile != nullptr)
                {
                    pOutContext->loadTargetBankInfos[i].address = bankFile;
                    pOutContext->loadTargetWaveArchiveInfos[i].address = warcFile;
                    pOutContext->isLoadIndividuals[i] = false;
                }
                else
                {
                    pOutContext->loadTargetBankInfos[i].address = nullptr;
                    pOutContext->loadTargetWaveArchiveInfos[i].address = nullptr;
                }
            }
        }
    }

    void SequenceSoundRuntime::SetupSequenceSoundPlayerStartInfo(
        detail::driver::SequenceSoundPlayer::StartInfo* pOutStartInfo,
        uint32_t sequenceOffset,
        const StartInfoReader& startInfoReader) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutStartInfo);

        int startOffset = startInfoReader.GetStartOffset();
        detail::driver::SequenceSoundPlayer::StartOffsetType startOffsetType;
        switch ( startInfoReader.GetStartOffsetType() )
        {
        case SoundStartable::StartInfo::StartOffsetType_MilliSeconds:
            startOffsetType = detail::driver::SequenceSoundPlayer::StartOffsetType_Millisec;
            break;
        case SoundStartable::StartInfo::StartOffsetType_Tick:
            startOffsetType = detail::driver::SequenceSoundPlayer::StartOffsetType_Tick;
            break;
        case SoundStartable::StartInfo::StartOffsetType_Sample:
            startOffsetType = detail::driver::SequenceSoundPlayer::StartOffsetType_Tick;
            startOffset = 0;
            break;
        default:
            startOffsetType = detail::driver::SequenceSoundPlayer::StartOffsetType_Tick;
            startOffset = 0;
            break;
        }

        pOutStartInfo->seqOffset = static_cast<int32_t>( sequenceOffset );
        pOutStartInfo->startOffsetType = startOffsetType;
        pOutStartInfo->startOffset = startOffset;
        pOutStartInfo->delayTime = startInfoReader.GetDelayTime();
        pOutStartInfo->delayCount = startInfoReader.GetDelayCount();
        pOutStartInfo->updateType = startInfoReader.GetUpdateType();
    }

}}} // namespace nn::atk::detail
