﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SequenceSoundPlayer.h>

#include <nn/atk/atk_SequenceTrack.h>
#include <nn/atk/atk_SequenceTrackAllocator.h>
#include <nn/atk/atk_DisposeCallbackManager.h>
#include <nn/atk/atk_HardwareManager.h>
#include <nn/atk/atk_SoundSystem.h>
#include <nn/atk/atk_NoteOnCallback.h>
#include <nn/atk/atk_SoundThread.h>
#include <nn/atk/atk_PlayerHeap.h>
#include <nn/atk/atk_TaskManager.h>
#include <nn/atk/atk_SoundPlayer.h>
#include <nn/atk/atk_SoundDataManager.h>
#include <nn/atk/atk_SequenceSoundFileReader.h>
#include <nn/atk/detail/atk_Macro.h>

namespace
{
// DSP プロセスの割り込み間隔の分母・分子。
// NUMERATOR / DENOMINATOR [msec] (5) が実際の割り込み間隔。
const uint64_t IntervalMsecDenominator = 0x10000;
uint64_t GetIntervalMsecNumerator(int updateFrameCount)
{
    NN_SDK_ASSERT_GREATER_EQUAL(updateFrameCount, 0);
    return static_cast<uint64_t>(updateFrameCount) * nn::atk::detail::driver::HardwareManager::SoundFrameIntervalMsec * IntervalMsecDenominator;
}
}

namespace nn {
namespace atk {
namespace detail {
namespace driver {

/* ========================================================================
        static variable
   ======================================================================== */

NN_DEFINE_STATIC_CONSTANT( const int SequenceSoundPlayer::PlayerVariableCount );
NN_DEFINE_STATIC_CONSTANT( const int SequenceSoundPlayer::GlobalVariableCount );
NN_DEFINE_STATIC_CONSTANT( const int SequenceSoundPlayer::TrackCountPerPlayer );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SequenceSoundPlayer::AllTrackBitFlag );
NN_DEFINE_STATIC_CONSTANT( const int SequenceSoundPlayer::VariableDefaultValue );
NN_DEFINE_STATIC_CONSTANT( const int SequenceSoundPlayer::DefaultTimebase );
NN_DEFINE_STATIC_CONSTANT( const int SequenceSoundPlayer::DefaultTempo );
NN_DEFINE_STATIC_CONSTANT( const int SequenceSoundPlayer::DefaultSkipIntervalTick );
volatile int16_t SequenceSoundPlayer::m_GlobalVariable[ GlobalVariableCount ];
volatile int32_t SequenceSoundPlayer::m_SkipIntervalTickPerFrame = DefaultSkipIntervalTick;

/* ========================================================================
        public function
   ======================================================================== */

void SequenceSoundPlayer::InitSequenceSoundPlayer() NN_NOEXCEPT
{
    for( int variableNo = 0 ; variableNo < GlobalVariableCount ; variableNo++ )
    {
        m_GlobalVariable[ variableNo ] = VariableDefaultValue;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         SequenceSoundPlayer

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
SequenceSoundPlayer::SequenceSoundPlayer() NN_NOEXCEPT
: m_ReleasePriorityFixFlag(false)
, m_PanRange(1.0f)
, m_TempoRatio(1.0f)
, m_TickFraction(0.0f)
, m_SkipTimeCounter(0.0f)
, m_DelayCount(0)
, m_SequenceUserprocCallback(NULL)
, m_pSequenceUserprocCallbackArg(NULL)
, m_TickCounter(0)
, m_IsInitialized(false)
, m_IsRegisterPlayerCallback(false)
, m_pLoaderManager(NULL)
, m_pLoader(NULL)
{
    for ( int varNo = 0; varNo < PlayerVariableCount ; varNo++ )
    {
        m_LocalVariable[ varNo ] = VariableDefaultValue;
    }

    for ( int trackNo = 0; trackNo < TrackCountPerPlayer ; trackNo++ )
    {
        m_pTracks[trackNo] = NULL;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         ~SequenceSoundPlayer

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
SequenceSoundPlayer::~SequenceSoundPlayer() NN_NOEXCEPT
{
    Finalize();
}

/*--------------------------------------------------------------------------------*
  Name:         Initialize

  Description:  パラメータを初期化します

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::Initialize(OutputReceiver* pOutputReceiver) NN_NOEXCEPT
{
    BasicSoundPlayer::Initialize( pOutputReceiver );

    SetStartedFlag(false);
    SetPauseFlag(false);
    m_ReleasePriorityFixFlag       = false;
    SetActiveFlag(false);

    m_TempoRatio                   = 1.0f;
    m_TickFraction                 = 0.0f;
    m_SkipTickCounter              = 0;
    m_SkipTimeCounter              = 0.0f;
    m_DelayCount                   = 0;
    m_PanRange                     = 1.0f;
    m_TickCounter                  = 0;
    m_UpdateType                   = UpdateType_AudioFrame;

    m_SequenceUserprocCallback     = NULL;
    m_pSequenceUserprocCallbackArg = NULL;

    m_ParserParam.tempo            = DefaultTempo;
    m_ParserParam.timebase         = DefaultTimebase;
    m_ParserParam.volume.InitValue(127);
    m_ParserParam.priority         = 64;
    m_ParserParam.callback         = NULL;

    for ( int varNo = 0; varNo < PlayerVariableCount ; varNo++ )
    {
        m_LocalVariable[ varNo ] = VariableDefaultValue;
    }

    for ( int trackNo = 0; trackNo < TrackCountPerPlayer ; trackNo++ )
    {
        m_pTracks[trackNo] = NULL;
    }
    m_IsRegisterPlayerCallback = false;
    m_IsInitialized = true;
}

/*--------------------------------------------------------------------------------*
  Name:         Finalize

  Description:  シーケンスプレイヤーのシャットダウン

  Arguments:    None.

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::Finalize() NN_NOEXCEPT
{
    SetFinishFlag(true);

    FinishPlayer();

    // 無効化リストから削除
    if ( IsActive() )
    {
        DisposeCallbackManager::GetInstance().UnregisterDisposeCallback( this );
        SetActiveFlag(false);
    }

    for ( int i = 0; i < SoundArchive::SequenceBankMax; i++ )
    {
        m_BankFileReader[i].Finalize();
        m_WarcFileReader[i].Finalize();
    }

    if (m_IsInitialized)
    {
        BasicSoundPlayer::Finalize();
        m_IsInitialized = false;
    }

    FreeLoader();
}

/*--------------------------------------------------------------------------------*
  Name:         Setup

  Description:  シーケンスの初期化を行います

  Arguments:

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::Setup( const SetupArg& arg ) NN_NOEXCEPT
{
    //-----------------------------------------------------------------------------
    // プレイヤー初期化
    m_ParserParam.callback = arg.callback;

    {
        //-----------------------------------------------------------------------------
        // 全トラックを確保することができるかどうか
        {
            int trackCount = 0;
            for( uint32_t trackBitMask = arg.allocTracks; trackBitMask != 0; trackBitMask >>= 1 )
            {
                if ( trackBitMask & 0x01 ) trackCount++;
            }
            int allocatableCount = arg.trackAllocator->GetAllocatableTrackCount();
            if (trackCount > allocatableCount)
            {
                NN_ATK_WARNING("Failed to start sequence sound for not enough sequence track instance.");
                Finalize();
                return;
            }
        }

        //-----------------------------------------------------------------------------
        // トラック確保
        {
            uint32_t trackBitMask = arg.allocTracks;
            for( int trackNo = 0; trackBitMask != 0; trackNo++, trackBitMask >>= 1 )
            {
                if ( ( trackBitMask & 0x01 ) == 0 ) continue;

                SequenceTrack* track = arg.trackAllocator->AllocTrack( this );
                NN_SDK_ASSERT_NOT_NULL( track );
                SetPlayerTrack( trackNo, track );
            }
        }
    }

    // NN_DETAIL_ATK_INFO( "track rest %d\n",trackAllocator->GetAllocatableTrackCount());

    m_pSequenceTrackAllocator = arg.trackAllocator;

    // SoundThread::GetInstance().SetSoundPlay( true );

//    return SETUP_SUCCESS;
}

void SequenceSoundPlayer::ForceTrackMute( uint32_t trackMask ) NN_NOEXCEPT
{
    for( int trackNo = 0; trackNo < TrackCountPerPlayer && trackMask != 0; trackNo++, trackMask >>= 1 )
    {
        if( ( trackMask & 0x01 ) == 0 )
        {
            continue;
        }

        auto* track = GetPlayerTrack( trackNo );
        if ( track == NULL )
        {
            continue;
        }

        track->ForceMute();
    }
}

/*--------------------------------------------------------------------------------*
  Name:         Start

  Description:  準備完了したシーケンスの再生を開始します

  Arguments:    playerNo - プレイヤー番号

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::Start() NN_NOEXCEPT
{
    // プレイヤーリストに追加
    SetStartedFlag(true);
}

/*--------------------------------------------------------------------------------*
  Name:         Stop

  Description:  シーケンスを止めます

  Arguments:    playerNo - プレイヤー番号

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::Stop() NN_NOEXCEPT
{
    FinishPlayer();
}

/*--------------------------------------------------------------------------------*
  Name:         Pause

  Description:  シーケンスを一時停止または再開します

  Arguments:    playerNo - プレイヤー番号
                flag - true で一時停止、false で再開します

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::Pause( bool flag ) NN_NOEXCEPT
{
    SetPauseFlag(flag);

    // 全トラックのチャンネルを停止
    SequenceTrack* track;
    for( int trackNo = 0; trackNo < TrackCountPerPlayer ; trackNo++ )
    {
        track = GetPlayerTrack( trackNo );
        if ( track == NULL ) continue;

        track->PauseAllChannel( flag );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         Skip

  Description:  シーケンスをスキップします
                注意：サウンドフレームを生成するタイマーを一時的に止めています

  Arguments:    playerNo - プレイヤー番号
                tick - スキップ量をティックで指定します

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::Skip( StartOffsetType offsetType, int offset ) NN_NOEXCEPT
{
    if ( !IsActive() )
    {
        return;
    }

    switch ( offsetType )
    {
    case StartOffsetType_Tick:
        m_SkipTickCounter += offset;
        break;
    case StartOffsetType_Millisec:
        m_SkipTimeCounter += static_cast<float>( offset );
        break;
    default:
        break;
    }
}

void SequenceSoundPlayer::SetTempoRatio( float tempoRatio ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( tempoRatio >= 0.0f );
    m_TempoRatio = tempoRatio;
}

void SequenceSoundPlayer::SetPanRange( float panRange ) NN_NOEXCEPT
{
    m_PanRange = panRange;
}

void SequenceSoundPlayer::SetChannelPriority( int priority ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( priority >= nn::atk::ChannelPriorityMin && priority <= nn::atk::ChannelPriorityMax );
    m_ParserParam.priority = static_cast<uint8_t>( priority );
}

void SequenceSoundPlayer::SetReleasePriorityFix( bool fix ) NN_NOEXCEPT
{
    m_ReleasePriorityFixFlag = fix;
}

void SequenceSoundPlayer::SetSequenceUserprocCallback( SequenceUserProcCallback callback, void* arg ) NN_NOEXCEPT
{
    m_SequenceUserprocCallback = callback;
    m_pSequenceUserprocCallbackArg = arg;
}

void SequenceSoundPlayer::CallSequenceUserprocCallback( uint16_t procId, SequenceTrack* track ) NN_NOEXCEPT
{
    if ( m_SequenceUserprocCallback == NULL )
    {
        return;
    }

    NN_SDK_ASSERT_NOT_NULL( track );

    SequenceTrack::ParserTrackParam& trackParam = track->GetParserTrackParam();

    SequenceUserProcCallbackParam param;
    param.localVariable  = GetVariablePtr( 0 );
    param.globalVariable = GetVariablePtr( 16 );
    param.trackVariable  = track->GetVariablePtr( 0 );
    param.cmpFlag = trackParam.cmpFlag;

    m_SequenceUserprocCallback(
        procId,
        &param,
        m_pSequenceUserprocCallbackArg
    );

    trackParam.cmpFlag = param.cmpFlag;
}


/*--------------------------------------------------------------------------------*
  Name:         GetLocalVariable

  Description:  シーケンスローカル変数を取得します。

  Arguments:    player - プレイヤー
                varNo - 変数番号

  Returns:      変数の値
 *--------------------------------------------------------------------------------*/
int16_t SequenceSoundPlayer::GetLocalVariable( int varNo ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( varNo, 0, PlayerVariableCount );

    return m_LocalVariable[ varNo ];
}

/*--------------------------------------------------------------------------------*
  Name:         GetGlobalVariable

  Description:  シーケンスグローバル変数を取得します。

  Arguments:    varNo - 変数番号

  Returns:      変数の値
 *--------------------------------------------------------------------------------*/
int16_t SequenceSoundPlayer::GetGlobalVariable( int varNo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( varNo, 0, GlobalVariableCount );

    return m_GlobalVariable[ varNo ];
}

/*--------------------------------------------------------------------------------*
  Name:         SetLocalVariable

  Description:  シーケンスローカル変数を設定します。

  Arguments:    player - プレイヤー
                varNo - 変数番号
                var - 変数値

  Returns:      変数の値
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::SetLocalVariable( int varNo, int16_t var ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( varNo, 0, PlayerVariableCount );

    m_LocalVariable[ varNo ] = var;
}

/*--------------------------------------------------------------------------------*
  Name:         SetGlobalVariable

  Description:  シーケンスグローバル変数を設定します。

  Arguments:    varNo - 変数番号
                var - 変数値

  Returns:      変数の値
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::SetGlobalVariable( int varNo, int16_t var ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( varNo, 0, GlobalVariableCount );

    m_GlobalVariable[ varNo ] = var;
}

/*--------------------------------------------------------------------------------*
  Name:         SetTrackMute

  Description:  トラックをミュートまたはミュート解除します

  Arguments:    trackBitFlag - ミュート設定するトラックのビットマスク
                flag         - true でミュート、false でミュート解除します

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::SetTrackMute( uint32_t trackBitFlag, SequenceMute mute ) NN_NOEXCEPT
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetMute, mute );
}

void SequenceSoundPlayer::SetTrackSilence( unsigned long trackBitFlag, bool silenceFlag, int fadeTimes ) NN_NOEXCEPT
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetSilence, silenceFlag, fadeTimes );
}

void SequenceSoundPlayer::SetTrackVolume( uint32_t trackBitFlag, float volume ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( volume >= 0.0f );
    SetTrackParam( trackBitFlag, &SequenceTrack::SetVolume, volume );
}

void SequenceSoundPlayer::SetTrackPitch( uint32_t trackBitFlag, float pitch ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pitch >= 0.0f );
    SetTrackParam( trackBitFlag, &SequenceTrack::SetPitch, pitch );
}

void SequenceSoundPlayer::SetTrackLpfFreq( uint32_t trackBitFlag, float lpfFreq ) NN_NOEXCEPT
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetLpfFreq, lpfFreq );
}

void SequenceSoundPlayer::SetTrackBiquadFilter( uint32_t trackBitFlag, int type, float value ) NN_NOEXCEPT
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetBiquadFilter, type, value );
}

bool SequenceSoundPlayer::SetTrackBankIndex( uint32_t trackBitFlag, int bankIndex ) NN_NOEXCEPT
{
    // SequenceSoundHandle::SetTrackBankIndex でも範囲チェックしているが、念のため。
    NN_SDK_ASSERT_RANGE( bankIndex, 0, static_cast<int>(SoundArchive::SequenceBankMax) );
    if ( m_BankFileReader[ bankIndex ].IsInitialized() == false )
    {
        return false;
    }

    SetTrackParam( trackBitFlag, &SequenceTrack::SetBankIndex, bankIndex );
    return true;
}

void SequenceSoundPlayer::SetTrackTranspose( uint32_t trackBitFlag, int8_t transpose ) NN_NOEXCEPT
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetTranspose, transpose );
}

void SequenceSoundPlayer::SetTrackVelocityRange( uint32_t trackBitFlag, uint8_t range ) NN_NOEXCEPT
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetVelocityRange, range );
}

void SequenceSoundPlayer::SetTrackOutputLine( uint32_t trackBitFlag, uint32_t outputLine ) NN_NOEXCEPT
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetOutputLine, static_cast<int32_t>(outputLine) );
}

void SequenceSoundPlayer::ResetTrackOutputLine( uint32_t trackBitFlag ) NN_NOEXCEPT
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetOutputLine, -1 );
}



void SequenceSoundPlayer::SetTrackTvVolume( uint32_t trackBitFlag, float volume ) NN_NOEXCEPT
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetTvVolume, volume );
}
void SequenceSoundPlayer::SetTrackChannelTvMixParameter( uint32_t trackBitFlag, uint32_t srcChNo, const MixParameter& param ) NN_NOEXCEPT
{
    for (int32_t ch = 0; ch < ChannelIndex_Count; ch++ )
    {
        SetTrackParam( trackBitFlag, &SequenceTrack::SetTvMixParameter, srcChNo, ch, param.ch[ch] );
    }
}
void SequenceSoundPlayer::SetTrackTvPan( uint32_t trackBitFlag, float pan ) NN_NOEXCEPT
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetTvPan, pan );
}
void SequenceSoundPlayer::SetTrackTvSurroundPan( uint32_t trackBitFlag, float surroundPan ) NN_NOEXCEPT
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetTvSurroundPan, surroundPan );
}
void SequenceSoundPlayer::SetTrackTvMainSend( uint32_t trackBitFlag, float send ) NN_NOEXCEPT
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetTvMainSend, send );
}
void SequenceSoundPlayer::SetTrackTvFxSend( uint32_t trackBitFlag, AuxBus bus, float send ) NN_NOEXCEPT
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetTvFxSend, bus, send );
}

/*--------------------------------------------------------------------------------*
  Name:         InvalidateData

  Description:  指定シーケンスデータを使用しているシーケンスを無効化します

  Arguments:    start - シーケンスデータ開始アドレス
                end   - シーケンスデータ終了アドレス

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::InvalidateData( const void* start, const void* end ) NN_NOEXCEPT
{
    if ( IsActive() )
    {
        for( int trackNo = 0; trackNo < TrackCountPerPlayer ; trackNo++ )
        {
            SequenceTrack* track = GetPlayerTrack( trackNo );
            if ( track == NULL ) continue;

            const uint8_t* cur = track->GetParserTrackParam().baseAddr;
            if ( start <= cur && cur <= end )
            {
                Finalize();
                break;
            }
        }
        for ( int i = 0; i < SoundArchive::SequenceBankMax; i++ )
        {
            const void* cur = m_BankFileReader[i].GetBankFileAddress();
            if ( start <= cur && cur <= end )
            {
                m_BankFileReader[i].Finalize();
            }
        }
    }
}

/*--------------------------------------------------------------------------------*
  Name:         GetPlayerTrack

  Description:  プレイヤーが保持しているトラックを取得します

  Arguments:    trackNo  - トラック番号

  Returns:      トラックポインタ
 *--------------------------------------------------------------------------------*/
SequenceTrack* SequenceSoundPlayer::GetPlayerTrack( int trackNo ) NN_NOEXCEPT
{
    if ( trackNo > TrackCountPerPlayer - 1 )
    {
        return NULL;
    }

    return m_pTracks[ trackNo ];
}

/*--------------------------------------------------------------------------------*
  Name:         GetPlayerTrack

  Description:  プレイヤーが保持しているトラックを取得します

  Arguments:    trackNo  - トラック番号

  Returns:      トラックポインタ
 *--------------------------------------------------------------------------------*/
const SequenceTrack* SequenceSoundPlayer::GetPlayerTrack( int trackNo ) const NN_NOEXCEPT
{
    if ( trackNo > TrackCountPerPlayer - 1 )
    {
        return NULL;
    }

    return m_pTracks[ trackNo ];
}

/*--------------------------------------------------------------------------------*
  Name:         CloseTrack

  Description:  プレイヤーのトラックを閉じます。
                トラックシーケンス完了時、シーケンス２重オープン時、
                プレイヤーの完了時に呼びだされます。

  Arguments:    player : プレイヤー

  Returns:      None
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::CloseTrack( int trackNo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( trackNo, 0, TrackCountPerPlayer );

    SequenceTrack* track = GetPlayerTrack( trackNo );
    if ( track == NULL )
    {
        return;
    }

    track->Close();
    m_pSequenceTrackAllocator->FreeTrack( m_pTracks[ trackNo ] );
    m_pTracks[ trackNo ] = NULL;
}

/*--------------------------------------------------------------------------------*
  Name:         SetPlayerTrack

  Description:  プレイヤーが管理するトラックを追加します

  Arguments:    trackNo  - トラック番号
                trackID  - トラックＩＤ

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::SetPlayerTrack( int trackNo, SequenceTrack* track ) NN_NOEXCEPT
{
    if ( trackNo > TrackCountPerPlayer - 1 )
    {
        return;
    }
    m_pTracks[ trackNo ] = track;
    track->SetPlayerTrackNo( trackNo );
}


/*--------------------------------------------------------------------------------*
  Name:         FinishPlayer

  Description:  プレイヤーの完了処理を行います。

  Arguments:    player : プレイヤーポインタ

  Returns:      None
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::FinishPlayer() NN_NOEXCEPT
{
    if (m_IsRegisterPlayerCallback)
    {
        SoundThread::GetInstance().UnregisterPlayerCallback( this );
        m_IsRegisterPlayerCallback = false;
    }

    if (IsStarted())
    {
        SetStartedFlag(false);
    }

    // 全トラック解放
    for( int trackNo = 0; trackNo < TrackCountPerPlayer ; trackNo++ )
    {
        CloseTrack( trackNo );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         UpdateChannelParam

  Description:  プレイヤーの全トラックのチャンネルパラメータを更新します

  Arguments:    player - プレイヤーポインタ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::UpdateChannelParam() NN_NOEXCEPT
{
    SequenceTrack* track;
    for( int trackNo = 0; trackNo < TrackCountPerPlayer ; trackNo++ )
    {
        track = GetPlayerTrack( trackNo );
        if ( track == NULL ) continue;
        track->UpdateChannelParam();
    }
}

/*--------------------------------------------------------------------------------*
  Name:         ParseNextTick

  Description:  プレイヤーのシーケンス処理を行います

  Arguments:    player - プレイヤーポインタ
                doNoteOn - ノートオンするかどうか

  Returns:      継続時には０を、完了時には１を返します
 *--------------------------------------------------------------------------------*/
int SequenceSoundPlayer::ParseNextTick( bool doNoteOn ) NN_NOEXCEPT
{
    m_ParserParam.volume.Update();

    bool activeFlag = false;

    for ( int trackNo = 0; trackNo < TrackCountPerPlayer ; trackNo++ )
    {
        SequenceTrack* track = GetPlayerTrack( trackNo );
        if ( track == NULL )
        {
            continue;
        }

        track->UpdateChannelLength();

        if ( track->ParseNextTick( doNoteOn ) < 0 )
        {
            // トラックのシーケンス終了
            CloseTrack( trackNo );
        }

        if ( track->IsOpened() )
        {
            activeFlag = true;
        }
    }

    if ( ! activeFlag )
    {
        return 1;
    }

    return 0;
}

/*--------------------------------------------------------------------------------*
  Name:         GetVariablePtr

  Description:  変数のポインタを取得します

  Arguments:    player - プレイヤーポインタ
                varNo    - 変数番号

  Returns:      変数のポインタ
 *--------------------------------------------------------------------------------*/
volatile int16_t* SequenceSoundPlayer::GetVariablePtr( int varNo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( varNo >= 0 && varNo <= PlayerVariableCount + GlobalVariableCount );

    if ( varNo < PlayerVariableCount )
    {
        return & m_LocalVariable[ varNo ] ;
    }
    else if ( varNo < PlayerVariableCount + GlobalVariableCount )
    {
        return & m_GlobalVariable[ varNo - PlayerVariableCount ];
    }
    else
    {
        return NULL;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         Update

  Description:  プレイヤーのフレーム処理を行います

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::Update(int updateFrameCount) NN_NOEXCEPT
{
    // データロード状態の管理
    {
        switch (m_ResState)
        {
        case ResState_Assigned:
            break;
        case ResState_RecvLoadReq:
            if (TryAllocLoader())
            {
                m_pLoader->Initialize(m_LoaderArg);
            }
            else
            {
                return; // リトライする
            }
            // /* FALLTHROUGH */
        case ResState_AppendLoadTask:
            NN_SDK_ASSERT_NOT_NULL(m_pLoader);
            if (m_pLoader->TryWait())
            {
                if (m_pLoader->IsLoadSuccess())
                {
                    // タスク完了したので、プリペア
                    const SequenceSoundLoader::Data& data = m_pLoader->GetData();
                    PrepareArg arg;
                    arg.seqFile = data.seqFile;
                    arg.seqOffset = m_StartInfo.seqOffset;
                    arg.delayTime = m_StartInfo.delayTime;
                    arg.delayCount = m_StartInfo.delayCount;
                    arg.updateType = m_StartInfo.updateType;
                    for (int i = 0; i < SoundArchive::SequenceBankMax; i++)
                    {
                        arg.bankFiles[i] = data.bankFiles[i];
                        arg.warcFiles[i] = data.warcFiles[i];
                        arg.warcIsIndividuals[i] = data.warcIsIndividuals[i];
                    }
                    PrepareForPlayerHeap(arg);
                    Skip(m_StartInfo.startOffsetType, m_StartInfo.startOffset);
                }
                else
                {
                    // データロードに失敗したので終了
                    SetFinishFlag(true);
                    FinishPlayer();
                    return;
                }
            }
            else
            {
                return; // リトライする
            }
            break;
        default:
            NN_SDK_ASSERT(false);
            break;
        }
    }

    if (m_DelayCount > 0)
    {
        --m_DelayCount;
        return;
    }

    // フラグチェック
    {
        if ( ! IsActive() )
        {
            return;
        }
        if ( ! IsStarted() )
        {
            return;
        }
    }

    if ( ( m_SkipTickCounter > 0 ) || ( m_SkipTimeCounter > 0.0f ) )
    {
        SkipTick();
    }
    else if ( !IsPause() )
    {
        UpdateTick(updateFrameCount);
    }

    UpdateChannelParam();
}

bool SequenceSoundPlayer::TryAllocLoader() NN_NOEXCEPT
{
    if (m_pLoaderManager == NULL)
    {
        return false;
    }

    SequenceSoundLoader* loader = m_pLoaderManager->Alloc();
    if (loader == NULL)
    {
        return false;
    }

    m_pLoader = loader;
    m_ResState = ResState_AppendLoadTask;
    return true;
}

void SequenceSoundPlayer::FreeLoader() NN_NOEXCEPT
{
    if (m_pLoader)
    {
        m_pLoaderManager->Free(m_pLoader);
        m_pLoader = NULL;
    }
}


/*--------------------------------------------------------------------------------*
  Name:         UpdateTick

  Description:  プレイヤーのティック処理を行います

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::UpdateTick(int updateFrameCount) NN_NOEXCEPT
{
    float tickPerMsec = CalcTickPerMsec();
    if ( tickPerMsec == 0.0f )
    {
        return;
    }

    uint64_t restMsec = GetIntervalMsecNumerator(updateFrameCount);
    uint64_t nextMsec = static_cast<uint64_t>(IntervalMsecDenominator * m_TickFraction / tickPerMsec);

    while ( nextMsec < restMsec )
    {
        restMsec -= nextMsec;

        bool result = ( ParseNextTick( true ) != 0 );

        if ( result )
        {
            FinishPlayer();
            SetFinishFlag(true);
            return;
        }
        ++m_TickCounter;

        tickPerMsec = CalcTickPerMsec();
        if ( tickPerMsec == 0.0f )
        {
            return;
        }
        nextMsec = static_cast<uint64_t>(IntervalMsecDenominator / tickPerMsec);
    }
    nextMsec -= restMsec;
    m_TickFraction = nextMsec * tickPerMsec / IntervalMsecDenominator;
}

/*--------------------------------------------------------------------------------*
  Name:         SkipTick

  Description:  ティックのスキップ処理を行います

  Arguments:    None.

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void SequenceSoundPlayer::SkipTick() NN_NOEXCEPT
{
    // 全トラックの全チャンネルをリリース
    for( int trackNo = 0; trackNo < TrackCountPerPlayer ; trackNo++ )
    {
        SequenceTrack* track = GetPlayerTrack( trackNo );
        if ( track == NULL ) continue;

        track->ReleaseAllChannel( SequenceTrack::PauseReleaseValue );
        track->FreeAllChannel();
    }

    // スキップ処理
    int skipCount = 0;
    while ( m_SkipTickCounter > 0 || m_SkipTimeCounter * CalcTickPerMsec() >= 1.0f )
    {
        if ( skipCount >= m_SkipIntervalTickPerFrame )
        {
            return;
        }

        if ( m_SkipTickCounter > 0 )
        {
            --m_SkipTickCounter;
        }
        else
        {
            float tickPerMsec = CalcTickPerMsec();
            NN_SDK_ASSERT( tickPerMsec > 0.0f ); // 0.0fの時、スキップは行われない
            float msecPerTick = 1.0f / tickPerMsec;
            m_SkipTimeCounter -= msecPerTick;
        }

        if ( ParseNextTick( false ) != 0 )
        {
            FinishPlayer();
            SetFinishFlag(true);
            return;
        }
        ++skipCount;
        ++m_TickCounter;
    }

    m_SkipTimeCounter = 0.0f;
}

Channel* SequenceSoundPlayer::NoteOn(
    uint8_t bankIndex,
    const NoteOnInfo& noteOnInfo
) NN_NOEXCEPT
{
    Channel* channel = m_ParserParam.callback->NoteOn(
        this,
        bankIndex,
        noteOnInfo
    );

    return channel;
}

void SequenceSoundPlayer::Prepare(const PrepareArg& arg) NN_NOEXCEPT
{
    if ( IsActive() )
    {
        FinishPlayer();
    }

    SequenceTrack* seqTrack = GetPlayerTrack( 0 );
    if ( seqTrack == NULL )
    {
        Finalize();
        return;
    }

    // bXseq
    {
        NN_SDK_ASSERT_NOT_NULL( arg.seqFile );
        SequenceSoundFileReader reader(arg.seqFile);
        const void* seqData = reader.GetSequenceData(); // 'DATA' ブロックボディ
        seqTrack->SetSeqData( seqData, arg.seqOffset );
        seqTrack->Open();
    }

    // bXbnk, bXwar
    for ( int i = 0; i < SoundArchive::SequenceBankMax; i++ )
    {
        m_BankFileReader[i].Initialize( arg.bankFiles[i] );
        m_WarcFileReader[i].Initialize( arg.warcFiles[i], arg.warcIsIndividuals[i] );
    }

    m_ResState = ResState_Assigned;
    m_DelayCount = arg.delayCount != 0 ? arg.delayCount : ToDelayCount(arg.delayTime);
    SetActiveFlag(true);
    DisposeCallbackManager::GetInstance().RegisterDisposeCallback( this );
    SoundThread::GetInstance().RegisterPlayerCallback( this );
    m_IsRegisterPlayerCallback = true;
    m_UpdateType = arg.updateType;
}

void SequenceSoundPlayer::PrepareForPlayerHeap(const PrepareArg& arg) NN_NOEXCEPT
{
    if ( IsActive() )
    {
        FinishPlayer();
    }

    SequenceTrack* seqTrack = GetPlayerTrack( 0 );
    if ( seqTrack == NULL )
    {
        Finalize();
        return;
    }

    // bXseq
    {
        NN_SDK_ASSERT_NOT_NULL( arg.seqFile );
        SequenceSoundFileReader reader(arg.seqFile);
        const void* seqData = reader.GetSequenceData(); // 'DATA' ブロックボディ
        seqTrack->SetSeqData( seqData, arg.seqOffset );
        seqTrack->Open();
    }

    // bXbnk, bXwar
    for ( int i = 0; i < SoundArchive::SequenceBankMax; i++ )
    {
        m_BankFileReader[i].Initialize( arg.bankFiles[i] );
        m_WarcFileReader[i].Initialize( arg.warcFiles[i], arg.warcIsIndividuals[i] );
    }

    m_ResState = ResState_Assigned;
    m_DelayCount = arg.delayCount != 0 ? arg.delayCount : ToDelayCount(arg.delayTime);
    SetActiveFlag(true);
    DisposeCallbackManager::GetInstance().RegisterDisposeCallback(this);
    m_IsRegisterPlayerCallback = true;
    m_UpdateType = arg.updateType;
}

void SequenceSoundPlayer::RequestLoad(
        const StartInfo& info,
        const SequenceSoundLoader::Arg& arg) NN_NOEXCEPT
{
    m_StartInfo = info; // 構造体コピー
    m_LoaderArg = arg; // 構造体コピー
    m_ResState = ResState_RecvLoadReq;
    m_DelayCount = 0; // Prepareが済んでからカウントを開始するように実装していますが、念のためにここでは0にしておく

    SoundThread::GetInstance().RegisterPlayerCallback(this);
    m_IsRegisterPlayerCallback = true;
}

nn::os::Tick SequenceSoundPlayer::GetProcessTick(const SoundProfile& profile) NN_NOEXCEPT
{
    nn::os::Tick totalTick(0);
    for ( int trackIndex = 0; trackIndex < TrackCountPerPlayer ; trackIndex++ )
    {
        // 全トラックの DSP 負荷の総和を取得
        SequenceTrack* pTrack = m_pTracks[trackIndex];
        if ( pTrack != nullptr )
        {
            totalTick += pTrack->GetProcessTick(profile);
        }
    }
    return totalTick;
}

void SequenceSoundPlayer::PrepareForMidi(
        const void* banks[],
        const void* warcs[],
        bool warcIsIndividuals[] ) NN_NOEXCEPT
{
    for ( int i = 0; i < SoundArchive::SequenceBankMax; i++ )
    {
        m_BankFileReader[i].Initialize( banks[i] );
        m_WarcFileReader[i].Initialize( warcs[i], warcIsIndividuals[i] );
    }
    m_ResState = ResState_Assigned;

    if ( !IsActive() )
    {
        DisposeCallbackManager::GetInstance().RegisterDisposeCallback( this );
        m_IsRegisterPlayerCallback = true;
        SetActiveFlag(true);
        SoundThread::GetInstance().RegisterPlayerCallback( this );
    }
}

void SequenceSoundPlayer::SetSkipIntervalTick( int intervalTick ) NN_NOEXCEPT
{
    m_SkipIntervalTickPerFrame = intervalTick;
}

int SequenceSoundPlayer::GetSkipIntervalTick() NN_NOEXCEPT
{
    return m_SkipIntervalTickPerFrame;
}







SequenceSoundLoader::LoadInfo::LoadInfo(
    const SoundArchive* arc,
    const SoundDataManager* mgr,
    LoadItemInfo* seq,
    LoadItemInfo* banks,
    SoundPlayer* player) NN_NOEXCEPT
: soundArchive(arc)
, soundDataManager(mgr)
, loadInfoSeq(seq)
, soundPlayer(player)
{
    for (int i = 0; i < SoundArchive::SequenceBankMax; i++)
    {
        loadInfoBanks[i] = &banks[i];
    }
}

SequenceSoundLoader::~SequenceSoundLoader() NN_NOEXCEPT
{
    m_Task.Wait();
    m_FreePlayerHeapTask.Wait();
}

void SequenceSoundLoader::Initialize(const Arg& arg) NN_NOEXCEPT
{
    m_Task.Wait();
    m_FreePlayerHeapTask.Wait();

    NN_SDK_ASSERT(SoundSystem::detail_IsTaskThreadEnabled(), "Task thread is not enabled.");
    m_Task.Initialize();
    m_Task.m_Arg = arg;
    m_Task.m_pPlayerHeap = NULL;
    m_Task.m_pPlayerHeapDataManager = &m_PlayerHeapDataManager;

    m_FreePlayerHeapTask.Initialize();
    m_FreePlayerHeapTask.m_Arg = arg;
    m_FreePlayerHeapTask.m_pPlayerHeap = nullptr;
    m_FreePlayerHeapTask.m_pPlayerHeapDataManager = &m_PlayerHeapDataManager;
}

void SequenceSoundLoader::Finalize() NN_NOEXCEPT
{
    m_FreePlayerHeapTask.m_pPlayerHeap = m_Task.m_pPlayerHeap;
    TaskManager::GetInstance().AppendTask(&m_FreePlayerHeapTask, TaskManager::TaskPriority_Middle);
}

bool SequenceSoundLoader::TryWait() NN_NOEXCEPT
{
    // プレイヤーヒープの確保
    if (m_Task.TryAllocPlayerHeap() == false)
    {
        return false;
    }

    // データのアペンド
    Task::Status status = m_Task.GetStatus();
    switch (status)
    {
    case Task::Status_Free:
        TaskManager::GetInstance().AppendTask(&m_Task, TaskManager::TaskPriority_Middle);
        return false;
    case Task::Status_Done:
    case Task::Status_Cancel:
        return true;
    default:
        return false;
    }
}

bool SequenceSoundLoader::IsInUse() NN_NOEXCEPT
{
    if (!m_Task.TryWait())
    {
        return true;
    }
    if (!m_FreePlayerHeapTask.TryWait())
    {
        return true;
    }

    return false;
}







void SequenceSoundLoader::DataLoadTask::Initialize() NN_NOEXCEPT
{
    Task::InitializeStatus();
    m_Data.Initialize();
    m_IsLoadSuccess = false;
}

bool SequenceSoundLoader::DataLoadTask::TryAllocPlayerHeap() NN_NOEXCEPT
{
    if (m_pPlayerHeap == NULL)
    {
        NN_SDK_ASSERT_NOT_NULL(m_Arg.soundPlayer);
        m_pPlayerHeap = m_Arg.soundPlayer->detail_AllocPlayerHeap();
        if (m_pPlayerHeap == NULL)
        {
            return false;
        }
    }
    return true;
}

void SequenceSoundLoader::DataLoadTask::Execute(TaskProfileLogger& logger) NN_NOEXCEPT
{
    NN_UNUSED( logger );
    NN_SDK_ASSERT_NOT_NULL(m_pPlayerHeap);
    NN_SDK_ASSERT_NOT_NULL(m_pPlayerHeapDataManager);

    m_pPlayerHeapDataManager->Initialize(m_Arg.soundArchive);

    if ( ( m_Arg.loadInfoSeq.address == NULL ) &&
         ( m_Arg.loadInfoSeq.itemId != SoundArchive::InvalidId ) )
    {
        SoundArchive::ItemId soundId = m_Arg.loadInfoSeq.itemId;

        if ( ! m_pPlayerHeapDataManager->LoadData(
                    soundId, m_pPlayerHeap, SoundArchiveLoader::LoadFlag_Seq ) )
        {
            NN_ATK_WARNING("failed to load SEQ(%08x) to PlayerHeap", soundId);
            m_pPlayerHeap->SetState(PlayerHeap::State_TaskFinished);
            m_IsLoadSuccess = false;
            return;
        }
        m_Arg.loadInfoSeq.address =
            m_pPlayerHeapDataManager->detail_GetFileAddressByItemId( soundId );
    }

    NN_SDK_ASSERT_NOT_NULL(m_Arg.loadInfoSeq.address);

    LoadItemInfo warcInfos[SeqBankMax];
    bool isLoadIndividuals[SeqBankMax] = {false};

    for ( int i = 0; i < SeqBankMax; i++ )
    {
        if ( ( m_Arg.loadInfoBanks[i].address == NULL ) &&
             ( m_Arg.loadInfoBanks[i].itemId != SoundArchive::InvalidId ) )
        {
            SoundArchive::ItemId bankId = m_Arg.loadInfoBanks[i].itemId;
            if ( ! m_pPlayerHeapDataManager->LoadData(
                        bankId, m_pPlayerHeap, SoundArchiveLoader::LoadFlag_Bank ) )
            {
                // ロードリクエストがあるのにロードできなかった場合
                NN_ATK_WARNING("failed to load BANK(%08x) to PlayerHeap", bankId);
                m_pPlayerHeap->SetState(PlayerHeap::State_TaskFinished);
                m_IsLoadSuccess = false;
                return;
            }
            m_Arg.loadInfoBanks[i].address =
                m_pPlayerHeapDataManager->detail_GetFileAddressByItemId( bankId );
        }

        if ( m_Arg.loadInfoBanks[i].itemId != SoundArchive::InvalidId &&
             m_Arg.loadInfoBanks[i].address != NULL )
        {
            // SoundDataManager でロード済みか？
            detail::Util::WaveArchiveLoadStatus status = Util::GetWaveArchiveOfBank(
                    warcInfos[i],
                    isLoadIndividuals[i],
                    m_Arg.loadInfoBanks[i].address,
                    *m_Arg.soundArchive,
                    *m_Arg.soundDataManager );

            switch ( status )
            {
            case detail::Util::WaveArchiveLoadStatus_Ok:
            case detail::Util::WaveArchiveLoadStatus_Noneed:
                // なにもしない
                break;
            case detail::Util::WaveArchiveLoadStatus_Partly:
            case detail::Util::WaveArchiveLoadStatus_NotYet:
                // ロードを試みる
                if ( ! m_pPlayerHeapDataManager->detail_LoadWaveArchiveByBankFile(
                            m_Arg.loadInfoBanks[i].address, m_pPlayerHeap ) )
                {
                    NN_ATK_WARNING(
                            "failed to load WARC of BANK(%08x) to PlayerHeap",
                            m_Arg.loadInfoBanks[i].itemId );
                    m_pPlayerHeap->SetState(PlayerHeap::State_TaskFinished);
                    m_IsLoadSuccess = false;
                    return;
                }
                // ロード成功時、warcInfos と isLoadIndividuals を埋める
                {
                    status = Util::GetWaveArchiveOfBank(
                        warcInfos[i],
                        isLoadIndividuals[i],
                        m_Arg.loadInfoBanks[i].address,
                        *m_Arg.soundArchive,
                        *m_pPlayerHeapDataManager );
                }
                break;
            case detail::Util::WaveArchiveLoadStatus_Error:
            default:
                NN_SDK_ASSERT( false ); // ここに来ることは無いはず
                break;
            }
        }
    }

    m_Data.seqFile = m_Arg.loadInfoSeq.address;
    for ( int i = 0; i < SeqBankMax; i++ )
    {
        m_Data.bankFiles[i] = m_Arg.loadInfoBanks[i].address;
        m_Data.warcFiles[i] = warcInfos[i].address;
        m_Data.warcIsIndividuals[i] = isLoadIndividuals[i];
    }

    m_pPlayerHeap->SetState(PlayerHeap::State_TaskFinished);
    m_IsLoadSuccess = true;
}







void SequenceSoundLoader::FreePlayerHeapTask::Initialize() NN_NOEXCEPT
{
    Task::InitializeStatus();
}

void SequenceSoundLoader::FreePlayerHeapTask::Execute(TaskProfileLogger& logger) NN_NOEXCEPT
{
    NN_UNUSED( logger );
    if (m_pPlayerHeap)
    {
        m_pPlayerHeap->Clear();
        m_Arg.soundPlayer->detail_FreePlayerHeap(m_pPlayerHeap);
    }

    NN_SDK_ASSERT_NOT_NULL(m_pPlayerHeapDataManager);
    m_pPlayerHeapDataManager->Finalize();

}

} // namespace nn::atk::detail::driver
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

