﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_MidiSequencePlayer.h>

#include <nn/atk/atk_Bank.h>
#include <nn/atk/atk_SequenceTrack.h>

#include <nn/atk/fnd/basis/atkfnd_Inlines.h>

namespace nn {
namespace atk {
namespace detail {
namespace driver {

NN_DEFINE_STATIC_CONSTANT( const int MidiSequencePlayer::NoteInfoCount );

/* ========================================================================
        public function
   ======================================================================== */

/*--------------------------------------------------------------------------------*
  Name:         MidiSequencePlayer

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
MidiSequencePlayer::MidiSequencePlayer() NN_NOEXCEPT
{
    for( int i = 0 ; i < NoteInfoCount ; i++ ) {
        m_NoteInfo[i].channel = NULL;
    }
}

MidiSequencePlayer::~MidiSequencePlayer() NN_NOEXCEPT
{
}

void MidiSequencePlayer::Prepare(
                                 const void* banks[],
                                 const void* warcs[],
                                 bool warcIsIndividuals[] ) NN_NOEXCEPT
{
    SequenceSoundPlayer::PrepareForMidi( banks, warcs, warcIsIndividuals );
}

void MidiSequencePlayer::Start() NN_NOEXCEPT
{
    SequenceSoundPlayer::Start();

    // 全てのトラックをOpenしないとトラックのチャンネルがUpdateされない
    for( int trackNo = 0; trackNo < TrackCountPerPlayer ; trackNo++ )
    {
        SequenceTrack* track = GetPlayerTrack( trackNo );
        if ( track == NULL ) continue;
        track->Open();
    }
}

void MidiSequencePlayer::SendMessage( uint8_t status, uint8_t data1, uint8_t data2 ) NN_NOEXCEPT
{
    uint8_t channel = static_cast<uint8_t>(status & 0x0f);

    switch ( status & 0xf0 )
    {
    case MidiMessage_NoteOff:
        NoteOff( channel, data1, data2 );
        break;

    case MidiMessage_NoteOn:
        if (data2 != 0)
        {
            NoteOn( channel, data1, data2 );
        }
        else
        {
            NoteOff( channel, data1, data2 );
        }
        break;

    case MidiMessage_PolyKeyPressure:
        break;

    case MidiMessage_ControlChange:
        if ( data1 < 120 )
        {
            HandleControlChangeMessage( channel, data1, data2 );
        }
        else
        {
            HandleChannelModeMessage( data1, data2 );
        }
        break;

    case MidiMessage_ProgramChange:
        HandleProgramChangeMessage( channel, data1 );
        break;

    case MidiMessage_ChannelPressure:
        break;

    case MidiMessage_PitchBend:
        HandlePitchBendMessage( channel, data1, data2 );
        break;

    case MidiMessage_System:
        switch (status)
        {
        case 0xfe:                    // active sencing
            break;
        case 0xff:                    // reset
            Reset();
            break;
        default:
            break;
        }
        break;

    default:
        break;
    }
}

void MidiSequencePlayer::SetProgramNumber( int channelIndex, int prgNo ) NN_NOEXCEPT
{
    SequenceTrack* track = GetPlayerTrack( channelIndex );
    track->GetParserTrackParam().prgNo = prgNo;
}

void MidiSequencePlayer::NoteOn( uint8_t channelIndex, uint8_t key, uint8_t velocity ) NN_NOEXCEPT
{
    SequenceTrack* track = GetPlayerTrack( channelIndex );
    if ( track == NULL ) return;

    SequenceTrack::ParserTrackParam& trackParam = track->GetParserTrackParam();

    key = static_cast<uint8_t>( nn::atk::detail::fnd::Clamp( key + trackParam.transpose, 0, 127 ) );

    if ( trackParam.monophonicFlag )
    {
        NoteInfo* noteInfo = FindNoteInfo( channelIndex );
        if ( noteInfo != NULL )
        {
            Channel* channel = noteInfo->channel;
            if ( ! channel->IsRelease() )
            {
                channel->SetKey( key );
                channel->SetVelocity( Bank::CalcChannelVelocityVolume(velocity) );
                noteInfo->key = key;
                return;
            }
            channel->Stop();
            channel->CallChannelCallback( Channel::ChannelCallbackStatus_Stopped );
            Channel::FreeChannel( channel );
            noteInfo->channel = NULL;
        }
    }

    Channel* channel = track->NoteOn( key, velocity, -1, false );
    if ( channel == NULL ) return;

    NoteInfo* noteInfo = FindFreeNoteInfo();
    NN_SDK_ASSERT_NOT_NULL( noteInfo );

    noteInfo->channel = channel;
    noteInfo->channelIndex = channelIndex;
    noteInfo->key = key;
}

void MidiSequencePlayer::NoteOff( uint8_t channelIndex, uint8_t key, uint8_t velocity ) NN_NOEXCEPT
{
    (void)velocity;

    SequenceTrack* track = GetPlayerTrack( channelIndex );
    if ( track == NULL ) return;
    SequenceTrack::ParserTrackParam& param = track->GetParserTrackParam();

    key = static_cast<uint8_t>( nn::atk::detail::fnd::Clamp( key + param.transpose, 0, 127 ) );

    for( ;; )
    {
        NoteInfo* noteInfo = FindNoteInfo( channelIndex, key );
        if ( noteInfo == NULL ) break;

        noteInfo->channel->SetLength( 0 );
        track->UpdateChannelRelease( noteInfo->channel );
        noteInfo->channel = NULL;
    }
}

void MidiSequencePlayer::HandleControlChangeMessage( uint8_t channelIndex, uint8_t control, uint8_t value ) NN_NOEXCEPT
{
    SequenceTrack* track = GetPlayerTrack( channelIndex );
    if ( track == NULL ) return;
    SequenceSoundPlayer* player = track->GetSequenceSoundPlayer();
    if ( player == NULL ) return;

    SequenceTrack::ParserTrackParam& param = track->GetParserTrackParam();
    SequenceSoundPlayer::ParserPlayerParam& playerParam = player->GetParserPlayerParam();

    switch ( control )
    {
    case MidiControlChange_BankSelect:
        if ( value <= 3 )   // value は uint8_t なので正であることが保証されている
        {
            param.bankIndex = value;
        }
        break;
    case MidiControlChange_ModDepth:
        param.lfoParam[0].depth = value / 128.0f;
        break;
    case MidiControlChange_InitPan:
        param.initPan = static_cast<int8_t>(value - 64);
        break;
    case MidiControlChange_PortaTime:
        param.portaTime = value;
        break;
    case MidiControlChange_DataEntry:
        switch ( m_ParameterControlInfo[ channelIndex ].mode )
        {
        case ParameterControlInfo::Mode_Rpn:
            HandleRpnMessage( channelIndex, value );
            break;
        case ParameterControlInfo::Mode_Nrpn:
            HandleNrpnMessage( channelIndex, value );
            break;
        default:
            break;
        }
        break;
    case MidiControlChange_Volume:
        param.volume.SetTarget( value, 0 );
        break;
    case MidiControlChange_Pan:
        param.pan.SetTarget( static_cast<int8_t>(value - 64), 0 );
        break;
    case MidiControlChange_Expression:
        param.volume2.SetTarget( value, 0 );
        break;
    case MidiControlChange_MainVolume:
        playerParam.volume.SetTarget( value, 0 );
        break;
    case MidiControlChange_Transpose:
        param.transpose = static_cast<int8_t>(value - 64);
        break;
    case MidiControlChange_Priority:
        param.priority = value;
        break;
    case MidiControlChange_Setvar0:
    case MidiControlChange_Setvar1:
    case MidiControlChange_Setvar2:
    case MidiControlChange_Setvar3:
        track->SetTrackVariable( control - MidiControlChange_Setvar0, value );
        break;

    case MidiControlChange_BendRange:
        param.bendRange = value;
        break;
    case MidiControlChange_ModSpeed:
        param.lfoParam[0].speed = value * 0.390625f;
        break;
    case MidiControlChange_ModType:
        param.lfoTarget[0] = value;
        break;
    case MidiControlChange_ModRange:
        param.lfoParam[0].range = value;
        break;
    case MidiControlChange_BiquadType:
        param.biquadType = value;
        break;
    case MidiControlChange_BiquadValue:
        param.biquadValue = static_cast<float>( value ) / 127.0f;
        break;
    case MidiControlChange_ModDelay:
        param.lfoParam[0].delay = static_cast<uint32_t>( value * 5 );
        break;
    case MidiControlChange_ModDelay10:
        param.lfoParam[0].delay = static_cast<uint32_t>( (value * 5) * 10 );
        break;
    case MidiControlChange_SweepPitch:
        param.sweepPitch = (value - 64) / 64.0f;
        break;
    case MidiControlChange_SweepPitch24:
        param.sweepPitch = (value - 64) / 64.0f * 24;
        break;
    case MidiControlChange_Damper:
        param.damperFlag = (value >= 64) ? true : false;
        break;
    case MidiControlChange_Porta:
        param.portaFlag = (value >= 64) ? true : false;
        break;
    case MidiControlChange_Monophonic:
        param.monophonicFlag = (value >= 64) ? true : false;
        if ( param.monophonicFlag )
        {
            NoteOffAll();
        }
        break;
    case MidiControlChange_Cutoff:
        param.lpfFreq = static_cast<float>( value - 64 ) / 64.0f;
        break;
    case MidiControlChange_PortaControl:
        param.portaKey = static_cast<uint8_t>(value + param.transpose);
        param.portaFlag = true;
        break;
    case MidiControlChange_Attack:
        param.attack = value;
        break;
    case MidiControlChange_Decay:
        param.decay = value;
        break;
    case MidiControlChange_Sustain:
        param.sustain = value;
        break;
    case MidiControlChange_Release:
        param.release = value;
        break;
    case MidiControlChange_EnvHold:
        param.envHold = value;
        break;
    case MidiControlChange_FxsendA:
        param.fxSend[ AuxBus_A ] = value;
        break;
    case MidiControlChange_FxsendB:
        param.fxSend[ AuxBus_B ] = value;
        break;
    case MidiControlChange_Mainsend:
        param.mainSend = value;
        break;
    case MidiControlChange_NrpnLsb:
        m_ParameterControlInfo[ channelIndex ].nrpnLsb = value;
        m_ParameterControlInfo[ channelIndex ].mode = ParameterControlInfo::Mode_Nrpn;
        break;
    case MidiControlChange_NrpnMsb:
        m_ParameterControlInfo[ channelIndex ].nrpnMsb = value;
        m_ParameterControlInfo[ channelIndex ].mode = ParameterControlInfo::Mode_Nrpn;
        break;
    case MidiControlChange_RpnLsb:
        m_ParameterControlInfo[ channelIndex ].rpnLsb = value;
        m_ParameterControlInfo[ channelIndex ].mode = ParameterControlInfo::Mode_Rpn;
        break;
    case MidiControlChange_RpnMsb:
        m_ParameterControlInfo[ channelIndex ].rpnMsb = value;
        m_ParameterControlInfo[ channelIndex ].mode = ParameterControlInfo::Mode_Rpn;
        break;
    case MidiControlChange_Frontbypass:
        param.frontBypassFlag = (value >= 64) ? true : false;
        break;
    default:
        break;
    }
} // NOLINT(impl/function_size)

void MidiSequencePlayer::HandleRpnMessage( uint8_t channelIndex, uint8_t value ) NN_NOEXCEPT
{
    int parameterNumber =
        ( static_cast<int>( m_ParameterControlInfo[ channelIndex ].rpnMsb ) << 8 ) +
        m_ParameterControlInfo[ channelIndex ].rpnLsb
        ;

    SequenceTrack* track = GetPlayerTrack( channelIndex );
    if ( track == NULL ) return;
    SequenceSoundPlayer* player = track->GetSequenceSoundPlayer();
    if ( player == NULL ) return;

    SequenceTrack::ParserTrackParam& param = track->GetParserTrackParam();

    switch ( parameterNumber )
    {
    case MidiRpn_PitchbendSensitivity:
        param.bendRange = value;
        break;
    case MidiRpn_Null:
        break;
    default:
        break;
    }
}

void MidiSequencePlayer::HandleNrpnMessage( uint8_t channelIndex, uint8_t value ) NN_NOEXCEPT
{
    int parameterNumber =
        ( static_cast<int>( m_ParameterControlInfo[ channelIndex ].nrpnMsb ) << 8 ) +
        m_ParameterControlInfo[ channelIndex ].nrpnLsb
        ;

    SequenceTrack* track = GetPlayerTrack( channelIndex );
    if ( track == NULL ) return;
    SequenceSoundPlayer* player = track->GetSequenceSoundPlayer();
    if ( player == NULL ) return;

    SequenceTrack::ParserTrackParam& param = track->GetParserTrackParam();

    (void)value;

    switch ( parameterNumber )
    {
    case MidiNrpn_EnvReset:
        param.attack = SequenceTrack::InvalidEnvelope;
        param.decay = SequenceTrack::InvalidEnvelope;
        param.sustain = SequenceTrack::InvalidEnvelope;
        param.release = SequenceTrack::InvalidEnvelope;
        param.envHold = SequenceTrack::InvalidEnvelope;
        break;
    case MidiNrpn_Null:
        break;
    default:
        break;
    }
}

void MidiSequencePlayer::HandleChannelModeMessage( uint8_t control, uint8_t value ) NN_NOEXCEPT
{
    (void)value;

    switch( control )
    {
    case MidiModeMessage_AllSoundOff:
        StopAllSound();
        break;
    case MidiModeMessage_ResetAllController:
        ResetAllController();
        break;
    case MidiModeMessage_LocalControl:
        break;
    case MidiModeMessage_AllNoteOff:
        NoteOffAll();
        break;
    case MidiModeMessage_OmniOff:
    case MidiModeMessage_OmniOn:
    case MidiModeMessage_MonoMode:
    case MidiModeMessage_PolyMode:
        break;
    default:
        break;
    }
}

void MidiSequencePlayer::HandleProgramChangeMessage( uint8_t channelIndex, uint8_t program ) NN_NOEXCEPT
{
    SetProgramNumber( channelIndex, program );
}

void MidiSequencePlayer::HandlePitchBendMessage( uint8_t channelIndex, uint8_t lsb, uint8_t msb ) NN_NOEXCEPT
{
    int16_t value = static_cast<int16_t>( ((msb << 7) | lsb) - 0x2000 );
    value >>= 6;

    SequenceTrack* track = GetPlayerTrack( channelIndex );
    if ( track == NULL ) return;
    SequenceTrack::ParserTrackParam& param = track->GetParserTrackParam();
    param.pitchBend.SetTarget( static_cast<int8_t>(value), 0 );
}

void MidiSequencePlayer::NoteOffAll() NN_NOEXCEPT
{
    for( int i = 0 ; i < NoteInfoCount ; i++ ) {
        if ( m_NoteInfo[i].channel != NULL )
        {
            m_NoteInfo[i].channel->Release();
            m_NoteInfo[i].channel = NULL;
        }
    }
}

void MidiSequencePlayer::StopAllSound() NN_NOEXCEPT
{
    for( int i = 0 ; i < NoteInfoCount ; i++ ) {
        if ( m_NoteInfo[i].channel != NULL )
        {
            m_NoteInfo[i].channel = NULL;
        }
    }

    for ( int channelIndex = 0 ; channelIndex < TrackCountPerPlayer ; channelIndex++ ) {
        SequenceTrack* track = GetPlayerTrack( channelIndex );
        if ( track != NULL ) {
            track->StopAllChannel();
        }
    }
}

void MidiSequencePlayer::ResetAllController() NN_NOEXCEPT
{
    for ( int channelIndex = 0 ; channelIndex < TrackCountPerPlayer ; channelIndex++ ) {
        SequenceTrack* track = GetPlayerTrack( channelIndex );
        if ( track != NULL ) {
            track->InitParam();
        }
    }
}

void MidiSequencePlayer::Reset() NN_NOEXCEPT
{
    for( int i = 0 ; i < NoteInfoCount ; i++ ) {
        if ( m_NoteInfo[i].channel != NULL )
        {
            m_NoteInfo[i].channel = NULL;
        }
    }

    for ( int channelIndex = 0 ; channelIndex < TrackCountPerPlayer ; channelIndex++ ) {
        SequenceTrack* track = GetPlayerTrack( channelIndex );
        if ( track != NULL ) {
            track->StopAllChannel();
            track->InitParam();
        }
    }
}

MidiSequencePlayer::NoteInfo* MidiSequencePlayer::FindFreeNoteInfo() NN_NOEXCEPT
{
    for( int i = 0 ; i < NoteInfoCount ; i++ ) {
        if ( m_NoteInfo[i].channel == NULL ) {
            return &m_NoteInfo[i];
        }
    }
    return NULL;
}

MidiSequencePlayer::NoteInfo* MidiSequencePlayer::FindNoteInfo( int channelIndex, int key ) NN_NOEXCEPT
{
    for( int i = 0 ; i < NoteInfoCount ; i++ ) {
        if ( m_NoteInfo[i].channel != NULL &&
             m_NoteInfo[i].channelIndex == channelIndex &&
             m_NoteInfo[i].key == key )
        {
            return &m_NoteInfo[i];
        }
    }
    return NULL;
}

MidiSequencePlayer::NoteInfo* MidiSequencePlayer::FindNoteInfo( int channelIndex ) NN_NOEXCEPT
{
    for( int i = 0 ; i < NoteInfoCount ; i++ ) {
        if ( m_NoteInfo[i].channel != NULL &&
             m_NoteInfo[i].channelIndex == channelIndex )
        {
            return &m_NoteInfo[i];
        }
    }
    return NULL;
}

void MidiSequencePlayer::ChannelCallback( Channel* channel ) NN_NOEXCEPT
{
    for( int i = 0 ; i < NoteInfoCount ; i++ ) {
        if ( m_NoteInfo[i].channel == channel ) {
            m_NoteInfo[i].channel = NULL;
        }
    }
}

} // namespace nn::atk::detail::driver
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

