﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_BankFileReader.h>
#include <nn/atk/detail/atk_Macro.h>

namespace nn {
namespace atk {
namespace detail {

namespace
{

const uint32_t SignatureInfoBlockBank  = NN_UTIL_CREATE_SIGNATURE_4( 'I', 'N', 'F', 'O' );

const uint32_t SupportedFileVersionBank = 0x00010000;
const uint32_t CurrentFileVersionBank   = 0x00010000;

bool IsValidFileHeaderBank( const void* bankFile ) NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader* header =
        reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( bankFile );

    // シグニチャ確認
    NN_SDK_ASSERT( header->signature == BankFileReader::SignatureFile,
            "invalid file signature. bank file is not available." );
    if ( header->signature != BankFileReader::SignatureFile ) return false;

    // バイトオーダーマーク確認
    NN_SDK_ASSERT( header->byteOrder == BinaryFileHeader::ValidByteOrderMark,
            "invalid file byte order mark. [expected:0x%04x][this:0x%04x]", BinaryFileHeader::ValidByteOrderMark, header->byteOrder );
    if ( header->byteOrder != BinaryFileHeader::ValidByteOrderMark )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SupportedFileVersionBank <= header->version) &&
                                   (header->version <= CurrentFileVersionBank) )
    {
        isSupportedVersion = true;

    }
    NN_SDK_ASSERT( isSupportedVersion,
            "bfbnk file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SupportedFileVersionBank, CurrentFileVersionBank, header->version
    );
    return isSupportedVersion;
}

} // anonymous namespace

NN_DEFINE_STATIC_CONSTANT( const uint32_t BankFileReader::SignatureFile );

BankFileReader::BankFileReader() NN_NOEXCEPT
: m_pHeader( NULL )
, m_pInfoBlockBody( NULL )
, m_IsInitialized( false )
{
}

BankFileReader::BankFileReader( const void* bankFile ) NN_NOEXCEPT
: m_pHeader( NULL )
, m_pInfoBlockBody( NULL )
, m_IsInitialized( false )
{
    Initialize( bankFile );
}

void BankFileReader::Initialize( const void* bankFile ) NN_NOEXCEPT
{
    if ( bankFile == NULL )
    {
        return;
    }

    if ( ! IsValidFileHeaderBank( bankFile ) )
    {
        return;
    }

    m_pHeader = reinterpret_cast<const BankFile::FileHeader*>( bankFile );

    const BankFile::InfoBlock* infoBlock = m_pHeader->GetInfoBlock();
    NN_SDK_ASSERT_NOT_NULL( infoBlock );
    NN_SDK_ASSERT( infoBlock->header.kind == SignatureInfoBlockBank );

    if ( infoBlock->header.kind != SignatureInfoBlockBank )
    {
        return;
    }

    m_pInfoBlockBody = &infoBlock->body;

    m_IsInitialized = true;
}

void BankFileReader::Finalize() NN_NOEXCEPT
{
    if ( m_IsInitialized )
    {
        m_pHeader = NULL;
        m_pInfoBlockBody = NULL;
        m_IsInitialized = false;
    }
}

bool BankFileReader::ReadVelocityRegionInfo(
        VelocityRegionInfo* info,
        int programNo,
        int key,
        int velocity ) const NN_NOEXCEPT
{
    if ( m_IsInitialized == false )
    {
        return false;
    }

    // イレギュラーな programNo を指定された場合
    if ( programNo < 0 || programNo >= m_pInfoBlockBody->GetInstrumentCount() )
    {
        return false;
    }

    // ベロシティリージョンを取得
    const BankFile::Instrument* instrument =
        m_pInfoBlockBody->GetInstrument( programNo );
    if ( instrument == NULL )
    {
        return false;
    }

    const BankFile::KeyRegion* keyRegion = instrument->GetKeyRegion( key );
    if ( keyRegion == NULL )
    {
        return false;
    }

    const BankFile::VelocityRegion* velocityRegion =
        keyRegion->GetVelocityRegion( velocity );
    if ( velocityRegion == NULL )
    {
        return false;
    }

    // 波形アーカイブ ID / 波形アーカイブ内インデックスを取得
    NN_SDK_ASSERT( velocityRegion->waveIdTableIndex < m_pInfoBlockBody->GetWaveIdCount() );
    const Util::WaveId* pWaveId =
        m_pInfoBlockBody->GetWaveId( velocityRegion->waveIdTableIndex );

    if ( pWaveId == NULL )
    {
        return false;
    }

    if ( pWaveId->waveIndex == 0xffffffff )   // 当該リージョンに波形ファイルが割り当たっていない
    {
        NN_ATK_WARNING(
                "This region [programNo(%d) key(%d) velocity(%d)] is not assigned wave file.",
                programNo, key, velocity );
        return false;
    }

    // info への書き込み
    info->waveArchiveId     = pWaveId->waveArchiveId;
    info->waveIndex         = pWaveId->waveIndex;
    const BankFile::RegionParameter* regionParameter = velocityRegion->GetRegionParameter();
    if ( regionParameter == NULL )
    {
        info->originalKey       = velocityRegion->GetOriginalKey();
        info->volume            = velocityRegion->GetVolume();
        info->pan               = velocityRegion->GetPan();
        info->pitch             = velocityRegion->GetPitch();
        info->isIgnoreNoteOff   = velocityRegion->IsIgnoreNoteOff();
        info->keyGroup          = velocityRegion->GetKeyGroup();
        info->interpolationType = velocityRegion->GetInterpolationType();
        info->adshrCurve        = velocityRegion->GetAdshrCurve();
    }
    else
    {
        info->originalKey       = regionParameter->originalKey;
        info->volume            = regionParameter->volume;
        info->pan               = regionParameter->pan;
        info->pitch             = regionParameter->pitch;
        info->isIgnoreNoteOff   = regionParameter->isIgnoreNoteOff;
        info->keyGroup          = regionParameter->keyGroup;
        info->interpolationType = regionParameter->interpolationType;
        info->adshrCurve        = regionParameter->adshrCurve;
    }

    return true;
}

const Util::WaveIdTable*
BankFileReader::GetWaveIdTable() const NN_NOEXCEPT
{
    if ( m_IsInitialized == false )
    {
        return NULL;
    }
    return &m_pInfoBlockBody->GetWaveIdTable();
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

