﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_Bank.h>
#include <nn/atk/atk_NoteOnCallback.h>      // NoteOnInfo
#include <nn/atk/atk_BankFileReader.h>
#include <nn/atk/atk_WaveArchiveFileReader.h>
#include <nn/atk/atk_WaveFileReader.h>
#include <nn/atk/atk_Util.h>                // GetWaveFile

namespace
{
// ベロシティの最大値の逆数です。割り算の掛け算化による高速化が狙いです。
const float VelocityMaxR = 1.0f / 127.0f;
// インスト音量の基準値の逆数です。割り算の掛け算化による高速化が狙いです。
const float InstrumentVolumeCriterionR = 1.0f / 127.0f;
} // anonymous namespace

namespace nn {
namespace atk {
namespace detail {
namespace driver {

/* ========================================================================
        public functions
   ======================================================================== */

/*--------------------------------------------------------------------------------*
  Name:         Bank

  Description:  コンストラクタ

  Arguments:    bankData - バンクデータへの参照

  Returns:      None.
 *--------------------------------------------------------------------------------*/
Bank::Bank() NN_NOEXCEPT
{
}

/*--------------------------------------------------------------------------------*
  Name:         ~Bank

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
Bank::~Bank() NN_NOEXCEPT
{
}

Channel* Bank::NoteOn(
    const BankFileReader& bankReader,
    const WaveArchiveFileReader& warcReader,
    const NoteOnInfo& noteOnInfo
) const NN_NOEXCEPT
{
    // VelocityRegionInfo の取得
    VelocityRegionInfo regionInfo;
    {
        if ( ! bankReader.ReadVelocityRegionInfo(
                &regionInfo,
                noteOnInfo.prgNo,
                noteOnInfo.key,
                noteOnInfo.velocity ) )
        {
            return NULL;
        }
    }

    // regnioInfo に応じた波形を取得する
    // (regionInfo.waveArchiveId に該当した波形アーカイブが、
    //  warcReader に登録されて引数で渡されてくる)
    const void* waveFile = warcReader.GetWaveFile( regionInfo.waveIndex );
    NN_SDK_ASSERT_NOT_NULL(waveFile);

    // 波形の詳細を取得、Channel 確保
    WaveInfo waveInfo;
    {
        WaveFileReader reader( waveFile );
        if ( ! reader.ReadWaveInfo( &waveInfo ) )
        {
            return NULL;
        }
    }

    Channel* pChannel = Channel::AllocChannel(
        std::min( static_cast<int>( waveInfo.channelCount ), 2 ),
        noteOnInfo.priority,
        noteOnInfo.channelCallback,
        noteOnInfo.channelCallbackData
    );
    if ( pChannel == NULL )
    {
        return NULL;
    }

    // 初期パラメータ設定
    pChannel->SetKey( static_cast<uint8_t>(noteOnInfo.key), regionInfo.originalKey );
    pChannel->SetVelocity( CalcChannelVelocityVolume(static_cast<uint8_t>(noteOnInfo.velocity))  );
    pChannel->SetInstrumentVolume( regionInfo.volume * InstrumentVolumeCriterionR );
    pChannel->SetTune( regionInfo.pitch );

    pChannel->SetAttack( regionInfo.adshrCurve.GetAttack() );
    pChannel->SetHold( regionInfo.adshrCurve.GetHold() );
    pChannel->SetDecay( regionInfo.adshrCurve.GetDecay() );
    pChannel->SetSustain( regionInfo.adshrCurve.GetSustain() );
    pChannel->SetRelease( regionInfo.adshrCurve.GetRelease() );

    float initPan = static_cast<float>( noteOnInfo.initPan + regionInfo.pan - 64 ) / 63.0f;
    pChannel->SetInitPan( initPan );
//    pChannel->SetInitSurroundPan( 0.0f );

    pChannel->SetKeyGroupId( regionInfo.keyGroup );
    pChannel->SetIsIgnoreNoteOff( regionInfo.isIgnoreNoteOff );
    pChannel->SetInterpolationType( regionInfo.interpolationType );
    pChannel->SetUpdateType( noteOnInfo.updateType );
    pChannel->SetOutputReceiver( noteOnInfo.pOutputReceiver );

    pChannel->Start( waveInfo, noteOnInfo.length, 0, false);

    return pChannel;
}

float Bank::CalcChannelVelocityVolume(uint8_t velocity) NN_NOEXCEPT
{
    return static_cast<float>(velocity) * VelocityMaxR;
}

} // namespace nn::atk::detail::driver
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

