﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_BiquadFilterPresets.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_VC)
#pragma warning(push)
#pragma warning(disable:4309)
// ↑ 係数値 (0x3ab3) が int16_t に格納されることで下記警告が出るが、これを抑制する。
// 「warning C4309:  '初期化中' : 定数値が切り捨てられました。」
#pragma warning(disable:4838)
// ↑ 係数値 (0x3ab3) が int16_t に格納されることにより VS2015 環境で下記警告が出るが、これを抑制する。
// 「warning C4838: 'int' から 'int16_t' への変換には縮小変換が必要です。」
#elif defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wc++11-narrowing"
#endif

namespace nn {
namespace atk {
namespace detail {

NN_DEFINE_STATIC_CONSTANT( const int BiquadFilterLpf::CoefficientsTableSize );
NN_DEFINE_STATIC_CONSTANT( const int BiquadFilterHpf::CoefficientsTableSize );
NN_DEFINE_STATIC_CONSTANT( const int BiquadFilterBpf512::CoefficientsTableSize );
NN_DEFINE_STATIC_CONSTANT( const int BiquadFilterBpf1024::CoefficientsTableSize );
NN_DEFINE_STATIC_CONSTANT( const int BiquadFilterBpf2048::CoefficientsTableSize );

//------------------------------------------------------------------------------
// サンプルレート 32kHz 用の係数定義
//------------------------------------------------------------------------------
// LPF係数
// （1/16オクターブ間隔、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterLpf::CoefficientsTable32000[ BiquadFilterLpf::CoefficientsTableSize ] =
{
    //    n0      n1      n2      d1      d2
    { 0x3ab3, 0x7566, 0x3ab3, 0x83b8, 0xc391 },     // ( 15689.372Hz )
    { 0x371b, 0x6e36, 0x371b, 0x8bf8, 0xca8c },
    { 0x33c7, 0x678d, 0x33c7, 0x9428, 0xd078 },
    { 0x30b0, 0x6161, 0x30b0, 0x9c32, 0xd582 },
    { 0x2dd2, 0x5ba5, 0x2dd2, 0xa40c, 0xd9cf },
    { 0x2b27, 0x564e, 0x2b27, 0xabae, 0xdd7d },
    { 0x28a9, 0x5152, 0x28a9, 0xb316, 0xe0a3 },
    { 0x2654, 0x4ca8, 0x2654, 0xba45, 0xe355 },
    { 0x2425, 0x4849, 0x2425, 0xc13a, 0xe5a2 },
    { 0x2217, 0x442f, 0x2217, 0xc7f8, 0xe797 },
    { 0x2029, 0x4051, 0x2029, 0xce7f, 0xe93e },
    { 0x1e57, 0x3cad, 0x1e57, 0xd4d3, 0xeaa2 },
    { 0x1c9f, 0x393d, 0x1c9f, 0xdaf5, 0xebc8 },
    { 0x1aff, 0x35fd, 0x1aff, 0xe0e6, 0xecb9 },
    { 0x1975, 0x32eb, 0x1975, 0xe6aa, 0xed78 },
    { 0x1801, 0x3002, 0x1801, 0xec40, 0xee0b },     // 8192Hz
    { 0x16a0, 0x2d40, 0x16a0, 0xf1ab, 0xee77 },
    { 0x1552, 0x2aa4, 0x1552, 0xf6ed, 0xeebe },
    { 0x1415, 0x282a, 0x1415, 0xfc05, 0xeee5 },
    { 0x12e8, 0x25d0, 0x12e8, 0x00f6, 0xeeee },
    { 0x11cb, 0x2396, 0x11cb, 0x05c1, 0xeedb },
    { 0x10bd, 0x217a, 0x10bd, 0x0a65, 0xeeb0 },
    { 0x0fbd, 0x1f79, 0x0fbd, 0x0ee5, 0xee6d },
    { 0x0eca, 0x1d93, 0x0eca, 0x1341, 0xee17 },
    { 0x0de3, 0x1bc7, 0x0de3, 0x177a, 0xedad },
    { 0x0d09, 0x1a13, 0x0d09, 0x1b90, 0xed33 },
    { 0x0c3b, 0x1876, 0x0c3b, 0x1f85, 0xeca9 },
    { 0x0b78, 0x16ef, 0x0b78, 0x2358, 0xec12 },
    { 0x0abf, 0x157e, 0x0abf, 0x270b, 0xeb6e },
    { 0x0a10, 0x1421, 0x0a10, 0x2a9d, 0xeabf },
    { 0x096b, 0x12d7, 0x096b, 0x2e11, 0xea06 },
    { 0x08d0, 0x119f, 0x08d0, 0x3166, 0xe945 },     // 4096Hz
    { 0x083d, 0x107a, 0x083d, 0x349d, 0xe87c },
    { 0x07b2, 0x0f65, 0x07b2, 0x37b7, 0xe7ad },
    { 0x0730, 0x0e60, 0x0730, 0x3ab4, 0xe6d8 },
    { 0x06b5, 0x0d6a, 0x06b5, 0x3d95, 0xe5ff },
    { 0x0642, 0x0c83, 0x0642, 0x405b, 0xe522 },
    { 0x05d5, 0x0baa, 0x05d5, 0x4306, 0xe443 },
    { 0x056f, 0x0adf, 0x056f, 0x4598, 0xe361 },
    { 0x0510, 0x0a1f, 0x0510, 0x4810, 0xe27e },
    { 0x04b6, 0x096c, 0x04b6, 0x4a70, 0xe19b },
    { 0x0462, 0x08c4, 0x0462, 0x4cb8, 0xe0b7 },
    { 0x0413, 0x0826, 0x0413, 0x4ee9, 0xdfd3 },
    { 0x03c9, 0x0793, 0x03c9, 0x5104, 0xdef1 },
    { 0x0384, 0x0709, 0x0384, 0x5309, 0xde10 },
    { 0x0344, 0x0688, 0x0344, 0x54f8, 0xdd31 },
    { 0x0308, 0x0610, 0x0308, 0x56d4, 0xdc54 },
    { 0x02d0, 0x059f, 0x02d0, 0x589c, 0xdb7a },     // 2048Hz
    { 0x029b, 0x0537, 0x029b, 0x5a51, 0xdaa3 },
    { 0x026b, 0x04d5, 0x026b, 0x5bf4, 0xd9cf },
    { 0x023d, 0x047a, 0x023d, 0x5d85, 0xd8fe },
    { 0x0213, 0x0426, 0x0213, 0x5f06, 0xd831 },
    { 0x01eb, 0x03d7, 0x01eb, 0x6076, 0xd768 },
    { 0x01c7, 0x038d, 0x01c7, 0x61d6, 0xd6a3 },
    { 0x01a5, 0x0349, 0x01a5, 0x6327, 0xd5e2 },
    { 0x0185, 0x030a, 0x0185, 0x646a, 0xd526 },
    { 0x0168, 0x02cf, 0x0168, 0x659f, 0xd46d },
    { 0x014c, 0x0299, 0x014c, 0x66c6, 0xd3ba },
    { 0x0133, 0x0266, 0x0133, 0x67e1, 0xd30a },
    { 0x011c, 0x0237, 0x011c, 0x68ef, 0xd25f },
    { 0x0106, 0x020c, 0x0106, 0x69f1, 0xd1b9 },
    { 0x00f2, 0x01e4, 0x00f2, 0x6ae9, 0xd117 },
    { 0x00df, 0x01be, 0x00df, 0x6bd5, 0xd07a },
    { 0x00ce, 0x019c, 0x00ce, 0x6cb7, 0xcfe1 },     // 1024Hz
    { 0x00be, 0x017c, 0x00be, 0x6d8f, 0xcf4d },
    { 0x00af, 0x015e, 0x00af, 0x6e5d, 0xcebd },
    { 0x00a1, 0x0143, 0x00a1, 0x6f23, 0xce31 },
    { 0x0095, 0x012a, 0x0095, 0x6fe0, 0xcdaa },
    { 0x0089, 0x0112, 0x0089, 0x7094, 0xcd27 },
    { 0x007e, 0x00fd, 0x007e, 0x7140, 0xcca8 },
    { 0x0074, 0x00e9, 0x0074, 0x71e5, 0xcc2e },
    { 0x006b, 0x00d6, 0x006b, 0x7283, 0xcbb7 },
    { 0x0063, 0x00c5, 0x0063, 0x7319, 0xcb44 },
    { 0x005b, 0x00b6, 0x005b, 0x73a9, 0xcad5 },
    { 0x0054, 0x00a7, 0x0054, 0x7433, 0xca6a },
    { 0x004d, 0x009a, 0x004d, 0x74b7, 0xca03 },
    { 0x0047, 0x008e, 0x0047, 0x7534, 0xc99f },
    { 0x0041, 0x0082, 0x0041, 0x75ac, 0xc93f },
    { 0x003c, 0x0078, 0x003c, 0x761f, 0xc8e2 },
    { 0x0037, 0x006e, 0x0037, 0x768d, 0xc889 },     // 512Hz
    { 0x0033, 0x0066, 0x0033, 0x76f6, 0xc833 },
    { 0x002f, 0x005d, 0x002f, 0x775a, 0xc7e0 },
    { 0x002b, 0x0056, 0x002b, 0x77ba, 0xc790 },
    { 0x0027, 0x004f, 0x0027, 0x7816, 0xc743 },
    { 0x0024, 0x0049, 0x0024, 0x786e, 0xc6f8 },
    { 0x0021, 0x0043, 0x0021, 0x78c2, 0xc6b1 },
    { 0x001f, 0x003d, 0x001f, 0x7912, 0xc66c },
    { 0x001c, 0x0038, 0x001c, 0x795f, 0xc62a },
    { 0x001a, 0x0034, 0x001a, 0x79a8, 0xc5ea },
    { 0x0018, 0x0030, 0x0018, 0x79ee, 0xc5ad },
    { 0x0016, 0x002c, 0x0016, 0x7a31, 0xc572 },
    { 0x0014, 0x0028, 0x0014, 0x7a71, 0xc53a },
    { 0x0012, 0x0025, 0x0012, 0x7aae, 0xc503 },
    { 0x0011, 0x0022, 0x0011, 0x7ae9, 0xc4cf },
    { 0x0010, 0x001f, 0x0010, 0x7b21, 0xc49d },
    { 0x000e, 0x001d, 0x000e, 0x7b57, 0xc46d },     // 256Hz
    { 0x000d, 0x001a, 0x000d, 0x7b8a, 0xc43e },
    { 0x000c, 0x0018, 0x000c, 0x7bbb, 0xc412 },
    { 0x000b, 0x0016, 0x000b, 0x7bea, 0xc3e7 },
    { 0x000a, 0x0014, 0x000a, 0x7c17, 0xc3be },
    { 0x0009, 0x0013, 0x0009, 0x7c42, 0xc396 },
    { 0x0009, 0x0011, 0x0009, 0x7c6b, 0xc371 },
    { 0x0008, 0x0010, 0x0008, 0x7c92, 0xc34c },
    { 0x0007, 0x000e, 0x0007, 0x7cb8, 0xc329 },
    { 0x0007, 0x000d, 0x0007, 0x7cdc, 0xc308 },
    { 0x0006, 0x000c, 0x0006, 0x7cfe, 0xc2e8 },
    { 0x0006, 0x000b, 0x0006, 0x7d1f, 0xc2c9 },
    { 0x0005, 0x000a, 0x0005, 0x7d3f, 0xc2ab },
    { 0x0005, 0x0009, 0x0005, 0x7d5d, 0xc28f },
    { 0x0004, 0x0009, 0x0004, 0x7d7a, 0xc274 },
    { 0x0004, 0x0008, 0x0004, 0x7d96, 0xc25a },
    { 0x0004, 0x0007, 0x0004, 0x7db0, 0xc241 }      // 128Hz
};


// HPF係数
// （1/16オクターブ間隔、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterHpf::CoefficientsTable32000[ BiquadFilterHpf::CoefficientsTableSize ] =
{
    //    n0      n1      n2      d1      d2
    { 0x3bb5, 0x8895, 0x3bb5, 0x7e7b, 0xc17f },     // 128Hz
    { 0x3bad, 0x88a6, 0x3bad, 0x7e69, 0xc18f },
    { 0x3ba5, 0x88b6, 0x3ba5, 0x7e57, 0xc1a1 },
    { 0x3b9c, 0x88c8, 0x3b9c, 0x7e44, 0xc1b3 },
    { 0x3b93, 0x88da, 0x3b93, 0x7e30, 0xc1c6 },
    { 0x3b89, 0x88ed, 0x3b89, 0x7e1c, 0xc1da },
    { 0x3b7f, 0x8901, 0x3b7f, 0x7e06, 0xc1ee },
    { 0x3b75, 0x8916, 0x3b75, 0x7df0, 0xc204 },
    { 0x3b6a, 0x892c, 0x3b6a, 0x7dd8, 0xc21a },
    { 0x3b5f, 0x8942, 0x3b5f, 0x7dbf, 0xc232 },
    { 0x3b53, 0x895a, 0x3b53, 0x7da6, 0xc24a },
    { 0x3b47, 0x8973, 0x3b47, 0x7d8b, 0xc264 },
    { 0x3b3a, 0x898d, 0x3b3a, 0x7d6f, 0xc27e },
    { 0x3b2c, 0x89a8, 0x3b2c, 0x7d51, 0xc29a },
    { 0x3b1e, 0x89c4, 0x3b1e, 0x7d33, 0xc2b7 },
    { 0x3b10, 0x89e1, 0x3b10, 0x7d13, 0xc2d5 },
    { 0x3b00, 0x8a00, 0x3b00, 0x7cf1, 0xc2f4 },     // 256Hz
    { 0x3af0, 0x8a20, 0x3af0, 0x7cce, 0xc315 },
    { 0x3ae0, 0x8a41, 0x3ae0, 0x7ca9, 0xc337 },
    { 0x3ace, 0x8a64, 0x3ace, 0x7c83, 0xc35a },
    { 0x3abc, 0x8a88, 0x3abc, 0x7c5b, 0xc37f },
    { 0x3aa9, 0x8aae, 0x3aa9, 0x7c31, 0xc3a6 },
    { 0x3a95, 0x8ad6, 0x3a95, 0x7c05, 0xc3ce },
    { 0x3a81, 0x8aff, 0x3a81, 0x7bd8, 0xc3f8 },
    { 0x3a6b, 0x8b2a, 0x3a6b, 0x7ba8, 0xc423 },
    { 0x3a55, 0x8b57, 0x3a55, 0x7b76, 0xc451 },
    { 0x3a3d, 0x8b86, 0x3a3d, 0x7b42, 0xc480 },
    { 0x3a25, 0x8bb7, 0x3a25, 0x7b0b, 0xc4b1 },
    { 0x3a0b, 0x8bea, 0x3a0b, 0x7ad2, 0xc4e4 },
    { 0x39f0, 0x8c1f, 0x39f0, 0x7a96, 0xc519 },
    { 0x39d5, 0x8c57, 0x39d5, 0x7a58, 0xc550 },
    { 0x39b8, 0x8c91, 0x39b8, 0x7a16, 0xc58a },
    { 0x3999, 0x8ccd, 0x3999, 0x79d2, 0xc5c6 },     // 512Hz
    { 0x397a, 0x8d0c, 0x397a, 0x798b, 0xc604 },
    { 0x3959, 0x8d4e, 0x3959, 0x7940, 0xc645 },
    { 0x3937, 0x8d93, 0x3937, 0x78f2, 0xc688 },
    { 0x3913, 0x8dda, 0x3913, 0x78a0, 0xc6ce },
    { 0x38ed, 0x8e25, 0x38ed, 0x784a, 0xc716 },
    { 0x38c7, 0x8e73, 0x38c7, 0x77f1, 0xc762 },
    { 0x389e, 0x8ec4, 0x389e, 0x7793, 0xc7b0 },
    { 0x3874, 0x8f19, 0x3874, 0x7731, 0xc802 },
    { 0x3847, 0x8f71, 0x3847, 0x76cb, 0xc856 },
    { 0x3819, 0x8fcd, 0x3819, 0x7660, 0xc8ae },
    { 0x37e9, 0x902d, 0x37e9, 0x75f0, 0xc909 },
    { 0x37b7, 0x9091, 0x37b7, 0x757a, 0xc968 },
    { 0x3783, 0x90f9, 0x3783, 0x74ff, 0xc9ca },
    { 0x374d, 0x9166, 0x374d, 0x747f, 0xca2f },
    { 0x3714, 0x91d7, 0x3714, 0x73f8, 0xca98 },
    { 0x36d9, 0x924d, 0x36d9, 0x736b, 0xcb05 },     // 1024Hz
    { 0x369c, 0x92c8, 0x369c, 0x72d8, 0xcb76 },
    { 0x365c, 0x9348, 0x365c, 0x723e, 0xcbeb },
    { 0x3619, 0x93cd, 0x3619, 0x719c, 0xcc64 },
    { 0x35d4, 0x9458, 0x35d4, 0x70f3, 0xcce1 },
    { 0x358c, 0x94e9, 0x358c, 0x7042, 0xcd63 },
    { 0x3540, 0x957f, 0x3540, 0x6f89, 0xcde8 },
    { 0x34f2, 0x961c, 0x34f2, 0x6ec7, 0xce73 },
    { 0x34a0, 0x96bf, 0x34a0, 0x6dfb, 0xcf01 },
    { 0x344b, 0x9769, 0x344b, 0x6d27, 0xcf94 },
    { 0x33f3, 0x981a, 0x33f3, 0x6c48, 0xd02c },
    { 0x3397, 0x98d2, 0x3397, 0x6b5f, 0xd0c9 },
    { 0x3337, 0x9992, 0x3337, 0x6a6a, 0xd16a },
    { 0x32d4, 0x9a59, 0x32d4, 0x696a, 0xd210 },
    { 0x326c, 0x9b28, 0x326c, 0x685e, 0xd2bb },
    { 0x3200, 0x9bff, 0x3200, 0x6746, 0xd36b },
    { 0x3190, 0x9cdf, 0x3190, 0x6620, 0xd41f },     // 2048Hz
    { 0x311c, 0x9dc8, 0x311c, 0x64ec, 0xd4d9 },
    { 0x30a3, 0x9eba, 0x30a3, 0x63a9, 0xd597 },
    { 0x3025, 0x9fb6, 0x3025, 0x6258, 0xd65a },
    { 0x2fa3, 0xa0bb, 0x2fa3, 0x60f6, 0xd721 },
    { 0x2f1b, 0xa1ca, 0x2f1b, 0x5f83, 0xd7ee },
    { 0x2e8e, 0xa2e4, 0x2e8e, 0x5dff, 0xd8be },
    { 0x2dfc, 0xa409, 0x2dfc, 0x5c68, 0xd993 },
    { 0x2d64, 0xa539, 0x2d64, 0x5abe, 0xda6d },
    { 0x2cc6, 0xa674, 0x2cc6, 0x58ff, 0xdb4a },
    { 0x2c22, 0xa7bb, 0x2c22, 0x572b, 0xdc2b },
    { 0x2b79, 0xa90e, 0x2b79, 0x5541, 0xdd10 },
    { 0x2ac9, 0xaa6e, 0x2ac9, 0x533f, 0xddf8 },
    { 0x2a12, 0xabdc, 0x2a12, 0x5125, 0xdee3 },
    { 0x2955, 0xad56, 0x2955, 0x4ef0, 0xdfd1 },
    { 0x2891, 0xaedf, 0x2891, 0x4ca1, 0xe0c0 },
    { 0x27c5, 0xb075, 0x27c5, 0x4a34, 0xe1b1 },     // 4096Hz
    { 0x26f3, 0xb21b, 0x26f3, 0x47aa, 0xe2a4 },
    { 0x2618, 0xb3cf, 0x2618, 0x4500, 0xe396 },
    { 0x2536, 0xb593, 0x2536, 0x4234, 0xe489 },
    { 0x244c, 0xb768, 0x244c, 0x3f45, 0xe57a },
    { 0x235a, 0xb94d, 0x235a, 0x3c31, 0xe669 },
    { 0x225f, 0xbb43, 0x225f, 0x38f6, 0xe756 },
    { 0x215b, 0xbd4a, 0x215b, 0x3591, 0xe83e },
    { 0x204e, 0xbf64, 0x204e, 0x3200, 0xe920 },
    { 0x1f38, 0xc191, 0x1f38, 0x2e40, 0xe9fc },
    { 0x1e17, 0xc3d1, 0x1e17, 0x2a4e, 0xeacf },
    { 0x1ced, 0xc625, 0x1ced, 0x2628, 0xeb97 },
    { 0x1bb9, 0xc88e, 0x1bb9, 0x21c8, 0xec52 },
    { 0x1a7a, 0xcb0b, 0x1a7a, 0x1d2d, 0xecfd },
    { 0x1930, 0xcd9f, 0x1930, 0x1850, 0xed96 },
    { 0x17dc, 0xd049, 0x17dc, 0x132e, 0xee18 },
    { 0x167b, 0xd30a, 0x167b, 0x0dc1, 0xee80 }      // 8192Hz
};


// BPF係数その１
// （中心周波数 512Hz、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterBpf512::CoefficientsTable32000[ BiquadFilterBpf512::CoefficientsTableSize ] =
{
    //    n0      n1      n2      d1      d2
    { 0x2efa, 0x0000, 0xd106, 0x21d7, 0x1df4 },     // 32 - 7844.686 Hz
    { 0x2e8f, 0x0000, 0xd171, 0x22ad, 0x1d1e },
    { 0x2e24, 0x0000, 0xd1dc, 0x2381, 0x1c49 },
    { 0x2dba, 0x0000, 0xd246, 0x2455, 0x1b74 },
    { 0x2d50, 0x0000, 0xd2b0, 0x2527, 0x1aa1 },
    { 0x2ce7, 0x0000, 0xd319, 0x25f9, 0x19ce },
    { 0x2c7e, 0x0000, 0xd382, 0x26cb, 0x18fc },
    { 0x2c15, 0x0000, 0xd3eb, 0x279c, 0x182a },
    { 0x2bad, 0x0000, 0xd453, 0x286c, 0x1759 },
    { 0x2b44, 0x0000, 0xd4bc, 0x293c, 0x1689 },
    { 0x2add, 0x0000, 0xd523, 0x2a0b, 0x15b9 },
    { 0x2a75, 0x0000, 0xd58b, 0x2ada, 0x14ea },
    { 0x2a0d, 0x0000, 0xd5f3, 0x2ba8, 0x141a },
    { 0x29a6, 0x0000, 0xd65a, 0x2c76, 0x134c },
    { 0x293f, 0x0000, 0xd6c1, 0x2d43, 0x127d },
    { 0x28d8, 0x0000, 0xd728, 0x2e11, 0x11af },
    { 0x2871, 0x0000, 0xd78f, 0x2ede, 0x10e2 },
    { 0x280a, 0x0000, 0xd7f6, 0x2faa, 0x1014 },
    { 0x27a4, 0x0000, 0xd85c, 0x3076, 0x0f47 },
    { 0x273d, 0x0000, 0xd8c3, 0x3142, 0x0e7a },
    { 0x26d7, 0x0000, 0xd929, 0x320e, 0x0dae },
    { 0x2671, 0x0000, 0xd98f, 0x32da, 0x0ce2 },
    { 0x260b, 0x0000, 0xd9f5, 0x33a5, 0x0c16 },
    { 0x25a5, 0x0000, 0xda5b, 0x3470, 0x0b4a },
    { 0x253f, 0x0000, 0xdac1, 0x353a, 0x0a7e },
    { 0x24da, 0x0000, 0xdb26, 0x3605, 0x09b3 },
    { 0x2474, 0x0000, 0xdb8c, 0x36cf, 0x08e8 },
    { 0x240f, 0x0000, 0xdbf1, 0x3799, 0x081d },
    { 0x23a9, 0x0000, 0xdc57, 0x3863, 0x0753 },
    { 0x2344, 0x0000, 0xdcbc, 0x392c, 0x0689 },
    { 0x22df, 0x0000, 0xdd21, 0x39f5, 0x05bf },
    { 0x227b, 0x0000, 0xdd85, 0x3abe, 0x04f5 },
    { 0x2216, 0x0000, 0xddea, 0x3b86, 0x042c },     // 64 - 3922.343 Hz
    { 0x21b1, 0x0000, 0xde4f, 0x3c4f, 0x0363 },
    { 0x214d, 0x0000, 0xdeb3, 0x3d17, 0x029a },
    { 0x20e9, 0x0000, 0xdf17, 0x3dde, 0x01d1 },
    { 0x2085, 0x0000, 0xdf7b, 0x3ea6, 0x0109 },
    { 0x2021, 0x0000, 0xdfdf, 0x3f6d, 0x0041 },
    { 0x1fbd, 0x0000, 0xe043, 0x4033, 0xff7a },
    { 0x1f59, 0x0000, 0xe0a7, 0x40fa, 0xfeb3 },
    { 0x1ef6, 0x0000, 0xe10a, 0x41c0, 0xfdec },
    { 0x1e93, 0x0000, 0xe16d, 0x4285, 0xfd25 },
    { 0x1e30, 0x0000, 0xe1d0, 0x434a, 0xfc5f },
    { 0x1dcd, 0x0000, 0xe233, 0x440f, 0xfb9a },
    { 0x1d6a, 0x0000, 0xe296, 0x44d4, 0xfad4 },
    { 0x1d08, 0x0000, 0xe2f8, 0x4597, 0xfa10 },
    { 0x1ca6, 0x0000, 0xe35a, 0x465b, 0xf94b },
    { 0x1c44, 0x0000, 0xe3bc, 0x471e, 0xf887 },
    { 0x1be2, 0x0000, 0xe41e, 0x47e1, 0xf7c4 },
    { 0x1b80, 0x0000, 0xe480, 0x48a3, 0xf701 },
    { 0x1b1f, 0x0000, 0xe4e1, 0x4964, 0xf63f },
    { 0x1abe, 0x0000, 0xe542, 0x4a25, 0xf57d },
    { 0x1a5e, 0x0000, 0xe5a2, 0x4ae6, 0xf4bb },
    { 0x19fd, 0x0000, 0xe603, 0x4ba6, 0xf3fa },
    { 0x199d, 0x0000, 0xe663, 0x4c65, 0xf33a },
    { 0x193d, 0x0000, 0xe6c3, 0x4d24, 0xf27a },
    { 0x18dd, 0x0000, 0xe723, 0x4de3, 0xf1bb },
    { 0x187e, 0x0000, 0xe782, 0x4ea0, 0xf0fc },
    { 0x181f, 0x0000, 0xe7e1, 0x4f5e, 0xf03e },
    { 0x17c0, 0x0000, 0xe840, 0x501a, 0xef81 },
    { 0x1762, 0x0000, 0xe89e, 0x50d6, 0xeec4 },
    { 0x1704, 0x0000, 0xe8fc, 0x5192, 0xee07 },
    { 0x16a6, 0x0000, 0xe95a, 0x524d, 0xed4c },
    { 0x1648, 0x0000, 0xe9b8, 0x5307, 0xec91 },
    { 0x15eb, 0x0000, 0xea15, 0x53c0, 0xebd6 },     // 128 - 1961.172 Hz
    { 0x158e, 0x0000, 0xea72, 0x5479, 0xeb1c },
    { 0x1532, 0x0000, 0xeace, 0x5532, 0xea63 },
    { 0x14d5, 0x0000, 0xeb2b, 0x55ea, 0xe9ab },
    { 0x1479, 0x0000, 0xeb87, 0x56a1, 0xe8f3 },
    { 0x141e, 0x0000, 0xebe2, 0x5757, 0xe83b },
    { 0x13c2, 0x0000, 0xec3e, 0x580d, 0xe785 },
    { 0x1367, 0x0000, 0xec99, 0x58c2, 0xe6ce },
    { 0x130d, 0x0000, 0xecf3, 0x5977, 0xe619 },
    { 0x12b2, 0x0000, 0xed4e, 0x5a2b, 0xe564 },
    { 0x1258, 0x0000, 0xeda8, 0x5ade, 0xe4b0 },
    { 0x11fe, 0x0000, 0xee02, 0x5b91, 0xe3fc },
    { 0x11a5, 0x0000, 0xee5b, 0x5c43, 0xe349 },
    { 0x114b, 0x0000, 0xeeb5, 0x5cf5, 0xe297 },
    { 0x10f2, 0x0000, 0xef0e, 0x5da6, 0xe1e5 },
    { 0x109a, 0x0000, 0xef66, 0x5e56, 0xe134 },
    { 0x1041, 0x0000, 0xefbf, 0x5f06, 0xe083 },
    { 0x0fe9, 0x0000, 0xf017, 0x5fb6, 0xdfd3 },
    { 0x0f91, 0x0000, 0xf06f, 0x6065, 0xdf23 },
    { 0x0f3a, 0x0000, 0xf0c6, 0x6113, 0xde74 },
    { 0x0ee2, 0x0000, 0xf11e, 0x61c1, 0xddc5 },
    { 0x0e8b, 0x0000, 0xf175, 0x626f, 0xdd16 },
    { 0x0e34, 0x0000, 0xf1cc, 0x631c, 0xdc69 },
    { 0x0dde, 0x0000, 0xf222, 0x63c8, 0xdbbb },
    { 0x0d87, 0x0000, 0xf279, 0x6475, 0xdb0e },
    { 0x0d31, 0x0000, 0xf2cf, 0x6520, 0xda61 },
    { 0x0cda, 0x0000, 0xf326, 0x65cc, 0xd9b5 },
    { 0x0c84, 0x0000, 0xf37c, 0x6677, 0xd909 },
    { 0x0c2f, 0x0000, 0xf3d1, 0x6722, 0xd85d },
    { 0x0bd9, 0x0000, 0xf427, 0x67cd, 0xd7b2 },
    { 0x0b83, 0x0000, 0xf47d, 0x6878, 0xd706 },
    { 0x0b2d, 0x0000, 0xf4d3, 0x6922, 0xd65b },
    { 0x0ad8, 0x0000, 0xf528, 0x69cc, 0xd5b0 },     // 256 - 980.586 Hz
    { 0x0a82, 0x0000, 0xf57e, 0x6a77, 0xd505 },
    { 0x0a2d, 0x0000, 0xf5d3, 0x6b21, 0xd45a },
    { 0x09d7, 0x0000, 0xf629, 0x6bcb, 0xd3af },
    { 0x0982, 0x0000, 0xf67e, 0x6c75, 0xd304 },
    { 0x092c, 0x0000, 0xf6d4, 0x6d1f, 0xd259 },
    { 0x08d7, 0x0000, 0xf729, 0x6dca, 0xd1ae },
    { 0x0881, 0x0000, 0xf77f, 0x6e74, 0xd102 },
    { 0x082b, 0x0000, 0xf7d5, 0x6f1f, 0xd056 },
    { 0x07d5, 0x0000, 0xf82b, 0x6fca, 0xcfaa },
    { 0x077f, 0x0000, 0xf881, 0x7076, 0xcefe },
    { 0x0729, 0x0000, 0xf8d7, 0x7122, 0xce51 },
    { 0x06d2, 0x0000, 0xf92e, 0x71ce, 0xcda4 },
    { 0x067b, 0x0000, 0xf985, 0x727b, 0xccf6 },
    { 0x0624, 0x0000, 0xf9dc, 0x7329, 0xcc48 },
    { 0x05cc, 0x0000, 0xfa34, 0x73d7, 0xcb98 },
    { 0x0574, 0x0000, 0xfa8c, 0x7487, 0xcae8 },
    { 0x051c, 0x0000, 0xfae4, 0x7537, 0xca38 },
    { 0x04c3, 0x0000, 0xfb3d, 0x75e7, 0xc986 },
    { 0x046a, 0x0000, 0xfb96, 0x7699, 0xc8d3 },
    { 0x0410, 0x0000, 0xfbf0, 0x774c, 0xc81f },
    { 0x03b5, 0x0000, 0xfc4b, 0x7800, 0xc76a },
    { 0x035a, 0x0000, 0xfca6, 0x78b6, 0xc6b4 },
    { 0x02fe, 0x0000, 0xfd02, 0x796d, 0xc5fc },
    { 0x02a2, 0x0000, 0xfd5e, 0x7a25, 0xc543 },
    { 0x0244, 0x0000, 0xfdbc, 0x7adf, 0xc488 }
};


// BPF係数その２
// （中心周波数 1024Hz、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterBpf1024::CoefficientsTable32000[ BiquadFilterBpf1024::CoefficientsTableSize ] =
{
    //    n0      n1      n2      d1      d2
    { 0x2f04, 0x0000, 0xd0fc, 0x21de, 0x1e08 },     // 16 - 7844.686 Hz
    { 0x2e99, 0x0000, 0xd167, 0x22b3, 0x1d32 },
    { 0x2e2f, 0x0000, 0xd1d1, 0x2386, 0x1c5d },
    { 0x2dc5, 0x0000, 0xd23b, 0x2459, 0x1b89 },
    { 0x2d5b, 0x0000, 0xd2a5, 0x252b, 0x1ab6 },
    { 0x2cf2, 0x0000, 0xd30e, 0x25fd, 0x19e4 },
    { 0x2c89, 0x0000, 0xd377, 0x26ce, 0x1912 },
    { 0x2c20, 0x0000, 0xd3e0, 0x279e, 0x1841 },
    { 0x2bb8, 0x0000, 0xd448, 0x286d, 0x1770 },
    { 0x2b50, 0x0000, 0xd4b0, 0x293c, 0x16a0 },
    { 0x2ae8, 0x0000, 0xd518, 0x2a0b, 0x15d0 },
    { 0x2a80, 0x0000, 0xd580, 0x2ad9, 0x1500 },
    { 0x2a19, 0x0000, 0xd5e7, 0x2ba7, 0x1431 },
    { 0x29b1, 0x0000, 0xd64f, 0x2c74, 0x1363 },
    { 0x294a, 0x0000, 0xd6b6, 0x2d41, 0x1294 },
    { 0x28e3, 0x0000, 0xd71d, 0x2e0e, 0x11c6 },
    { 0x287c, 0x0000, 0xd784, 0x2eda, 0x10f8 },     // 32 - 5547.031 Hz
    { 0x2815, 0x0000, 0xd7eb, 0x2fa6, 0x102a },
    { 0x27af, 0x0000, 0xd851, 0x3072, 0x0f5d },
    { 0x2748, 0x0000, 0xd8b8, 0x313e, 0x0e90 },
    { 0x26e1, 0x0000, 0xd91f, 0x3209, 0x0dc3 },
    { 0x267b, 0x0000, 0xd985, 0x32d4, 0x0cf6 },
    { 0x2614, 0x0000, 0xd9ec, 0x339f, 0x0c29 },
    { 0x25ae, 0x0000, 0xda52, 0x346a, 0x0b5c },
    { 0x2548, 0x0000, 0xdab8, 0x3535, 0x0a8f },
    { 0x24e1, 0x0000, 0xdb1f, 0x35ff, 0x09c3 },
    { 0x247b, 0x0000, 0xdb85, 0x36ca, 0x08f6 },
    { 0x2415, 0x0000, 0xdbeb, 0x3794, 0x082a },
    { 0x23af, 0x0000, 0xdc51, 0x385e, 0x075d },
    { 0x2349, 0x0000, 0xdcb7, 0x3928, 0x0691 },
    { 0x22e2, 0x0000, 0xdd1e, 0x39f3, 0x05c5 },
    { 0x227c, 0x0000, 0xdd84, 0x3abd, 0x04f8 },
    { 0x2216, 0x0000, 0xddea, 0x3b86, 0x042c },     // 64 - 3922.343 Hz
    { 0x21b0, 0x0000, 0xde50, 0x3c50, 0x035f },
    { 0x2149, 0x0000, 0xdeb7, 0x3d1a, 0x0293 },
    { 0x20e3, 0x0000, 0xdf1d, 0x3de4, 0x01c6 },
    { 0x207d, 0x0000, 0xdf83, 0x3eae, 0x00f9 },
    { 0x2016, 0x0000, 0xdfea, 0x3f78, 0x002c },
    { 0x1fb0, 0x0000, 0xe050, 0x4042, 0xff5f },
    { 0x1f49, 0x0000, 0xe0b7, 0x410c, 0xfe92 },
    { 0x1ee2, 0x0000, 0xe11e, 0x41d7, 0xfdc4 },
    { 0x1e7b, 0x0000, 0xe185, 0x42a1, 0xfcf7 },
    { 0x1e14, 0x0000, 0xe1ec, 0x436c, 0xfc29 },
    { 0x1dad, 0x0000, 0xe253, 0x4436, 0xfb5b },
    { 0x1d46, 0x0000, 0xe2ba, 0x4501, 0xfa8c },
    { 0x1cdf, 0x0000, 0xe321, 0x45cd, 0xf9bd },
    { 0x1c77, 0x0000, 0xe389, 0x4698, 0xf8ee },
    { 0x1c0f, 0x0000, 0xe3f1, 0x4764, 0xf81e },
    { 0x1ba7, 0x0000, 0xe459, 0x4831, 0xf74d },     // 128 - 2773.515 Hz
    { 0x1b3e, 0x0000, 0xe4c2, 0x48fd, 0xf67c },
    { 0x1ad5, 0x0000, 0xe52b, 0x49cb, 0xf5ab },
    { 0x1a6c, 0x0000, 0xe594, 0x4a99, 0xf4d8 },
    { 0x1a03, 0x0000, 0xe5fd, 0x4b67, 0xf405 },
    { 0x1998, 0x0000, 0xe668, 0x4c37, 0xf331 },
    { 0x192e, 0x0000, 0xe6d2, 0x4d07, 0xf25c },
    { 0x18c3, 0x0000, 0xe73d, 0x4dd8, 0xf186 },
    { 0x1857, 0x0000, 0xe7a9, 0x4eaa, 0xf0af },
    { 0x17eb, 0x0000, 0xe815, 0x4f7d, 0xefd6 },
    { 0x177e, 0x0000, 0xe882, 0x5051, 0xeefc },
    { 0x1710, 0x0000, 0xe8f0, 0x5127, 0xee21 },
    { 0x16a2, 0x0000, 0xe95e, 0x51fe, 0xed44 },
    { 0x1632, 0x0000, 0xe9ce, 0x52d7, 0xec65 },
    { 0x15c2, 0x0000, 0xea3e, 0x53b2, 0xeb84 },
    { 0x1550, 0x0000, 0xeab0, 0x548e, 0xeaa1 },
    { 0x14de, 0x0000, 0xeb22, 0x556d, 0xe9bc },     // 256 - 1961.172 Hz
    { 0x146a, 0x0000, 0xeb96, 0x564e, 0xe8d4 },
    { 0x13f5, 0x0000, 0xec0b, 0x5731, 0xe7e9 },
    { 0x137e, 0x0000, 0xec82, 0x5817, 0xe6fc },
    { 0x1306, 0x0000, 0xecfa, 0x5900, 0xe60b },
    { 0x128c, 0x0000, 0xed74, 0x59ed, 0xe517 },
    { 0x1210, 0x0000, 0xedf0, 0x5adc, 0xe41f },
    { 0x1192, 0x0000, 0xee6e, 0x5bd0, 0xe323 },
    { 0x1111, 0x0000, 0xeeef, 0x5cc8, 0xe223 },
    { 0x108f, 0x0000, 0xef71, 0x5dc4, 0xe11e },
    { 0x100a, 0x0000, 0xeff6, 0x5ec5, 0xe013 },
    { 0x0f82, 0x0000, 0xf07e, 0x5fcb, 0xdf04 },
    { 0x0ef7, 0x0000, 0xf109, 0x60d7, 0xddee },
    { 0x0e69, 0x0000, 0xf197, 0x61e9, 0xdcd1 },
    { 0x0dd7, 0x0000, 0xf229, 0x6301, 0xdbae },
    { 0x0d41, 0x0000, 0xf2bf, 0x6421, 0xda83 },
    { 0x0ca8, 0x0000, 0xf358, 0x6549, 0xd94f },     // 512 - 1386.758 Hz
    { 0x0c09, 0x0000, 0xf3f7, 0x6679, 0xd813 },
    { 0x0b66, 0x0000, 0xf49a, 0x67b3, 0xd6cd },
    { 0x0abe, 0x0000, 0xf542, 0x68f6, 0xd57c },
    { 0x0a10, 0x0000, 0xf5f0, 0x6a45, 0xd420 },
    { 0x095b, 0x0000, 0xf6a5, 0x6b9f, 0xd2b7 },
    { 0x08a0, 0x0000, 0xf760, 0x6d06, 0xd141 },
    { 0x07de, 0x0000, 0xf822, 0x6e7b, 0xcfbc },
    { 0x0714, 0x0000, 0xf8ec, 0x6fff, 0xce27 },
    { 0x0640, 0x0000, 0xf9c0, 0x7195, 0xcc81 },
    { 0x0564, 0x0000, 0xfa9c, 0x733c, 0xcac7 },
    { 0x047d, 0x0000, 0xfb83, 0x74f7, 0xc8f9 },
    { 0x038a, 0x0000, 0xfc76, 0x76c8, 0xc714 }
};


// BPF係数その３
// （中心周波数 2048Hz、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterBpf2048::CoefficientsTable32000[ BiquadFilterBpf2048::CoefficientsTableSize ] =
{
    //    n0      n1      n2      d1      d2
    { 0x3f42, 0x0000, 0xc0be, 0x0136, 0x3e83 },     // 32 - 15689.372 Hz
    { 0x3e8a, 0x0000, 0xc176, 0x02a3, 0x3d14 },
    { 0x3dd9, 0x0000, 0xc227, 0x0401, 0x3bb3 },
    { 0x3d2f, 0x0000, 0xc2d1, 0x0553, 0x3a5e },
    { 0x3c8b, 0x0000, 0xc375, 0x0699, 0x3916 },
    { 0x3bec, 0x0000, 0xc414, 0x07d5, 0x37d7 },
    { 0x3b51, 0x0000, 0xc4af, 0x0907, 0x36a2 },
    { 0x3abb, 0x0000, 0xc545, 0x0a31, 0x3576 },
    { 0x3a28, 0x0000, 0xc5d8, 0x0b53, 0x3451 },
    { 0x399a, 0x0000, 0xc666, 0x0c6d, 0x3333 },
    { 0x390e, 0x0000, 0xc6f2, 0x0d81, 0x321c },
    { 0x3885, 0x0000, 0xc77b, 0x0e8f, 0x310b },
    { 0x37ff, 0x0000, 0xc801, 0x0f97, 0x2fff },
    { 0x377c, 0x0000, 0xc884, 0x109b, 0x2ef7 },
    { 0x36fa, 0x0000, 0xc906, 0x119a, 0x2df5 },
    { 0x367b, 0x0000, 0xc985, 0x1294, 0x2cf6 },
    { 0x35fe, 0x0000, 0xca02, 0x138b, 0x2bfb },     // 64 - 11094.062 Hz
    { 0x3582, 0x0000, 0xca7e, 0x147e, 0x2b04 },
    { 0x3508, 0x0000, 0xcaf8, 0x156e, 0x2a0f },
    { 0x348f, 0x0000, 0xcb71, 0x165b, 0x291e },
    { 0x3417, 0x0000, 0xcbe9, 0x1745, 0x282f },
    { 0x33a1, 0x0000, 0xcc5f, 0x182d, 0x2742 },
    { 0x332c, 0x0000, 0xccd4, 0x1913, 0x2657 },
    { 0x32b7, 0x0000, 0xcd49, 0x19f6, 0x256f },
    { 0x3244, 0x0000, 0xcdbc, 0x1ad7, 0x2488 },
    { 0x31d1, 0x0000, 0xce2f, 0x1bb7, 0x23a2 },
    { 0x315f, 0x0000, 0xcea1, 0x1c95, 0x22be },
    { 0x30ee, 0x0000, 0xcf12, 0x1d72, 0x21dc },
    { 0x307d, 0x0000, 0xcf83, 0x1e4d, 0x20fa },
    { 0x300c, 0x0000, 0xcff4, 0x1f28, 0x2019 },
    { 0x2f9c, 0x0000, 0xd064, 0x2001, 0x1f39 },
    { 0x2f2d, 0x0000, 0xd0d3, 0x20d9, 0x1e59 },
    { 0x2ebd, 0x0000, 0xd143, 0x21b1, 0x1d7a },     // 128 - 7844.686 Hz
    { 0x2e4e, 0x0000, 0xd1b2, 0x2288, 0x1c9c },
    { 0x2ddf, 0x0000, 0xd221, 0x235e, 0x1bbe },
    { 0x2d70, 0x0000, 0xd290, 0x2434, 0x1ae0 },
    { 0x2d01, 0x0000, 0xd2ff, 0x250a, 0x1a02 },
    { 0x2c92, 0x0000, 0xd36e, 0x25df, 0x1923 },
    { 0x2c23, 0x0000, 0xd3dd, 0x26b4, 0x1845 },
    { 0x2bb3, 0x0000, 0xd44d, 0x2789, 0x1767 },
    { 0x2b44, 0x0000, 0xd4bc, 0x285e, 0x1688 },
    { 0x2ad4, 0x0000, 0xd52c, 0x2934, 0x15a8 },
    { 0x2a64, 0x0000, 0xd59c, 0x2a09, 0x14c8 },
    { 0x29f4, 0x0000, 0xd60c, 0x2adf, 0x13e8 },
    { 0x2983, 0x0000, 0xd67d, 0x2bb6, 0x1306 },
    { 0x2912, 0x0000, 0xd6ee, 0x2c8d, 0x1224 },
    { 0x28a0, 0x0000, 0xd760, 0x2d64, 0x1140 },
    { 0x282e, 0x0000, 0xd7d2, 0x2e3c, 0x105c },
    { 0x27bb, 0x0000, 0xd845, 0x2f15, 0x0f76 },     // 256 - 5547.031 Hz
    { 0x2747, 0x0000, 0xd8b9, 0x2ff0, 0x0e8e },
    { 0x26d3, 0x0000, 0xd92d, 0x30cb, 0x0da5 },
    { 0x265d, 0x0000, 0xd9a3, 0x31a7, 0x0cbb },
    { 0x25e7, 0x0000, 0xda19, 0x3285, 0x0bce },
    { 0x2570, 0x0000, 0xda90, 0x3364, 0x0ae0 },
    { 0x24f7, 0x0000, 0xdb09, 0x3445, 0x09ef },
    { 0x247e, 0x0000, 0xdb82, 0x3528, 0x08fc },
    { 0x2403, 0x0000, 0xdbfd, 0x360c, 0x0806 },
    { 0x2387, 0x0000, 0xdc79, 0x36f3, 0x070d },
    { 0x2309, 0x0000, 0xdcf7, 0x37dc, 0x0612 },
    { 0x228a, 0x0000, 0xdd76, 0x38c8, 0x0513 },
    { 0x2208, 0x0000, 0xddf8, 0x39b6, 0x0411 },
    { 0x2185, 0x0000, 0xde7b, 0x3aa8, 0x030b },
    { 0x2100, 0x0000, 0xdf00, 0x3b9d, 0x0201 },
    { 0x2079, 0x0000, 0xdf87, 0x3c95, 0x00f2 },
    { 0x1ff0, 0x0000, 0xe010, 0x3d91, 0xffdf },     // 512 - 3922.343 Hz
    { 0x1f63, 0x0000, 0xe09d, 0x3e91, 0xfec7 },
    { 0x1ed5, 0x0000, 0xe12b, 0x3f96, 0xfda9 },
    { 0x1e43, 0x0000, 0xe1bd, 0x409f, 0xfc85 },
    { 0x1dae, 0x0000, 0xe252, 0x41ae, 0xfb5b },
    { 0x1d15, 0x0000, 0xe2eb, 0x42c3, 0xfa2a },
    { 0x1c79, 0x0000, 0xe387, 0x43de, 0xf8f2 },
    { 0x1bd9, 0x0000, 0xe427, 0x44ff, 0xf7b1 },
    { 0x1b34, 0x0000, 0xe4cc, 0x4628, 0xf668 },
    { 0x1a8b, 0x0000, 0xe575, 0x4759, 0xf515 },
    { 0x19dc, 0x0000, 0xe624, 0x4893, 0xf3b8 },
    { 0x1928, 0x0000, 0xe6d8, 0x49d6, 0xf250 },
    { 0x186e, 0x0000, 0xe792, 0x4b24, 0xf0dc },
    { 0x17ad, 0x0000, 0xe853, 0x4c7d, 0xef5a },
    { 0x16e5, 0x0000, 0xe91b, 0x4de3, 0xedca },
    { 0x1615, 0x0000, 0xe9eb, 0x4f56, 0xec2a },
    { 0x153d, 0x0000, 0xeac3, 0x50d7, 0xea79 },     // 1024 - 2773.515 Hz
    { 0x145a, 0x0000, 0xeba6, 0x526a, 0xe8b5 },
    { 0x136e, 0x0000, 0xec92, 0x540e, 0xe6dc },
    { 0x1276, 0x0000, 0xed8a, 0x55c6, 0xe4ec },
    { 0x1172, 0x0000, 0xee8e, 0x5794, 0xe2e3 },
    { 0x105f, 0x0000, 0xefa1, 0x597a, 0xe0be },
    { 0x0f3d, 0x0000, 0xf0c3, 0x5b7b, 0xde7a },
    { 0x0e0a, 0x0000, 0xf1f6, 0x5d9b, 0xdc14 },
    { 0x0cc3, 0x0000, 0xf33d, 0x5fdc, 0xd987 },
    { 0x0b67, 0x0000, 0xf499, 0x6242, 0xd6cf },
    { 0x09f3, 0x0000, 0xf60d, 0x64d3, 0xd3e7 },
    { 0x0864, 0x0000, 0xf79c, 0x6793, 0xd0c8 },
    { 0x06b6, 0x0000, 0xf94a, 0x6a89, 0xcd6c }
};

//------------------------------------------------------------------------------
// サンプルレート 48kHz 用の係数定義
//------------------------------------------------------------------------------
// LPF係数
// （1/16オクターブ間隔、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterLpfNw4fCompatible48k::CoefficientsTable48000[ BiquadFilterLpfNw4fCompatible48k::CoefficientsTableSize ] =
{
    //    n0      n1      n2      d1      d2
    { 0x2162, 0x42c4, 0x2162, 0xca58, 0xe838 },   //( 15689.372151Hz )
    { 0x1f7e, 0x3efc, 0x1f7e, 0xd0cc, 0xe9c7 },   //( 15024.194244Hz )
    { 0x1db6, 0x3b6b, 0x1db6, 0xd70e, 0xeb13 },   //( 14387.217698Hz )
    { 0x1c06, 0x380d, 0x1c06, 0xdd1e, 0xec26 },   //( 13777.246868Hz )
    { 0x1a6f, 0x34de, 0x1a6f, 0xe2ff, 0xed04 },   //( 13193.136799Hz )
    { 0x18ed, 0x31da, 0x18ed, 0xe8b2, 0xedb3 },   //( 12633.791082Hz )
    { 0x1780, 0x2f00, 0x1780, 0xee39, 0xee38 },   //( 12098.159788Hz )
    { 0x1626, 0x2c4c, 0x1626, 0xf395, 0xee96 },   //( 11585.237503Hz )
    { 0x14de, 0x29bc, 0x14de, 0xf8c7, 0xeed1 },   //( 11094.061441Hz )
    { 0x13a7, 0x274e, 0x13a7, 0xfdd1, 0xeeec },   //( 10623.709632Hz )
    { 0x1280, 0x2500, 0x1280, 0x02b3, 0xeeeb },   //( 10173.299197Hz )
    { 0x1168, 0x22d1, 0x1168, 0x0770, 0xeecf },   //( 9741.984686Hz )
    { 0x105f, 0x20be, 0x105f, 0x0c08, 0xee9b },   //( 9328.956496Hz )
    { 0x0f64, 0x1ec8, 0x0f64, 0x107b, 0xee52 },   //( 8933.439346Hz )
    { 0x0e75, 0x1ceb, 0x0e75, 0x14ca, 0xedf4 },   //( 8554.690826Hz )
    { 0x0d94, 0x1b27, 0x0d94, 0x18f6, 0xed84 },   //( 8192.000000Hz )
    { 0x0cbe, 0x197c, 0x0cbe, 0x1d00, 0xed04 },   //( 7844.686075Hz )
    { 0x0bf4, 0x17e7, 0x0bf4, 0x20e8, 0xec75 },   //( 7512.097122Hz )
    { 0x0b34, 0x1668, 0x0b34, 0x24b0, 0xebd9 },   //( 7193.608849Hz )
    { 0x0a7f, 0x14fe, 0x0a7f, 0x2857, 0xeb31 },   //( 6888.623434Hz )
    { 0x09d4, 0x13a8, 0x09d4, 0x2bde, 0xea7e },   //( 6596.568400Hz )
    { 0x0932, 0x1265, 0x0932, 0x2f47, 0xe9c3 },   //( 6316.895541Hz )
    { 0x089a, 0x1134, 0x089a, 0x3291, 0xe8fe },   //( 6049.079894Hz )
    { 0x080a, 0x1014, 0x080a, 0x35be, 0xe833 },   //( 5792.618751Hz )
    { 0x0783, 0x0f05, 0x0783, 0x38cd, 0xe762 },   //( 5547.030720Hz )
    { 0x0703, 0x0e06, 0x0703, 0x3bc0, 0xe68c },   //( 5311.854816Hz )
    { 0x068b, 0x0d16, 0x068b, 0x3e97, 0xe5b1 },   //( 5086.649598Hz )
    { 0x061a, 0x0c34, 0x061a, 0x4154, 0xe4d3 },   //( 4870.992343Hz )
    { 0x05b0, 0x0b60, 0x05b0, 0x43f6, 0xe3f3 },   //( 4664.478248Hz )
    { 0x054c, 0x0a98, 0x054c, 0x467e, 0xe311 },   //( 4466.719673Hz )
    { 0x04ef, 0x09dd, 0x04ef, 0x48ee, 0xe22d },   //( 4277.345413Hz )
    { 0x0497, 0x092e, 0x0497, 0x4b45, 0xe14a },   //( 4096.000000Hz )
    { 0x0445, 0x088a, 0x0445, 0x4d84, 0xe066 },   //( 3922.343038Hz )
    { 0x03f8, 0x07f0, 0x03f8, 0x4fad, 0xdf83 },   //( 3756.048561Hz )
    { 0x03b0, 0x0760, 0x03b0, 0x51c0, 0xdea1 },   //( 3596.804424Hz )
    { 0x036d, 0x06da, 0x036d, 0x53bd, 0xddc1 },   //( 3444.311717Hz )
    { 0x032e, 0x065c, 0x032e, 0x55a6, 0xdce3 },   //( 3298.284200Hz )
    { 0x02f3, 0x05e7, 0x02f3, 0x577a, 0xdc07 },   //( 3158.447770Hz )
    { 0x02bd, 0x0579, 0x02bd, 0x593b, 0xdb2e },   //( 3024.539947Hz )
    { 0x0289, 0x0513, 0x0289, 0x5aea, 0xda57 },   //( 2896.309376Hz )
    { 0x025a, 0x04b4, 0x025a, 0x5c86, 0xd985 },   //( 2773.515360Hz )
    { 0x022e, 0x045b, 0x022e, 0x5e11, 0xd8b5 },   //( 2655.927408Hz )
    { 0x0204, 0x0409, 0x0204, 0x5f8c, 0xd7ea },   //( 2543.324799Hz )
    { 0x01de, 0x03bc, 0x01de, 0x60f6, 0xd722 },   //( 2435.496172Hz )
    { 0x01ba, 0x0374, 0x01ba, 0x6251, 0xd65f },   //( 2332.239124Hz )
    { 0x0199, 0x0332, 0x0199, 0x639d, 0xd59f },   //( 2233.359836Hz )
    { 0x017a, 0x02f4, 0x017a, 0x64da, 0xd4e4 },   //( 2138.672706Hz )
    { 0x015e, 0x02bb, 0x015e, 0x660a, 0xd42d },   //( 2048.000000Hz )
    { 0x0143, 0x0286, 0x0143, 0x672d, 0xd37b },   //( 1961.171519Hz )
    { 0x012a, 0x0255, 0x012a, 0x6843, 0xd2cd },   //( 1878.024280Hz )
    { 0x0114, 0x0227, 0x0114, 0x694d, 0xd224 },   //( 1798.402212Hz )
    { 0x00fe, 0x01fd, 0x00fe, 0x6a4c, 0xd17f },   //( 1722.155858Hz )
    { 0x00eb, 0x01d6, 0x00eb, 0x6b3f, 0xd0df },   //( 1649.142100Hz )
    { 0x00d9, 0x01b1, 0x00d9, 0x6c27, 0xd043 },   //( 1579.223885Hz )
    { 0x00c8, 0x0190, 0x00c8, 0x6d06, 0xcfac },   //( 1512.269973Hz )
    { 0x00b8, 0x0171, 0x00b8, 0x6dda, 0xcf19 },   //( 1448.154688Hz )
    { 0x00aa, 0x0154, 0x00aa, 0x6ea5, 0xce8b },   //( 1386.757680Hz )
    { 0x009d, 0x013a, 0x009d, 0x6f68, 0xce01 },   //( 1327.963704Hz )
    { 0x0091, 0x0121, 0x0091, 0x7021, 0xcd7b },   //( 1271.662400Hz )
    { 0x0085, 0x010a, 0x0085, 0x70d3, 0xccfa },   //( 1217.748086Hz )
    { 0x007b, 0x00f5, 0x007b, 0x717d, 0xcc7d },   //( 1166.119562Hz )
    { 0x0071, 0x00e2, 0x0071, 0x721f, 0xcc03 },   //( 1116.679918Hz )
    { 0x0068, 0x00d0, 0x0068, 0x72ba, 0xcb8e },   //( 1069.336353Hz )
    { 0x0060, 0x00c0, 0x0060, 0x734e, 0xcb1d },   //( 1024.000000Hz )
    { 0x0058, 0x00b0, 0x0058, 0x73dc, 0xcab0 },   //( 980.585759Hz )
    { 0x0051, 0x00a2, 0x0051, 0x7463, 0xca46 },   //( 939.012140Hz )
    { 0x004b, 0x0095, 0x004b, 0x74e4, 0xc9e0 },   //( 899.201106Hz )
    { 0x0045, 0x008a, 0x0045, 0x7560, 0xc97d },   //( 861.077929Hz )
    { 0x003f, 0x007f, 0x003f, 0x75d6, 0xc91f },   //( 824.571050Hz )
    { 0x003a, 0x0074, 0x003a, 0x7647, 0xc8c3 },   //( 789.611943Hz )
    { 0x0036, 0x006b, 0x0036, 0x76b3, 0xc86b },   //( 756.134987Hz )
    { 0x0031, 0x0063, 0x0031, 0x771b, 0xc816 },   //( 724.077344Hz )
    { 0x002d, 0x005b, 0x002d, 0x777d, 0xc7c4 },   //( 693.378840Hz )
    { 0x002a, 0x0053, 0x002a, 0x77dc, 0xc775 },   //( 663.981852Hz )
    { 0x0026, 0x004d, 0x0026, 0x7836, 0xc729 },   //( 635.831200Hz )
    { 0x0023, 0x0046, 0x0023, 0x788c, 0xc6df },   //( 608.874043Hz )
    { 0x0020, 0x0041, 0x0020, 0x78df, 0xc699 },   //( 583.059781Hz )
    { 0x001e, 0x003b, 0x001e, 0x792e, 0xc655 },   //( 558.339959Hz )
    { 0x001b, 0x0037, 0x001b, 0x7979, 0xc614 },   //( 534.668177Hz )
    { 0x0019, 0x0032, 0x0019, 0x79c1, 0xc5d5 },   //( 512.000000Hz )
    { 0x0017, 0x002e, 0x0017, 0x7a06, 0xc599 },   //( 490.292880Hz )
    { 0x0015, 0x002a, 0x0015, 0x7a48, 0xc55f },   //( 469.506070Hz )
    { 0x0013, 0x0027, 0x0013, 0x7a87, 0xc527 },   //( 449.600553Hz )
    { 0x0012, 0x0024, 0x0012, 0x7ac4, 0xc4f1 },   //( 430.538965Hz )
    { 0x0010, 0x0021, 0x0010, 0x7afe, 0xc4be },   //( 412.285525Hz )
    { 0x000f, 0x001e, 0x000f, 0x7b35, 0xc48c },   //( 394.805971Hz )
    { 0x000e, 0x001c, 0x000e, 0x7b69, 0xc45d },   //( 378.067493Hz )
    { 0x000d, 0x0019, 0x000d, 0x7b9c, 0xc42f },   //( 362.038672Hz )
    { 0x000c, 0x0017, 0x000c, 0x7bcc, 0xc403 },   //( 346.689420Hz )
    { 0x000b, 0x0015, 0x000b, 0x7bfa, 0xc3d9 },   //( 331.990926Hz )
    { 0x000a, 0x0014, 0x000a, 0x7c27, 0xc3b1 },   //( 317.915600Hz )
    { 0x0009, 0x0012, 0x0009, 0x7c51, 0xc38a },   //( 304.437021Hz )
    { 0x0008, 0x0011, 0x0008, 0x7c79, 0xc364 },   //( 291.529890Hz )
    { 0x0008, 0x000f, 0x0008, 0x7ca0, 0xc341 },   //( 279.169980Hz )
    { 0x0007, 0x000e, 0x0007, 0x7cc5, 0xc31e },   //( 267.334088Hz )
    { 0x0006, 0x000d, 0x0006, 0x7ce9, 0xc2fd },   //( 256.000000Hz )
    { 0x0006, 0x000c, 0x0006, 0x7d0a, 0xc2de },   //( 245.146440Hz )
    { 0x0005, 0x000b, 0x0005, 0x7d2b, 0xc2bf },   //( 234.753035Hz )
    { 0x0005, 0x000a, 0x0005, 0x7d4a, 0xc2a2 },   //( 224.800277Hz )
    { 0x0005, 0x0009, 0x0005, 0x7d68, 0xc286 },   //( 215.269482Hz )
    { 0x0004, 0x0008, 0x0004, 0x7d84, 0xc26b },   //( 206.142762Hz )
    { 0x0004, 0x0008, 0x0004, 0x7d9f, 0xc252 },   //( 197.402986Hz )
    { 0x0004, 0x0007, 0x0004, 0x7db9, 0xc239 },   //( 189.033747Hz )
    { 0x0003, 0x0006, 0x0003, 0x7dd2, 0xc221 },   //( 181.019336Hz )
    { 0x0003, 0x0006, 0x0003, 0x7dea, 0xc20a },   //( 173.344710Hz )
    { 0x0003, 0x0005, 0x0003, 0x7e01, 0xc1f5 },   //( 165.995463Hz )
    { 0x0003, 0x0005, 0x0003, 0x7e17, 0xc1e0 },   //( 158.957800Hz )
    { 0x0002, 0x0005, 0x0002, 0x7e2b, 0xc1cc },   //( 152.218511Hz )
    { 0x0002, 0x0004, 0x0002, 0x7e3f, 0xc1b9 },   //( 145.764945Hz )
    { 0x0002, 0x0004, 0x0002, 0x7e53, 0xc1a6 },   //( 139.584990Hz )
    { 0x0002, 0x0004, 0x0002, 0x7e65, 0xc195 },   //( 133.667044Hz )
    { 0x0002, 0x0003, 0x0002, 0x7e76, 0xc184 },   //( 128.000000Hz )
};


// HPF係数
// （1/16オクターブ間隔、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterHpfNw4fCompatible48k::CoefficientsTable48000[ BiquadFilterHpfNw4fCompatible48k::CoefficientsTableSize ] =
{
    //    n0      n1      n2      d1      d2
    { 0x3bf2, 0x881d, 0x3bf2, 0x7efd, 0xc101 },   //( 128.000000Hz )
    { 0x3bed, 0x8828, 0x3bed, 0x7ef1, 0xc10c },   //( 133.667044Hz )
    { 0x3be7, 0x8833, 0x3be7, 0x7ee5, 0xc118 },   //( 139.584990Hz )
    { 0x3be1, 0x883f, 0x3be1, 0x7ed9, 0xc124 },   //( 145.764945Hz )
    { 0x3bdb, 0x884b, 0x3bdb, 0x7ecc, 0xc131 },   //( 152.218511Hz )
    { 0x3bd5, 0x8858, 0x3bd5, 0x7ebe, 0xc13e },   //( 158.957800Hz )
    { 0x3bce, 0x8865, 0x3bce, 0x7eb0, 0xc14c },   //( 165.995463Hz )
    { 0x3bc7, 0x8873, 0x3bc7, 0x7ea1, 0xc15b },   //( 173.344710Hz )
    { 0x3bc0, 0x8882, 0x3bc0, 0x7e91, 0xc16a },   //( 181.019336Hz )
    { 0x3bb8, 0x8891, 0x3bb8, 0x7e81, 0xc17a },   //( 189.033747Hz )
    { 0x3bb0, 0x88a1, 0x3bb0, 0x7e70, 0xc18a },   //( 197.402986Hz )
    { 0x3ba8, 0x88b1, 0x3ba8, 0x7e5e, 0xc19b },   //( 206.142762Hz )
    { 0x3b9f, 0x88c2, 0x3b9f, 0x7e4b, 0xc1ad },   //( 215.269482Hz )
    { 0x3b96, 0x88d4, 0x3b96, 0x7e38, 0xc1c0 },   //( 224.800277Hz )
    { 0x3b8d, 0x88e7, 0x3b8d, 0x7e23, 0xc1d4 },   //( 234.753035Hz )
    { 0x3b83, 0x88fb, 0x3b83, 0x7e0e, 0xc1e8 },   //( 245.146440Hz )
    { 0x3b79, 0x890f, 0x3b79, 0x7df8, 0xc1fd },   //( 256.000000Hz )
    { 0x3b6e, 0x8925, 0x3b6e, 0x7de1, 0xc213 },   //( 267.334088Hz )
    { 0x3b63, 0x893b, 0x3b63, 0x7dc8, 0xc22a },   //( 279.169980Hz )
    { 0x3b57, 0x8952, 0x3b57, 0x7daf, 0xc242 },   //( 291.529890Hz )
    { 0x3b4b, 0x896b, 0x3b4b, 0x7d95, 0xc25c },   //( 304.437021Hz )
    { 0x3b3e, 0x8984, 0x3b3e, 0x7d79, 0xc276 },   //( 317.915600Hz )
    { 0x3b31, 0x899f, 0x3b31, 0x7d5c, 0xc291 },   //( 331.990926Hz )
    { 0x3b23, 0x89ba, 0x3b23, 0x7d3e, 0xc2ad },   //( 346.689420Hz )
    { 0x3b15, 0x89d7, 0x3b15, 0x7d1e, 0xc2cb },   //( 362.038672Hz )
    { 0x3b06, 0x89f5, 0x3b06, 0x7cfd, 0xc2ea },   //( 378.067493Hz )
    { 0x3af6, 0x8a15, 0x3af6, 0x7cdb, 0xc30a },   //( 394.805971Hz )
    { 0x3ae6, 0x8a36, 0x3ae6, 0x7cb7, 0xc32c },   //( 412.285525Hz )
    { 0x3ad4, 0x8a58, 0x3ad4, 0x7c91, 0xc34f },   //( 430.538965Hz )
    { 0x3ac3, 0x8a7c, 0x3ac3, 0x7c6a, 0xc373 },   //( 449.600553Hz )
    { 0x3ab0, 0x8aa1, 0x3ab0, 0x7c40, 0xc399 },   //( 469.506070Hz )
    { 0x3a9c, 0x8ac8, 0x3a9c, 0x7c15, 0xc3c0 },   //( 490.292880Hz )
    { 0x3a88, 0x8af1, 0x3a88, 0x7be8, 0xc3ea },   //( 512.000000Hz )
    { 0x3a73, 0x8b1b, 0x3a73, 0x7bb9, 0xc414 },   //( 534.668177Hz )
    { 0x3a5d, 0x8b48, 0x3a5d, 0x7b88, 0xc441 },   //( 558.339959Hz )
    { 0x3a46, 0x8b76, 0x3a46, 0x7b55, 0xc470 },   //( 583.059781Hz )
    { 0x3a2e, 0x8ba6, 0x3a2e, 0x7b1f, 0xc4a0 },   //( 608.874043Hz )
    { 0x3a14, 0x8bd8, 0x3a14, 0x7ae7, 0xc4d2 },   //( 635.831200Hz )
    { 0x39fa, 0x8c0d, 0x39fa, 0x7aac, 0xc507 },   //( 663.981852Hz )
    { 0x39df, 0x8c43, 0x39df, 0x7a6e, 0xc53d },   //( 693.378840Hz )
    { 0x39c2, 0x8c7d, 0x39c2, 0x7a2e, 0xc576 },   //( 724.077344Hz )
    { 0x39a4, 0x8cb8, 0x39a4, 0x79eb, 0xc5b1 },   //( 756.134987Hz )
    { 0x3985, 0x8cf6, 0x3985, 0x79a5, 0xc5ee },   //( 789.611943Hz )
    { 0x3965, 0x8d37, 0x3965, 0x795b, 0xc62e },   //( 824.571050Hz )
    { 0x3943, 0x8d7b, 0x3943, 0x790e, 0xc670 },   //( 861.077929Hz )
    { 0x3920, 0x8dc1, 0x3920, 0x78be, 0xc6b5 },   //( 899.201106Hz )
    { 0x38fb, 0x8e0b, 0x38fb, 0x786a, 0xc6fd },   //( 939.012140Hz )
    { 0x38d5, 0x8e58, 0x38d5, 0x7812, 0xc747 },   //( 980.585759Hz )
    { 0x38ad, 0x8ea8, 0x38ad, 0x77b5, 0xc795 },   //( 1024.000000Hz )
    { 0x3883, 0x8efb, 0x3883, 0x7755, 0xc7e5 },   //( 1069.336353Hz )
    { 0x3858, 0x8f52, 0x3858, 0x76f0, 0xc839 },   //( 1116.679918Hz )
    { 0x382a, 0x8fad, 0x382a, 0x7687, 0xc88f },   //( 1166.119562Hz )
    { 0x37fb, 0x900b, 0x37fb, 0x7618, 0xc8e9 },   //( 1217.748086Hz )
    { 0x37ca, 0x906e, 0x37ca, 0x75a5, 0xc946 },   //( 1271.662400Hz )
    { 0x3796, 0x90d4, 0x3796, 0x752c, 0xc9a7 },   //( 1327.963704Hz )
    { 0x3761, 0x913f, 0x3761, 0x74ae, 0xca0b },   //( 1386.757680Hz )
    { 0x3729, 0x91af, 0x3729, 0x7429, 0xca73 },   //( 1448.154688Hz )
    { 0x36ef, 0x9223, 0x36ef, 0x739f, 0xcadf },   //( 1512.269973Hz )
    { 0x36b2, 0x929c, 0x36b2, 0x730e, 0xcb4e },   //( 1579.223885Hz )
    { 0x3673, 0x931a, 0x3673, 0x7276, 0xcbc2 },   //( 1649.142100Hz )
    { 0x3632, 0x939e, 0x3632, 0x71d7, 0xcc39 },   //( 1722.155858Hz )
    { 0x35ed, 0x9427, 0x35ed, 0x7131, 0xccb5 },   //( 1798.402212Hz )
    { 0x35a6, 0x94b5, 0x35a6, 0x7083, 0xcd35 },   //( 1878.024280Hz )
    { 0x355c, 0x954a, 0x355c, 0x6fcc, 0xcdb9 },   //( 1961.171519Hz )
    { 0x350e, 0x95e4, 0x350e, 0x6f0d, 0xce41 },   //( 2048.000000Hz )
    { 0x34be, 0x9685, 0x34be, 0x6e46, 0xcece },   //( 2138.672706Hz )
    { 0x346a, 0x972d, 0x346a, 0x6d74, 0xcf60 },   //( 2233.359836Hz )
    { 0x3413, 0x97db, 0x3413, 0x6c99, 0xcff6 },   //( 2332.239124Hz )
    { 0x33b8, 0x9890, 0x33b8, 0x6bb4, 0xd091 },   //( 2435.496172Hz )
    { 0x335a, 0x994d, 0x335a, 0x6ac3, 0xd131 },   //( 2543.324799Hz )
    { 0x32f8, 0x9a11, 0x32f8, 0x69c8, 0xd1d5 },   //( 2655.927408Hz )
    { 0x3292, 0x9ade, 0x3292, 0x68c0, 0xd27e },   //( 2773.515360Hz )
    { 0x3227, 0x9bb2, 0x3227, 0x67ac, 0xd32c },   //( 2896.309376Hz )
    { 0x31b9, 0x9c8f, 0x31b9, 0x668b, 0xd3df },   //( 3024.539947Hz )
    { 0x3146, 0x9d75, 0x3146, 0x655c, 0xd497 },   //( 3158.447770Hz )
    { 0x30cf, 0x9e63, 0x30cf, 0x641f, 0xd553 },   //( 3298.284200Hz )
    { 0x3053, 0x9f5b, 0x3053, 0x62d3, 0xd614 },   //( 3444.311717Hz )
    { 0x2fd2, 0xa05d, 0x2fd2, 0x6177, 0xd6da },   //( 3596.804424Hz )
    { 0x2f4c, 0xa169, 0x2f4c, 0x600a, 0xd7a5 },   //( 3756.048561Hz )
    { 0x2ec1, 0xa27f, 0x2ec1, 0x5e8c, 0xd874 },   //( 3922.343038Hz )
    { 0x2e31, 0xa39f, 0x2e31, 0x5cfc, 0xd947 },   //( 4096.000000Hz )
    { 0x2d9b, 0xa4cb, 0x2d9b, 0x5b59, 0xda1f },   //( 4277.345413Hz )
    { 0x2cff, 0xa602, 0x2cff, 0x59a2, 0xdafb },   //( 4466.719673Hz )
    { 0x2c5e, 0xa745, 0x2c5e, 0x57d6, 0xdbdb },   //( 4664.478248Hz )
    { 0x2bb6, 0xa894, 0x2bb6, 0x55f4, 0xdcbe },   //( 4870.992343Hz )
    { 0x2b09, 0xa9ef, 0x2b09, 0x53fb, 0xdda5 },   //( 5086.649598Hz )
    { 0x2a55, 0xab58, 0x2a55, 0x51e9, 0xde8f },   //( 5311.854816Hz )
    { 0x299a, 0xaccd, 0x299a, 0x4fbe, 0xdf7c },   //( 5547.030720Hz )
    { 0x28d8, 0xae51, 0x28d8, 0x4d78, 0xe06b },   //( 5792.618751Hz )
    { 0x280f, 0xafe2, 0x280f, 0x4b17, 0xe15c },   //( 6049.079894Hz )
    { 0x273f, 0xb182, 0x273f, 0x4897, 0xe24e },   //( 6316.895541Hz )
    { 0x2668, 0xb331, 0x2668, 0x45f9, 0xe340 },   //( 6596.568400Hz )
    { 0x2589, 0xb4f0, 0x2589, 0x4339, 0xe433 },   //( 6888.623434Hz )
    { 0x24a1, 0xb6be, 0x24a1, 0x4057, 0xe525 },   //( 7193.608849Hz )
    { 0x23b2, 0xb89d, 0x23b2, 0x3d51, 0xe615 },   //( 7512.097122Hz )
    { 0x22ba, 0xba8d, 0x22ba, 0x3a24, 0xe702 },   //( 7844.686075Hz )
    { 0x21b9, 0xbc8f, 0x21b9, 0x36ce, 0xe7ec },   //( 8192.000000Hz )
};


// BPF係数その１
// （中心周波数 512Hz、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterBpf512Nw4fCompatible48k::CoefficientsTable48000[ BiquadFilterBpf512Nw4fCompatible48k::CoefficientsTableSize ] =
{
    //    n0      n1      n2      d1      d2
    { 0x2772, 0x0000, 0xd88f, 0x30ff, 0x0ee4 },   //( 32.000000 - 7844.686075 Hz )
    { 0x270d, 0x0000, 0xd8f4, 0x31c9, 0x0e19 },   //( 32.700709 - 7676.590629 Hz )
    { 0x26a7, 0x0000, 0xd95a, 0x3293, 0x0d4f },   //( 33.416761 - 7512.097122 Hz )
    { 0x2642, 0x0000, 0xd9bf, 0x335c, 0x0c85 },   //( 34.148493 - 7351.128371 Hz )
    { 0x25de, 0x0000, 0xda23, 0x3425, 0x0bbb },   //( 34.896247 - 7193.608849 Hz )
    { 0x2579, 0x0000, 0xda88, 0x34ee, 0x0af2 },   //( 35.660376 - 7039.464645 Hz )
    { 0x2515, 0x0000, 0xdaec, 0x35b7, 0x0a29 },   //( 36.441236 - 6888.623434 Hz )
    { 0x24b0, 0x0000, 0xdb51, 0x367f, 0x0961 },   //( 37.239195 - 6741.014439 Hz )
    { 0x244c, 0x0000, 0xdbb5, 0x3747, 0x0899 },   //( 38.054628 - 6596.568400 Hz )
    { 0x23e9, 0x0000, 0xdc18, 0x380e, 0x07d1 },   //( 38.887916 - 6455.217542 Hz )
    { 0x2385, 0x0000, 0xdc7c, 0x38d5, 0x070a },   //( 39.739450 - 6316.895541 Hz )
    { 0x2321, 0x0000, 0xdce0, 0x399c, 0x0643 },   //( 40.609631 - 6181.537495 Hz )
    { 0x22be, 0x0000, 0xdd43, 0x3a62, 0x057c },   //( 41.498866 - 6049.079894 Hz )
    { 0x225b, 0x0000, 0xdda6, 0x3b28, 0x04b6 },   //( 42.407573 - 5919.460585 Hz )
    { 0x21f8, 0x0000, 0xde09, 0x3bed, 0x03f0 },   //( 43.336178 - 5792.618751 Hz )
    { 0x2195, 0x0000, 0xde6c, 0x3cb2, 0x032b },   //( 44.285116 - 5668.494877 Hz )
    { 0x2133, 0x0000, 0xdece, 0x3d77, 0x0266 },   //( 45.254834 - 5547.030720 Hz )
    { 0x20d1, 0x0000, 0xdf30, 0x3e3b, 0x01a1 },   //( 46.245786 - 5428.169290 Hz )
    { 0x206f, 0x0000, 0xdf92, 0x3efe, 0x00dd },   //( 47.258437 - 5311.854816 Hz )
    { 0x200d, 0x0000, 0xdff4, 0x3fc1, 0x001a },   //( 48.293262 - 5198.032721 Hz )
    { 0x1fac, 0x0000, 0xe055, 0x4084, 0xff58 },   //( 49.350746 - 5086.649598 Hz )
    { 0x1f4a, 0x0000, 0xe0b7, 0x4146, 0xfe96 },   //( 50.431387 - 4977.653186 Hz )
    { 0x1ee9, 0x0000, 0xe118, 0x4208, 0xfdd4 },   //( 51.535691 - 4870.992343 Hz )
    { 0x1e89, 0x0000, 0xe178, 0x42c9, 0xfd12 },   //( 52.664175 - 4766.617022 Hz )
    { 0x1e28, 0x0000, 0xe1d9, 0x4389, 0xfc52 },   //( 53.817371 - 4664.478248 Hz )
    { 0x1dc8, 0x0000, 0xe239, 0x4449, 0xfb91 },   //( 54.995818 - 4564.528098 Hz )
    { 0x1d68, 0x0000, 0xe299, 0x4508, 0xfad2 },   //( 56.200069 - 4466.719673 Hz )
    { 0x1d09, 0x0000, 0xe2f8, 0x45c6, 0xfa13 },   //( 57.430690 - 4371.007081 Hz )
    { 0x1caa, 0x0000, 0xe357, 0x4684, 0xf955 },   //( 58.688259 - 4277.345413 Hz )
    { 0x1c4b, 0x0000, 0xe3b6, 0x4742, 0xf897 },   //( 59.973364 - 4185.690721 Hz )
    { 0x1bed, 0x0000, 0xe414, 0x47fe, 0xf7da },   //( 61.286610 - 4096.000000 Hz )
    { 0x1b8f, 0x0000, 0xe472, 0x48ba, 0xf71e },   //( 62.628612 - 4008.231166 Hz )
    { 0x1b31, 0x0000, 0xe4d0, 0x4975, 0xf663 },   //( 64.000000 - 3922.343038 Hz )
    { 0x1ad4, 0x0000, 0xe52d, 0x4a2f, 0xf5a8 },   //( 65.401418 - 3838.295315 Hz )
    { 0x1a77, 0x0000, 0xe58a, 0x4ae9, 0xf4ee },   //( 66.833522 - 3756.048561 Hz )
    { 0x1a1a, 0x0000, 0xe5e7, 0x4ba1, 0xf435 },   //( 68.296986 - 3675.564186 Hz )
    { 0x19be, 0x0000, 0xe643, 0x4c59, 0xf37d },   //( 69.792495 - 3596.804424 Hz )
    { 0x1962, 0x0000, 0xe69f, 0x4d10, 0xf2c5 },   //( 71.320752 - 3519.732323 Hz )
    { 0x1907, 0x0000, 0xe6fa, 0x4dc7, 0xf20f },   //( 72.882473 - 3444.311717 Hz )
    { 0x18ac, 0x0000, 0xe755, 0x4e7c, 0xf159 },   //( 74.478391 - 3370.507219 Hz )
    { 0x1852, 0x0000, 0xe7af, 0x4f31, 0xf0a4 },   //( 76.109255 - 3298.284200 Hz )
    { 0x17f8, 0x0000, 0xe809, 0x4fe4, 0xeff0 },   //( 77.775831 - 3227.608771 Hz )
    { 0x179e, 0x0000, 0xe863, 0x5097, 0xef3d },   //( 79.478900 - 3158.447770 Hz )
    { 0x1745, 0x0000, 0xe8bc, 0x5149, 0xee8b },   //( 81.219261 - 3090.768748 Hz )
    { 0x16ec, 0x0000, 0xe915, 0x51f9, 0xedda },   //( 82.997731 - 3024.539947 Hz )
    { 0x1694, 0x0000, 0xe96d, 0x52a9, 0xed2a },   //( 84.815145 - 2959.730293 Hz )
    { 0x163d, 0x0000, 0xe9c4, 0x5358, 0xec7a },   //( 86.672355 - 2896.309376 Hz )
    { 0x15e5, 0x0000, 0xea1c, 0x5406, 0xebcc },   //( 88.570232 - 2834.247438 Hz )
    { 0x158f, 0x0000, 0xea72, 0x54b3, 0xeb1f },   //( 90.509668 - 2773.515360 Hz )
    { 0x1539, 0x0000, 0xeac8, 0x555f, 0xea72 },   //( 92.491572 - 2714.084645 Hz )
    { 0x14e3, 0x0000, 0xeb1e, 0x560a, 0xe9c7 },   //( 94.516873 - 2655.927408 Hz )
    { 0x148e, 0x0000, 0xeb73, 0x56b4, 0xe91d },   //( 96.586523 - 2599.016360 Hz )
    { 0x1439, 0x0000, 0xebc8, 0x575d, 0xe874 },   //( 98.701493 - 2543.324799 Hz )
    { 0x13e5, 0x0000, 0xec1c, 0x5804, 0xe7cc },   //( 100.862774 - 2488.826593 Hz )
    { 0x1392, 0x0000, 0xec6f, 0x58ab, 0xe725 },   //( 103.071381 - 2435.496172 Hz )
    { 0x133f, 0x0000, 0xecc2, 0x5951, 0xe67f },   //( 105.328351 - 2383.308511 Hz )
    { 0x12ec, 0x0000, 0xed15, 0x59f5, 0xe5da },   //( 107.634741 - 2332.239124 Hz )
    { 0x129a, 0x0000, 0xed67, 0x5a99, 0xe536 },   //( 109.991635 - 2282.264049 Hz )
    { 0x1249, 0x0000, 0xedb8, 0x5b3c, 0xe493 },   //( 112.400138 - 2233.359836 Hz )
    { 0x11f8, 0x0000, 0xee09, 0x5bdd, 0xe3f1 },   //( 114.861381 - 2185.503541 Hz )
    { 0x11a8, 0x0000, 0xee59, 0x5c7d, 0xe350 },   //( 117.376518 - 2138.672706 Hz )
    { 0x1158, 0x0000, 0xeea9, 0x5d1c, 0xe2b1 },   //( 119.946729 - 2092.845360 Hz )
    { 0x1109, 0x0000, 0xeef8, 0x5dbb, 0xe212 },   //( 122.573220 - 2048.000000 Hz )
    { 0x10ba, 0x0000, 0xef47, 0x5e58, 0xe175 },   //( 125.257224 - 2004.115583 Hz )
    { 0x106c, 0x0000, 0xef95, 0x5ef4, 0xe0d9 },   //( 128.000000 - 1961.171519 Hz )
    { 0x101e, 0x0000, 0xefe3, 0x5f8f, 0xe03d },   //( 130.802835 - 1919.147657 Hz )
    { 0x0fd1, 0x0000, 0xf030, 0x6029, 0xdfa3 },   //( 133.667044 - 1878.024280 Hz )
    { 0x0f85, 0x0000, 0xf07c, 0x60c1, 0xdf0a },   //( 136.593971 - 1837.782093 Hz )
    { 0x0f39, 0x0000, 0xf0c8, 0x6159, 0xde72 },   //( 139.584990 - 1798.402212 Hz )
    { 0x0eed, 0x0000, 0xf114, 0x61f0, 0xdddb },   //( 142.641503 - 1759.866161 Hz )
    { 0x0ea2, 0x0000, 0xf15f, 0x6285, 0xdd45 },   //( 145.764945 - 1722.155858 Hz )
    { 0x0e58, 0x0000, 0xf1a9, 0x631a, 0xdcb0 },   //( 148.956782 - 1685.253610 Hz )
    { 0x0e0e, 0x0000, 0xf1f3, 0x63ad, 0xdc1d },   //( 152.218511 - 1649.142100 Hz )
    { 0x0dc4, 0x0000, 0xf23d, 0x6440, 0xdb8a },   //( 155.551662 - 1613.804385 Hz )
    { 0x0d7b, 0x0000, 0xf286, 0x64d1, 0xdaf8 },   //( 158.957800 - 1579.223885 Hz )
    { 0x0d33, 0x0000, 0xf2ce, 0x6562, 0xda67 },   //( 162.438523 - 1545.384374 Hz )
    { 0x0ceb, 0x0000, 0xf316, 0x65f1, 0xd9d7 },   //( 165.995463 - 1512.269973 Hz )
    { 0x0ca4, 0x0000, 0xf35d, 0x6680, 0xd949 },   //( 169.630290 - 1479.865146 Hz )
    { 0x0c5d, 0x0000, 0xf3a4, 0x670d, 0xd8bb },   //( 173.344710 - 1448.154688 Hz )
    { 0x0c16, 0x0000, 0xf3eb, 0x679a, 0xd82e },   //( 177.140465 - 1417.123719 Hz )
    { 0x0bd0, 0x0000, 0xf431, 0x6826, 0xd7a2 },   //( 181.019336 - 1386.757680 Hz )
    { 0x0b8b, 0x0000, 0xf476, 0x68b0, 0xd717 },   //( 184.983143 - 1357.042323 Hz )
    { 0x0b46, 0x0000, 0xf4bb, 0x693a, 0xd68d },   //( 189.033747 - 1327.963704 Hz )
    { 0x0b01, 0x0000, 0xf500, 0x69c3, 0xd603 },   //( 193.173047 - 1299.508180 Hz )
    { 0x0abd, 0x0000, 0xf544, 0x6a4b, 0xd57b },   //( 197.402986 - 1271.662400 Hz )
    { 0x0a79, 0x0000, 0xf588, 0x6ad3, 0xd4f3 },   //( 201.725548 - 1244.413297 Hz )
    { 0x0a36, 0x0000, 0xf5cb, 0x6b59, 0xd46d },   //( 206.142762 - 1217.748086 Hz )
    { 0x09f3, 0x0000, 0xf60e, 0x6bdf, 0xd3e7 },   //( 210.656701 - 1191.654255 Hz )
    { 0x09b0, 0x0000, 0xf651, 0x6c64, 0xd362 },   //( 215.269482 - 1166.119562 Hz )
    { 0x096e, 0x0000, 0xf693, 0x6ce8, 0xd2dd },   //( 219.983270 - 1141.132024 Hz )
    { 0x092c, 0x0000, 0xf6d5, 0x6d6b, 0xd259 },   //( 224.800277 - 1116.679918 Hz )
    { 0x08eb, 0x0000, 0xf716, 0x6dee, 0xd1d6 },   //( 229.722762 - 1092.751770 Hz )
    { 0x08a9, 0x0000, 0xf758, 0x6e70, 0xd154 },   //( 234.753035 - 1069.336353 Hz )
    { 0x0869, 0x0000, 0xf798, 0x6ef2, 0xd0d2 },   //( 239.893457 - 1046.422680 Hz )
    { 0x0828, 0x0000, 0xf7d9, 0x6f72, 0xd051 },   //( 245.146440 - 1024.000000 Hz )
    { 0x07e8, 0x0000, 0xf819, 0x6ff3, 0xcfd1 },   //( 250.514448 - 1002.057792 Hz )
    { 0x07a8, 0x0000, 0xf859, 0x7072, 0xcf51 },   //( 256.000000 - 980.585759 Hz )
    { 0x0768, 0x0000, 0xf899, 0x70f2, 0xced1 },   //( 261.605670 - 959.573829 Hz )
    { 0x0729, 0x0000, 0xf8d8, 0x7170, 0xce52 },   //( 267.334088 - 939.012140 Hz )
    { 0x06e9, 0x0000, 0xf918, 0x71ef, 0xcdd3 },   //( 273.187943 - 918.891046 Hz )
    { 0x06aa, 0x0000, 0xf957, 0x726d, 0xcd55 },   //( 279.169980 - 899.201106 Hz )
    { 0x066b, 0x0000, 0xf996, 0x72ea, 0xccd7 },   //( 285.283006 - 879.933081 Hz )
    { 0x062c, 0x0000, 0xf9d5, 0x7367, 0xcc5a },   //( 291.529890 - 861.077929 Hz )
    { 0x05ee, 0x0000, 0xfa13, 0x73e4, 0xcbdd },   //( 297.913564 - 842.626805 Hz )
    { 0x05af, 0x0000, 0xfa52, 0x7461, 0xcb60 },   //( 304.437021 - 824.571050 Hz )
    { 0x0571, 0x0000, 0xfa90, 0x74dd, 0xcae3 },   //( 311.103324 - 806.902193 Hz )
    { 0x0533, 0x0000, 0xface, 0x755a, 0xca67 },   //( 317.915600 - 789.611943 Hz )
    { 0x04f5, 0x0000, 0xfb0c, 0x75d6, 0xc9ea },   //( 324.877045 - 772.692187 Hz )
    { 0x04b6, 0x0000, 0xfb4b, 0x7652, 0xc96e },   //( 331.990926 - 756.134987 Hz )
    { 0x0478, 0x0000, 0xfb89, 0x76ce, 0xc8f2 },   //( 339.260581 - 739.932573 Hz )
    { 0x043a, 0x0000, 0xfbc7, 0x774a, 0xc875 },   //( 346.689420 - 724.077344 Hz )
    { 0x03fc, 0x0000, 0xfc05, 0x77c6, 0xc7f9 },   //( 354.280930 - 708.561860 Hz )
    { 0x03be, 0x0000, 0xfc43, 0x7842, 0xc77d },   //( 362.038672 - 693.378840 Hz )
    { 0x0380, 0x0000, 0xfc81, 0x78be, 0xc700 },   //( 369.966287 - 678.521161 Hz )
    { 0x0341, 0x0000, 0xfcc0, 0x793b, 0xc683 },   //( 378.067493 - 663.981852 Hz )
    { 0x0303, 0x0000, 0xfcfe, 0x79b8, 0xc606 },   //( 386.346093 - 649.754090 Hz )
    { 0x02c4, 0x0000, 0xfd3d, 0x7a35, 0xc589 },   //( 394.805971 - 635.831200 Hz )
    { 0x0285, 0x0000, 0xfd7c, 0x7ab2, 0xc50b },   //( 403.451096 - 622.206648 Hz )
    { 0x0246, 0x0000, 0xfdbb, 0x7b30, 0xc48d },   //( 412.285525 - 608.874043 Hz )
    { 0x0207, 0x0000, 0xfdfa, 0x7bae, 0xc40f },   //( 421.313402 - 595.827128 Hz )
    { 0x01c7, 0x0000, 0xfe3a, 0x7c2d, 0xc38f },   //( 430.538965 - 583.059781 Hz )
    { 0x0187, 0x0000, 0xfe7a, 0x7cad, 0xc310 },   //( 439.966540 - 570.566012 Hz )
};


// BPF係数その２
// （中心周波数 1024Hz、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterBpf1024Nw4fCompatible48k::CoefficientsTable48000[ BiquadFilterBpf1024Nw4fCompatible48k::CoefficientsTableSize ] =
{
    //    n0      n1      n2      d1      d2
    { 0x277b, 0x0000, 0xd886, 0x30fb, 0x0ef7 },   //( 16.000000 - 7844.686075 Hz )
    { 0x2716, 0x0000, 0xd8eb, 0x31c4, 0x0e2c },   //( 16.708381 - 7676.590629 Hz )
    { 0x26b1, 0x0000, 0xd950, 0x328e, 0x0d63 },   //( 17.448124 - 7512.097122 Hz )
    { 0x264d, 0x0000, 0xd9b4, 0x3357, 0x0c99 },   //( 18.220618 - 7351.128371 Hz )
    { 0x25e8, 0x0000, 0xda19, 0x341f, 0x0bd0 },   //( 19.027314 - 7193.608849 Hz )
    { 0x2583, 0x0000, 0xda7e, 0x34e8, 0x0b07 },   //( 19.869725 - 7039.464645 Hz )
    { 0x251f, 0x0000, 0xdae2, 0x35b0, 0x0a3e },   //( 20.749433 - 6888.623434 Hz )
    { 0x24bb, 0x0000, 0xdb46, 0x3677, 0x0976 },   //( 21.668089 - 6741.014439 Hz )
    { 0x2457, 0x0000, 0xdbaa, 0x373e, 0x08ae },   //( 22.627417 - 6596.568400 Hz )
    { 0x23f3, 0x0000, 0xdc0e, 0x3805, 0x07e7 },   //( 23.629218 - 6455.217542 Hz )
    { 0x2390, 0x0000, 0xdc71, 0x38cc, 0x071f },   //( 24.675373 - 6316.895541 Hz )
    { 0x232c, 0x0000, 0xdcd5, 0x3992, 0x0659 },   //( 25.767845 - 6181.537495 Hz )
    { 0x22c9, 0x0000, 0xdd38, 0x3a58, 0x0592 },   //( 26.908685 - 6049.079894 Hz )
    { 0x2266, 0x0000, 0xdd9b, 0x3b1d, 0x04cc },   //( 28.100035 - 5919.460585 Hz )
    { 0x2203, 0x0000, 0xddfe, 0x3be3, 0x0406 },   //( 29.344129 - 5792.618751 Hz )
    { 0x21a0, 0x0000, 0xde61, 0x3ca7, 0x0340 },   //( 30.643305 - 5668.494877 Hz )
    { 0x213e, 0x0000, 0xdec3, 0x3d6c, 0x027b },   //( 32.000000 - 5547.030720 Hz )
    { 0x20db, 0x0000, 0xdf26, 0x3e30, 0x01b7 },   //( 33.416761 - 5428.169290 Hz )
    { 0x2079, 0x0000, 0xdf88, 0x3ef3, 0x00f2 },   //( 34.896247 - 5311.854816 Hz )
    { 0x2017, 0x0000, 0xdfea, 0x3fb6, 0x002e },   //( 36.441236 - 5198.032721 Hz )
    { 0x1fb5, 0x0000, 0xe04c, 0x4079, 0xff6c },   //( 38.054628 - 5086.649598 Hz )
    { 0x1f54, 0x0000, 0xe0ad, 0x413b, 0xfea9 },   //( 39.739450 - 4977.653186 Hz )
    { 0x1ef2, 0x0000, 0xe10f, 0x41fd, 0xfde6 },   //( 41.498866 - 4870.992343 Hz )
    { 0x1e91, 0x0000, 0xe170, 0x42be, 0xfd24 },   //( 43.336178 - 4766.617022 Hz )
    { 0x1e30, 0x0000, 0xe1d1, 0x437f, 0xfc62 },   //( 45.254834 - 4664.478248 Hz )
    { 0x1dd0, 0x0000, 0xe231, 0x443f, 0xfba0 },   //( 47.258437 - 4564.528098 Hz )
    { 0x1d6f, 0x0000, 0xe292, 0x44ff, 0xfadf },   //( 49.350746 - 4466.719673 Hz )
    { 0x1d0f, 0x0000, 0xe2f2, 0x45bf, 0xfa1f },   //( 51.535691 - 4371.007081 Hz )
    { 0x1caf, 0x0000, 0xe352, 0x467e, 0xf95f },   //( 53.817371 - 4277.345413 Hz )
    { 0x1c4f, 0x0000, 0xe3b2, 0x473c, 0xf89f },   //( 56.200069 - 4185.690721 Hz )
    { 0x1bef, 0x0000, 0xe412, 0x47fa, 0xf7e0 },   //( 58.688259 - 4096.000000 Hz )
    { 0x1b90, 0x0000, 0xe471, 0x48b8, 0xf721 },   //( 61.286610 - 4008.231166 Hz )
    { 0x1b31, 0x0000, 0xe4d0, 0x4975, 0xf663 },   //( 64.000000 - 3922.343038 Hz )
    { 0x1ad2, 0x0000, 0xe52f, 0x4a31, 0xf5a5 },   //( 66.833522 - 3838.295315 Hz )
    { 0x1a73, 0x0000, 0xe58e, 0x4aed, 0xf4e8 },   //( 69.792495 - 3756.048561 Hz )
    { 0x1a15, 0x0000, 0xe5ec, 0x4ba9, 0xf42b },   //( 72.882473 - 3675.564186 Hz )
    { 0x19b7, 0x0000, 0xe64a, 0x4c64, 0xf36e },   //( 76.109255 - 3596.804424 Hz )
    { 0x1959, 0x0000, 0xe6a8, 0x4d1f, 0xf2b2 },   //( 79.478900 - 3519.732323 Hz )
    { 0x18fb, 0x0000, 0xe706, 0x4dd9, 0xf1f7 },   //( 82.997731 - 3444.311717 Hz )
    { 0x189d, 0x0000, 0xe764, 0x4e92, 0xf13b },   //( 86.672355 - 3370.507219 Hz )
    { 0x1840, 0x0000, 0xe7c1, 0x4f4c, 0xf081 },   //( 90.509668 - 3298.284200 Hz )
    { 0x17e3, 0x0000, 0xe81e, 0x5004, 0xefc6 },   //( 94.516873 - 3227.608771 Hz )
    { 0x1786, 0x0000, 0xe87b, 0x50bd, 0xef0c },   //( 98.701493 - 3158.447770 Hz )
    { 0x1729, 0x0000, 0xe8d8, 0x5175, 0xee52 },   //( 103.071381 - 3090.768748 Hz )
    { 0x16cc, 0x0000, 0xe935, 0x522d, 0xed99 },   //( 107.634741 - 3024.539947 Hz )
    { 0x166f, 0x0000, 0xe992, 0x52e4, 0xece0 },   //( 112.400138 - 2959.730293 Hz )
    { 0x1613, 0x0000, 0xe9ee, 0x539b, 0xec27 },   //( 117.376518 - 2896.309376 Hz )
    { 0x15b7, 0x0000, 0xea4a, 0x5452, 0xeb6e },   //( 122.573220 - 2834.247438 Hz )
    { 0x155a, 0x0000, 0xeaa7, 0x5508, 0xeab6 },   //( 128.000000 - 2773.515360 Hz )
    { 0x14fe, 0x0000, 0xeb03, 0x55be, 0xe9fe },   //( 133.667044 - 2714.084645 Hz )
    { 0x14a2, 0x0000, 0xeb5f, 0x5675, 0xe945 },   //( 139.584990 - 2655.927408 Hz )
    { 0x1446, 0x0000, 0xebbb, 0x572b, 0xe88d },   //( 145.764945 - 2599.016360 Hz )
    { 0x13ea, 0x0000, 0xec17, 0x57e1, 0xe7d5 },   //( 152.218511 - 2543.324799 Hz )
    { 0x138e, 0x0000, 0xec73, 0x5897, 0xe71d },   //( 158.957800 - 2488.826593 Hz )
    { 0x1332, 0x0000, 0xeccf, 0x594d, 0xe664 },   //( 165.995463 - 2435.496172 Hz )
    { 0x12d5, 0x0000, 0xed2c, 0x5a03, 0xe5ac },   //( 173.344710 - 2383.308511 Hz )
    { 0x1279, 0x0000, 0xed88, 0x5ab9, 0xe4f3 },   //( 181.019336 - 2332.239124 Hz )
    { 0x121c, 0x0000, 0xede5, 0x5b70, 0xe43a },   //( 189.033747 - 2282.264049 Hz )
    { 0x11c0, 0x0000, 0xee41, 0x5c27, 0xe380 },   //( 197.402986 - 2233.359836 Hz )
    { 0x1163, 0x0000, 0xee9e, 0x5cde, 0xe2c6 },   //( 206.142762 - 2185.503541 Hz )
    { 0x1105, 0x0000, 0xeefc, 0x5d96, 0xe20c },   //( 215.269482 - 2138.672706 Hz )
    { 0x10a8, 0x0000, 0xef59, 0x5e4f, 0xe150 },   //( 224.800277 - 2092.845360 Hz )
    { 0x1049, 0x0000, 0xefb8, 0x5f08, 0xe094 },   //( 234.753035 - 2048.000000 Hz )
    { 0x0feb, 0x0000, 0xf016, 0x5fc2, 0xdfd7 },   //( 245.146440 - 2004.115583 Hz )
    { 0x0f8c, 0x0000, 0xf075, 0x607e, 0xdf18 },   //( 256.000000 - 1961.171519 Hz )
    { 0x0f2c, 0x0000, 0xf0d5, 0x613a, 0xde59 },   //( 267.334088 - 1919.147657 Hz )
    { 0x0ecb, 0x0000, 0xf136, 0x61f8, 0xdd98 },   //( 279.169980 - 1878.024280 Hz )
    { 0x0e6a, 0x0000, 0xf197, 0x62b7, 0xdcd5 },   //( 291.529890 - 1837.782093 Hz )
    { 0x0e08, 0x0000, 0xf1f9, 0x6377, 0xdc11 },   //( 304.437021 - 1798.402212 Hz )
    { 0x0da5, 0x0000, 0xf25c, 0x643a, 0xdb4b },   //( 317.915600 - 1759.866161 Hz )
    { 0x0d41, 0x0000, 0xf2c0, 0x64fe, 0xda83 },   //( 331.990926 - 1722.155858 Hz )
    { 0x0cdc, 0x0000, 0xf325, 0x65c5, 0xd9b9 },   //( 346.689420 - 1685.253610 Hz )
    { 0x0c76, 0x0000, 0xf38b, 0x668e, 0xd8ec },   //( 362.038672 - 1649.142100 Hz )
    { 0x0c0e, 0x0000, 0xf3f3, 0x6759, 0xd81d },   //( 378.067493 - 1613.804385 Hz )
    { 0x0ba5, 0x0000, 0xf45c, 0x6828, 0xd74a },   //( 394.805971 - 1579.223885 Hz )
    { 0x0b3a, 0x0000, 0xf4c7, 0x68f9, 0xd674 },   //( 412.285525 - 1545.384374 Hz )
    { 0x0acd, 0x0000, 0xf534, 0x69ce, 0xd59b },   //( 430.538965 - 1512.269973 Hz )
    { 0x0a5e, 0x0000, 0xf5a3, 0x6aa7, 0xd4be },   //( 449.600553 - 1479.865146 Hz )
    { 0x09ee, 0x0000, 0xf613, 0x6b83, 0xd3dc },   //( 469.506070 - 1448.154688 Hz )
    { 0x097b, 0x0000, 0xf686, 0x6c64, 0xd2f7 },   //( 490.292880 - 1417.123719 Hz )
    { 0x0905, 0x0000, 0xf6fc, 0x6d4a, 0xd20c },   //( 512.000000 - 1386.757680 Hz )
    { 0x088d, 0x0000, 0xf774, 0x6e35, 0xd11c },   //( 534.668177 - 1357.042323 Hz )
    { 0x0812, 0x0000, 0xf7ef, 0x6f25, 0xd026 },   //( 558.339959 - 1327.963704 Hz )
    { 0x0794, 0x0000, 0xf86d, 0x701c, 0xcf2a },   //( 583.059781 - 1299.508180 Hz )
    { 0x0713, 0x0000, 0xf8ee, 0x7119, 0xce27 },   //( 608.874043 - 1271.662400 Hz )
    { 0x068e, 0x0000, 0xf973, 0x721d, 0xcd1d },   //( 635.831200 - 1244.413297 Hz )
    { 0x0605, 0x0000, 0xf9fc, 0x7329, 0xcc0b },   //( 663.981852 - 1217.748086 Hz )
    { 0x0577, 0x0000, 0xfa8a, 0x743d, 0xcaf0 },   //( 693.378840 - 1191.654255 Hz )
    { 0x04e6, 0x0000, 0xfb1b, 0x755a, 0xc9cc },   //( 724.077344 - 1166.119562 Hz )
    { 0x044f, 0x0000, 0xfbb2, 0x7681, 0xc89e },   //( 756.134987 - 1141.132024 Hz )
    { 0x03b2, 0x0000, 0xfc4f, 0x77b3, 0xc765 },   //( 789.611943 - 1116.679918 Hz )
    { 0x0310, 0x0000, 0xfcf1, 0x78f0, 0xc621 },   //( 824.571050 - 1092.751770 Hz )
    { 0x0267, 0x0000, 0xfd9a, 0x7a39, 0xc4d0 },   //( 861.077929 - 1069.336353 Hz )
};


// BPF係数その３
// （中心周波数 2048Hz、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterBpf2048Nw4fCompatible48k::CoefficientsTable48000[ BiquadFilterBpf2048Nw4fCompatible48k::CoefficientsTableSize ] =
{
    //    n0      n1      n2      d1      d2
    { 0x34cb, 0x0000, 0xcb36, 0x1642, 0x2997 },   //( 32.000000 - 15689.372151 Hz )
    { 0x3455, 0x0000, 0xcbac, 0x172d, 0x28aa },   //( 33.416761 - 15353.181259 Hz )
    { 0x33e0, 0x0000, 0xcc21, 0x1816, 0x27c0 },   //( 34.896247 - 15024.194244 Hz )
    { 0x336c, 0x0000, 0xcc95, 0x18fc, 0x26d8 },   //( 36.441236 - 14702.256742 Hz )
    { 0x32f9, 0x0000, 0xcd08, 0x19e0, 0x25f2 },   //( 38.054628 - 14387.217698 Hz )
    { 0x3287, 0x0000, 0xcd7a, 0x1ac2, 0x250f },   //( 39.739450 - 14078.929290 Hz )
    { 0x3216, 0x0000, 0xcdeb, 0x1ba2, 0x242d },   //( 41.498866 - 13777.246868 Hz )
    { 0x31a6, 0x0000, 0xce5b, 0x1c81, 0x234d },   //( 43.336178 - 13482.028877 Hz )
    { 0x3137, 0x0000, 0xceca, 0x1d5d, 0x226e },   //( 45.254834 - 13193.136799 Hz )
    { 0x30c9, 0x0000, 0xcf38, 0x1e39, 0x2191 },   //( 47.258437 - 12910.435083 Hz )
    { 0x305b, 0x0000, 0xcfa6, 0x1f13, 0x20b5 },   //( 49.350746 - 12633.791082 Hz )
    { 0x2fed, 0x0000, 0xd014, 0x1feb, 0x1fdb },   //( 51.535691 - 12363.074991 Hz )
    { 0x2f81, 0x0000, 0xd080, 0x20c3, 0x1f01 },   //( 53.817371 - 12098.159788 Hz )
    { 0x2f14, 0x0000, 0xd0ed, 0x2199, 0x1e29 },   //( 56.200069 - 11838.921171 Hz )
    { 0x2ea9, 0x0000, 0xd158, 0x226e, 0x1d51 },   //( 58.688259 - 11585.237503 Hz )
    { 0x2e3d, 0x0000, 0xd1c4, 0x2343, 0x1c7b },   //( 61.286610 - 11336.989753 Hz )
    { 0x2dd2, 0x0000, 0xd22f, 0x2417, 0x1ba4 },   //( 64.000000 - 11094.061441 Hz )
    { 0x2d67, 0x0000, 0xd29a, 0x24ea, 0x1acf },   //( 66.833522 - 10856.338581 Hz )
    { 0x2cfd, 0x0000, 0xd304, 0x25bc, 0x19fa },   //( 69.792495 - 10623.709632 Hz )
    { 0x2c93, 0x0000, 0xd36e, 0x268e, 0x1926 },   //( 72.882473 - 10396.065441 Hz )
    { 0x2c29, 0x0000, 0xd3d8, 0x275f, 0x1852 },   //( 76.109255 - 10173.299197 Hz )
    { 0x2bbf, 0x0000, 0xd442, 0x2830, 0x177e },   //( 79.478900 - 9955.306373 Hz )
    { 0x2b55, 0x0000, 0xd4ac, 0x2900, 0x16ab },   //( 82.997731 - 9741.984686 Hz )
    { 0x2aec, 0x0000, 0xd515, 0x29d0, 0x15d8 },   //( 86.672355 - 9533.234043 Hz )
    { 0x2a83, 0x0000, 0xd57e, 0x2aa0, 0x1505 },   //( 90.509668 - 9328.956496 Hz )
    { 0x2a19, 0x0000, 0xd5e8, 0x2b6f, 0x1433 },   //( 94.516873 - 9129.056195 Hz )
    { 0x29b0, 0x0000, 0xd651, 0x2c3f, 0x1360 },   //( 98.701493 - 8933.439346 Hz )
    { 0x2947, 0x0000, 0xd6ba, 0x2d0e, 0x128e },   //( 103.071381 - 8742.014162 Hz )
    { 0x28de, 0x0000, 0xd723, 0x2ddd, 0x11bb },   //( 107.634741 - 8554.690826 Hz )
    { 0x2874, 0x0000, 0xd78d, 0x2eab, 0x10e9 },   //( 112.400138 - 8371.381442 Hz )
    { 0x280b, 0x0000, 0xd7f6, 0x2f7a, 0x1016 },   //( 117.376518 - 8192.000000 Hz )
    { 0x27a2, 0x0000, 0xd85f, 0x3049, 0x0f44 },   //( 122.573220 - 8016.462333 Hz )
    { 0x2738, 0x0000, 0xd8c9, 0x3118, 0x0e71 },   //( 128.000000 - 7844.686075 Hz )
    { 0x26cf, 0x0000, 0xd932, 0x31e7, 0x0d9e },   //( 133.667044 - 7676.590629 Hz )
    { 0x2665, 0x0000, 0xd99c, 0x32b6, 0x0ccb },   //( 139.584990 - 7512.097122 Hz )
    { 0x25fc, 0x0000, 0xda05, 0x3385, 0x0bf7 },   //( 145.764945 - 7351.128371 Hz )
    { 0x2592, 0x0000, 0xda6f, 0x3454, 0x0b23 },   //( 152.218511 - 7193.608849 Hz )
    { 0x2528, 0x0000, 0xdad9, 0x3524, 0x0a4f },   //( 158.957800 - 7039.464645 Hz )
    { 0x24bd, 0x0000, 0xdb44, 0x35f3, 0x097b },   //( 165.995463 - 6888.623434 Hz )
    { 0x2453, 0x0000, 0xdbae, 0x36c3, 0x08a6 },   //( 173.344710 - 6741.014439 Hz )
    { 0x23e8, 0x0000, 0xdc19, 0x3794, 0x07d0 },   //( 181.019336 - 6596.568400 Hz )
    { 0x237d, 0x0000, 0xdc84, 0x3865, 0x06fa },   //( 189.033747 - 6455.217542 Hz )
    { 0x2312, 0x0000, 0xdcef, 0x3936, 0x0623 },   //( 197.402986 - 6316.895541 Hz )
    { 0x22a6, 0x0000, 0xdd5b, 0x3a08, 0x054c },   //( 206.142762 - 6181.537495 Hz )
    { 0x223a, 0x0000, 0xddc7, 0x3ada, 0x0473 },   //( 215.269482 - 6049.079894 Hz )
    { 0x21cd, 0x0000, 0xde34, 0x3bad, 0x039a },   //( 224.800277 - 5919.460585 Hz )
    { 0x2160, 0x0000, 0xdea1, 0x3c81, 0x02c1 },   //( 234.753035 - 5792.618751 Hz )
    { 0x20f3, 0x0000, 0xdf0e, 0x3d55, 0x01e6 },   //( 245.146440 - 5668.494877 Hz )
    { 0x2085, 0x0000, 0xdf7c, 0x3e2a, 0x010a },   //( 256.000000 - 5547.030720 Hz )
    { 0x2017, 0x0000, 0xdfea, 0x3f00, 0x002d },   //( 267.334088 - 5428.169290 Hz )
    { 0x1fa7, 0x0000, 0xe05a, 0x3fd7, 0xff50 },   //( 279.169980 - 5311.854816 Hz )
    { 0x1f38, 0x0000, 0xe0c9, 0x40af, 0xfe71 },   //( 291.529890 - 5198.032721 Hz )
    { 0x1ec7, 0x0000, 0xe13a, 0x4188, 0xfd90 },   //( 304.437021 - 5086.649598 Hz )
    { 0x1e56, 0x0000, 0xe1ab, 0x4263, 0xfcae },   //( 317.915600 - 4977.653186 Hz )
    { 0x1de4, 0x0000, 0xe21d, 0x433e, 0xfbca },   //( 331.990926 - 4870.992343 Hz )
    { 0x1d72, 0x0000, 0xe28f, 0x441c, 0xfae4 },   //( 346.689420 - 4766.617022 Hz )
    { 0x1cfe, 0x0000, 0xe303, 0x44fa, 0xf9fd },   //( 362.038672 - 4664.478248 Hz )
    { 0x1c89, 0x0000, 0xe378, 0x45db, 0xf913 },   //( 378.067493 - 4564.528098 Hz )
    { 0x1c13, 0x0000, 0xe3ee, 0x46bd, 0xf828 },   //( 394.805971 - 4466.719673 Hz )
    { 0x1b9d, 0x0000, 0xe464, 0x47a1, 0xf73a },   //( 412.285525 - 4371.007081 Hz )
    { 0x1b24, 0x0000, 0xe4dd, 0x4888, 0xf64a },   //( 430.538965 - 4277.345413 Hz )
    { 0x1aab, 0x0000, 0xe556, 0x4971, 0xf557 },   //( 449.600553 - 4185.690721 Hz )
    { 0x1a30, 0x0000, 0xe5d1, 0x4a5c, 0xf461 },   //( 469.506070 - 4096.000000 Hz )
    { 0x19b3, 0x0000, 0xe64e, 0x4b4a, 0xf368 },   //( 490.292880 - 4008.231166 Hz )
    { 0x1935, 0x0000, 0xe6cc, 0x4c3b, 0xf26b },   //( 512.000000 - 3922.343038 Hz )
    { 0x18b5, 0x0000, 0xe74c, 0x4d2f, 0xf16b },   //( 534.668177 - 3838.295315 Hz )
    { 0x1833, 0x0000, 0xe7ce, 0x4e27, 0xf068 },   //( 558.339959 - 3756.048561 Hz )
    { 0x17af, 0x0000, 0xe852, 0x4f22, 0xef5f },   //( 583.059781 - 3675.564186 Hz )
    { 0x1729, 0x0000, 0xe8d8, 0x5022, 0xee53 },   //( 608.874043 - 3596.804424 Hz )
    { 0x16a0, 0x0000, 0xe961, 0x5126, 0xed41 },   //( 635.831200 - 3519.732323 Hz )
    { 0x1615, 0x0000, 0xe9ec, 0x522e, 0xec2b },   //( 663.981852 - 3444.311717 Hz )
    { 0x1587, 0x0000, 0xea7a, 0x533c, 0xeb0e },   //( 693.378840 - 3370.507219 Hz )
    { 0x14f5, 0x0000, 0xeb0c, 0x5450, 0xe9ec },   //( 724.077344 - 3298.284200 Hz )
    { 0x1461, 0x0000, 0xeba0, 0x5569, 0xe8c3 },   //( 756.134987 - 3227.608771 Hz )
    { 0x13c9, 0x0000, 0xec38, 0x5689, 0xe792 },   //( 789.611943 - 3158.447770 Hz )
    { 0x132d, 0x0000, 0xecd4, 0x57b0, 0xe65a },   //( 824.571050 - 3090.768748 Hz )
    { 0x128c, 0x0000, 0xed75, 0x58de, 0xe51a },   //( 861.077929 - 3024.539947 Hz )
    { 0x11e8, 0x0000, 0xee19, 0x5a15, 0xe3d0 },   //( 899.201106 - 2959.730293 Hz )
    { 0x113e, 0x0000, 0xeec3, 0x5b55, 0xe27d },   //( 939.012140 - 2896.309376 Hz )
    { 0x108f, 0x0000, 0xef72, 0x5c9f, 0xe11f },   //( 980.585759 - 2834.247438 Hz )
    { 0x0fda, 0x0000, 0xf027, 0x5df4, 0xdfb5 },   //( 1024.000000 - 2773.515360 Hz )
    { 0x0f1f, 0x0000, 0xf0e2, 0x5f54, 0xde3f },   //( 1069.336353 - 2714.084645 Hz )
    { 0x0e5d, 0x0000, 0xf1a4, 0x60c0, 0xdcbb },   //( 1116.679918 - 2655.927408 Hz )
    { 0x0d94, 0x0000, 0xf26d, 0x623b, 0xdb29 },   //( 1166.119562 - 2599.016360 Hz )
    { 0x0cc2, 0x0000, 0xf33f, 0x63c5, 0xd986 },   //( 1217.748086 - 2543.324799 Hz )
    { 0x0be8, 0x0000, 0xf419, 0x655f, 0xd7d1 },   //( 1271.662400 - 2488.826593 Hz )
    { 0x0b04, 0x0000, 0xf4fd, 0x670b, 0xd608 },   //( 1327.963704 - 2435.496172 Hz )
    { 0x0a15, 0x0000, 0xf5ec, 0x68cb, 0xd42b },   //( 1386.757680 - 2383.308511 Hz )
    { 0x091b, 0x0000, 0xf6e6, 0x6aa1, 0xd236 },   //( 1448.154688 - 2332.239124 Hz )
    { 0x0813, 0x0000, 0xf7ee, 0x6c8e, 0xd028 },   //( 1512.269973 - 2282.264049 Hz )
    { 0x06fe, 0x0000, 0xf903, 0x6e96, 0xcdfd },   //( 1579.223885 - 2233.359836 Hz )
    { 0x05d9, 0x0000, 0xfa28, 0x70bb, 0xcbb2 },   //( 1649.142100 - 2185.503541 Hz )
    { 0x04a2, 0x0000, 0xfb5f, 0x7301, 0xc946 },   //( 1722.155858 - 2138.672706 Hz )
};

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_VC)
#pragma warning( pop )
#elif defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
#pragma clang diagnostic pop
#endif

/*--------------------------------------------------------------------------------*
  Name:         GetCoefficients

  Description:

  Arguments:    type -
                value -
                coef -

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void BiquadFilterLpf::GetCoefficients(
    BiquadFilterCallback::Coefficients* pOutValue,
    int type,
    float value
) const NN_NOEXCEPT
{
    (void)type;
    int coefSize = CoefficientsTableSize;
    int coefIndex = static_cast<int>( static_cast<float>( coefSize - 1 ) * value );

    coefIndex = nn::atk::detail::fnd::Clamp( coefIndex, 0, CoefficientsTableSize - 1 );
    *pOutValue = CoefficientsTable32000[ coefIndex ];
}

/*--------------------------------------------------------------------------------*
  Name:         GetCoefficients

  Description:

  Arguments:    type -
                value -
                coef -

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void BiquadFilterHpf::GetCoefficients(
    BiquadFilterCallback::Coefficients* pOutValue,
    int type,
    float value
) const NN_NOEXCEPT
{
    (void)type;
    int coefSize = CoefficientsTableSize;
    int coefIndex = static_cast<int>( static_cast<float>( coefSize - 1 ) * value );

    coefIndex = nn::atk::detail::fnd::Clamp( coefIndex, 0, CoefficientsTableSize - 1 );
    *pOutValue = CoefficientsTable32000[ coefIndex ];
}

/*--------------------------------------------------------------------------------*
  Name:         GetCoefficients

  Description:

  Arguments:    type -
                value -
                coef -

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void BiquadFilterBpf512::GetCoefficients(
    BiquadFilterCallback::Coefficients* pOutValue,
    int type,
    float value
) const NN_NOEXCEPT
{
    (void)type;

    value *= ( 2.0f - value );  // 係数の取り方に重み付け

    int coefSize = CoefficientsTableSize;
    int coefIndex = static_cast<int>( static_cast<float>( coefSize - 1 ) * value );

    coefIndex = nn::atk::detail::fnd::Clamp( coefIndex, 0, CoefficientsTableSize - 1 );
    *pOutValue = CoefficientsTable32000[ coefIndex ];
}

/*--------------------------------------------------------------------------------*
  Name:         GetCoefficients

  Description:

  Arguments:    type -
                value -
                coef -

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void BiquadFilterBpf1024::GetCoefficients(
    BiquadFilterCallback::Coefficients* pOutValue,
    int type,
    float value
) const NN_NOEXCEPT
{
    (void)type;

    value *= ( 2.0f - value );   // 係数の取り方に重み付け

    int coefSize = CoefficientsTableSize;
    int coefIndex = static_cast<int>( static_cast<float>( coefSize - 1 ) * value );

    coefIndex = nn::atk::detail::fnd::Clamp( coefIndex, 0, CoefficientsTableSize - 1 );
    *pOutValue = CoefficientsTable32000[ coefIndex ];
}

/*--------------------------------------------------------------------------------*
  Name:         GetCoefficients

  Description:

  Arguments:    type -
                value -
                coef -

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void BiquadFilterBpf2048::GetCoefficients(
    BiquadFilterCallback::Coefficients* pOutValue,
    int type,
    float value
) const NN_NOEXCEPT
{
    (void)type;

    value *= ( 2.0f - value );   // 係数の取り方に重み付け

    int coefSize = CoefficientsTableSize;
    int coefIndex = static_cast<int>( static_cast<float>( coefSize - 1 ) * value );

    coefIndex = nn::atk::detail::fnd::Clamp( coefIndex, 0, CoefficientsTableSize - 1 );
    *pOutValue = CoefficientsTable32000[ coefIndex ];
}

void BiquadFilterLpfNw4fCompatible48k::GetCoefficients(
    BiquadFilterCallback::Coefficients* pOutValue,
    int type,
    float value
) const NN_NOEXCEPT
{
    (void)type;
    int coefSize = CoefficientsTableSize;
    int coefIndex = static_cast<int>( static_cast<float>( coefSize - 1 ) * value );

    coefIndex = nn::atk::detail::fnd::Clamp( coefIndex, 0, CoefficientsTableSize - 1 );
    *pOutValue = CoefficientsTable48000[ coefIndex ];
}

void BiquadFilterHpfNw4fCompatible48k::GetCoefficients(
    BiquadFilterCallback::Coefficients* pOutValue,
    int type,
    float value
) const NN_NOEXCEPT
{
    (void)type;
    int coefSize = CoefficientsTableSize;
    int coefIndex = static_cast<int>( static_cast<float>( coefSize - 1 ) * value );

    coefIndex = nn::atk::detail::fnd::Clamp( coefIndex, 0, CoefficientsTableSize - 1 );
    *pOutValue = CoefficientsTable48000[ coefIndex ];
}

void BiquadFilterBpf512Nw4fCompatible48k::GetCoefficients(
    BiquadFilterCallback::Coefficients* pOutValue,
    int type,
    float value
) const NN_NOEXCEPT
{
    (void)type;

    value *= ( 2.0f - value );  // 係数の取り方に重み付け

    int coefSize = CoefficientsTableSize;
    int coefIndex = static_cast<int>( static_cast<float>( coefSize - 1 ) * value );

    coefIndex = nn::atk::detail::fnd::Clamp( coefIndex, 0, CoefficientsTableSize - 1 );
    *pOutValue = CoefficientsTable48000[ coefIndex ];
}

void BiquadFilterBpf1024Nw4fCompatible48k::GetCoefficients(
    BiquadFilterCallback::Coefficients* pOutValue,
    int type,
    float value
) const NN_NOEXCEPT
{
    (void)type;

    value *= ( 2.0f - value );   // 係数の取り方に重み付け

    int coefSize = CoefficientsTableSize;
    int coefIndex = static_cast<int>( static_cast<float>( coefSize - 1 ) * value );

    coefIndex = nn::atk::detail::fnd::Clamp( coefIndex, 0, CoefficientsTableSize - 1 );
    *pOutValue = CoefficientsTable48000[ coefIndex ];
}

void BiquadFilterBpf2048Nw4fCompatible48k::GetCoefficients(
    BiquadFilterCallback::Coefficients* pOutValue,
    int type,
    float value
) const NN_NOEXCEPT
{
    (void)type;

    value *= ( 2.0f - value );   // 係数の取り方に重み付け

    int coefSize = CoefficientsTableSize;
    int coefIndex = static_cast<int>( static_cast<float>( coefSize - 1 ) * value );

    coefIndex = nn::atk::detail::fnd::Clamp( coefIndex, 0, CoefficientsTableSize - 1 );
    *pOutValue = CoefficientsTable48000[ coefIndex ];
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

