﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/detail/atk_SoundArchiveManager.h>
#include <nn/atk/detail/atk_Macro.h>

namespace nn { namespace atk { namespace detail {

    SoundArchiveManager::SoundArchiveManager() NN_NOEXCEPT
    : m_pMainSoundArchive(nullptr)
    , m_pMainSoundDataManager(nullptr)
    , m_ContainerList()
    , m_pCurrentSoundArchive(nullptr)
    , m_pCurrentSoundDataManager(nullptr)
    , m_pParametersHook(nullptr)
    {
    }

    SoundArchiveManager::~SoundArchiveManager() NN_NOEXCEPT
    {
    }

    void SoundArchiveManager::Initialize(const SoundArchive* pSoundArchive, const SoundDataManager* pSoundDataManager) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( pSoundArchive );
        m_pMainSoundArchive = pSoundArchive;
        m_pMainSoundDataManager = pSoundDataManager;

        m_ContainerList.Clear();

        ChangeTargetArchive(nullptr);
    }

    void SoundArchiveManager::Finalize() NN_NOEXCEPT
    {
        m_pCurrentSoundDataManager = nullptr;
        m_pCurrentSoundArchive = nullptr;

        m_ContainerList.Clear();

        m_pMainSoundDataManager = nullptr;
        m_pMainSoundArchive = nullptr;

        m_pParametersHook = nullptr;
    }

    void SoundArchiveManager::Add(AddonSoundArchiveContainer& container) NN_NOEXCEPT
    {
        m_ContainerList.PushBack(container);
    }

    void SoundArchiveManager::Remove(AddonSoundArchiveContainer& container) NN_NOEXCEPT
    {
        m_ContainerList.Remove(container);
    }

    bool SoundArchiveManager::IsAvailable() const NN_NOEXCEPT
    {
        if (m_pMainSoundArchive == nullptr)
        {
            return false;
        }

        bool isAvailable = m_pMainSoundArchive->IsAvailable();

        bool isSoundArchiveListAvailable = true;
        for ( ContainerList::ConstIterator iterator = m_ContainerList.Begin();
              iterator != m_ContainerList.End();
              ++iterator )
        {
            if (!iterator->GetSoundArchive()->IsAvailable())
            {
                isSoundArchiveListAvailable = false;
            }
        }

        return isAvailable && isSoundArchiveListAvailable;
    }

    void SoundArchiveManager::ChangeTargetArchive(const char* soundArchiveName) NN_NOEXCEPT
    {
        if (soundArchiveName == nullptr)
        {
            m_pCurrentSoundArchive = m_pMainSoundArchive;
            m_pCurrentSoundDataManager = m_pMainSoundDataManager;
        }
        else
        {
            m_pCurrentSoundArchive = m_pMainSoundArchive; // 見つからなかった場合はメインとするための初期化
            m_pCurrentSoundDataManager = m_pMainSoundDataManager;

            for ( ContainerList::ConstIterator iterator = m_ContainerList.Begin();
                  iterator != m_ContainerList.End();
                  ++iterator )
            {
                if ( iterator->IsSameName(soundArchiveName) )
                {
                    m_pCurrentSoundArchive = iterator->GetSoundArchive();
                    m_pCurrentSoundDataManager = iterator->GetSoundDataManager();

                    break;
                }
            }

            if ( m_pCurrentSoundArchive == m_pMainSoundArchive )
            {
                NN_ATK_WARNING("use main archive because specified name is not found");
            }
        }
    }

    const nn::atk::SoundArchive* SoundArchiveManager::GetAddonSoundArchive(const char* soundArchiveName) const NN_NOEXCEPT
    {
        if (soundArchiveName == nullptr)
        {
            NN_ATK_WARNING("nullptr is specified as soundArchiveName");
            return nullptr;
        }

        for ( auto iterator = m_ContainerList.Begin(); iterator != m_ContainerList.End(); ++iterator )
        {
            if ( iterator->IsSameName(soundArchiveName) )
            {
                return iterator->GetSoundArchive();
            }
        }

        return nullptr;
    }

    const nn::atk::SoundDataManager* SoundArchiveManager::GetAddonSoundDataManager(const char* soundArchiveName) const NN_NOEXCEPT
    {
        if (soundArchiveName == nullptr)
        {
            NN_ATK_WARNING("nullptr is specified as soundArchiveName");
            return nullptr;
        }

        for ( auto iterator = m_ContainerList.Begin(); iterator != m_ContainerList.End(); ++iterator )
        {
            if ( iterator->IsSameName(soundArchiveName) )
            {
                return iterator->GetSoundDataManager();
            }
        }

        return nullptr;
    }

    const detail::AddonSoundArchiveContainer* SoundArchiveManager::GetAddonSoundArchiveContainer(int index) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(index, 0, GetAddonSoundArchiveCount());
        auto iterator = m_ContainerList.Begin();
        for ( int i = 0; i < index; ++i )
        {
            ++iterator;
        }
        return &(*iterator);
    }

    detail::AddonSoundArchiveContainer* SoundArchiveManager::GetAddonSoundArchiveContainer(int index) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(index, 0, GetAddonSoundArchiveCount());
        auto iterator = m_ContainerList.Begin();
        for ( int i = 0; i < index; ++i )
        {
            ++iterator;
        }
        return &(*iterator);
    }

    void SoundArchiveManager::SetParametersHook(detail::SoundArchiveParametersHook* parametersHook) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pMainSoundArchive);
        SoundArchive* mainArchive = const_cast<SoundArchive*>(m_pMainSoundArchive);
        mainArchive->detail_SetParametersHook(parametersHook);

        for ( auto iterator = m_ContainerList.Begin(); iterator != m_ContainerList.End(); ++iterator )
        {
            SoundArchive* archive = const_cast<SoundArchive*>(iterator->GetSoundArchive());
            NN_SDK_ASSERT_NOT_NULL(archive);
            archive->detail_SetParametersHook(parametersHook);
        }

        m_pParametersHook = parametersHook;
    }

    detail::SoundArchiveParametersHook* SoundArchiveManager::GetParametersHook() const NN_NOEXCEPT
    {
        return m_pParametersHook;
    }

}}}

