﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/detail/atk_OpusDecoder.h>
#include <nn/util/util_BytePtr.h>
#include <nn/atk/detail/atk_Macro.h>

//#define NN_ATK_ENABLE_DEBUG_OPUS_INFO_DUMP

//#define NN_ATK_ENABLE_DEBUG_TRACE_FUNC_OPUS_DECODER
#if defined(NN_ATK_ENABLE_DEBUG_TRACE_FUNC_OPUS_DECODER)
#define NN_ATK_DEBUG_TRACE_FUNC_MSG(AdditionalMessage) \
            NN_DETAIL_ATK_INFO("[%p](%s) %s%s\n", this, NN_ATK_FILENAME, __FUNCTION__, AdditionalMessage)
#define NN_ATK_DEBUG_TRACE_FUNC() \
            NN_ATK_DEBUG_TRACE_FUNC_MSG("")
#else
#define NN_ATK_DEBUG_TRACE_FUNC_MSG(AdditionalMessage) static_cast<void>(0)
#define NN_ATK_DEBUG_TRACE_FUNC() static_cast<void>(0)
#endif // NN_ATK_ENABLE_DEBUG_TRACE_FUNC_OPUS_DECODER

namespace nn { namespace atk {

    detail::driver::OpusDecoderManager g_OpusDecoderManager;

    void InitializeOpusDecoder( void* buffer, size_t bufferSize ) NN_NOEXCEPT
    {
        InitializeOpusDecoder(buffer, bufferSize, detail::driver::OpusDecoderManager::DefaultOpusDecoderCount);
    }

    void InitializeOpusDecoder( void* buffer, size_t bufferSize, int decoderCount ) NN_NOEXCEPT
    {
        if (!g_OpusDecoderManager.IsInitialized())
        {
            g_OpusDecoderManager.Initialize(buffer, bufferSize, decoderCount);
            nn::atk::detail::driver::StreamSoundLoader::RegisterStreamDataDecoderManager( &g_OpusDecoderManager );
        }
        else
        {
            NN_ATK_WARNING("OpusDecoder is already initialized.");
        }
    }

    void FinalizeOpusDecoder() NN_NOEXCEPT
    {
        if (g_OpusDecoderManager.IsInitialized())
        {
            nn::atk::detail::driver::StreamSoundLoader::UnregisterStreamDataDecoderManager( &g_OpusDecoderManager );
            g_OpusDecoderManager.Finalize();
        }
    }

    size_t GetRequiredOpusDecoderBufferSize() NN_NOEXCEPT
    {
        return GetRequiredOpusDecoderBufferSize(detail::driver::OpusDecoderManager::DefaultOpusDecoderCount);
    }

    size_t GetRequiredOpusDecoderBufferSize(int decoderCount) NN_NOEXCEPT
    {
        return g_OpusDecoderManager.GetRequiredMemorySize(decoderCount);
    }

}}

namespace nn { namespace atk { namespace detail { namespace driver {

    OpusDecoder::OpusDecoder() NN_NOEXCEPT
        : m_Buffer(nullptr)
    {
    }

    OpusDecoder::~OpusDecoder() NN_NOEXCEPT
    {
        Finalize();
    }

    void OpusDecoder::Initialize( void* buffer, size_t bufferSize ) NN_NOEXCEPT
    {
        NN_ATK_DEBUG_TRACE_FUNC();
        m_Buffer = buffer;
        m_OpusDecoderImpl.Initialize(buffer, bufferSize, 1, &m_Decoder);
    }

    void* OpusDecoder::Finalize() NN_NOEXCEPT
    {
        NN_ATK_DEBUG_TRACE_FUNC();
        m_OpusDecoderImpl.Finalize();

        void* pWorkBufferHead = m_Buffer;
        m_Buffer = nullptr;

        return pWorkBufferHead;
    }

    bool OpusDecoder::ReadDataInfo( DataInfo* pOutValue, detail::fnd::FileStream* pFileStream ) NN_NOEXCEPT
    {
        NN_ATK_DEBUG_TRACE_FUNC();
        return m_OpusDecoderImpl.ReadDataInfo(pOutValue, pFileStream);
    }

    bool OpusDecoder::Skip( detail::fnd::FileStream* pFileStream ) NN_NOEXCEPT
    {
        return m_OpusDecoderImpl.Skip(pFileStream);
    }

    bool OpusDecoder::Decode(int16_t* pOutDecodedBufferAddresses[], detail::fnd::FileStream* pFileStream, int channelCount, DecodeType decodeType ) NN_NOEXCEPT
    {
        NN_ATK_DEBUG_TRACE_FUNC();
        return m_OpusDecoderImpl.Decode(pOutDecodedBufferAddresses, pFileStream, channelCount, decodeType);
    }

    void OpusDecoder::ResetDecodeProfile() NN_NOEXCEPT
    {
        m_OpusDecoderImpl.ResetDecodeProfile();
    }

    IStreamDataDecoder::DecodeProfile OpusDecoder::GetDecodeProfile() const NN_NOEXCEPT
    {
        return m_OpusDecoderImpl.GetDecodeProfile();
    }

}}}}

namespace nn { namespace atk { namespace detail { namespace driver {

    NN_DEFINE_STATIC_CONSTANT(const int OpusDecoderManager::DefaultOpusDecoderCount);

    OpusDecoderManager::OpusDecoderManager() NN_NOEXCEPT
    {
    }

    OpusDecoderManager::~OpusDecoderManager() NN_NOEXCEPT
    {
    }

    void OpusDecoderManager::Initialize(void* buffer, size_t bufferSize, int decoderCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(bufferSize, GetRequiredMemorySize(decoderCount));
        m_DecoderManagerImpl.Initialize(buffer, bufferSize, GetWorkBufferSizePerDecoder(), decoderCount, detail::StreamFileType_Opus, detail::DecodeMode_Cpu );
    }

    void OpusDecoderManager::Finalize() NN_NOEXCEPT
    {
        m_DecoderManagerImpl.Finalize();
    }

    size_t OpusDecoderManager::GetRequiredMemorySize(int decoderCount) NN_NOEXCEPT
    {
        nn::codec::OpusDecoder decoder;

        NN_SDK_REQUIRES_GREATER(decoderCount, 0);
        return (sizeof(OpusDecoder) + GetWorkBufferSizePerDecoder()) * decoderCount;
    }

    bool OpusDecoderManager::IsInitialized() const NN_NOEXCEPT
    {
        return m_DecoderManagerImpl.IsInitialized();
    }

    IStreamDataDecoder* OpusDecoderManager::AllocDecoder() NN_NOEXCEPT
    {
        NN_ATK_DEBUG_TRACE_FUNC();

        return m_DecoderManagerImpl.AllocDecoder();
    }

    void OpusDecoderManager::FreeDecoder(IStreamDataDecoder* pDecoder) NN_NOEXCEPT
    {
        NN_ATK_DEBUG_TRACE_FUNC();

        return m_DecoderManagerImpl.FreeDecoder(pDecoder);
    }

    int OpusDecoderManager::GetStreamFileType() const NN_NOEXCEPT
    {
        return m_DecoderManagerImpl.GetStreamFileType();
    }

    int OpusDecoderManager::GetDecodeMode() const NN_NOEXCEPT
    {
        return m_DecoderManagerImpl.GetDecodeMode();
    }

    size_t OpusDecoderManager::GetWorkBufferSizePerDecoder() NN_NOEXCEPT
    {
        size_t size = 0;
        size += nn::atk::detail::driver::OpusDecoderImpl::InputBufferSize;
        size += nn::atk::detail::driver::OpusDecoderImpl::DecodedBufferSizeMax;
        // HACK： 2016/02/04 現在一番重い組み合わせ
        int sampleRate = 48000;
        int channelCount = 2;

        nn::codec::OpusDecoder decoder;
        size += decoder.GetWorkBufferSize(sampleRate, channelCount);

        return size;
    }

}}}}

