﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_BasicSound.h>
#include <nn/atk/atk_BasicSoundPlayer.h>
#include <nn/atk/atk_SoundPlayer.h>
#include <nn/atk/atk_ExternalSoundPlayer.h>
#include <nn/atk/atk_SoundActor.h>
#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/atk_SoundHandle.h>
#include <nn/atk/detail/atk_Macro.h>
#include <cstring>  // std::memcpy


namespace {

void SetSoundPlayerCalculationValues(nn::atk::SoundParamCalculationValues::SoundPlayerParam& param, const nn::atk::SoundPlayer& player) NN_NOEXCEPT
{
    param.volume   = player.GetVolume();
    param.lpf      = player.GetLowPassFilterFrequency();
    param.bqfType  = player.GetBiquadFilterType();
    param.bqfValue = player.GetBiquadFilterValue();

    for(int i = 0; i < nn::atk::OutputDevice_Count; i++)
    {
        const nn::atk::OutputDevice device = static_cast<nn::atk::OutputDevice>(i);
        param.outputVolume[device]   = player.GetOutputVolume( device );
        param.outputMainSend[device] = player.GetOutputMainSend( device );

        for(int k = 0; k < nn::atk::AuxBus_Count; k++)
        {
            const nn::atk::AuxBus bus = static_cast<nn::atk::AuxBus>(k);
            param.outputEffectSend[device][bus] = player.GetOutputFxSend( device, bus );
        }
    }
}
void SetSound3DCalculationValues(nn::atk::SoundParamCalculationValues::Sound3DParam& param, const nn::atk::SoundParam& ambientParam) NN_NOEXCEPT
{
    param.volume         = ambientParam.GetVolume();
    param.pitch          = ambientParam.GetPitch();
    param.lpf            = ambientParam.GetLpf();
    param.bqfType        = ambientParam.GetBiquadFilterType();
    param.bqfValue       = ambientParam.GetBiquadFilterValue();
    param.outputLineFlag = ambientParam.GetOutputLineFlag();
    param.playerPriority = ambientParam.GetPriority();

    for(int i = 0; i < nn::atk::OutputDevice_Count; i++)
    {
        const nn::atk::OutputAmbientParam* pAmbientParam = nullptr;
        const nn::atk::OutputDevice device = static_cast<nn::atk::OutputDevice>(i);
        switch( device )
        {
            case nn::atk::OutputDevice_Main:
                pAmbientParam = &ambientParam.GetTvParam();
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
        }

        param.outputVolume[device]      = pAmbientParam->GetVolume();
        param.outputPan[device]         = pAmbientParam->GetPan();
        param.outputSurroundPan[device] = pAmbientParam->GetSurroundPan();

        for(int k = 0; k < nn::atk::AuxBus_Count; k++)
        {
            const nn::atk::AuxBus bus = static_cast<nn::atk::AuxBus>(k);
            param.outputEffectSend[device][bus] = pAmbientParam->GetEffectSend( bus );
        }
    }
}
void SetSoundActorCalculationValues(nn::atk::SoundParamCalculationValues::SoundActorParam& param, const nn::atk::detail::SoundActorParam& actorParam) NN_NOEXCEPT
{
    param.volume = actorParam.volume;
    param.pitch  = actorParam.pitch;
    param.lpf    = actorParam.lpf;

    for(int i = 0; i < nn::atk::OutputDevice_Count; i++)
    {
        const nn::atk::OutputDevice device = static_cast<nn::atk::OutputDevice>(i);
        switch( device )
        {
            case nn::atk::OutputDevice_Main:
                param.outputVolume[device] = actorParam.tvVolume;
                param.outputPan[device]    = actorParam.tvPan;
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }
}

}

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT( const int SoundAmbientParam::OutputLineFlagInherit );

namespace detail {

uint32_t BasicSound::g_LastInstanceId = 0;

/*--------------------------------------------------------------------------------*
  Name:         BasicSound

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
BasicSound::BasicSound() NN_NOEXCEPT
: m_State(State_Constructed)
, m_pUserParam(NULL)
, m_UserParamSize(0)
, m_SoundStopCallback(NULL)
{
    for ( int i = 0; i < OutputDevice_Count; i++ )
    {
        m_pOutputAdditionalParam[i] = nullptr;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         Initialize

  Description:  初期化

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
bool BasicSound::Initialize(OutputReceiver* pOutputReceiver) NN_NOEXCEPT
{
    // パラメータの初期化
    {
        m_InstanceId = g_LastInstanceId++;

        m_pPlayerHeap = NULL;
        m_pGeneralHandle = NULL;
        m_pTempGeneralHandle = NULL;
        m_pSoundPlayer = NULL;
        m_pSoundActor = NULL;
        m_pExtSoundPlayer = NULL;
        m_pSoundArchive = NULL;
        m_Id = nn::atk::InvalidSoundId;

        m_AmbientInfo.paramUpdateCallback = NULL;
        m_AmbientInfo.argUpdateCallback = NULL;
        m_AmbientInfo.argAllocatorCallback = NULL;
        m_AmbientInfo.arg = NULL;
        m_AmbientInfo.argSize = 0;

        m_PlayerState         = PlayerState_Init;
        m_PauseState          = PauseState_Normal;
        m_MuteState           = MuteState_Normal;
        m_UnPauseFlag         = false;
        m_PauseMode           = PauseMode_Default;
        m_StartFlag           = false;
        m_StartedFlag         = false;
        m_AutoStopFlag        = false;
        m_FadeOutFlag         = false;
        m_PlayerAvailableFlag = false;

        m_AutoStopCounter     = 0;
        m_UpdateCounter       = 0;
        m_PlayingCounter      = 0;

        m_FadeVolume.InitValue( 1.0f );
        m_PauseFadeVolume.InitValue( 1.0f );
        m_MuteFadeVolume.InitValue( 1.0f );

        m_InitVolume        = 1.0f;
        m_Pitch             = 1.0f;
        m_LpfFreq           = 0.0f;
        m_BiquadFilterType  = BiquadFilterType_Inherit;
        m_BiquadFilterValue = 0.0f;

        m_OutputLineFlag = ( m_pSoundPlayer != NULL ) ?
            m_pSoundPlayer->GetDefaultOutputLine() :
            OutputLine_Main;

        m_CommonParam.Initialize();
        for ( int i = 0; i < OutputDevice_Count; i++ )
        {
            m_OutputParam[i].Initialize();
            if(m_pOutputAdditionalParam[i] != nullptr)
            {
                m_pOutputAdditionalParam[i]->Reset();
            }
        }

        // AmbientParam
        m_AmbientParam.Initialize();
        m_ActorParam.Reset();
        if (m_UserParamSize > 0)
        {
            std::memset(m_pUserParam, 0x0, m_UserParamSize);
        }
    }

    // プレイヤーの初期化
    {
        driver::BasicSoundPlayer* basicPlayer = GetBasicSoundPlayerHandle();
        DriverCommand& cmdmgr = DriverCommand::GetInstance();

        DriverCommandPlayerInit* command =
            cmdmgr.AllocCommand<DriverCommandPlayerInit>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            NN_ATK_WARNING("failed to alloc command (DriverCommandPlayerInit) because of driver command shortage.");
            return false; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_PlayerInit;
        command->player = basicPlayer;
        command->pOutputReceiver = pOutputReceiver;
        command->availableFlagPtr = &m_PlayerAvailableFlag;
        cmdmgr.PushCommand(command);
        basicPlayer->InitializeEvent();
    }

    // サブミックスのリファレンスカウントを増加させます。
    // 増加は AppThread で行われる必要があるため BasicSound の初期化時に行います。
    // 減少は BasicSoundPlayer の終了処理時に行われます。
    pOutputReceiver->AddReferenceCount( 1 );
    m_pOutputReceiver = pOutputReceiver;

    m_State = State_Initialized;

    return true;
}

void BasicSound::SetPriority( int priority, int ambientPriority ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( priority >= nn::atk::PlayerPriorityMin && priority <= nn::atk::PlayerPriorityMax );

    m_Priority = static_cast<uint8_t>(priority);
    m_AmbientParam.SetPriority( ambientPriority );
}
void BasicSound::GetPriority( int* priority, int* ambientPriority ) const NN_NOEXCEPT
{
    if ( priority != NULL )
    {
        *priority = m_Priority;
    }
    if ( ambientPriority != NULL )
    {
        *ambientPriority = m_AmbientParam.GetPriority();
    }
}

void BasicSound::ClearIsFinalizedForCannotAllocatedResourceFlag() NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandPlayerClearResourceFlag* command =
        cmdmgr.AllocCommand<DriverCommandPlayerClearResourceFlag>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_PlayerClearResourceFlag;
    command->player = GetBasicSoundPlayerHandle();
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         Finalize

  Description:

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::Finalize() NN_NOEXCEPT
{
    if (m_State < State_Initialized)
    {
        return;
    }
    if (m_State >= State_Finalized)
    {
        return;
    }

    // IDを無効化
    SetId( nn::atk::InvalidSoundId );

    // ハンドルとの切断
    if ( IsAttachedGeneralHandle() )
    {
        DetachGeneralHandle();
    }
    if ( IsAttachedTempGeneralHandle() )
    {
        DetachTempGeneralHandle();
    }
    if ( IsAttachedTempSpecialHandle() )
    {
        DetachTempSpecialHandle();
    }

    // プレイヤーとの切断
    if ( m_pSoundPlayer != NULL ) m_pSoundPlayer->detail_RemoveSound( this );
    if ( m_pExtSoundPlayer != NULL ) m_pExtSoundPlayer->RemoveSound( this );

    if ( m_AmbientInfo.argAllocatorCallback != NULL )
    {
        m_AmbientInfo.argAllocatorCallback->detail_FreeAmbientArg( m_AmbientInfo.arg, this );
        m_AmbientInfo.arg = NULL;
    }

    {
        // プレイヤーの停止
        if ( m_StartedFlag )
        {
            DriverCommand& cmdmgr = DriverCommand::GetInstance();

            DriverCommandPlayer* command =
                cmdmgr.AllocCommand<DriverCommandPlayer>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
            if (command == nullptr)
            {
                return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
            }
#else
            NN_SDK_ASSERT_NOT_NULL(command);
#endif
            command->id = DriverCommandId_PlayerStop;
            command->player = GetBasicSoundPlayerHandle();
            command->flag = m_FadeOutFlag;
            cmdmgr.PushCommand(command);

            m_StartedFlag = false;
        }
        m_PlayerAvailableFlag = false;
        m_PlayerState = PlayerState_Stop;

        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandPlayer* command =
            cmdmgr.AllocCommand<DriverCommandPlayer>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_PlayerFinalize;
        command->player = GetBasicSoundPlayerHandle();
        cmdmgr.PushCommand(command);
    }

    m_pOutputReceiver = nullptr;

    if (m_SoundStopCallback != NULL)
    {
        m_SoundStopCallback();
        m_SoundStopCallback = NULL;
    }

    m_FadeOutFlag = false;
    m_State = State_Finalized;
}

/*--------------------------------------------------------------------------------*
  Name:         StartPrepared

  Description:  プリペアが完了していたらサウンドをスタートする

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::StartPrepared() NN_NOEXCEPT
{
    m_StartFlag = true;
}

/*--------------------------------------------------------------------------------*
  Name:         Stop

  Description:  サウンド停止

  Arguments:    fadeFrames - フェードアウトフレーム

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::Stop( int fadeFrames ) NN_NOEXCEPT
{
    if ( ( fadeFrames <= 0 ) || ( m_PauseState == PauseState_Paused ) ||
         ( m_StartFlag == false && m_StartedFlag == false ) )   // StartPrepared 前
    {
        Finalize();
        return;
    }

    int frames = static_cast<int>( fadeFrames * m_FadeVolume.GetValue() );
    m_FadeVolume.SetTarget( 0.0f, frames );
    SetPlayerPriority( nn::atk::PlayerPriorityMin );
    m_AutoStopFlag = false;
    m_PauseState = PauseState_Normal;
    m_UnPauseFlag = false;
    m_PauseMode = PauseMode_Default;
    m_FadeOutFlag = true;
    m_MuteState = MuteState_Normal;
}

void BasicSound::ForceStop() NN_NOEXCEPT
{
    m_FadeOutFlag = true; // フラグを立てると音量を 0 にして停止する
    Finalize();
}

/*--------------------------------------------------------------------------------*
  Name:         Pause

  Description:  サウンドの一時停止または再開

  Arguments:    flag           - 一時停止または再開
                fadeFrames     - フェードフレーム数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::Pause( bool flag, int fadeFrames ) NN_NOEXCEPT
{
    Pause(flag, fadeFrames, nn::atk::PauseMode_Default);
}

/*--------------------------------------------------------------------------------*
  Name:         Pause

  Description:  サウンドの一時停止または再開

  Arguments:    flag           - 一時停止または再開
                fadeFrames     - フェードフレーム数
                pauseMode      - ポーズの設定

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::Pause( bool flag, int fadeFrames, PauseMode pauseMode ) NN_NOEXCEPT
{
    int frames;
    if ( flag )
    {
        // 一時停止処理
        switch( m_PauseState ) {
        case PauseState_Normal:
        case PauseState_Pausing:
        case PauseState_Unpausing:
            frames = static_cast<int>( fadeFrames * m_PauseFadeVolume.GetValue() );
            if ( frames <= 0 ) frames = 1;
            m_PauseFadeVolume.SetTarget( 0.0f, frames);
            m_PauseState = PauseState_Pausing;
            m_UnPauseFlag = false;
            m_PauseMode = pauseMode;
            break;
        case PauseState_Paused:
            // do nothing
            return;
        default:
            NN_SDK_ASSERT( false, "Unexpected pause state %d", m_PauseState );
            return;
        }
    }
    else
    {
        switch( m_PauseState ) {
        case PauseState_Normal:
            // do nothing
            return;
        case PauseState_Pausing:
        case PauseState_Unpausing:
        case PauseState_Paused:
            frames = static_cast<int>( fadeFrames * ( 1.0f - m_PauseFadeVolume.GetValue() ) );
            if ( frames <= 0 ) frames = 1;
            m_PauseFadeVolume.SetTarget( 1.0f, frames);
            m_PauseState = PauseState_Unpausing;
            m_UnPauseFlag = true;
            m_PauseMode = pauseMode;
            break;
        default:
            NN_SDK_ASSERT( false, "Unexpected pause state %d", m_PauseState );
            return;
        }
    }
}

/*--------------------------------------------------------------------------------*
  Name:         Mute

  Description:  サウンドのミュートまたはミュート解除

  Arguments:    flag       - ミュートまたはミュート解除
                fadeFrames - フェードフレーム数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::Mute(bool flag, int fadeFrames) NN_NOEXCEPT
{
    int frames;
    if ( flag )
    {
        // ミュート処理
        switch ( m_MuteState )
        {
        case MuteState_Normal:
        case MuteState_Muting:
        case MuteState_Unmuting:
            frames = static_cast<int>( fadeFrames * m_MuteFadeVolume.GetValue() );
            if ( frames <= 0 ) frames = 1;
            m_MuteFadeVolume.SetTarget( 0.0f, frames );
            m_MuteState = MuteState_Muting;
            break;
        case MuteState_Muted:
            // do nothing
            return;
        default:
            NN_SDK_ASSERT( false, "Unexpected mute state %d", m_MuteState );
            return;
        }
    }
    else
    {
        switch ( m_MuteState )
        {
        case MuteState_Normal:
            // do nothing
            return;
        case MuteState_Muting:
        case MuteState_Unmuting:
        case MuteState_Muted:
            frames = static_cast<int>( fadeFrames * ( 1.0f - m_MuteFadeVolume.GetValue() ) );
            if ( frames <= 0 ) frames = 1;
            m_MuteFadeVolume.SetTarget( 1.0f, frames );
            m_MuteState = MuteState_Unmuting;
            break;
        default:
            NN_SDK_ASSERT( false, "Unexpected mute state %d", m_MuteState );
            return;
        }
    }
}

/*--------------------------------------------------------------------------------*
  Name:         SetAutoStopCounter

  Description:  サウンドが自動停止するまでのフレーム数を設定する

  Arguments:    frames - 自動停止するまでのフーレム数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::SetAutoStopCounter( int frames ) NN_NOEXCEPT
{
    m_AutoStopCounter = frames;
    m_AutoStopFlag = ( frames > 0 );
}

/*--------------------------------------------------------------------------------*
  Name:         FadeIn

  Description:  フェードイン

  Arguments:    frames - 変化フレーム数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::FadeIn( int frames ) NN_NOEXCEPT
{
    if ( m_FadeOutFlag )
    {
        return;
    }
    if ( m_UpdateCounter == 0 )
    {
        m_FadeVolume.InitValue( 0.0f );
        m_FadeVolume.SetTarget( 1.0f, frames );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         IsPause

  Description:  一時停止中かどうかを取得

  Arguments:    None.

  Returns:      一時停止中ならば true
 *--------------------------------------------------------------------------------*/
bool BasicSound::IsPause() const NN_NOEXCEPT
{
    return (m_PauseState == PauseState_Pausing) || (m_PauseState == PauseState_Paused);
}

/*--------------------------------------------------------------------------------*
  Name:         IsMute

  Description:  ミュート中かどうかを取得

  Arguments:    None.

  Returns:      ミュート中ならば true
 *--------------------------------------------------------------------------------*/
bool BasicSound::IsMute() const NN_NOEXCEPT
{
    return (m_MuteState == MuteState_Muting) || (m_MuteState == MuteState_Muted);
}

/*--------------------------------------------------------------------------------*
  Name:         Update

  Description:  サウンドの更新

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::Update() NN_NOEXCEPT
{
    driver::BasicSoundPlayer* player = GetBasicSoundPlayerHandle();
    NN_SDK_ASSERT_NOT_NULL(player);

    if ( m_PlayerAvailableFlag && GetBasicSoundPlayerHandle()->IsPlayFinished() )
    {
        Finalize();
        return;
    }

    if ( player->IsFinalizedForCannotAllocateResource() )
    {
        ClearIsFinalizedForCannotAllocatedResourceFlag();
        Finalize();
        return;
    }

    if (IsPrepared() == false)
    {
        return;
    }

    // 自動停止カウンター処理
    if ( m_AutoStopFlag )
    {
        if ( m_AutoStopCounter == 0 )
        {
            if ( ( m_PauseState == PauseState_Normal ) ||
                 ( m_PauseState == PauseState_Unpausing ) )
            {
                Stop( 0 );
                return;
            }
        }
        else
        {
            --m_AutoStopCounter;
        }
    }

    // スタート前の処理
    bool playerStartFlag = false;
    if ( ! m_StartedFlag )
    {
        if ( ! m_StartFlag ) return;
        if ( ! IsPrepared() ) return;

        m_PlayerState = PlayerState_Play;
        playerStartFlag = true;
    }

    // カウンタ更新
    if ( m_PlayerState == PlayerState_Play )
    {
        if ( m_UpdateCounter < 0xffffffff ) m_UpdateCounter++;
    }
    if ( m_PlayerState == PlayerState_Play && m_PauseState != PauseState_Paused )
    {
        if ( m_PlayingCounter < 0xffffffff ) m_PlayingCounter++;
    }

    // 通常のポーズの終了判定は、前フレームのフェード処理の終了判定で行う
    bool pauseFadeIsFinished = m_PauseFadeVolume.IsFinished();

    // フェードフレームの更新
    switch ( m_PauseState )
    {
    case PauseState_Pausing:
        m_PauseFadeVolume.Update();
        break;
    case PauseState_Unpausing:
        m_PauseFadeVolume.Update();
        UpdateMoveValue();
        break;
    case PauseState_Normal:
        UpdateMoveValue();
        break;
    default:
        break;
    }

    // 外部パラメータの更新
    if ( m_AmbientInfo.argUpdateCallback != NULL )
    {
        m_AmbientInfo.argUpdateCallback->detail_UpdateAmbientArg(
            m_AmbientInfo.arg,
            this
        );
    }

    if ( m_AmbientInfo.paramUpdateCallback != NULL )
    {
        SoundAmbientParam ambientParam;
        if ( m_UpdateCounter > 0 )
        {
            ambientParam.SetVolume( m_AmbientParam.GetVolume() );
            ambientParam.SetPitch( m_AmbientParam.GetPitch() );
            ambientParam.SetLowPassFilter( m_AmbientParam.GetLpf() );
            ambientParam.SetBiquadFilterValue( m_AmbientParam.GetBiquadFilterValue() );
            ambientParam.SetBiquadFilterType( m_AmbientParam.GetBiquadFilterType() );
            ambientParam.SetUserData( m_AmbientParam.GetUserData() );
            ambientParam.SetPriority( m_AmbientParam.GetPriority() );

            ambientParam.SetOutputLineFlag( m_AmbientParam.GetOutputLineFlag() );
            ambientParam.SetTvParam( m_AmbientParam.GetTvParam() );
        }
        else
        {
            ambientParam.SetUserData( 0 );
        }

        m_AmbientInfo.paramUpdateCallback->detail_UpdateAmbientParam(
            m_AmbientInfo.arg,
            m_Id,
            &ambientParam
        );

        m_AmbientParam.SetVolume( ambientParam.GetVolume() );
        m_AmbientParam.SetPitch( ambientParam.GetPitch() );
        m_AmbientParam.SetLpf( ambientParam.GetLowPassFilter() );
        m_AmbientParam.SetBiquadFilterValue( ambientParam.GetBiquadFilterValue() );
        m_AmbientParam.SetBiquadFilterType( ambientParam.GetBiquadFilterType() );
        m_AmbientParam.SetUserData( ambientParam.GetUserData() );
        m_AmbientParam.SetPriority( ambientParam.GetPriority() );

        m_AmbientParam.SetOutputLineFlag( ambientParam.GetOutputLineFlag() );
        m_AmbientParam.SetTvParam( ambientParam.GetTvParam() );
    }

    if ( m_pSoundActor != NULL )
    {
        m_ActorParam = m_pSoundActor->detail_GetActorParam();
    }

    UpdateParam();

    // フェードアウト完了チェック
    if ( m_FadeOutFlag && m_FadeVolume.IsFinished() )
    {
        m_FadeOutFlag = false;
        Finalize();
        return;
    }

    DriverCommand& cmdmgr = DriverCommand::GetInstance();

    // プレイヤーの開始処理
    if ( playerStartFlag )
    {
        DriverCommandPlayer* command =
            cmdmgr.AllocCommand<DriverCommandPlayer>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_PlayerStart;
        command->player = GetBasicSoundPlayerHandle();
        cmdmgr.PushCommand(command);

        m_StartedFlag = true;
        m_StartFlag = false;
    }

    // ポーズステータスチェック
    if ( m_PauseMode == PauseMode_PauseImmediately )
    {
        pauseFadeIsFinished = m_PauseFadeVolume.IsFinished();
    }
    if ( m_PauseState == PauseState_Pausing )
    {
        if (pauseFadeIsFinished)
        {
            DriverCommandPlayer* command =
                cmdmgr.AllocCommand<DriverCommandPlayer>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
            if (command == nullptr)
            {
                return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
            }
#else
            NN_SDK_ASSERT_NOT_NULL(command);
#endif
            command->id = DriverCommandId_PlayerPause;
            command->player = GetBasicSoundPlayerHandle();
            command->flag = true;
            cmdmgr.PushCommand(command);

            m_PauseState = PauseState_Paused;
        }
    }
    else if ( m_PauseState == PauseState_Unpausing )
    {
        if ( m_PauseFadeVolume.IsFinished() )
        {
            m_PauseState = PauseState_Normal;
        }
    }
    if ( m_UnPauseFlag ) {
        DriverCommandPlayer* command =
            cmdmgr.AllocCommand<DriverCommandPlayer>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_PlayerPause;
        command->player = GetBasicSoundPlayerHandle();
        command->flag = false;
        cmdmgr.PushCommand(command);

        m_UnPauseFlag = false;
    }

    // ミュートステータスチェック
    if ( m_MuteState == MuteState_Muting )
    {
        if ( m_MuteFadeVolume.IsFinished() )
        {
            m_MuteState = MuteState_Muted;
        }
    }
    else if ( m_MuteState == MuteState_Unmuting )
    {
        if ( m_MuteFadeVolume.IsFinished() )
        {
            m_MuteState = MuteState_Normal;
        }
    }
} // NOLINT(impl/function_size)

void BasicSound::UpdateMoveValue() NN_NOEXCEPT
{
    m_FadeVolume.Update();
    m_MuteFadeVolume.Update();
    m_CommonParam.Update();
}

void BasicSound::UpdateParam() NN_NOEXCEPT
{
    OnUpdateParam();    // 各サウンドごとの処理

    // パラメータの更新
    const float volume = CalculateVolume();
    const float pitch = CalculatePitch();
    const float lpfFreq = CalculateLpfFrequency();
    const uint32_t outputLineFlag = CalculateOutLineFlag();

    int biquadFilterType;
    float biquadFilterValue;
    CalculateBiquadFilter( &biquadFilterType, &biquadFilterValue );

    OutputParam main;
    CalculateOutputParam( &main, OutputDevice_Main );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();

    DriverCommandPlayerParam* command =
            cmdmgr.AllocCommand<DriverCommandPlayerParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id                = DriverCommandId_PlayerParam;
    command->player            = GetBasicSoundPlayerHandle();
    command->volume            = volume;
    command->pitch             = pitch;
    command->lpfFreq           = lpfFreq;
    command->biquadFilterType  = biquadFilterType;
    command->biquadFilterValue = biquadFilterValue;
    command->outputLineFlag    = outputLineFlag;
    command->tvParam           = main;
    cmdmgr.PushCommand(command);

    // 追加パラメーターに関するコマンド送信
    if(m_pOutputAdditionalParam[OutputDevice_Main] == nullptr)
    {
        return;
    }

    // AdditionalSend
    if(m_pOutputAdditionalParam[OutputDevice_Main]->IsAdditionalSendEnabled())
    {
        SendArray* pSend = m_pOutputAdditionalParam[OutputDevice_Main]->GetAdditionalSendAddr();
        SoundPlayer* pSoundPlayer = GetSoundPlayer();
        for(int i = 0; i < pSend->GetCount(); ++i)
        {
            const int Bus = i + DefaultBusCount;
            const float PlayerSend =  pSoundPlayer != nullptr ? pSoundPlayer->GetSend(Bus) : 0;
            DriverCommandPlayerAdditionalSend* sendCommand = cmdmgr.AllocCommand<DriverCommandPlayerAdditionalSend>();
            sendCommand->id              = DriverCommandId_PlayerAdditionalSend;
            sendCommand->player          = GetBasicSoundPlayerHandle();
            sendCommand->bus             = Bus;
            sendCommand->send            = pSend->GetValue(i) + PlayerSend;
            cmdmgr.PushCommand(sendCommand);
        }
    }

    // BusMixVolume
    if(m_pOutputAdditionalParam[OutputDevice_Main]->IsBusMixVolumeEnabled())
    {
        const bool IsUsed = m_pOutputAdditionalParam[OutputDevice_Main]->IsBusMixVolumeUsed();

        DriverCommandPlayerBusMixVolumeUsed* busMixUsedCommand =
            cmdmgr.AllocCommand<DriverCommandPlayerBusMixVolumeUsed>();
        busMixUsedCommand->id                = DriverCommandId_PlayerBusMixVolumeUsed;
        busMixUsedCommand->player            = GetBasicSoundPlayerHandle();
        busMixUsedCommand->isUsed            = IsUsed;
        cmdmgr.PushCommand(busMixUsedCommand);

        if(IsUsed)
        {
            OutputBusMixVolume busMixParam;
            CalculateOutputBusMixVolume( &busMixParam, OutputDevice_Main );
            DriverCommandPlayerBusMixVolume* busMixCommand =
                cmdmgr.AllocCommand<DriverCommandPlayerBusMixVolume>();
            busMixCommand->id                = DriverCommandId_PlayerBusMixVolume;
            busMixCommand->player            = GetBasicSoundPlayerHandle();
            busMixCommand->tvBusMixVolume    = busMixParam;
            cmdmgr.PushCommand(busMixCommand);

            for (int i = 0; i < m_pOutputAdditionalParam[OutputDevice_Main]->GetBusMixVolumePacketAddr()->GetBusCount(); ++i)
            {
                DriverCommandPlayerBusMixVolumeEnabled* busMixEnabledCommand =
                    cmdmgr.AllocCommand<DriverCommandPlayerBusMixVolumeEnabled>();
                busMixEnabledCommand->id                = DriverCommandId_PlayerBusMixVolumeEnabled;
                busMixEnabledCommand->player            = GetBasicSoundPlayerHandle();
                busMixEnabledCommand->bus               = i;
                busMixEnabledCommand->isEnabled         = m_pOutputAdditionalParam[OutputDevice_Main]->IsBusMixVolumeEnabledForBus(i);
                cmdmgr.PushCommand(busMixEnabledCommand);
            }
        }
    }

    // VolumeThroughMode
    if(m_pOutputAdditionalParam[OutputDevice_Main]->IsVolumeThroughModeEnabled())
    {
        DriverCommandPlayerVolumeThroughModeUsed* volumeThroughModeEnabledCommand = cmdmgr.AllocCommand<DriverCommandPlayerVolumeThroughModeUsed>();
        volumeThroughModeEnabledCommand->id                         = DriverCommandId_PlayerVolumeThroughModeUsed;
        volumeThroughModeEnabledCommand->player                     = GetBasicSoundPlayerHandle();
        volumeThroughModeEnabledCommand->isVolumeThroughModeUsed    = IsVolumeThroughModeUsed();
        cmdmgr.PushCommand(volumeThroughModeEnabledCommand);

        if(volumeThroughModeEnabledCommand->isVolumeThroughModeUsed)
        {
            const float binaryVolume = m_InitVolume;

            VolumeThroughModePacket* pPacket = m_pOutputAdditionalParam[OutputDevice_Main]->GetVolumeThroughModePacketAddr();
            DriverCommandPlayerBinaryVolume* binaryVolumeCommand = cmdmgr.AllocCommand<DriverCommandPlayerBinaryVolume>();
            binaryVolumeCommand->id       = DriverCommandId_PlayerBinaryVolume;
            binaryVolumeCommand->player   = GetBasicSoundPlayerHandle();
            binaryVolumeCommand->volume   = binaryVolume;
            cmdmgr.PushCommand(binaryVolumeCommand);

            for(int i = 0; i < pPacket->GetBusCount(); ++i)
            {
                DriverCommandPlayerVolumeThroughMode* volumeThroughModeCommand = cmdmgr.AllocCommand<DriverCommandPlayerVolumeThroughMode>();
                volumeThroughModeCommand->id                = DriverCommandId_PlayerVolumeThroughMode;
                volumeThroughModeCommand->player            = GetBasicSoundPlayerHandle();
                volumeThroughModeCommand->bus               = i;
                volumeThroughModeCommand->volumeThroughMode = pPacket->GetVolumeThroughMode(i);
                cmdmgr.PushCommand(volumeThroughModeCommand);
            }
        }
    }
} // NOLINT(impl/function_size)

float BasicSound::CalculateVolume() const NN_NOEXCEPT
{
    float volume;
    if ( m_MuteState == MuteState_Muted )
    {
        volume = 0;
    }
    else
    {
        volume  = 1.0f;

        // バイナリのボリューム設定の適用を行います。
        // この箇所でボリューム設定が適用されない場合、 MultiVoice までボリューム計算が遅延されます
        if (!IsVolumeThroughModeUsed())
        {
            volume *= m_InitVolume;
        }

        volume *= GetSoundPlayer()->GetVolume();
        volume *= m_CommonParam.GetVolume();
        volume *= m_FadeVolume.GetValue();
        volume *= m_PauseFadeVolume.GetValue();
        volume *= m_MuteFadeVolume.GetValue();
        volume *= m_AmbientParam.GetVolume();
        volume *= m_ActorParam.volume;
    }
    return volume;
}
float BasicSound::CalculatePitch() const NN_NOEXCEPT
{
    float pitch = m_Pitch;
    pitch *= m_AmbientParam.GetPitch();
    pitch *= m_ActorParam.pitch;
    return pitch;
}
float BasicSound::CalculateLpfFrequency() const NN_NOEXCEPT
{
    float lpfFreq = m_LpfFreq;
    lpfFreq += m_AmbientParam.GetLpf();
    lpfFreq += GetSoundPlayer()->GetLowPassFilterFrequency();
    lpfFreq += m_ActorParam.lpf;
    return lpfFreq;
}
void BasicSound::CalculateBiquadFilter(int* pOutBiquadType, float* pOutBiquadValue) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutBiquadType );
    NN_SDK_ASSERT_NOT_NULL( pOutBiquadValue );

    int biquadFilterType = m_BiquadFilterType;
    float biquadFilterValue = m_BiquadFilterValue;
    if ( biquadFilterType == BiquadFilterType_Inherit )
    {
        biquadFilterType = GetSoundPlayer()->GetBiquadFilterType();
        biquadFilterValue = GetSoundPlayer()->GetBiquadFilterValue();

        if( biquadFilterType == BiquadFilterType_Inherit )
        {
            biquadFilterType = m_ActorParam.biquadFilterType;
            biquadFilterValue = m_ActorParam.biquadFilterValue;

            if( biquadFilterType == BiquadFilterType_Inherit )
            {
                biquadFilterType = m_AmbientParam.GetBiquadFilterType();
                biquadFilterValue = m_AmbientParam.GetBiquadFilterValue();
            }
        }
    }

    *pOutBiquadType = biquadFilterType;
    *pOutBiquadValue = biquadFilterValue;
}
uint32_t BasicSound::CalculateOutLineFlag() const NN_NOEXCEPT
{
    uint32_t outputLineFlag = m_OutputLineFlag;
    if ( m_AmbientParam.GetOutputLineFlag() != SoundAmbientParam::OutputLineFlagInherit )
    {
        outputLineFlag = m_AmbientParam.GetOutputLineFlag();
    }
    return outputLineFlag;
}
void BasicSound::CalculateOutputParam(OutputParam* pOutParam, OutputDevice device) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutParam );
    NN_SDK_ASSERT_RANGE( device, 0, OutputDevice_Count );

    const OutputAmbientParam* pAmbientParam;
    float actorVolume, actorPan;
    switch( device )
    {
        case OutputDevice_Main:
            pAmbientParam = &m_AmbientParam.GetTvParam();
            actorVolume   = m_ActorParam.tvVolume;
            actorPan      = m_ActorParam.tvPan;
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
    }

    *pOutParam = m_OutputParam[device];
    ApplyCommonParam( *pOutParam );  // 全体パラメータの適用

    const SoundPlayer* pSoundPlayer = GetSoundPlayer();
    pOutParam->volume   *= pSoundPlayer->GetOutputVolume( device )
                        *  pAmbientParam->GetVolume()
                        *  actorVolume;
    pOutParam->pan      += pAmbientParam->GetPan()
                        +  actorPan;
    pOutParam->span     += pAmbientParam->GetSurroundPan();
    pOutParam->send[Util::GetSubMixBusFromMainBus()] += pSoundPlayer->GetOutputMainSend( device );

    for ( int i = 0; i < AuxBus_Count; i++ )
    {
        pOutParam->send[Util::GetSubMixBus(i)] += pSoundPlayer->GetOutputFxSend( device, static_cast<AuxBus>(i) )
                             +  pAmbientParam->GetEffectSend(i);
    }
}

void BasicSound::CalculateOutputBusMixVolume(OutputBusMixVolume* pOutParam, OutputDevice device) const NN_NOEXCEPT
{
    *pOutParam = m_pOutputAdditionalParam[device]->GetBusMixVolume();
}

void BasicSound::ApplyCommonParam( OutputParam& param ) const NN_NOEXCEPT
{
    // param.volume *= m_CommonParam.GetVolume();
    param.pan += m_CommonParam.pan;
    param.span += m_CommonParam.span;
    param.mixMode = m_CommonParam.mixMode;
    for ( int i = 0; i < DefaultBusCount; i++ )
    {
        param.send[i] += m_CommonParam.send[i];
    }
}

/*--------------------------------------------------------------------------------*
  Name:         AttachPlayerHeap

  Description:  PlayerHeapを結びつける

  Arguments:    pHeap - 結びつけるPlayerHeap

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void BasicSound::AttachPlayerHeap( PlayerHeap* pHeap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pHeap );
    NN_SDK_ASSERT( m_pPlayerHeap == NULL );

    m_pPlayerHeap = pHeap;
}

/*--------------------------------------------------------------------------------*
  Name:         DetachPlayerHeap

  Description:  PlayerHeapを切り離す

  Arguments:    heap - 切り離すPlayerHeap

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void BasicSound::DetachPlayerHeap( PlayerHeap* pHeap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pHeap );
    NN_SDK_ASSERT( pHeap == m_pPlayerHeap );
    (void)pHeap;

    m_pPlayerHeap = NULL;
}

/*--------------------------------------------------------------------------------*
  Name:         AttachSoundPlayer

  Description:  SoundPlayerを結びつける

  Arguments:    player - 結びつけるSoundPlayer

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void BasicSound::AttachSoundPlayer( SoundPlayer* player ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( player );
    NN_SDK_ASSERT( m_pSoundPlayer == NULL );

    m_pSoundPlayer = player;
}

/*--------------------------------------------------------------------------------*
  Name:         DetachSoundPlayer

  Description:  SoundPlayerを切り離す

  Arguments:    player - 切り離すSoundPlayer

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void BasicSound::DetachSoundPlayer( SoundPlayer* player ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( player );
    NN_SDK_ASSERT( player == m_pSoundPlayer );
    (void)player;

    m_pSoundPlayer = NULL;
}

/*--------------------------------------------------------------------------------*
  Name:         AttachSoundActor

  Description:  SoundActorを結びつける

  Arguments:    actor - 結びつけるSoundActor

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void BasicSound::AttachSoundActor( SoundActor* actor ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( actor );
    NN_SDK_ASSERT( m_pSoundActor == NULL );

    m_pSoundActor = actor;
}

/*--------------------------------------------------------------------------------*
  Name:         DetachSoundActor

  Description:  SoundActorを切り離す

  Arguments:    actor - 切り離すSoundActor

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void BasicSound::DetachSoundActor( SoundActor* actor ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( actor );
    NN_SDK_ASSERT( actor == m_pSoundActor );
    (void)actor;

    m_pSoundActor = NULL;
}

/*--------------------------------------------------------------------------------*
  Name:         AttachExternalSoundPlayer

  Description:  ExternalSoundPlayerを結びつける

  Arguments:    extPlayer - 結びつけるExternalSoundPlayer

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void BasicSound::AttachExternalSoundPlayer( ExternalSoundPlayer* extPlayer ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( extPlayer );
    NN_SDK_ASSERT( m_pExtSoundPlayer == NULL );

    m_pExtSoundPlayer = extPlayer;
}

/*--------------------------------------------------------------------------------*
  Name:         DetachExternalSoundPlayer

  Description:  ExternalSoundPlayerを切り離す

  Arguments:    extPlayer - 切り離すExternalSoundPlayer

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void BasicSound::DetachExternalSoundPlayer( ExternalSoundPlayer* extPlayer ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( extPlayer );
    NN_SDK_ASSERT( extPlayer == m_pExtSoundPlayer );
    (void)extPlayer;

    m_pExtSoundPlayer = NULL;
}

/*--------------------------------------------------------------------------------*
  Name:         GetRemainingFadeFrames

  Description:  フェードの残りフレームを取得する

  Arguments:    なし

  Returns:      残りフレーム
 *--------------------------------------------------------------------------------*/
int BasicSound::GetRemainingFadeFrames() const NN_NOEXCEPT
{
    return m_FadeVolume.GetRemainingCount();
}

/*--------------------------------------------------------------------------------*
  Name:         GetRemainingPauseFadeFrames

  Description:  一時停止フェードの残りフレームを取得する

  Arguments:    なし

  Returns:      残りフレーム
 *--------------------------------------------------------------------------------*/
int BasicSound::GetRemainingPauseFadeFrames() const NN_NOEXCEPT
{
    return m_PauseFadeVolume.GetRemainingCount();
}

/*--------------------------------------------------------------------------------*
  Name:         GetRemainingMuteFadeFrames

  Description:  ミュートフェードの残りフレームを取得する

  Arguments:    なし

  Returns:      残りフレーム
 *--------------------------------------------------------------------------------*/
int BasicSound::GetRemainingMuteFadeFrames() const NN_NOEXCEPT
{
    return m_MuteFadeVolume.GetRemainingCount();
}


void BasicSound::CalculateSoundParamCalculationValues(SoundParamCalculationValues* pOutValue) const NN_NOEXCEPT
{
    // TODO: バスごと MixVolume パラメーターをセンドする対応
    NN_SDK_ASSERT_NOT_NULL( pOutValue );
    NN_SDK_ASSERT_NOT_NULL( GetSoundPlayer() );

    pOutValue->soundArchiveParam.volume = m_InitVolume;
    SetSoundPlayerCalculationValues( pOutValue->soundPlayerParam, *GetSoundPlayer() );
    SetSound3DCalculationValues( pOutValue->sound3DParam, m_AmbientParam );
    SetSoundActorCalculationValues( pOutValue->soundActorParam, m_ActorParam );

    pOutValue->soundHandleParam.volume         = m_CommonParam.GetVolume();
    pOutValue->soundHandleParam.pitch          = m_Pitch;
    pOutValue->soundHandleParam.lpf            = m_LpfFreq;
    pOutValue->soundHandleParam.bqfType        = m_BiquadFilterType;
    pOutValue->soundHandleParam.bqfValue       = m_BiquadFilterValue;
    pOutValue->soundHandleParam.outputLineFlag = m_OutputLineFlag;
    pOutValue->soundHandleParam.mixMode        = m_CommonParam.mixMode;
    pOutValue->soundHandleParam.pan            = m_CommonParam.pan;
    pOutValue->soundHandleParam.surroundPan    = m_CommonParam.span;
    pOutValue->soundHandleParam.mainSend       = m_CommonParam.send[Util::GetSubMixBusFromMainBus()];
    pOutValue->soundHandleParam.playerPriority = m_Priority;
    for(int i = 0; i < AuxBus_Count; i++)
    {
        pOutValue->soundHandleParam.effectSend[i] = m_CommonParam.send[Util::GetSubMixBus(i)];
    }
    for(int i = 0; i < OutputDevice_Count; i++)
    {
        const OutputDevice device = static_cast<OutputDevice>(i);
        pOutValue->soundHandleParam.outputVolume[device]      = m_OutputParam[device].volume;
        pOutValue->soundHandleParam.outputPan[device]         = m_OutputParam[device].pan;
        pOutValue->soundHandleParam.outputSurroundPan[device] = m_OutputParam[device].span;
        pOutValue->soundHandleParam.outputMainSend[device]    = m_OutputParam[device].send[Util::GetSubMixBusFromMainBus()];

        for(int k = 0; k < AuxBus_Count; k++)
        {
            const AuxBus bus = static_cast<AuxBus>(k);
            pOutValue->soundHandleParam.outputEffectSend[device][bus] = m_OutputParam[device].send[Util::GetSubMixBus(k)];
        }
        for(int k = 0; k < WaveChannelMax; k++)
        {
            pOutValue->soundHandleParam.outputMixParameter[device][k] = m_OutputParam[device].mixParameter[k];
        }
    }

    // TODO: volume が返す値をどうするかを考える
    pOutValue->resultParam.volume         = CalculateVolume();
    pOutValue->resultParam.pitch          = CalculatePitch();
    pOutValue->resultParam.lpf            = CalculateLpfFrequency();
    pOutValue->resultParam.outputLineFlag = CalculateOutLineFlag();
    pOutValue->resultParam.playerPriority = CalcCurrentPlayerPriority();
    CalculateBiquadFilter( &pOutValue->resultParam.bqfType, &pOutValue->resultParam.bqfValue );
    for(int i = 0; i < OutputDevice_Count; i++)
    {
        CalculateOutputParam( &pOutValue->resultParam.outputParamResult[i], static_cast<OutputDevice>(i) );
    }

    pOutValue->fadeVolumeParam.stopFadeVolume  = m_FadeVolume.GetValue();
    pOutValue->fadeVolumeParam.pauseFadeVolume = m_PauseFadeVolume.GetValue();
    pOutValue->fadeVolumeParam.muteFadeVolume  = m_MuteFadeVolume.GetValue();
    pOutValue->fadeVolumeParam.isMuted         = m_MuteState == MuteState_Muted;
}


/*--------------------------------------------------------------------------------*
  Name:         SetPlayerPriority

  Description:  プレイヤープライオリティを設定する

  Arguments:    priority - プレイヤープライオリティ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::SetPlayerPriority( int priority ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( priority >= nn::atk::PlayerPriorityMin && priority <= nn::atk::PlayerPriorityMax );
    m_Priority = static_cast<uint8_t>( priority );
    if ( m_pSoundPlayer != NULL )
    {
        m_pSoundPlayer->detail_SortPriorityList( this );
    }

    OnUpdatePlayerPriority();
}

/*--------------------------------------------------------------------------------*
  Name:         SetInitialVolume

  Description:  初期ボリュームの設定

  Arguments:    volume - ボリューム

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::SetInitialVolume( float volume ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( volume >= 0.0f );
    if ( volume < 0.0f ) volume = 0.0f;
    m_InitVolume = volume;
}
float BasicSound::GetInitialVolume() const NN_NOEXCEPT
{
    return m_InitVolume;
}

/*--------------------------------------------------------------------------------*
  Name:         SetVolume

  Description:  ボリュームの変更

  Arguments:    volume - ボリューム

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::SetVolume( float volume, int frames ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( volume >= 0.0f );
    if ( volume < 0.0f ) volume = 0.0f;
    m_CommonParam.SetVolume(volume ,frames);
}
float BasicSound::GetVolume() const NN_NOEXCEPT
{
    return m_CommonParam.GetVolume();
}

/*--------------------------------------------------------------------------------*
  Name:         SetPitch

  Description:  音程の変更

  Arguments:    pitch - 変更する比率

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::SetPitch( float pitch ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pitch >= 0.0f );
    m_Pitch = pitch;
}
float BasicSound::GetPitch() const NN_NOEXCEPT
{
    return m_Pitch;
}

/*--------------------------------------------------------------------------------*
  Name:         SetLpfFreq

  Description:  ローパスフィルタのカットオフ周波数の変更

  Arguments:    lpfFreq - カットオフ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::SetLpfFreq( float lpfFreq ) NN_NOEXCEPT
{
    m_LpfFreq = lpfFreq;
}
float BasicSound::GetLpfFreq() const NN_NOEXCEPT
{
    return m_LpfFreq;
}

/*--------------------------------------------------------------------------------*
  Name:         SetBiquadFilter

  Description:  Biquadフィルタの設定

  Arguments:    type  - フィルタの種類 (-1 - 127)
                value - フィルタの値 (0.0f - 1.0f)

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::SetBiquadFilter( int type, float value ) NN_NOEXCEPT
{
    m_BiquadFilterType = static_cast<int8_t>( type );
    m_BiquadFilterValue = value;
}
void BasicSound::GetBiquadFilter( int* type, float* value ) const NN_NOEXCEPT
{
    if ( type != NULL )
    {
        *type = m_BiquadFilterType;
    }
    if ( value != NULL )
    {
        *value = m_BiquadFilterValue;
    }
}

void BasicSound::SetOutputLine( uint32_t lineFlag ) NN_NOEXCEPT
{
    m_OutputLineFlag = lineFlag;
}

uint32_t BasicSound::GetOutputLine() const NN_NOEXCEPT
{
    return m_OutputLineFlag;
}

void BasicSound::ResetOutputLine() NN_NOEXCEPT
{
    m_OutputLineFlag = m_pSoundPlayer->GetDefaultOutputLine();
}


//
// 全体向け設定
//
void BasicSound::SetMixMode(MixMode mixMode) NN_NOEXCEPT
{
    m_CommonParam.mixMode = mixMode;
}

MixMode BasicSound::GetMixMode() NN_NOEXCEPT
{
    return m_CommonParam.mixMode;
}
void BasicSound::SetPan( float pan ) NN_NOEXCEPT
{
    m_CommonParam.pan = pan;
}
float BasicSound::GetPan() const NN_NOEXCEPT
{
    return m_CommonParam.pan;
}

void BasicSound::SetSurroundPan( float span ) NN_NOEXCEPT
{
    m_CommonParam.span = span;
}
float BasicSound::GetSurroundPan() const NN_NOEXCEPT
{
    return m_CommonParam.span;
}

void BasicSound::SetMainSend( float send ) NN_NOEXCEPT
{
    m_CommonParam.send[Util::GetSubMixBusFromMainBus()] = send;
}
float BasicSound::GetMainSend() const NN_NOEXCEPT
{
    return m_CommonParam.send[Util::GetSubMixBusFromMainBus()];
}

void BasicSound::SetFxSend( AuxBus bus, float send ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( bus, 0, AuxBus_Count );
    m_CommonParam.send[Util::GetSubMixBus(bus)] = send;
}
float BasicSound::GetFxSend( AuxBus bus ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( bus, 0, AuxBus_Count );
    return m_CommonParam.send[Util::GetSubMixBus(bus)];
}

void BasicSound::SetSend(int bus, float send) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( bus, 0, OutputReceiver::BusCountMax );
    if (bus < DefaultBusCount)
    {
        m_CommonParam.send[bus] = send;
        return;
    }

    for(int device = 0; device < OutputDevice_Count; device++)
    {
        SetOutputAdditionalSend(static_cast<OutputDevice>(device), bus, send);
    }
}

float BasicSound::GetSend(int bus) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( bus, 0, OutputReceiver::BusCountMax );
    if (bus < DefaultBusCount)
    {
        return m_CommonParam.send[bus];
    }

    // OutputDevice が 2 以上に増える場合、共通の AdditionalSend を新たに作成する必要がある
    return GetOutputAdditionalSend(OutputDevice_Main, bus);
}

void BasicSound::SetVolumeThroughMode(int bus, uint8_t modeBitFlag) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( bus, 0, OutputReceiver::BusCountMax );
    for(int i = 0; i < OutputDevice_Count; i++)
    {
        SetOutputVolumeThroughMode( static_cast<OutputDevice>(i), bus, modeBitFlag);
    }
}

uint8_t BasicSound::GetVolumeThroughMode(int bus) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( bus, 0, OutputReceiver::BusCountMax );
    // OutputDevice が 2 以上に増える場合、共通の VolumeThroughMode を新たに作成する必要がある
    return GetOutputVolumeThroughMode(OutputDevice_Main, bus);
}

bool BasicSound::IsVolumeThroughModeUsed() const NN_NOEXCEPT
{
    bool isVolumeThroughMode = false;
    for (int i = 0; i < OutputDevice_Count; ++i)
    {
        if(m_pOutputAdditionalParam[i] == nullptr || !m_pOutputAdditionalParam[i]->IsVolumeThroughModeEnabled())
        {
            return false;
        }

        for (int j = 0; j < m_pOutputReceiver->GetBusCount(); ++j)
        {
            if(m_pOutputAdditionalParam[i]->GetVolumeThroughModePacketAddr()->TryGetVolumeThroughMode(j) != 0)
            {
                isVolumeThroughMode = true;
                break;
            }
        }
    }
    return isVolumeThroughMode;
}

int BasicSound::GetSendBusCount() NN_NOEXCEPT
{
    if(m_pOutputReceiver == nullptr)
    {
        return 0;
    }
    return m_pOutputReceiver->GetBusCount();
}

int BasicSound::GetSendChannelCount() NN_NOEXCEPT
{
    if(m_pOutputReceiver == nullptr)
    {
        return 0;
    }
    return m_pOutputReceiver->GetChannelCount();
}

//
// 出力先別設定
//
void BasicSound::SetOutputAdditionalParamAddr(OutputDevice device, OutputAdditionalParam* addr, OutputAdditionalParam* addrForPlayer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    m_pOutputAdditionalParam[device] = addr;
    GetBasicSoundPlayerHandle()->SetTvAdditionalParamAddr(addrForPlayer);
}

void BasicSound::SetOutputVolume(OutputDevice device, float volume) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    m_OutputParam[device].volume = volume;
}


void BasicSound::SetOutputBusMixVolume(OutputDevice device, int srcChNo, int subMixBus, ChannelMixVolume param) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(subMixBus, 0, OutputReceiver::BusCountMax);
    NN_SDK_ASSERT(srcChNo < WaveChannelMax);
    NN_SDK_REQUIRES_NOT_NULL(m_pOutputAdditionalParam[device]);
    NN_SDK_REQUIRES(m_pOutputAdditionalParam[device]->IsBusMixVolumeEnabled());

    if (m_pOutputReceiver == nullptr)
    {
        return;
    }
    NN_SDK_REQUIRES( subMixBus < m_pOutputReceiver->GetBusCount() );

    const int ChannelCount = std::min(param.GetChannelCount(), m_pOutputReceiver->GetChannelCount());

    for ( int i = 0; i < ChannelCount; i++ )
    {
        const float channelVolume = param.GetChannelVolume(i);
        m_pOutputAdditionalParam[device]->SetBusMixVolume(srcChNo, Util::GetOutputReceiverMixBufferIndex(m_pOutputReceiver, subMixBus, i), channelVolume);
    }
}

void BasicSound::SetOutputChannelMixParameter(OutputDevice device, uint32_t srcChNo, MixParameter param) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    NN_SDK_ASSERT(srcChNo < WaveChannelMax);

    for ( int i = 0; i < ChannelIndex_Count; i++ )
    {
        m_OutputParam[device].mixParameter[srcChNo].ch[i] = param.ch[i];
    }
}

void BasicSound::SetOutputVolumeThroughMode(OutputDevice device, int bus, uint8_t modeBitFlag) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    NN_SDK_REQUIRES_NOT_NULL(m_pOutputAdditionalParam[device]);
    NN_SDK_REQUIRES(m_pOutputAdditionalParam[device]->IsVolumeThroughModeEnabled());

    VolumeThroughModePacket* pVolumeThroughMode = m_pOutputAdditionalParam[device]->GetVolumeThroughModePacketAddr();
    NN_SDK_REQUIRES_RANGE( bus, 0, pVolumeThroughMode->GetBusCount());
    if (m_pOutputReceiver == nullptr)
    {
        return;
    }
    NN_SDK_REQUIRES( bus < m_pOutputReceiver->GetBusCount() );
    pVolumeThroughMode->SetVolumeThroughMode(bus, modeBitFlag);
}

void BasicSound::SetOutputPan(OutputDevice device, float pan) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    m_OutputParam[device].pan = pan;
}

void BasicSound::SetOutputSurroundPan(OutputDevice device, float span) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    m_OutputParam[device].span = span;
}

void BasicSound::SetOutputMainSend(OutputDevice device, float send) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    m_OutputParam[device].send[Util::GetSubMixBusFromMainBus()] = send;
}

void BasicSound::SetOutputFxSend(OutputDevice device, AuxBus bus, float send) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    NN_SDK_ASSERT_RANGE(bus, 0, AuxBus_Count);
    m_OutputParam[device].send[Util::GetSubMixBus(bus)] = send;
}

void BasicSound::SetOutputAdditionalSend(OutputDevice device, int bus, float send) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    if(m_pOutputAdditionalParam[device] == nullptr || !m_pOutputAdditionalParam[device]->IsAdditionalSendEnabled())
    {
        // カスタムサブミックス不使用、またはカスタムサブミックス使用で 4 バス以下の場合
        NN_SDK_REQUIRES_RANGE(bus, 0, AuxBus_Count + 1);
    }
    // カスタムサブミックス使用、4バスを超える場合
    SendArray* pAdditionalSend = m_pOutputAdditionalParam[device]->GetAdditionalSendAddr();
    NN_SDK_REQUIRES_RANGE( bus, DefaultBusCount, pAdditionalSend->GetCount() + DefaultBusCount);
    if (m_pOutputReceiver == nullptr)
    {
        return;
    }
    NN_SDK_REQUIRES( bus < m_pOutputReceiver->GetBusCount() );
    pAdditionalSend->SetValue(Util::GetAdditionalSendIndex(bus), send);
}

void BasicSound::SetOutputBusMixVolumeEnabled(OutputDevice device, int subMixBus, bool isEnabled) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(subMixBus, 0, OutputReceiver::BusCountMax);
    NN_SDK_REQUIRES_NOT_NULL(m_pOutputAdditionalParam[device]);
    NN_SDK_REQUIRES(m_pOutputAdditionalParam[device]->IsBusMixVolumeEnabled());
    if (m_pOutputReceiver == nullptr)
    {
        return;
    }
    NN_SDK_REQUIRES_RANGE( subMixBus, 0, m_pOutputReceiver->GetBusCount());
    OutputAdditionalParam* pAdditionalParam = m_pOutputAdditionalParam[device];
    NN_SDK_REQUIRES_RANGE( subMixBus, 0, pAdditionalParam->GetBusMixVolumePacketAddr()->GetBusCount());

    pAdditionalParam->SetBusMixVolumeEnabledForBus(subMixBus, isEnabled);
    pAdditionalParam->SetBusMixVolumeUsed(false);
    for (int i = 0; i < pAdditionalParam->GetBusMixVolumePacketAddr()->GetBusCount(); ++i)
    {
        if(pAdditionalParam->IsBusMixVolumeEnabledForBus(subMixBus))
        {
            pAdditionalParam->SetBusMixVolumeUsed(true);
            return;
        }
    }
}

float BasicSound::GetOutputVolume(OutputDevice device) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    return m_OutputParam[device].volume;
}
MixParameter BasicSound::GetOutputChannelMixParameter(OutputDevice device, uint32_t srcChNo) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    return m_OutputParam[device].mixParameter[srcChNo];
}
float BasicSound::GetOutputPan(OutputDevice device) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    return m_OutputParam[device].pan;
}
float BasicSound::GetOutputSurroundPan(OutputDevice device) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    return m_OutputParam[device].span;
}
float BasicSound::GetOutputMainSend(OutputDevice device) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    return m_OutputParam[device].send[Util::GetSubMixBusFromMainBus()];
}
float BasicSound::GetOutputFxSend(OutputDevice device, AuxBus bus) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    NN_SDK_ASSERT_RANGE(bus, 0, AuxBus_Count);
    return m_OutputParam[device].send[Util::GetSubMixBus(bus)];
}

bool BasicSound::IsOutputBusMixVolumeEnabled(OutputDevice device, int subMixBus) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    NN_SDK_ASSERT_RANGE(subMixBus, 0, OutputReceiver::BusCountMax);
    NN_SDK_REQUIRES_NOT_NULL(m_pOutputAdditionalParam[device]);
    NN_SDK_REQUIRES(m_pOutputAdditionalParam[device]->IsBusMixVolumeEnabled());
    NN_SDK_REQUIRES_RANGE( subMixBus, 0, m_pOutputAdditionalParam[device]->GetBusMixVolumePacketAddr()->GetBusCount());

    if (m_pOutputReceiver == nullptr)
    {
        return false;
    }
    return m_pOutputAdditionalParam[device]->IsBusMixVolumeEnabledForBus(subMixBus);
}

float BasicSound::GetOutputAdditionalSend(OutputDevice device, int bus) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    NN_SDK_REQUIRES_NOT_NULL(m_pOutputAdditionalParam[device]);
    NN_SDK_REQUIRES(m_pOutputAdditionalParam[device]->IsAdditionalSendEnabled());

    SendArray* pAdditionalSend = m_pOutputAdditionalParam[device]->GetAdditionalSendAddr();
    NN_SDK_REQUIRES_RANGE( bus, DefaultBusCount, pAdditionalSend->GetCount() + DefaultBusCount);
    return pAdditionalSend->GetValue(Util::GetAdditionalSendIndex(bus));
}

uint8_t BasicSound::GetOutputVolumeThroughMode(OutputDevice device, int bus) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);
    NN_SDK_REQUIRES_NOT_NULL(m_pOutputAdditionalParam[device]);
    NN_SDK_REQUIRES(m_pOutputAdditionalParam[device]->IsVolumeThroughModeEnabled());

    VolumeThroughModePacket* pVolumeThroughMode = m_pOutputAdditionalParam[device]->GetVolumeThroughModePacketAddr();
    NN_SDK_REQUIRES_RANGE( bus, 0, pVolumeThroughMode->GetBusCount());
    return pVolumeThroughMode->GetVolumeThroughMode(bus);
}

ChannelMixVolume BasicSound::GetOutputBusMixVolume(OutputDevice device, int srcChNo, int subMixBus) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(subMixBus, 0, OutputReceiver::BusCountMax);
    NN_SDK_REQUIRES_NOT_NULL(m_pOutputAdditionalParam[device]);
    NN_SDK_REQUIRES(m_pOutputAdditionalParam[device]->IsBusMixVolumeEnabled());

    ChannelMixVolume result;
    if(m_pOutputReceiver == nullptr)
    {
        return result;
    }

    const int ChannelCount = m_pOutputReceiver->GetChannelCount();
    NN_SDK_ASSERT_RANGE(subMixBus, 0, m_pOutputReceiver->GetBusCount());
    NN_SDK_ASSERT_RANGE(srcChNo, 0, m_pOutputReceiver->GetChannelCount());
    result.SetChannelCount(ChannelCount);
    for(int i = 0; i < ChannelCount; i++)
    {
        const float volume = m_pOutputAdditionalParam[device]->GetBusMixVolume(srcChNo, subMixBus);
        result.SetChannelVolume(i, volume);
    }

    return result;
}

void BasicSound::SetPanMode( PanMode mode ) NN_NOEXCEPT
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandPlayerPanParam* command =
            cmdmgr.AllocCommand<DriverCommandPlayerPanParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_PlayerPanmode;
        command->player = GetBasicSoundPlayerHandle();
        command->panMode = static_cast<uint8_t>(mode);
        cmdmgr.PushCommand(command);
    }
}

void BasicSound::SetPanCurve( PanCurve curve ) NN_NOEXCEPT
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandPlayerPanParam* command =
            cmdmgr.AllocCommand<DriverCommandPlayerPanParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_PlayerPancurve;
        command->player = GetBasicSoundPlayerHandle();
        command->panCurve = static_cast<uint8_t>(curve);
        cmdmgr.PushCommand(command);
    }
}

/*--------------------------------------------------------------------------------*
  Name:         SetAmbientInfo

  Description:  アンビエント情報を設定する

  Arguments:

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void BasicSound::SetAmbientInfo( const AmbientInfo& ambientArgInfo ) NN_NOEXCEPT
{
    // AmbientArg の複製を作成
    NN_SDK_ASSERT_NOT_NULL( ambientArgInfo.argAllocatorCallback );
    void* ambientArg =
        ambientArgInfo.argAllocatorCallback->detail_AllocAmbientArg( ambientArgInfo.argSize );
    if ( ambientArg == NULL )
    {
        NN_ATK_WARNING("Failed to alloc AmbientArg.");
        return;
    }
    std::memcpy( ambientArg, ambientArgInfo.arg, ambientArgInfo.argSize );

    // AmbientInfo の設定
    m_AmbientInfo = ambientArgInfo;
    m_AmbientInfo.arg = ambientArg;
}

/*--------------------------------------------------------------------------------*
  Name:         GetAmbientPriority [static]

  Description:  アンビエントプライオリティの取得

  Arguments:    ambientInfo - アンビエント情報
                soundId - サウンドID

  Returns:      アンビエントプライオリティ値を返す
  *--------------------------------------------------------------------------------*/
int BasicSound::GetAmbientPriority( const AmbientInfo& ambientInfo, uint32_t soundId ) NN_NOEXCEPT
{
    if ( ambientInfo.paramUpdateCallback == NULL ) return 0;

    int priority = ambientInfo.paramUpdateCallback->detail_GetAmbientPriority(
        ambientInfo.arg,
        soundId
    );

    return priority;
}

// ハンドル関数
bool BasicSound::IsAttachedGeneralHandle() NN_NOEXCEPT
{
    return m_pGeneralHandle != NULL;
}

bool BasicSound::IsAttachedTempGeneralHandle() NN_NOEXCEPT
{
    return m_pTempGeneralHandle != NULL;
}

void BasicSound::DetachGeneralHandle() NN_NOEXCEPT
{
    m_pGeneralHandle->DetachSound();
}

void BasicSound::DetachTempGeneralHandle() NN_NOEXCEPT
{
    m_pTempGeneralHandle->DetachSound();
}

void BasicSound::SetId( uint32_t id ) NN_NOEXCEPT
{
    m_Id = id;
}

void BasicSound::SetSetupTick(const nn::os::Tick& tick) NN_NOEXCEPT
{
    m_SetupTick = tick;
}

void BasicSound::SetSoundArchive(const SoundArchive* soundArchive) NN_NOEXCEPT
{
    m_pSoundArchive = soundArchive;
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

