﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_AnimSoundImpl.h>

#include <cmath>
#include <nn/atk/atk_AnimEventPlayer.h>
#include <nn/atk/atk_BasicSound.h>
#include <nn/atk/atk_SoundArchive.h>

namespace nn {
namespace atk {
namespace detail {

AnimSoundImpl::AnimSoundImpl(
    SoundStartable& starter,
    AnimEventPlayer* eventPlayers,
    int eventPlayerCount ) NN_NOEXCEPT
: m_Starter( starter )
, m_Reader()
, m_pEventPlayers( eventPlayers )
, m_EventCallback( NULL )
, m_EventCallbackArg( NULL )
, m_CurrentFrame( 0.0f )
, m_EventPlayerCount( eventPlayerCount )
, m_IsActive( false )
, m_IsInitFrame( false )
, m_IsReset( false )
, m_LoopCounter( 0 )
, m_BaseStep( 1.0f )
, m_CurrentSpeed( 1.0f )
{
}

AnimSoundImpl::~AnimSoundImpl() NN_NOEXCEPT
{
    Finalize();
}

bool AnimSoundImpl::Initialize( const void* bcasdFile ) NN_NOEXCEPT
{
    if ( m_IsActive )
    {
        Finalize();
    }

    NN_SDK_ASSERT_NOT_NULL( bcasdFile );

    bool result = m_Reader.Initialize( bcasdFile );
    if ( ! result )
    {
        return false;
    }
    m_FrameSize = m_Reader.GetFrameSize();
    NN_SDK_ASSERT( m_FrameSize != 0 );

    m_IsInitFrame = true;
    m_IsActive = true;

    m_CurrentFrame = 0.0f;
    m_LoopCounter = 0;

    return true;
}

void AnimSoundImpl::Finalize() NN_NOEXCEPT
{
    if ( ! m_IsActive )
    {
        return;
    }

    for ( int i = 0; i < m_EventPlayerCount; i++ )
    {
        m_pEventPlayers[i].Finalize();
    }

    m_Reader.Finalize();
    m_IsActive = false;
}

bool AnimSoundImpl::ConvertSoundId( const SoundArchive& arc ) NN_NOEXCEPT
{
    if ( ! m_IsActive )
    {
        return false;
    }

    uint32_t eventCount = m_Reader.GetEventCount();
    for ( uint32_t i = 0; i < eventCount; i++ )  // TODO: Foreach 的なメソッドに置き換えたい
    {
        const AnimSoundFile::AnimEvent* event = m_Reader.GetAnimEvent( i );
        if ( ! event )
        {
            return false;
        }

        const AnimSoundFile::EventInfo* eventInfo = event->GetEventInfo();
        if ( ! eventInfo )
        {
            return false;
        }

        const char* soundLabel = eventInfo->GetSoundLabel();
        if ( ! soundLabel )
        {
            return false;
        }

        // ここだけ書き換える必要があるため、const を外す
        const_cast<AnimSoundFile::EventInfo*>(eventInfo)->placeForSoundId =
            arc.GetItemId( soundLabel );
    }

    return true;
}

void AnimSoundImpl::ResetFrame( float frame, int loopCounter ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( frame >= 0.0f && frame < static_cast<float>( m_FrameSize ) );

    if ( frame >= static_cast<float>( m_FrameSize ) )
    {
        frame = static_cast<float>( m_FrameSize - 1 );
    }

    m_CurrentFrame = frame;
    m_LoopCounter = loopCounter;

    m_IsReset = true;
    m_IsInitFrame = false;
}

void AnimSoundImpl::UpdateFrame( float frame, PlayDirection direction ) NN_NOEXCEPT
{
    // 自動停止をチェック
    for ( int i = 0; i < m_EventPlayerCount; i++ )
    {
        m_pEventPlayers[i].UpdateFrame();
    }

    NN_SDK_ASSERT( frame >= 0.0f && frame <= static_cast<float>( m_FrameSize ) );

    if ( frame >= static_cast<float>( m_FrameSize ) )
    {
        frame = static_cast<float>( m_FrameSize - 1 );
    }

    // 初期化直後の処理
    if ( m_IsInitFrame )
    {
        if ( direction == PlayDirection_Forward )
        {
            ResetFrame( 0.0f, 0 );
        }
        else
        {
            ResetFrame( m_FrameSize - 1.0f, 0 );
        }
    }

    // スピード更新
    m_CurrentSpeed = ( frame - m_CurrentFrame ) / m_BaseStep;

    // 更新処理
    if ( direction == PlayDirection_Forward )
    {
        UpdateForward( frame );
    }
    else
    {
        UpdateBackward( frame );
    }

    m_CurrentFrame = frame;
    m_IsReset = false;
}

void AnimSoundImpl::UpdateForward( float frame ) NN_NOEXCEPT
{
    int32_t intBaseFrame = static_cast<int32_t>( std::floor( m_CurrentFrame ) );
    int32_t intTargetFrame = static_cast<int32_t>( std::floor( frame ) );

    if ( m_IsReset )
    {
        if ( m_CurrentFrame == static_cast<float>( intBaseFrame ) )
        {
            intBaseFrame -= 1;
        }
    }

    if ( intBaseFrame == intTargetFrame )
    {
        return;
    }

    for ( int32_t t = intBaseFrame + 1; ; t++ )
    {
        // ループ判定
        if ( t == m_FrameSize )
        {
            t -= m_FrameSize;
            m_LoopCounter++;
        }

        // 現フレームのイベントを探して実行
        UpdateOneFrame( t, PlayDirection_Forward );

        if ( t == intTargetFrame )
        {
            break;
        }
    }
}
void AnimSoundImpl::UpdateBackward( float frame ) NN_NOEXCEPT
{
    int32_t intBaseFrame = static_cast<int32_t>( std::ceil( m_CurrentFrame ) );
    int32_t intTargetFrame = static_cast<int32_t>( std::ceil( frame ) );

    // TODO: 分かりやすい処理に書き換える
    if ( intBaseFrame >= m_FrameSize )
    {
        intBaseFrame -= m_FrameSize;
    }
    if ( intTargetFrame >= m_FrameSize )
    {
        intTargetFrame -= m_FrameSize;
    }

    if ( m_IsReset )
    {
        if ( m_CurrentFrame == static_cast<float>( intBaseFrame ) )
        {
            intBaseFrame += 1;
        }
    }

    if ( intBaseFrame == intTargetFrame )
    {
        return;
    }

    for ( int32_t t = intBaseFrame - 1; ; t-- )
    {
        // ループ判定
        if ( t == -1 )
        {
            t += m_FrameSize;
            m_LoopCounter--;
        }

        // 現フレームのイベントを探して実行
        UpdateOneFrame( t, PlayDirection_Backward );

        if ( t == intTargetFrame ) break;
    }
}

void AnimSoundImpl::UpdateOneFrame( int32_t current, PlayDirection direction ) NN_NOEXCEPT
{
    // 現フレームのイベントを探して実行
    uint32_t eventCount = m_Reader.GetEventCount();
    for ( uint32_t i = 0; i < eventCount; i++ )
    {
        // イベント取得
        const AnimSoundFile::AnimEvent* event = m_Reader.GetAnimEvent( i );
        if ( event == NULL )
        {
            continue;
        }

        // ループカウンタ判定
        if ( ! IsPlayableLoopCount( event->frameInfo ) ) continue;

        // イベント実行
        if ( event->frameInfo.frameFlag & AnimSoundFile::FrameInfo::FrameFlag_TriggerEvent )
        {
            // トリガイベント
            UpdateTrigger( *event, current, direction );
        }
        else
        {
            // レンジイベント
            UpdateRange( *event, current, direction );
        }
    }
}

void AnimSoundImpl::UpdateTrigger(
    const AnimSoundFile::AnimEvent& event,
    int32_t current,
    PlayDirection direction
) NN_NOEXCEPT
{
    if ( event.frameInfo.frameFlag & AnimSoundFile::FrameInfo::FrameFlag_StartFrameInf )
    {
        return;
    }
    const AnimSoundFile::EventInfo* eventInfo = event.GetEventInfo();
    if ( eventInfo == NULL )
    {
        return;
    }

    // 再生方向判定
    bool isEnableForPlayDirection = false;
    if ( eventInfo->playDirection == AnimSoundFile::EventInfo::PlayDirection_Both ||
         ( direction == PlayDirection_Forward  &&
           eventInfo->playDirection == AnimSoundFile::EventInfo::PlayDirection_Forward  ) ||
         ( direction == PlayDirection_Backward  &&
           eventInfo->playDirection == AnimSoundFile::EventInfo::PlayDirection_Backward  )
       )
    {
        isEnableForPlayDirection = true;
    }
    if ( ! isEnableForPlayDirection )
    {
        return;
    }

    if ( event.frameInfo.frameFlag & AnimSoundFile::FrameInfo::FrameFlag_EndFrameInf )
    {
        // 終了フレームが無いワンショットのトリガイベント
        if ( event.frameInfo.startFrame == current )
        {
            StopEvent( *eventInfo );
            if ( m_EventCallback != NULL )
            {
                m_EventCallback(
                    EventType_TriggerStart,
                    current,
                    eventInfo->GetSoundLabel(),
                    eventInfo->userParam,
                    m_EventCallbackArg
                );
            }
            StartEvent( *eventInfo, true );
        }
    }
    else
    {
        // 終了フレームがあるトリガイベント

        // 開始判定
        if ( event.frameInfo.startFrame == current )
        {
            StopEvent( *eventInfo );
            if ( m_EventCallback != NULL )
            {
                m_EventCallback(
                    EventType_TriggerStart,
                    current,
                    eventInfo->GetSoundLabel(),
                    eventInfo->userParam,
                    m_EventCallbackArg
                );
            }
            StartEvent( *eventInfo, true );
        }

        // 終了判定
        long endFrame;
        if ( direction == PlayDirection_Forward )
        {
            endFrame = event.frameInfo.endFrame;
        }
        else // PLAY_BACKWARD
        {
            endFrame = event.frameInfo.startFrame
                - ( event.frameInfo.endFrame - event.frameInfo.startFrame );
            endFrame = std::max( endFrame, 0L );
        }
        if ( endFrame == current )
        {
            if ( m_EventCallback != NULL )
            {
                m_EventCallback(
                    EventType_TriggerStop,
                    current,
                    eventInfo->GetSoundLabel(),
                    eventInfo->userParam,
                    m_EventCallbackArg
                );
            }
            StopEvent( *eventInfo );
        }
    }
}

void AnimSoundImpl::UpdateRange(
    const AnimSoundFile::AnimEvent& event,
    int32_t current,
    PlayDirection direction
) NN_NOEXCEPT
{
    const AnimSoundFile::EventInfo* eventInfo = event.GetEventInfo();
    if ( eventInfo == NULL )
    {
        return;
    }

    // 再生方向判定
    bool isEnableForPlayDirection = false;
    if ( eventInfo->playDirection == AnimSoundFile::EventInfo::PlayDirection_Both ||
         ( direction == PlayDirection_Forward  &&
           eventInfo->playDirection == AnimSoundFile::EventInfo::PlayDirection_Forward  ) ||
         ( direction == PlayDirection_Backward  &&
           eventInfo->playDirection == AnimSoundFile::EventInfo::PlayDirection_Backward  )
       )
    {
        isEnableForPlayDirection = true;
    }
    if ( ! isEnableForPlayDirection )
    {
        return;
    }

    if ( direction == PlayDirection_Forward )
    {
        UpdateForwardRange( event, current );
    }
    else // PLAY_BACKWARD
    {
        UpdateBackwardRange( event, current );
    }
}

void AnimSoundImpl::UpdateForwardRange(
    const AnimSoundFile::AnimEvent& event,
    int32_t current
) NN_NOEXCEPT
{
    const AnimSoundFile::EventInfo* eventInfo = event.GetEventInfo();
    if ( eventInfo == NULL )
    {
        return;
    }

    const AnimSoundFile::FrameInfo& frameInfo = event.frameInfo;
    if ( frameInfo.frameFlag & AnimSoundFile::FrameInfo::FrameFlag_StartFrameInf )
    {
        if ( frameInfo.frameFlag & AnimSoundFile::FrameInfo::FrameFlag_EndFrameInf )
        {
            // 開始フレームも終了フレームも無いレンジイベント
            HoldEvent( *eventInfo, true );
        }
        else
        {
            // 開始フレームが無いレンジイベント
            if ( frameInfo.endFrame == current )
            {
                if ( m_EventCallback != NULL )
                {
                    m_EventCallback(
                        EventType_RangeStop,
                        current,
                        eventInfo->GetSoundLabel(),
                        eventInfo->userParam,
                        m_EventCallbackArg );
                }
            }
            if ( m_LoopCounter < frameInfo.loopOffset )
            {
                HoldEvent( *eventInfo, true );
            }
            else if ( m_LoopCounter == frameInfo.loopOffset )
            {
                if ( current < frameInfo.endFrame )
                {
                    HoldEvent( *eventInfo, true );
                }
                else
                {
                    StopEvent( *eventInfo );
                }
            }
        }
    }
    else if ( frameInfo.frameFlag & AnimSoundFile::FrameInfo::FrameFlag_EndFrameInf )
    {
        // 終了フレームが無いレンジイベント
        if ( frameInfo.startFrame == current )
        {
            if ( m_EventCallback != NULL )
            {
                m_EventCallback(
                    EventType_RangeStart,
                    current,
                    eventInfo->GetSoundLabel(),
                    eventInfo->userParam,
                    m_EventCallbackArg
                );
            }
        }
        if ( m_LoopCounter > frameInfo.loopOffset )
        {
            HoldEvent( *eventInfo, true );
        }
        else if ( m_LoopCounter == frameInfo.loopOffset )
        {
            if ( frameInfo.startFrame <= current )
            {
                HoldEvent( *eventInfo, true );
            }
        }
    }
    else
    {
        // 開始フレームも終了フレームもあるレンジイベント
        NN_SDK_ASSERT( frameInfo.startFrame < frameInfo.endFrame );
        if ( frameInfo.startFrame == current )
        {
            if ( m_EventCallback != NULL )
            {
                m_EventCallback(
                    EventType_RangeStart,
                    current,
                    eventInfo->GetSoundLabel(),
                    eventInfo->userParam,
                    m_EventCallbackArg
                );
            }
        }
        if ( frameInfo.endFrame == current )
        {
            if ( m_EventCallback != NULL )
            {
                m_EventCallback(
                    EventType_RangeStop,
                    current,
                    eventInfo->GetSoundLabel(),
                    eventInfo->userParam,
                    m_EventCallbackArg
                );
            }
        }
        if ( ( frameInfo.startFrame <= current ) &&
             ( current < frameInfo.endFrame ) )
        {
            HoldEvent( *eventInfo, true );
        }
        else
        {
            StopEvent( *eventInfo );
        }
    }
} // NOLINT(impl/function_size)

void AnimSoundImpl::UpdateBackwardRange(
    const AnimSoundFile::AnimEvent& event,
    int32_t current
) NN_NOEXCEPT
{
    const AnimSoundFile::EventInfo* eventInfo = event.GetEventInfo();
    if ( eventInfo == NULL )
    {
        return;
    }

    const AnimSoundFile::FrameInfo& frameInfo = event.frameInfo;
    if ( frameInfo.frameFlag & AnimSoundFile::FrameInfo::FrameFlag_StartFrameInf )
    {
        if ( frameInfo.frameFlag & AnimSoundFile::FrameInfo::FrameFlag_EndFrameInf )
        {
            // 開始フレームも終了フレームも無いレンジイベント
            HoldEvent( *eventInfo, true );
        }
        else
        {
            // 開始フレームが無いレンジイベント
            if ( frameInfo.endFrame == current )
            {
                if ( m_EventCallback != NULL )
                {
                    m_EventCallback(
                        EventType_RangeStart,
                        current,
                        eventInfo->GetSoundLabel(),
                        eventInfo->userParam,
                        m_EventCallbackArg
                    );
                }
            }
            if ( m_LoopCounter < frameInfo.loopOffset )
            {
                HoldEvent( *eventInfo, true );
            }
            else if ( m_LoopCounter == frameInfo.loopOffset )
            {
                if ( current <= frameInfo.endFrame )
                {
                    HoldEvent( *eventInfo, true );
                }
            }
        }
    }
    else if ( frameInfo.frameFlag & AnimSoundFile::FrameInfo::FrameFlag_EndFrameInf )
    {
        // 終了フレームが無いレンジイベント
        if ( frameInfo.startFrame == current )
        {
            if ( m_EventCallback != NULL )
            {
                m_EventCallback(
                    EventType_RangeStop,
                    current,
                    eventInfo->GetSoundLabel(),
                    eventInfo->userParam,
                    m_EventCallbackArg
                );
            }
        }
        if ( m_LoopCounter > frameInfo.loopOffset )
        {
            HoldEvent( *eventInfo, true );
        }
        else if ( m_LoopCounter == frameInfo.loopOffset )
        {
            if ( frameInfo.startFrame < current )
            {
                HoldEvent( *eventInfo, true );
            }
            else StopEvent( *eventInfo );
        }
    }
    else
    {
        // 開始フレームも終了フレームもあるレンジイベント
        NN_SDK_ASSERT( frameInfo.startFrame < frameInfo.endFrame );
        if ( frameInfo.startFrame == current )
        {
            if ( m_EventCallback != NULL )
            {
                m_EventCallback(
                    EventType_RangeStop,
                    current,
                    eventInfo->GetSoundLabel(),
                    eventInfo->userParam,
                    m_EventCallbackArg
                );
            }
        }
        if ( frameInfo.endFrame == current )
        {
            if ( m_EventCallback != NULL )
            {
                m_EventCallback(
                    EventType_RangeStart,
                    current,
                    eventInfo->GetSoundLabel(),
                    eventInfo->userParam,
                    m_EventCallbackArg
                );
            }
        }
        if ( ( frameInfo.startFrame < current ) && ( current <= frameInfo.endFrame ) )
        {
            HoldEvent( *eventInfo, true );
        }
        else
        {
            StopEvent( *eventInfo );
        }
    }
} // NOLINT(impl/function_size)

void AnimSoundImpl::StartEvent( const AnimSoundFile::EventInfo& info, bool isStopWhenFinalize ) NN_NOEXCEPT
{
    // 空いているハンドルを探す
    // 全て埋まっている時は、一番プライオリティが低いサウンドを止める
    int lowestPriority = nn::atk::PlayerPriorityMax + 1;
    int lowestIndex = -1;
    int i;
    for ( i = 0; i < m_EventPlayerCount; i++ )
    {
        if ( !m_pEventPlayers[i].IsAttachedSound() )
        {
            break;
        }
        int priority = m_pEventPlayers[i].GetPlayingSoundPriority();
        if ( lowestPriority > priority )
        {
            lowestIndex = i;
            lowestPriority = priority;
        }
    }
    if ( i == m_EventPlayerCount )
    {
        m_pEventPlayers[lowestIndex].ForceStop();
        i = lowestIndex;
    }

    // 再生
    m_pEventPlayers[i].StartEvent( info, m_Starter, isStopWhenFinalize );
    WritePlaySpeedToSequenceVariable( i, info );
}

void AnimSoundImpl::HoldEvent( const AnimSoundFile::EventInfo& info, bool isStopWhenFinalize ) NN_NOEXCEPT
{
    // イベントが再生中なら何もしない
    // 新規再生用に一番プライオリティが低いサウンドを探す
    int lowestPriority = nn::atk::PlayerPriorityMax + 1;
    int lowestIndex = -1;
    int i;
    for ( i = 0; i < m_EventPlayerCount; i++ )
    {
        if ( m_pEventPlayers[i].IsPlaying( info ) )
        {
            return; // 再生中なので何もしない
        }
        if ( !m_pEventPlayers[i].IsAttachedSound() )
        {
            lowestIndex = i;
            lowestPriority = -1;
        }
        else
        {
            int priority = m_pEventPlayers[i].GetPlayingSoundPriority();
            if ( lowestPriority > priority )
            {
                lowestIndex = i;
                lowestPriority = priority;
            }
        }
    }

    // 再生中のイベントが無かったので新規に再生
    if ( i == m_EventPlayerCount )
    {
        m_pEventPlayers[lowestIndex].ForceStop();
        i = lowestIndex;
    }

    // 再生
    m_pEventPlayers[i].HoldEvent( info, m_Starter, isStopWhenFinalize );
    WritePlaySpeedToSequenceVariable( i, info );
}

void AnimSoundImpl::WritePlaySpeedToSequenceVariable(
        int eventPlayerNo, const AnimSoundFile::EventInfo& info ) NN_NOEXCEPT
{
    uint8_t sequenceVariableNo = 0;  // とりあえず 0 で初期化
    if ( info.GetSequenceVariable( &sequenceVariableNo ) )
    {
        m_pEventPlayers[eventPlayerNo].WritePlaySpeedToSequenceVariable(
                sequenceVariableNo,
                m_CurrentSpeed );
    }
}

void AnimSoundImpl::StopEvent( const AnimSoundFile::EventInfo& info ) NN_NOEXCEPT
{
    for ( int i = 0; i < m_EventPlayerCount; i++ )
    {
        m_pEventPlayers[i].StopEvent( info );
    }
}

void AnimSoundImpl::StopAllSound() NN_NOEXCEPT
{
    for ( int i = 0; i < m_EventPlayerCount; i++ )
    {
        m_pEventPlayers[i].ForceStop();
    }
}

bool AnimSoundImpl::IsPlayableLoopCount( const AnimSoundFile::FrameInfo& info ) NN_NOEXCEPT
{
    // レンジイベントで、開始フレームもしくは終了フレームがinfの場合は
    // ループカウントが反映されない
    if ( ! ( info.frameFlag & AnimSoundFile::FrameInfo::FrameFlag_TriggerEvent ) )
    {
        if ( ( info.frameFlag & AnimSoundFile::FrameInfo::FrameFlag_StartFrameInf ) ||
             ( info.frameFlag & AnimSoundFile::FrameInfo::FrameFlag_EndFrameInf )
           )
        {
            return true;
        }
    }

    int loopOffset = std::max( static_cast<int>( info.loopOffset ), 0 );
    int loopCounter = fnd::Abs( m_LoopCounter );
    if ( info.loopInterval == 0 )
    {
        if ( loopCounter < loopOffset )
        {
            return false;
        }
    }
    else
    {
        if ( loopCounter < loopOffset )
        {
            return false;
        }
        if ( ( loopCounter - loopOffset ) % info.loopInterval != 0 )
        {
            return false;
        }
    }
    return true;
}

void AnimSoundImpl::Dump( const void* bcasdFile ) NN_NOEXCEPT
{
    AnimSoundFileReader reader;
    bool ret = reader.Initialize( bcasdFile );
    if ( ret == false )
    {
        return;
    }

    reader.Dump();
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

