﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BinTypes.h>
#include <nn/atk/atk_BinaryFileFormat.h>
#include <nn/atk/atk_AnimSoundFileReader.h>

namespace nn {
namespace atk {
namespace detail {


namespace {

const uint32_t SignatureFile       = NN_UTIL_CREATE_SIGNATURE_4( 'F', 'A', 'S', 'D' );
// const uint32_t SignatureDataBlock = NN_UTIL_CREATE_SIGNATURE_4( 'D', 'A', 'T', 'A' );

const uint32_t SupportedFileVersion = 0x01000000;  // サポートする最低バージョン
const uint32_t CurrentFileVersion   = 0x01000000;  // サポートする最新バージョン

bool IsValidFileHeader( const void* bfasdFile ) NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( bfasdFile );

    // シグニチャ確認
    NN_SDK_ASSERT( header.signature == SignatureFile, "invalid file signature." );
    if ( header.signature != SignatureFile )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SupportedFileVersion <= header.version) &&
                                   (header.version <= CurrentFileVersion) )
    {
        isSupportedVersion = true;

    }
    NN_SDK_ASSERT( isSupportedVersion,
            "bfasd file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SupportedFileVersion, CurrentFileVersion, header.version
    );
    return isSupportedVersion;
}

} // anonymous namespace

AnimSoundFileReader::AnimSoundFileReader() NN_NOEXCEPT
: m_pAnimEventTable( NULL )
, m_FrameSize( 0 )
{}

bool AnimSoundFileReader::Initialize( const void* bfasdFile ) NN_NOEXCEPT
{
    if ( ! IsValidFileHeader( bfasdFile ) )
    {
        return false;
    }

    const AnimSoundFile::FileHeader* header =
        reinterpret_cast<const AnimSoundFile::FileHeader*>( bfasdFile );

    const AnimSoundFile::DataBlockBody* body = &header->GetDataBlock()->body;
    NN_SDK_ASSERT_NOT_NULL( body );

    if ( ! body )
    {
        return false;
    }

    const AnimSoundFile::AnimEventTable* table = body->GetAnimEventTable();
    NN_SDK_ASSERT_NOT_NULL( table );
    if ( ! table )
    {
        return false;
    }

    m_pAnimEventTable = table;
    m_FrameSize = body->frameSize;
    return true;
}

void AnimSoundFileReader::Finalize() NN_NOEXCEPT
{
    m_pAnimEventTable = NULL;
    m_FrameSize = 0;
}

void AnimSoundFileReader::Dump() NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP) // Dump 用途
    NN_DETAIL_ATK_INFO("*** %s begin ***\n", __FUNCTION__);
    NN_DETAIL_ATK_INFO("FrameSize : %3d\n", GetFrameSize());

    uint32_t eventCount = GetEventCount();
    NN_DETAIL_ATK_INFO("EventCount: %3d\n", eventCount);
    for ( uint32_t i = 0; i < eventCount; i++ )
    {
        const AnimSoundFile::AnimEvent* event = GetAnimEvent( i );
        if ( event == NULL )
        {
            NN_DETAIL_ATK_INFO("[%3d] event is null\n", i);
        }
        else
        {
            const AnimSoundFile::FrameInfo* frameInfo = &event->frameInfo;
            uint8_t frameFlag = frameInfo->frameFlag;
            // PC 版でうまく表示されないので一旦 auto 変数に格納する
            int32_t startFrame = frameInfo->startFrame;
            int32_t endFrame = frameInfo->endFrame;
            NN_DETAIL_ATK_INFO("[%3d] start(%d) end(%d) eventType(%s)\n",
                    i, startFrame, endFrame,
                    (frameFlag & AnimSoundFile::FrameInfo::FrameFlag_TriggerEvent) ?
                        "TRIG" : "RANGE");

            const AnimSoundFile::EventInfo* eventInfo = event->GetEventInfo();
            uint32_t optionFlag = eventInfo->optionFlag;
            float pitch = eventInfo->pitch;
            uint32_t userParam = eventInfo->userParam;
            NN_DETAIL_ATK_INFO("      (%s) vol(%d) pitch(%2.6f) userParam(0x%08x)\n",
                    eventInfo->GetSoundLabel(), eventInfo->volume,
                    pitch, userParam);
            NN_DETAIL_ATK_INFO("      notStopSoundWhenAnimFinish?(%d) seqVarEnable?(%d) seqVar(%d)\n",
                    (optionFlag & AnimSoundFile::EventInfo::
                        OptionFlag_IsNotStopSoundWhenAnimationFinish) ? 1 : 0,
                    (optionFlag & AnimSoundFile::EventInfo::
                        OptionFlag_IsEnableSequenceVariable) ? 1 : 0,
                    eventInfo->sequenceVariableNo );
            const char* PlayDirectionString[] = { "BOTH", "FORWARD", "BACKWARD" };
            NN_DETAIL_ATK_INFO("      playDirection(%s)\n",
                    PlayDirectionString[eventInfo->playDirection] );

        }
    }
    NN_DETAIL_ATK_INFO("*** %s end *****\n", __FUNCTION__);
#endif // NN_SDK_BUILD_DEBUG || NN_SDK_BUILD_DEVELOP
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

