﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SoundStartable.h>
#include <nn/atk/atk_SoundHandle.h>
#include <nn/atk/atk_Debug.h>

namespace nn {
namespace atk {

/*--------------------------------------------------------------------------------*
  Name:         detail_StartSound

  Description:  指定したサウンド番号のサウンドを再生する

  Arguments:    handle  - サウンドハンドル
                soundId - サウンドＩＤ
                startInfo - 再生パラメータ

  Returns:      結果コード
 *--------------------------------------------------------------------------------*/
SoundStartable::StartResult SoundStartable::StartSound(
    SoundHandle* handle,
    uint32_t soundId,
    const char* soundArchiveName,
    const StartInfo* startInfo
) NN_NOEXCEPT
{
    NN_ATK_LOG_APICALL_ATK("StartSound [ID:%08x]", soundId);

    StartResult result = detail_SetupSound(
        handle,
        soundId,
        false,
        soundArchiveName,
        startInfo
    );
    if ( ! result.IsSuccess() ) return result;
    handle->StartPrepared();
    StartResult result2( StartResult::ResultCode_Success );
    return result2;
}

SoundStartable::StartResult SoundStartable::StartSound(
    SoundHandle* handle,
    uint32_t soundId,
    const StartInfo* startInfo
) NN_NOEXCEPT
{
    return StartSound(handle, soundId, nullptr, startInfo);
}

/*--------------------------------------------------------------------------------*
  Name:         detail_StartSound

  Description:  指定したサウンド番号のサウンドを再生する

  Arguments:    handle  - サウンドハンドル
                soundName - サウンド名
                startInfo - 再生パラメータ

  Returns:      結果コード
 *--------------------------------------------------------------------------------*/
SoundStartable::StartResult SoundStartable::StartSound(
    SoundHandle* handle,
    const char* soundName,
    const char* soundArchiveName,
    const StartInfo* startInfo
) NN_NOEXCEPT
{
    uint32_t soundId = detail_GetItemId( soundName, soundArchiveName );
    NN_ATK_LOG_APICALL_ATK("StartSound [Name:%s ID:%08x]", soundName,soundId);

    if ( soundId == SoundArchive::InvalidId )
    {
        StartResult result( StartResult::ResultCode_ErrorInvalidLabelString );
        return result;
    }
    return StartSound(handle, soundId, soundArchiveName, startInfo);
}

SoundStartable::StartResult SoundStartable::StartSound(
    SoundHandle* handle,
    const char* soundName,
    const StartInfo* startInfo
) NN_NOEXCEPT
{
    return StartSound(handle, soundName, nullptr, startInfo);
}

/*--------------------------------------------------------------------------------*
  Name:         detail_HoldSound

  Description:  指定したサウンド番号のサウンドを再生する

  Arguments:    handle  - サウンドハンドル
                soundNo - サウンド番号
                startInfo - 再生パラメータ

  Returns:      結果コード
 *--------------------------------------------------------------------------------*/
SoundStartable::StartResult SoundStartable::HoldSound(
    SoundHandle* handle,
    uint32_t soundId,
    const char* soundArchiveName,
    const StartInfo* startInfo
) NN_NOEXCEPT
{
    if ( handle->IsAttachedSound() && ( handle->GetId() == soundId ) )
    {
        handle->detail_GetAttachedSound()->SetAutoStopCounter( 1 );
        if ( (startInfo != NULL) && (startInfo->enableFlag & StartInfo::EnableFlagBit_FadeFrame) )
        {
            handle->Stop( startInfo->fadeFrame );
        }
        StartResult result( StartResult::ResultCode_Success );
        return result;
    }

    StartResult result = detail_SetupSound(
        handle,
        soundId,
        true,
        soundArchiveName,
        startInfo
    );
    if ( ! result.IsSuccess() ) return result;
    handle->StartPrepared();
    handle->detail_GetAttachedSound()->SetAutoStopCounter( 1 );
    if ( (startInfo != NULL) && (startInfo->enableFlag & StartInfo::EnableFlagBit_FadeFrame) )
    {
        handle->Stop( startInfo->fadeFrame );
    }
    StartResult result2( StartResult::ResultCode_Success );
    return result2;
}

SoundStartable::StartResult SoundStartable::HoldSound(
    SoundHandle* handle,
    uint32_t soundId,
    const StartInfo* startInfo
) NN_NOEXCEPT
{
    return HoldSound( handle, soundId, nullptr, startInfo );
}

/*--------------------------------------------------------------------------------*
  Name:         detail_HoldSound

  Description:  指定したサウンド番号のサウンドを再生する

  Arguments:    handle  - サウンドハンドル
                soundLabel - サウンドラベル
                startInfo - 再生パラメータ

  Returns:      結果コード
 *--------------------------------------------------------------------------------*/
SoundStartable::StartResult SoundStartable::HoldSound(
    SoundHandle* handle,
    const char* soundLabel,
    const StartInfo* startInfo
) NN_NOEXCEPT
{
    return HoldSound( handle, soundLabel, nullptr, startInfo );
}

SoundStartable::StartResult SoundStartable::HoldSound(
    SoundHandle* handle,
    const char* soundLabel,
    const char* soundArchiveName,
    const StartInfo* startInfo
) NN_NOEXCEPT
{
    uint32_t soundId = detail_GetItemId( soundLabel, soundArchiveName );
    if ( soundId == SoundArchive::InvalidId )
    {
        StartResult result( StartResult::ResultCode_ErrorInvalidLabelString );
        return result;
    }
    return HoldSound( handle, soundId, soundArchiveName, startInfo );
}

/*--------------------------------------------------------------------------------*
  Name:         detail_PrepareSound

  Description:  指定したサウンド番号のサウンドを再生する

  Arguments:    handle  - サウンドハンドル
                soundNo - サウンド番号
                startInfo - 再生パラメータ

  Returns:      結果コード
 *--------------------------------------------------------------------------------*/
SoundStartable::StartResult SoundStartable::PrepareSound(
    SoundHandle* handle,
    uint32_t soundId,
    const char* soundArchiveName,
    const StartInfo* startInfo
) NN_NOEXCEPT
{
    return detail_SetupSound(
        handle,
        soundId,
        false,
        soundArchiveName,
        startInfo
    );
}

SoundStartable::StartResult SoundStartable::PrepareSound(
    SoundHandle* handle,
    uint32_t soundId,
    const StartInfo* startInfo
) NN_NOEXCEPT
{
    return PrepareSound(handle, soundId, nullptr, startInfo);
}

/*--------------------------------------------------------------------------------*
  Name:         detail_PrepareSound

  Description:  指定したサウンド番号のサウンドを再生する

  Arguments:    handle  - サウンドハンドル
                soundLabel - サウンドラベル
                startInfo - 再生パラメータ

  Returns:      結果コード
 *--------------------------------------------------------------------------------*/
SoundStartable::StartResult SoundStartable::PrepareSound(
    SoundHandle* handle,
    const char* soundName,
    const char* soundArchiveName,
    const StartInfo* startInfo
) NN_NOEXCEPT
{
    uint32_t soundId = detail_GetItemId( soundName, soundArchiveName );
    if ( soundId == SoundArchive::InvalidId )
    {
        StartResult result( StartResult::ResultCode_ErrorInvalidLabelString );
        return result;
    }
    return PrepareSound( handle, soundId, soundArchiveName, startInfo );
}

SoundStartable::StartResult SoundStartable::PrepareSound(
    SoundHandle* handle,
    const char* soundName,
    const StartInfo* startInfo
) NN_NOEXCEPT
{
    return PrepareSound( handle, soundName, nullptr, startInfo );
}

} // namespace nn::atk
} // namespace nn

