﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_Sound3DCalculator.h>

#include <nn/util/util_Vector.h>
#include <nn/atk/atk_Sound3DManager.h>      // nn::atk::Sound3DParam
#include <nn/atk/atk_Sound3DListener.h>

namespace nn {
namespace atk {

/* ------------------------------------------------------------------------
        inline function
   ------------------------------------------------------------------------ */
namespace {

// (x1,y1) (x2,y2) の2点を通る線形関数において、xに対応するyの値を求める
inline float SolveLinerFunction( float x, float x1, float x2, float y1, float y2 ) NN_NOEXCEPT
{
    if ( x1 == x2 ) return ( y1 + y2 ) / 2.0f;
    float divider = x1 - x2;
    return x * ( y1 - y2 ) / divider + ( x1 * y2 - x2 * y1 ) / divider;
}

}

/* ------------------------------------------------------------------------
        member function
   ------------------------------------------------------------------------ */

/*--------------------------------------------------------------------------------*
  Name:         CalcVolumeAndPriority

  Description:  3Dサウンドの音量と優先度を計算する

  Arguments:    manager - 3Dサウンドマネージャー
                listener - 3Dサウンドリスナー
                actorParam - 3Dサウンドアクターのパラメータ構造体
                pOutVolume - 計算した音量を格納するポインタ
                pOutPriority - 計算した優先度を格納するポインタ

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void Sound3DCalculator::CalculateVolumeAndPriority(
    float* pOutVolume,
    int* pOutPriority,
    const Sound3DManager& manager,
    const Sound3DListener& listener,
    const Sound3DParam& actorParam
) NN_NOEXCEPT
{
    //------------------------------------------------------------------
    // リスナーから見たアクターの位置を取得
    nn::util::Vector3fType position;
    nn::util::VectorSubtract( &position, actorParam.position, listener.GetPosition() );
    const float actorDistance = nn::util::VectorLength( position );

    CalculateVolumeAndPriority(
        pOutVolume,
        pOutPriority,
        manager,
        listener,
        actorParam,
        actorDistance );
}

void Sound3DCalculator::CalculateVolumeAndPriority(
    float* pOutVolume,
    int* pOutPriority,
    const Sound3DManager& manager,
    const Sound3DListener& listener,
    const Sound3DParam& actorParam,
    float  actorDistance
) NN_NOEXCEPT
{
    CalcVolumeAndPriorityImpl(
        actorDistance,
        static_cast<SoundArchive::Sound3DInfo::DecayCurve>(actorParam.decayCurve),
        actorParam.decayRatio,
        manager.GetMaxPriorityReduction(),
        listener.GetMaxVolumeDistance(),
        listener.GetUnitDistance(),
        pOutVolume,
        pOutPriority
    );
}

/*--------------------------------------------------------------------------------*
  Name:         CalcPan

  Description:  3Dサウンドのパンを計算する

  Arguments:    manager - 3Dサウンドマネージャー
                listener - 3Dサウンドリスナー
                actorParam - 3Dサウンドアクターのパラメータ構造体
                calcPanParam - パン計算に使用する構造体
                pOutPan - 計算したパンを格納するポインタ
                pOutSPan - 計算したサラウンドパンを格納するポインタ

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void Sound3DCalculator::CalculatePan(
    float* pOutPan,
    float* pOutSPan,
    const Sound3DManager& manager,
    const Sound3DListener& listener,
    const Sound3DParam& actorParam,
    const CalculatePanParam& calcPanParam
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutPan );
    NN_SDK_ASSERT_NOT_NULL( pOutSPan );

    //------------------------------------------------------------------
    // リスナーから見たアクターの位置を取得
    const nn::util::Matrix4x3fType& listenerMatrix = listener.GetMatrix();
    nn::util::Vector3fType position;
    nn::util::VectorTransform( &position, actorParam.position, listenerMatrix );
    const float actorDistance = nn::util::VectorLength( position );

    CalculatePanImpl(
        position,
        listener.GetInteriorSize(),
        actorDistance,
        manager.GetPanRange(),
        calcPanParam.stereoSpeakerAngle,
        calcPanParam.surroundSpeakerFrontAngle,
        calcPanParam.surroundSpeakerRearAngle,
        calcPanParam.surroundPanOffset,
        pOutPan,
        pOutSPan
    );
}

/*--------------------------------------------------------------------------------*
  Name:         CalcPitch

  Description:  3Dサウンドの音程を計算する

  Arguments:    manager - 3Dサウンドマネージャー
                listener - 3Dサウンドリスナー
                actorParam - 3Dサウンドアクターのパラメータ構造体
                pitchPtr - 計算した音程を格納するポインタ

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void Sound3DCalculator::CalculatePitch(
    float* pOutValue,
    const Sound3DManager& manager,
    const Sound3DListener& listener,
    const Sound3DParam& actorParam
) NN_NOEXCEPT
{
    //------------------------------------------------------------------
    // リスナーから見たアクターの位置を取得
    nn::util::Vector3fType actorPosition;
    nn::util::VectorSubtract( &actorPosition, actorParam.position, listener.GetPosition() );
    const float actorDistance = nn::util::VectorLength( actorPosition );

    CalculatePitch(
        pOutValue,
        manager,
        listener,
        actorParam,
        actorPosition,
        actorDistance );
}

void Sound3DCalculator::CalculatePitch(
    float* pOutValue,
    const Sound3DManager& manager,
    const Sound3DListener& listener,
    const Sound3DParam& actorParam,
    const nn::util::Vector3fType& actorPosition,
    float actorDistance
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutValue );

    nn::util::Vector3fType position = actorPosition;

    const float sonicVelocity = manager.GetSonicVelocity();
    if ( sonicVelocity == 0.0f )
    {
        *pOutValue = 1.0f;
        return;
    }

    if ( actorDistance > 0.0f )
    {
        nn::util::VectorMultiply(&position, position, 1.0f / actorDistance);
    }

    const float dopplerFactor = actorParam.dopplerFactor / 32.0f;

    float actorVelocity;
    float listenerVelocity;
    if ( actorDistance > 0.0f )
    {
        actorVelocity = - nn::util::VectorDot( position, actorParam.velocity );
        listenerVelocity = - nn::util::VectorDot( position, listener.GetVelocity() );
    }
    else
    {
        actorVelocity = - nn::util::VectorLength( actorParam.velocity );
        listenerVelocity = nn::util::VectorLength( listener.GetVelocity() );
    }
    actorVelocity *= dopplerFactor;
    listenerVelocity *= dopplerFactor;

    float pitch;
    if ( listenerVelocity > sonicVelocity ) {
        pitch = 0.0f;
    }
    else if ( actorVelocity >= sonicVelocity ) {
        pitch = 65535.0f;
    }
    else {
        pitch = ( sonicVelocity - listenerVelocity ) / ( sonicVelocity - actorVelocity );
    }

    *pOutValue = pitch;
}

/*--------------------------------------------------------------------------------*
  Name:         CalcBiquadValue

  Description:  Biquadフィルタの値を計算する

  Arguments:    manager - 3Dサウンドマネージャー
                listener - 3Dサウンドリスナー
                actorParam - 3Dサウンドアクターのパラメータ構造体
                pitchPtr - 計算したBiquadフィルタ値を格納するポインタ

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void Sound3DCalculator::CalculateBiquadFilterValue(
    float* pOutValue,
    const Sound3DManager& manager,
    const Sound3DListener& listener,
    const Sound3DParam& actorParam
) NN_NOEXCEPT
{
    //------------------------------------------------------------------
    // リスナーから見たアクターの位置を取得
    nn::util::Vector3fType position;
    nn::util::VectorSubtract( &position, actorParam.position, listener.GetPosition() );
    const float actorDistance = nn::util::VectorLength( position );

    CalculateBiquadFilterValue(
        pOutValue,
        manager,
        listener,
        actorParam,
        actorDistance );
}

void Sound3DCalculator::CalculateBiquadFilterValue(
    float* pOutValue,
    const Sound3DManager& manager,
    const Sound3DListener& listener,
    const Sound3DParam& actorParam,
    float actorDistance
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutValue );
    NN_UNUSED( manager );
    NN_UNUSED( actorParam );

    // Biquadフィルタの値はボリューム距離に追従
    float biquadFilterValue = 0.0f;
    float maxVolumeDistance = listener.GetMaxVolumeDistance();
    float maxBiquadFilterValue = listener.GetMaxBiquadFilterValue();

    if( actorDistance > maxVolumeDistance )
    {
        biquadFilterValue =
            ( actorDistance - maxVolumeDistance ) /
            listener.GetUnitDistance() *
            listener.GetUnitBiquadFilterValue();

        if ( biquadFilterValue > maxBiquadFilterValue )
        {
            biquadFilterValue = maxBiquadFilterValue;
        }
    }

    *pOutValue = biquadFilterValue;
}

void Sound3DCalculator::CalcVolumeAndPriorityImpl(
    float actorDistance,
    SoundArchive::Sound3DInfo::DecayCurve decayCurve,
    float decayRatio,
    int maxPriorityReduction,
    float maxVolumeDistance,
    float unitDistance,
    float* volumePtr,
    int* priorityPtr
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( volumePtr );
    NN_SDK_ASSERT_NOT_NULL( priorityPtr );

    static const float MaxVolume = 1.0f;
    float volume = MaxVolume;

    if ( actorDistance > maxVolumeDistance )
    {
        switch ( decayCurve )
        {
        case DecayCurve_Log:
            volume = std::powf( decayRatio, ( actorDistance - maxVolumeDistance ) / unitDistance );
            break;
        case DecayCurve_Linear:
            volume = 1.0f - ( actorDistance - maxVolumeDistance ) / unitDistance * ( 1.0f - decayRatio );
            if ( volume < 0.0f ) volume = 0.0f;
            break;
        default:
            break;
        }
    }

    *volumePtr = volume;
    *priorityPtr = - static_cast<int>( ( 1.0f - volume ) * maxPriorityReduction );
}

void Sound3DCalculator::CalculatePanImpl(
    const nn::util::Vector3fType& position,
    float interiorSize,
    float actorDistance,
    float panRange,
    float stereoSpeakerAngle,
    float surroundSpeakerFrontAngle,
    float surroundSpeakerRearAngle,
    float surroundPanOffset,
    float* panPtr,
    float* spanPtr
) NN_NOEXCEPT
{
#if 0
    switch ( SoundSystem::GetOutputMode() )
    {
#if 0 // TODO: 3D Surround
    case OutputMode_Dpl2:
    case OutputMode_Surround:
        CalcPanSurround(
            position,
            interiorSize,
            actorDistance,
            panRange,
            surroundSpeakerFrontAngle,
            surroundSpeakerRearAngle,
            surroundPanOffset,
            panPtr,
            spanPtr
        );
        break;
#endif
    case OutputMode_Stereo:
        CalcPanStereo(
            position,
            interiorSize,
            actorDistance,
            panRange,
            speakerAngleStereo,
            panPtr,
            spanPtr
        );
        break;
    case OutputMode_Monaural:
    default:
        *panPtr = 0.0f;
        *spanPtr = 0.0f;
        break;
    }
#else
    // TORIAEZU 必ずサラウンドで計算する
    NN_UNUSED(stereoSpeakerAngle);
    CalculatePanSurround( position, interiorSize, actorDistance, panRange,
            surroundSpeakerFrontAngle, surroundSpeakerRearAngle,
            surroundPanOffset, panPtr, spanPtr );
#endif
}

void Sound3DCalculator::CalculatePanSurround(
    const nn::util::Vector3fType& position,
    float interiorSize,
    float actorDistance,
    float panRange,
    float surroundSpeakerFrontAngle,
    float surroundSpeakerRearAngle,
    float surroundPanOffset,
    float* panPtr,
    float* surroundPanPtr
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( panPtr );
    NN_SDK_ASSERT_NOT_NULL( surroundPanPtr );

    // アクターの角度と距離を求める
    float angle;
    float distance;
    CalculateAngleAndDistance(
        position,
        actorDistance,
        interiorSize,
        &angle,
        &distance
    );

    NN_SDK_ASSERT( surroundSpeakerFrontAngle >= 0.0f && surroundSpeakerFrontAngle <= nn::util::FloatPi / 2.0f );
    NN_SDK_ASSERT( surroundSpeakerRearAngle >= nn::util::FloatPi / 2.0f && surroundSpeakerRearAngle <= nn::util::FloatPi );

    float x = 0.0f;
    float z = 0.0f;

    // スピーカーの角度をパン座標の角度に変換する
    const float angleRearLeft   = -surroundSpeakerRearAngle;
    const float angleFrontLeft  = -surroundSpeakerFrontAngle;
    const float angleFrontRight = surroundSpeakerFrontAngle;
    const float angleRearRight  = surroundSpeakerRearAngle;

    if ( angle < angleRearLeft )
    {
        x = SolveLinerFunction( angle, -nn::util::FloatPi, angleRearLeft, 0.0f, -1.0f );
        z = 1.0f;
    }
    else if ( angle < -nn::util::FloatPi / 2.0f )
    {
        x = -1.0f;
        z = SolveLinerFunction( angle, angleRearLeft, -nn::util::FloatPi / 2.0f, 1.0f, 0.0f );
    }
    else if ( angle < angleFrontLeft )
    {
        x = -1.0f;
        z = SolveLinerFunction( angle, -nn::util::FloatPi / 2.0f, angleFrontLeft, 0.0f, -1.0f );
    }
    else if ( angle < angleFrontRight )
    {
        x = SolveLinerFunction( angle, angleFrontLeft, angleFrontRight, -1.0f, 1.0f );
        z = -1.0f;
    }
    else if ( angle < nn::util::FloatPi / 2.0f )
    {
        x = 1.0f;
        z = SolveLinerFunction( angle, angleFrontRight, nn::util::FloatPi / 2.0f, -1.0f, 0.0f );
    }
    else if ( angle < angleRearRight )
    {
        x = 1.0f;
        z = SolveLinerFunction( angle, nn::util::FloatPi / 2.0f, angleRearRight, 0.0f, 1.0f );
    }
    else
    {
        x = SolveLinerFunction( angle, angleRearRight, nn::util::FloatPi, 1.0f, 0.0f );
        z = 1.0f;
    }

    // 原点の補正
    float a = ( std::cosf( surroundSpeakerFrontAngle ) + std::cosf( surroundSpeakerRearAngle ) ) / 2.0f;
    float speakerOffset = a / ( a + ( -std::cosf( surroundSpeakerRearAngle ) ) );

    const float center_x = 0.0f;
    const float center_z = speakerOffset;

    // パンを設定する
    x *= panRange;
    z *= panRange;
    *panPtr = x * distance + center_x * ( 1.0f - distance );
    *surroundPanPtr = z * distance + center_z * ( 1.0f - distance ) + 1.0f + surroundPanOffset;
}

#if 0 // 不要かも
void Sound3DCalculator::CalcPanStereo(
    const nn::util::Vector3fType& position,
    float interiorSize,
    float actorDistance,
    float panRange,
    float speakerAngleStereo,
    float* panPtr,
    float* spanPtr
)
{
    NN_SDK_ASSERT_NOT_NULL( panPtr );
    // NN_SDK_ASSERT_NOT_NULL( spanPtr );    // TODO:
    NN_UNUSED( spanPtr );

    // アクターの角度と距離を求める
    float angle;
    float distance;
    CalcAngleAndDistance(
        position,
        actorDistance,
        interiorSize,
        &angle,
        &distance
    );

    NN_SDK_ASSERT( speakerAngleStereo >= 0.0f && speakerAngleStereo <= nn::util::FloatPi / 2.0f );

    float x = 0.0f;

    const float angleRearLeft   = - nn::util::FloatPi + speakerAngleStereo;
    const float angleFrontLeft  = -speakerAngleStereo;
    const float angleFrontRight = speakerAngleStereo;
    const float angleRearRight  = nn::util::FloatPi - speakerAngleStereo;

    if ( angle < angleRearLeft )
    {
        x = SolveLinerFunction( angle, -nn::util::FloatPi, angleRearLeft, 0.0f, -1.0f );
    }
    else if ( angle < angleFrontLeft )
    {
        x = -1.0f;
    }
    else if ( angle < angleFrontRight )
    {
        x = SolveLinerFunction( angle, angleFrontLeft, angleFrontRight, -1.0f, 1.0f );
    }
    else if ( angle < angleRearRight )
    {
        x = 1.0f;
    }
    else
    {
        x = SolveLinerFunction( angle, angleRearRight, nn::util::FloatPi, 1.0f, 0.0f );
    }

    // パンを設定する
    x *= panRange;
    *panPtr = x * distance;
    // *spanPtr = 0.0f;     // TODO:
}
#endif

void Sound3DCalculator::CalculateAngleAndDistance(
    const nn::util::Vector3fType& position,
    float actorDistance,
    float interiorSize,
    float* anglePtr,
    float* distancePtr
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( anglePtr );
    NN_SDK_ASSERT_NOT_NULL( distancePtr );

    nn::util::Vector3fType interiorPosition;

    if ( actorDistance == 0.0f )
    {
        nn::util::VectorZero(&interiorPosition);
    }
    else
    {
        nn::util::Vector3fType yPlanePoint = position;
        nn::util::VectorSetY( &yPlanePoint, 0.0f );
        float d = nn::util::VectorLength( yPlanePoint );
        if ( d > interiorSize )
        {
            // 音源がインテリアサイズを超えた場合はインテリアサイズ内に音源を置く
            nn::util::VectorSetX(&yPlanePoint, nn::util::VectorGetX(yPlanePoint) * (interiorSize / d));
            nn::util::VectorSetZ(&yPlanePoint, nn::util::VectorGetZ(yPlanePoint) * (interiorSize / d));
        }

        float yPlaneDistance = nn::util::VectorLength( yPlanePoint );
        nn::util::VectorSetX(&interiorPosition, nn::util::VectorGetX(position) * yPlaneDistance / actorDistance);
        nn::util::VectorSetZ(&interiorPosition, nn::util::VectorGetZ(position) * yPlaneDistance / actorDistance);
        nn::util::VectorSetY(&interiorPosition, 0.0f);
    }

    *anglePtr = std::atan2f( nn::util::VectorGetX(interiorPosition), -nn::util::VectorGetZ(interiorPosition) );
    *distancePtr = nn::util::VectorLength( interiorPosition ) / interiorSize;
}

} // namespace nn::atk
} // namespace nn
