﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_Sound3DActor.h>

#include <nn/atk/atk_SoundArchivePlayer.h>
#include <nn/atk/atk_Sound3DManager.h>
#include <nn/atk/atk_SoundHandle.h>
#include <nn/atk/atk_Global.h>          // nn::atk::DecayCurve_Log
#include <nn/util/util_Vector.h>

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT( const size_t Sound3DActor::BufferAlignSize );

/*--------------------------------------------------------------------------------*
  Name:         Sound3DActor

  Description:  コンストラクタ

  Arguments:    player - サウンドアーカイブプレイヤー
                manager - ３Ｄサウンドマネージャー

  Returns:      None.
 *--------------------------------------------------------------------------------*/
Sound3DActor::Sound3DActor( SoundArchivePlayer* pPlayer, Sound3DManager* pManager ) NN_NOEXCEPT
: m_p3dManager( NULL )
, m_pArchivePlayer( NULL )
, m_UserParam( 0 )
, m_ResetPositionFlag( true )
, m_IsInitialized( false )
, m_IsFinalized( true )
{
    NN_ABORT_UNLESS_ALIGNED( &m_Position, Sound3DManager::NnUtilMathTypeAlignSize );
    NN_ABORT_UNLESS_ALIGNED( &m_Velocity, Sound3DManager::NnUtilMathTypeAlignSize );

    nn::util::VectorZero(&m_Position);
    nn::util::VectorZero(&m_Velocity);
    Initialize( pPlayer, pManager );
}

Sound3DActor::Sound3DActor( SoundArchivePlayer* pPlayer, Sound3DManager* pManager, ActorPlayer* pActorPlayer ) NN_NOEXCEPT
: m_p3dManager( NULL )
, m_pArchivePlayer( NULL )
, m_UserParam( 0 )
, m_ResetPositionFlag( true )
, m_IsInitialized( false )
, m_IsFinalized( true )
{
    NN_ABORT_UNLESS_ALIGNED( &m_Position, Sound3DManager::NnUtilMathTypeAlignSize );
    NN_ABORT_UNLESS_ALIGNED( &m_Velocity, Sound3DManager::NnUtilMathTypeAlignSize );

    nn::util::VectorZero(&m_Position);
    nn::util::VectorZero(&m_Velocity);
    Initialize( pPlayer, pManager, pActorPlayer );
}

Sound3DActor::Sound3DActor() NN_NOEXCEPT
: m_p3dManager( NULL )
, m_pArchivePlayer( NULL )
, m_UserParam( 0 )
, m_ResetPositionFlag( true )
, m_IsInitialized( false )
, m_IsFinalized( true )
{
    NN_ABORT_UNLESS_ALIGNED( &m_Position, Sound3DManager::NnUtilMathTypeAlignSize );
    NN_ABORT_UNLESS_ALIGNED( &m_Velocity, Sound3DManager::NnUtilMathTypeAlignSize );

    nn::util::VectorZero(&m_Position);
    nn::util::VectorZero(&m_Velocity);
}

/*--------------------------------------------------------------------------------*
  Name:         ~Sound3DActor

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
Sound3DActor::~Sound3DActor() NN_NOEXCEPT
{
    Finalize();
}

void Sound3DActor::Initialize( SoundArchivePlayer* pPlayer, Sound3DManager* pManager ) NN_NOEXCEPT
{
    Initialize( pPlayer, pManager, NULL );
}

void Sound3DActor::Initialize( SoundArchivePlayer* pPlayer, Sound3DManager* pManager, ActorPlayer* pActorPlayer ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pPlayer);
    NN_SDK_ASSERT_NOT_NULL(pManager);

    if ( m_IsInitialized ) return;

    if ( pActorPlayer == NULL )
    {
        SoundActor::Initialize( pPlayer );
    }
    else
    {
        SoundActor::Initialize( pPlayer, pActorPlayer );
    }

    m_p3dManager = pManager;
    m_pArchivePlayer = pPlayer;

    m_IsInitialized = true;
    m_IsFinalized = false;
}

void Sound3DActor::Finalize() NN_NOEXCEPT
{
    if ( m_IsFinalized ) return;

    // サウンドからの Update が来ないようにする
    ClearUpdateCallback clearUpdateCallback( this );
    SoundActor::ForEachSound( clearUpdateCallback );
    SoundActor::Finalize();

    m_IsFinalized = true;
    m_IsInitialized = false;
    m_p3dManager = NULL;
    m_pArchivePlayer = NULL;
}

/*--------------------------------------------------------------------------------*
  Name:         SetupSound [virtual]

  Description:  再生の実装関数

  Arguments:    handle  - サウンドハンドル
                soundId - サウンドＩＤ
                startInfo - サウンド再生パラメータ
                setupArg - セットアップ引数

  Returns:      結果コード
 *--------------------------------------------------------------------------------*/
SoundStartable::StartResult Sound3DActor::SetupSound(
    SoundHandle* handle,
    uint32_t soundId,
    const StartInfo* startInfo,
    void* setupArg
) NN_NOEXCEPT
{
    return Sound3DActor::SetupSound(
        handle,
        soundId,
        nullptr,
        startInfo,
        setupArg
    );
}

SoundStartable::StartResult Sound3DActor::SetupSound(
    SoundHandle* handle,
    uint32_t soundId,
    const char* soundArchiveName,
    const StartInfo* startInfo,
    void* setupArg
) NN_NOEXCEPT
{
    if ( m_IsInitialized == false )
    {
        return StartResult( StartResult::ResultCode_ErrorActorNotInitialized );
    }

    // 初期パラメータ設定
    Sound3DParam param;
    param.position = m_Position;
    param.velocity = m_Velocity;
    param.actorUserParam = m_UserParam;
    if ( m_pArchivePlayer != NULL )
    {
        const SoundArchive* pSoundArchive = (soundArchiveName == nullptr)
            ? &m_pArchivePlayer->GetSoundArchive()
            : m_pArchivePlayer->GetAddonSoundArchive(soundArchiveName);
        NN_SDK_ASSERT_NOT_NULL(pSoundArchive);

        SoundArchive::Sound3DInfo p;
        if ( pSoundArchive->ReadSound3DInfo( &p, soundId ) )
        {
            param.controlFlag = p.flags;
            param.decayRatio = p.decayRatio;
            param.dopplerFactor = p.dopplerFactor;

            // decayCurve
            switch ( p.decayCurve )
            {
            case SoundArchive::Sound3DInfo::DecayCurve_Log:
                param.decayCurve = nn::atk::DecayCurve_Log;
                break;
            case SoundArchive::Sound3DInfo::DecayCurve_Linear:
                param.decayCurve = nn::atk::DecayCurve_Linear;
                break;
            default:
                param.decayCurve = nn::atk::DecayCurve_Log;
                break;
            }
        }
        param.soundUserParam = pSoundArchive->GetSoundUserParam( soundId );
    }

    detail::BasicSound::AmbientInfo argInfo = {
        m_p3dManager,
        this,
        m_p3dManager,
        &param,
        sizeof( param )
    };

    SoundStartable::StartResult result = SoundActor::detail_SetupSoundWithAmbientInfo(
        handle,
        soundId,
        soundArchiveName,
        startInfo,
        &argInfo,
        setupArg
    );

    if ( handle->IsAttachedSound() )
    {
        handle->detail_GetAttachedSound()->SetPanCurve( PanCurve_Sincos );
    }

    return result;
}

/*--------------------------------------------------------------------------------*
  Name:         SetPosition

  Description:  アクターの位置を設定する

  Arguments:    position - 位置を表す３次元ベクトル

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void Sound3DActor::SetPosition( const nn::util::Vector3fType& position ) NN_NOEXCEPT
{
    if ( ! m_ResetPositionFlag )
    {
        nn::util::VectorSubtract( &m_Velocity, position, m_Position );
    }
    m_Position = position;
    m_ResetPositionFlag = false;
}

/*--------------------------------------------------------------------------------*
  Name:         ResetPosition

  Description:  アクターの位置情報をリセットする

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void Sound3DActor::ResetPosition() NN_NOEXCEPT
{
    m_ResetPositionFlag = true;
    nn::util::VectorZero(&m_Position);
    nn::util::VectorZero(&m_Velocity);
}

/*--------------------------------------------------------------------------------*
  Name:         SetVelocity

  Description:  アクターの速度を設定する

  Arguments:    velocity - 速度を表す３次元ベクトル

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void Sound3DActor::SetVelocity( const nn::util::Vector3fType& velocity ) NN_NOEXCEPT
{
    m_Velocity = velocity;
}

/*--------------------------------------------------------------------------------*
  Name:         Update

  Description:  アクターのパラメータを更新する

  Arguments:    arg - 引数
                sound - サウンドインスタンス

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void Sound3DActor::detail_UpdateAmbientArg( void* arg, const detail::BasicSound* sound ) NN_NOEXCEPT
{
    (void)sound;

    Sound3DParam* param = static_cast< Sound3DParam* >( arg );
    param->position = m_Position;
    param->velocity = m_Velocity;
    param->actorUserParam = m_UserParam;
}

Sound3DActor::ClearUpdateCallback::ClearUpdateCallback(const SoundActor* pSoundActor) NN_NOEXCEPT
    : m_pSoundActor(pSoundActor)
{
}
/*--------------------------------------------------------------------------------*
  Name:         ClearUpdateCallback

  Description:  アンビエント引数の更新コールバックのクリア

  Arguments:    handle - クリア対象のサウンドハンドル

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void Sound3DActor::ClearUpdateCallback::operator() ( SoundHandle& handle ) NN_NOEXCEPT
{
    if ( handle.IsAttachedSound() && handle.detail_GetAttachedSound()->GetSoundActor() == m_pSoundActor)
    {
        handle.detail_GetAttachedSound()->ClearAmbientArgUpdateCallback();
    }
}

} // namespace nn::atk
} // namespace nn
