﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_ThreadCommon.h>

#include <nn/font/font_ScalableFont.h>
#include <nn/font/font_TextureCache.h>

#include <nn/ui2d/viewer/ui2d_Config.h>
#if defined(NN_UI2D_VIEWER_ENABLED)

#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

#include <nn/ui2d.h>
#include <nn/font.h>

#include <nn/ui2d/viewer/ui2d_AnimationManager2.h>
#include <nn/ui2d/viewer/ui2d_ScalableFontManager.h>
#include <nn/ui2d/viewer/ui2d_Screen.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <windows.h>
#endif

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/ui2d/viewer/win/ui2d_DirResourceAccessorWin.h>
#include <nn/ui2d/viewer/win/ui2d_ScreenShotWin.h>
#else
#include <nn/ui2d/viewer/ui2d_FindableArcResourceAccessor.h>
#endif

#include <cstdarg>


#include <nn/ui2d/viewer/ui2d_Viewer.h>

namespace nn
{
namespace ui2d
{
namespace viewer
{


//! @brief アニメーションの一覧を保持します。
//!
typedef nn::util::IntrusiveList<Screen, nn::util::IntrusiveListMemberNodeTraits<Screen, &Screen::m_Link> > ScreenList;

class ViewerScreen;
//---------------------------------------------------------------------------
// リソース管理・スクリーンの生成・管理
// アプリケーション作成の基盤として使えるように、各種拡張ポイントの提供（もちろん標準ビューアーも作れる）
// Screen と ScreenManager は nn::ui2d 以下に引っ越します（libnn_ui2d で提供されるもの）
class ScreenManager
{
public:
    //---------------------------------------------------------------------------
    ScreenManager();
    virtual ~ScreenManager();

    //--------------------------------------------------------------------------------------------------
    void AttatchArchive(const char* pArcPath)
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        m_ResourceAccessor.Attach(pArcPath);
#else
        NN_SDK_ASSERT(false);
        NN_UNUSED(pArcPath);
#endif
    }

    //--------------------------------------------------------------------------------------------------
    void AttatchArchive(void* pArcData, const char* pPath)
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        NN_SDK_ASSERT(false);
        NN_UNUSED(pArcData);
        NN_UNUSED(pPath);
#else
        {
            if (!m_ResourceAccessor.Attach(pArcData, pPath))
            {
                NN_SDK_ASSERT(false, "can not attach layout archive.\n");
            }
        }
#endif
    }

    //---------------------------------------------------------------------------
    void* DetachArchive(nn::gfx::Device* pDevice)
    {
        void* pArcData = NULL;

        // 既にアタッチされていたらDetach
        if (m_ResourceAccessor.IsAttached())
        {
#if defined(NN_BUILD_CONFIG_OS_WIN)
#else
            pArcData = const_cast<void*>(m_ResourceAccessor.GetArchiveDataStart());
#endif
            m_ResourceAccessor.UnregisterTextureViewFromDescriptorPool(m_UnregisterTextureViewSlot, m_pUserDataForDescriptorSlotAllocator);
            m_ResourceAccessor.Detach();
            m_ResourceAccessor.Finalize(pDevice);
        }

        return pArcData;
    }

    //---------------------------------------------------------------------------
    nn::ui2d::viewer::Screen* CreateScreen();

    //---------------------------------------------------------------------------
    nn::ui2d::viewer::Screen* CreateScreenAnimMode(
        nn::gfx::Device* pDevice, const void* lytRes, const char* pName,
        const nn::ui2d::Layout::BuildOption& buildOption,
        BuildResultInformation* pBuildResultInformation);

    //---------------------------------------------------------------------------
    nn::ui2d::viewer::Screen* CreateScreen(nn::gfx::Device* pDevice, const void* lytRes, const char* pName,
        BuildResultInformation* pBuildResultInformation);

    //---------------------------------------------------------------------------
    void DoRegisterResourceTextureView()
    {
        m_ResourceAccessor.RegisterTextureViewToDescriptorPool(m_RegisterTextureViewSlot, m_pUserDataForDescriptorSlotAllocator);
    }

    ScreenList& GetScreenList()
    {
        return m_ScreenList;
    }

    const ScreenList& GetScreenList() const
    {
        return m_ScreenList;
    }

    //---------------------------------------------------------------------------
    // TODO:Viewer専用派生クラスに移動
    void InitializeComplexFontSet(nn::gfx::Device* pDevice)
    {
        // スケーラブルフォント関連の初期化
        {
            m_ScalableFontManager.InitializeComplexFontSet(
                pDevice, m_ResourceAccessor, m_RegisterTextureViewSlot, m_pUserDataForDescriptorSlotAllocator);

            // リソースとして登録
            for (int i = 0; i < m_ScalableFontManager.GetComplexFontCount(); i++)
            {
                auto pCpxFont = m_ScalableFontManager.GetComplexFontByIndex(i);
                m_ResourceAccessor.RegisterFont(pCpxFont->name, pCpxFont->pComplexFontTree);
            }
        }
    }

    //---------------------------------------------------------------------------
    // TODO:Viewer専用派生クラスに移動
    // スケーラブルフォントのテクスチャキャッシュ更新
    void RegisterAndCompleteAllGlyphsForScalableFontTextureCache(nn::ui2d::viewer::Screen* pScreen);

    //---------------------------------------------------------------------------
    ScalableFontManager                     m_ScalableFontManager; // TODO:Viewer専用派生クラスに移動
    nn::ui2d::GraphicsResource*             pGraphicsResource;

    // TODO:リソースアクセサのマネージャー（もっと高機能でインタフェースが整ったクラスを用意する）
#if defined(NN_BUILD_CONFIG_OS_WIN)
    DirResourceAccessor&                    m_ResourceAccessor;
#else
    FindableArcResourceAccessor             m_ResourceAccessor;
#endif

    RegisterTextureViewSlot                 m_RegisterTextureViewSlot;
    UnregisterTextureViewSlot               m_UnregisterTextureViewSlot;
    void*                                   m_pUserDataForDescriptorSlotAllocator;

    // スクリーンのリスト
    ScreenList m_ScreenList;

    // TODO:コンスタントバッファ
};

} // namespace viewer
} // namespace ui2d
} // namespace nn

#endif // NN_UI2D_VIEWER_ENABLED
