﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_ThreadCommon.h>

#include <nn/font/font_ScalableFont.h>
#include <nn/font/font_TextureCache.h>

#include <nn/ui2d/viewer/ui2d_Config.h>
#if defined(NN_UI2D_VIEWER_ENABLED)

#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

#include <nn/ui2d.h>
#include <nn/font.h>

#include <nn/ui2d/viewer/ui2d_AnimationManager2.h>
#include <nn/ui2d/viewer/ui2d_ScalableFontManager.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <windows.h>
#endif

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/ui2d/viewer/win/ui2d_DirResourceAccessorWin.h>
#include <nn/ui2d/viewer/win/ui2d_ScreenShotWin.h>
#else
#include <nn/ui2d/viewer/ui2d_FindableArcResourceAccessor.h>
#endif

#include <nn/ui2d/ui2d_StateMachine.h>

#include <cstdarg>


#include <nn/ui2d/viewer/ui2d_Viewer.h>

namespace nn
{
namespace ui2d
{
namespace viewer
{

class ViewerScreen;
class ScreenManager;

//---------------------------------------------------------------------------
// スクリーンが満たすべき、基本的な特性を決める基底クラスです。
class ScreenBase
{
public:
    virtual void UpdateLayout(int constantBufferIndex) = 0;
    virtual void DrawCaptureTexture(nn::gfx::Device* pDevice, nn::gfx::CommandBuffer& commandBuffer) = 0;
    virtual void DrawLayout(nn::gfx::CommandBuffer& commandBuffer) = 0;
};

//---------------------------------------------------------------------------
// 一つのボディレイアウトのすべてを管理する(アニメーション含め)
class Screen : public ScreenBase
{
public:
    //--------------------------------------------------------------------------------------------------
    struct AnimationPlayingState
    {
        bool isAnimationPlayingOld = true;
        float animationFrameOld = 0;
        char animationNameOld[nn::ui2d::AnimTagNameStrMax + 1] = {};
    };

public:
    Screen();

    //---------------------------------------------------------------------------
    void Initialize(ScreenManager* pScreenManager);

    //---------------------------------------------------------------------------
    nn::ui2d::Size GetViewSize() const
    {
        if (m_Layout != NULL)
        {
            return m_Layout->GetLayoutSize();
        }
        else
        {
            return nn::ui2d::Size::Create(1920.f, 1080.f);
        }
    }

    //---------------------------------------------------------------------------
    nn::ui2d::Layout* GetLayout()
    {
        return m_Layout;
    }

    const nn::ui2d::Layout* GetLayout() const
    {
        return m_Layout;
    }

    nn::ui2d::DrawInfo* GetDrawInfo()
    {
        return &m_DrawInfo;
    }

    const nn::ui2d::DrawInfo* GetDrawInfo() const
    {
        return &m_DrawInfo;
    }

    nn::ui2d::viewer::ScreenManager* GetScreenManager()
    {
        return m_pScreenManager;
    }

    const nn::ui2d::viewer::ScreenManager* GetScreenManager() const
    {
        return m_pScreenManager;
    }

    void SetIsPerspectiveProjection(bool isPerspectiveProjection)
    {
        m_IsPerspectiveProjection = isPerspectiveProjection;
    }

    void SetPerspectiveFovyDeg(float perspectiveFovyDeg)
    {
        m_PerspectiveFovyDeg = perspectiveFovyDeg;
    }

    //----------------------------------------------
    void SetPerspectiveNear(float perspectiveNear)
    {
        m_PerspectiveNear = perspectiveNear;
    }

    //----------------------------------------------
    void SetPerspectiveFar(float perspectiveFar)
    {
        m_PerspectiveFar = perspectiveFar;
    }

    //----------------------------------------------
    nn::ui2d::ControlCreator* GetControlCreator() const
    {
        return m_pControlCreator;
    }

    //----------------------------------------------
    Screen& SetControlCreator(nn::ui2d::ControlCreator* pControlCreator)
    {
        m_pControlCreator = pControlCreator;
        return *this;
    }

    //-------------------------------------------------------------------------
    nn::ui2d::ButtonGroup& GetButtonGroup()
    {
        return m_ButtonGroup;
    }

    nn::ui2d::ControlList& GetControlList()
    {
        return m_ControlList;
    }

    //-------------------------------------------------------------------------
    nn::ui2d::viewer::AnimationManager2& GetAnimationManager()
    {
        return m_AnimationMgr;
    }

    const nn::ui2d::ButtonGroup& GetButtonGroup() const
    {
        return m_ButtonGroup;
    }

    const nn::ui2d::ControlList& GetControlList() const
    {
        return m_ControlList;
    }

    //---------------------------------------------------------------------------
    virtual void UpdateLayout(int constantBufferIndex)
    {
        if (m_Layout == nullptr)
        {
            return;
        }

        UpdateControl_();

        m_DrawInfo.Map(constantBufferIndex);

        m_Layout->AnimateAndUpdateAnimFrame();
        m_DrawInfo.SetProjectionMtx(m_ProjectionMatrix);
        m_DrawInfo.SetViewMtx(m_CameraViewMatrix);
        m_Layout->Calculate(m_DrawInfo, m_IsViewMtxDirty);

        OnPostCalculate(m_Layout);

        m_AnimateOnce = true;
        m_IsViewMtxDirty = false;

        m_DrawInfo.Unmap();

        m_DrawInfo.SetGpuAccessBufferIndex(constantBufferIndex);
    }

    //---------------------------------------------------------------------------
    void UpdateControl_()
    {
        float step = GetAnimationStep();
        for (auto it = m_ControlList.begin(), it_end = m_ControlList.end(); it != it_end; ++it) {
            it->Update(step);
        }
    }

    void TestUpdateControlGroupInput(const nn::util::Float2* pPos, bool isDown, bool isRelease /* = false */)
    {
        for (auto it = m_ControlList.begin(), it_end = m_ControlList.end(); it != it_end; ++it) {
            it->UpdateUserInput(pPos, isDown, isRelease);
        }
    }

    //---------------------------------------------------------------------------
    float GetAnimationStep()
    {
        return 1.0f;
    }

    //---------------------------------------------------------------------------
    virtual void DrawCaptureTexture(nn::gfx::Device* pDevice, nn::gfx::CommandBuffer& commandBuffer)
    {
        if (m_Layout == nullptr)
        {
            return;
        }

        m_DrawInfo.SetProjectionMtx(m_ProjectionMatrix);
        m_DrawInfo.SetViewMtx(m_CameraViewMatrix);

        m_Layout->DrawCaptureTexture(pDevice, m_DrawInfo, commandBuffer);
    }

    //--------------------------------------------------------------------------------------------------
    virtual void DrawLayout(nn::gfx::CommandBuffer& commandBuffer)
    {
        if (m_Layout == nullptr)
        {
            return;
        }

        if (!m_AnimateOnce)
        {
            return;
        }

        m_DrawInfo.SetProjectionMtx(m_ProjectionMatrix);
        m_DrawInfo.SetViewMtx(m_CameraViewMatrix);

        m_Layout->Draw(m_DrawInfo, commandBuffer);
    }

    //--------------------------------------------------------------------------------------------------

    void FinalizeResources(nn::gfx::Device* pDevice)
    {
        // レイアウトリソースの解放
        if (m_Layout == nullptr)
        {
            return;
        }

        m_ButtonGroup.FreeAll();

        // m_ControlList
        {
            for (ControlList::iterator it = m_ControlList.begin(); it != m_ControlList.end(); )
            {
                auto currIter = it;
                it++;

                m_ControlList.erase(currIter);

                currIter->Finalize(pDevice);
                ui2d::Layout::DeleteObj(&(*currIter));
            }
        }

        m_AnimationMgr.Finalize();
        m_Layout->Finalize(pDevice);
        nn::ui2d::Layout::DeleteObj(m_Layout);
        m_Layout = NULL;
    }

    //--------------------------------------------------------------------------------------------------
    bool GetAnimationPlayingState(AnimationPlayingState* pState) const
    {
        if(m_Layout == nullptr)
        {
            return false;
        }

        if (m_AnimationMgr.GetAnimationFrameMax() == 0)
        {
            return false;
        }

        pState->isAnimationPlayingOld = m_AnimationMgr.IsAnimationPlaying();
        pState->animationFrameOld = m_AnimationMgr.GetCurrentAnimationFrame();
        if (m_AnimationMgr.GetCurrentAnimationNo() > 0)
        {
            nn::util::Strlcpy(pState->animationNameOld, m_AnimationMgr.GetCurrentAnimationName(), nn::ui2d::AnimTagNameStrMax + 1);
        }

        return true;
    }

    //--------------------------------------------------------------------------------------------------
    void SetAnimationPlayingState(nn::gfx::Device* pDevice, const AnimationPlayingState& state)
    {
        // 以前の状態を引き継ぐ
        int animationNo = m_AnimationMgr.GetAnimationNoByTagName(state.animationNameOld);
        if (animationNo < m_AnimationMgr.GetAnimationCount())
        {
            m_AnimationMgr.SetCurrentAnimationNo(pDevice, animationNo);
        }

        m_AnimationMgr.SetCurrentAnimationFrame(state.animationFrameOld);
        if (state.isAnimationPlayingOld)
        {
            m_AnimationMgr.StartAnimation();
        }
    }

    //--------------------------------------------------------------------------------------------------
    void BuildLayout(
        nn::gfx::Device* pDevice,
        const void* lytRes,
        nn::ui2d::ResourceAccessor* pResAcsr,
        nn::ui2d::IFindableResourceAccessor* pFindableResAcsr,
        const nn::ui2d::Layout::BuildOption* pBuildOption,
        BuildResultInformation* pBuildResultInformation)
    {
        m_AnimateOnce = false;

        NN_SDK_ASSERT(m_Layout == NULL);
        m_Layout = nn::ui2d::Layout::AllocateAndConstruct<nn::ui2d::Layout>();

        {

            m_Layout->Build(pBuildResultInformation, pDevice, pResAcsr, NULL, NULL, lytRes, *pBuildOption);
            m_AnimationMgr.Setup(pDevice, m_Layout, pFindableResAcsr);
        }

        nn::ui2d::Size size = m_Layout->GetLayoutSize();
        SetViewSize(size, size);
        m_Layout->ResetFirstFrameCaptureUpdatdFlag();
    }

    //--------------------------------------------------------------------------------------------------
    void BuildLayout(nn::gfx::Device* pDevice, const void* lytRes, nn::ui2d::ResourceAccessor* pResAcsr, BuildResultInformation* pBuildResultInformation)
    {
        m_AnimateOnce = false;

        NN_SDK_ASSERT(m_Layout == NULL);
        m_Layout = nn::ui2d::Layout::AllocateAndConstruct<nn::ui2d::Layout>();

        {
            if (m_pControlCreator == NULL)
            {
                nn::ui2d::DefaultControlCreator controlCreater(&m_ButtonGroup);
                m_Layout->Build(pBuildResultInformation, pDevice, pResAcsr, &controlCreater, lytRes);
            }
            else
            {
                m_Layout->Build(pBuildResultInformation, pDevice, pResAcsr, m_pControlCreator, lytRes);
            }
        }

        nn::ui2d::Size size = m_Layout->GetLayoutSize();
        SetViewSize(size, size);

        m_Layout->ResetFirstFrameCaptureUpdatdFlag();
    }

    //--------------------------------------------------------------------------------------------------
    void SetViewSize(nn::ui2d::Size& targetViewSize, nn::ui2d::Size& mainViewportSize)
    {
        m_IsViewMtxDirty = true;

        // drawInfoへの行列の設定
        if (m_Layout == nullptr)
        {
            return;
        }

        nn::ui2d::Size viewSizeValue = targetViewSize;
        nn::ui2d::Size layoutSize = m_Layout->GetLayoutSize();
        float widthRate = layoutSize.width / viewSizeValue.width;
        float heightRate = layoutSize.height / viewSizeValue.height;
        float halfWidth = mainViewportSize.width / 2.f;
        float halfHeight = mainViewportSize.height / 2.f;

        if (m_IsPerspectiveProjection)
        {
            float fovy = nn::util::DegreeToRadian(m_PerspectiveFovyDeg);

            float height = halfHeight * heightRate / nn::util::TanTable(nn::util::RadianToAngleIndex(fovy / 2.f));
            float aspect = (halfWidth * widthRate) / (halfHeight * heightRate);

            nn::util::MatrixPerspectiveFieldOfViewRightHanded(&m_ProjectionMatrix, fovy, aspect, m_PerspectiveNear, m_PerspectiveFar);

            nn::util::Vector3fType pos;
            nn::util::VectorSet(&pos, 0.f, 0.f, height);
            nn::util::Vector3fType up;
            nn::util::VectorSet(&up, 0.f, 1.f, 0.f);
            nn::util::Vector3fType target;
            nn::util::VectorSet(&target, 0.f, 0.f, 0.f);

            nn::util::MatrixLookAtRightHanded(&m_CameraViewMatrix, pos, target, up);
        }
        else
        {
            float left = -halfWidth * widthRate;
            float right = halfWidth * widthRate;
            float top = halfHeight * heightRate;
            float bottom = -halfHeight * heightRate;
            float nnear = 0.0f;
            float ffar = 500.0f;

            nn::util::MatrixOrthographicOffCenterRightHanded(&m_ProjectionMatrix, left, right, bottom, top, nnear, ffar);

            float cx = (left + right) * 0.5f;
            float cy = (top + bottom) * 0.5f;
            nn::util::Vector3fType pos;
            nn::util::VectorSet(&pos, cx, cy, nnear);
            nn::util::Vector3fType up;
            nn::util::VectorSet(&up, 0.f, 1.f, 0.f);
            nn::util::Vector3fType target;
            nn::util::VectorSet(&target, cx, cy, nnear - 1);

            nn::util::MatrixLookAtRightHanded(&m_CameraViewMatrix, pos, target, up);
        }
    }

public:
    // レイアウトクラスが管理するためのリンク情報です。
    //! @brief 内部用機能のため使用禁止です。
    nn::util::IntrusiveListNode m_Link;

protected:


    //---------------------------------------------------------------------------
    ScreenManager*          m_pScreenManager;

    nn::ui2d::DrawInfo      m_DrawInfo;
    nn::ui2d::Layout*       m_Layout;

    //----------------------------------------------
    // 拡張ポイント
    // 従来は、コールバックベースでしたが、オーバーライドベースで提供します。
    virtual void OnPostCalculate(nn::ui2d::Layout* pLayout) { NN_UNUSED(pLayout); };

    //----------------------------------------------
    // アニメーション、ボタン関係
    AnimationManager2               m_AnimationMgr;
    nn::ui2d::ButtonGroup           m_ButtonGroup;
    nn::ui2d::ControlCreator*       m_pControlCreator;
    nn::ui2d::ControlList           m_ControlList;

    //----------------------------------------------
    nn::util::MatrixT4x4fType   m_ProjectionMatrix;
    nn::util::MatrixT4x3fType   m_CameraViewMatrix;

    bool                        m_IsPerspectiveProjection;
    float                       m_PerspectiveFovyDeg;
    float                       m_PerspectiveNear;
    float                       m_PerspectiveFar;

    bool                        m_IsViewMtxDirty;

    //----------------------------------------------
    bool                        m_AnimateOnce; // 一回はアニメーションを行ったか
};

} // namespace viewer
} // namespace ui2d
} // namespace nn

#endif // NN_UI2D_VIEWER_ENABLED
