﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ui2d/viewer/ui2d_Config.h>
#if defined(NN_UI2D_VIEWER_ENABLED)

#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

#include <nn/ui2d.h>
#include <nn/font.h>

#include <nn/ui2d/ui2d_StateMachine.h>

namespace nn
{
namespace ui2d
{

//---------------------------------------------------------------------------
//! @brief ステートマシン用の、アニメーションリソースを実行時に作るクラス（ライブラリ内部用のため利用しないでください）
class RuntimeResAnimationBuilder
{
private:
    size_t CalcContentBlockSize_(const FeatureParamater& featureParamater, int featureIndex, const Transition& transition)
    {
        size_t resAnimationContentBlockSize = sizeof(nn::ui2d::ResAnimationContent);
        const int InfoCount = featureParamater.m_FeatureParamaterTypeCount;
        for (int infoIdx = 0; infoIdx < InfoCount; infoIdx++)
        {
            resAnimationContentBlockSize +=
                sizeof(uint32_t) +
                sizeof(nn::ui2d::ResAnimationInfo);

            const int TargetCount = featureParamater.m_pFeatureParamaterTypes[infoIdx].m_ParamTypeCount;
            for (int targetIdx = 0; targetIdx < TargetCount; targetIdx++)
            {
                resAnimationContentBlockSize +=
                    sizeof(uint32_t) +
                    sizeof(nn::ui2d::ResAnimationTarget);

                const int KeyCount = 1;
                for (int keyIdx = 0; keyIdx < KeyCount; keyIdx++)
                {
                    resAnimationContentBlockSize += sizeof(nn::ui2d::ResParameterizedAnim);

                    const TransitionTimeLineTrack* pTrack = transition.m_pTimeLine->GetTrackByFeatureParamaterIndex(featureIndex);
                    if (pTrack == nullptr)
                    {
                        NN_SDK_ASSERT(false);
                        continue;
                    }
                    // トラックが持つキーの個数 + 1 がアニメーション（区間）の個数。
                    const int parameterizedAnimCount = pTrack->m_TransitionSectionCount + 1;
                    resAnimationContentBlockSize += sizeof(uint32_t) * parameterizedAnimCount;
                    resAnimationContentBlockSize += sizeof(nn::ui2d::ResParameterizedAnimParameter) + parameterizedAnimCount;

                    // イベントキー用のリソース
                    const int eventKeyCount = pTrack->CalculateEventCount();
                    resAnimationContentBlockSize += sizeof(ResParameterizedAnimParameterEvent) * eventKeyCount;
                }
            }
        }

        return resAnimationContentBlockSize * 2;
    }

    size_t CalcAnimationBlockSize_(const StateLayer& stateLayer, const Transition& transition)
    {
        size_t totalSize = sizeof(nn::ui2d::ResAnimationContent);

        // fileName
        //totalSize += sizeof(uint32_t) * 1;
        //totalSize += nn::util::align_up(strlen(pDummyFileName), sizeof(uint32_t));

        // offset to contents
        totalSize += sizeof(uint32_t) * stateLayer.m_FeatureParamaterSet.size();

        // contents
        int featureIndex = 0;
        auto endIter = stateLayer.m_FeatureParamaterSet.end();
        for (auto iter = stateLayer.m_FeatureParamaterSet.begin(); iter != endIter; iter++)
        {
            totalSize += CalcContentBlockSize_(*iter, featureIndex, transition);
            featureIndex++;
        }

        return totalSize;
    }

    //----------------------------------------------------------
    void WriteResParameterizedAnim_(
        const StateLayer& stateLayer,
        const FeatureParamater& featureParamater,
        int parameterizedAnimCount,
        const Transition* pPrevTransition, float prevTransitionFrame,
        Transition* pNextTransition,
        const int contentIdx, int infoIdx, int targetIdx,
        nn::util::BytePtr* pBytptr)
    {
        const bool IsEventParamater = featureParamater.m_AnimContentType == AnimContentType_StateMachine;
        auto pParamAnimParam = pBytptr->Get<nn::ui2d::ResParameterizedAnim>();

        pParamAnimParam->parameterizedAnimCount = static_cast<uint16_t>(parameterizedAnimCount);
        pBytptr->Advance(sizeof(nn::ui2d::ResParameterizedAnim));

        uint32_t* pOffsToParams = pBytptr->Get<uint32_t>();
        pBytptr->Advance(sizeof(uint32_t) * pParamAnimParam->parameterizedAnimCount);

        auto pFirstParam = pBytptr->Get<nn::ui2d::ResParameterizedAnimParameter>();
        for (int paramIdx = 0; paramIdx < pParamAnimParam->parameterizedAnimCount; paramIdx++)
        {
            pOffsToParams[paramIdx] = static_cast<uint32_t>(nn::util::BytePtr(pParamAnimParam).Distance(pBytptr->Get()));
            auto pParam = pBytptr->Get<nn::ui2d::ResParameterizedAnimParameter>();
            pBytptr->Advance(sizeof(nn::ui2d::ResParameterizedAnimParameter));

            if (IsEventParamater)
            {
                auto pParamEvent = pBytptr->Get<nn::ui2d::ResParameterizedAnimParameterEvent>();
                pBytptr->Advance(sizeof(nn::ui2d::ResParameterizedAnimParameterEvent));
                pParam->pEvent = pParamEvent;
                pNextTransition->m_pTimeLine->SetupParametrizedAnimationEvent(pParam, contentIdx, paramIdx - 1);
            }
            else
            {
                const State* pStatePrev = stateLayer.FindStateByName(pNextTransition->m_pPrevStateName);
                const State* pStateNext = stateLayer.FindStateByName(pNextTransition->m_pNextStateName);

                if (pStatePrev == nullptr || pStateNext == nullptr)
                {
                    continue;
                }

                pNextTransition->m_pTimeLine->SetupParametrizedAnimation(
                    pParam,
                    contentIdx,
                    infoIdx,
                    targetIdx,
                    paramIdx - 1,
                    pStatePrev->m_FeatureParamaterStoreSet,
                    pStateNext->m_FeatureParamaterStoreSet);
            }
        }

        // 現在再生中のアニメーションの結果を、スタート値として上書きします。
        if (pPrevTransition != nullptr && !IsEventParamater)
        {
            const State* pCurrentStateNext = stateLayer.FindStateByName(pPrevTransition->m_pNextStateName);
            if (pCurrentStateNext != nullptr)
            {
                nn::ui2d::ResParameterizedAnimParameter temp = nn::ui2d::ResParameterizedAnimParameter();

                pPrevTransition->m_pTimeLine->SetupParametrizedAnimationFromTime(
                    &temp, contentIdx, infoIdx, targetIdx, prevTransitionFrame,
                    pPrevTransition->m_FeatureParamaterStoreSetOfStartValue,
                    pCurrentStateNext->m_FeatureParamaterStoreSet);

                // スタート値を記憶しておきます。
                pFirstParam->value.startValue = GetParameterizedAnimValueAtFrameClamped(prevTransitionFrame, &temp);
                pNextTransition->m_FeatureParamaterStoreSetOfStartValue.SetData(contentIdx, infoIdx, targetIdx, pFirstParam->value.startValue);
            }
        }
    }

    //----------------------------------------------------------
    // ResAnimationContent を実行時に作るクラス
    void WriteContent_(const StateLayer& stateLayer, const FeatureParamater& featureParamater, Transition* pNextTransition, const Transition* pPrevTransition, float prevTransitionFrame, nn::util::BytePtr* pBytptr)
    {
        NN_UNUSED(pPrevTransition);
        NN_UNUSED(prevTransitionFrame);

        // メモリの先頭からデータを書き込んでいく。
        {
            auto pContent = pBytptr->Get<nn::ui2d::ResAnimationContent>();
            pBytptr->Advance(sizeof(nn::ui2d::ResAnimationContent));

            const int contentIdx = stateLayer.FindFeatureParamaterByName(featureParamater.m_pName, featureParamater.m_AnimContentType);

            // Content 書く
            const int InfoCount = featureParamater.m_FeatureParamaterTypeCount;
            {
                pContent->count = static_cast<uint8_t>(InfoCount);
                pContent->type = featureParamater.m_AnimContentType;
                nn::util::Strlcpy(pContent->name, featureParamater.m_pName, sizeof(pContent->name));
            }

            nn::util::BytePtr pBaseForInfo(pContent);
            uint32_t* pOffsToInfo = pBytptr->Get<uint32_t>();
            pBytptr->Advance(sizeof(uint32_t) * InfoCount);

            //----------------------------------------------
            // ResAnimationInfo
            for (int infoIdx = 0; infoIdx < InfoCount; infoIdx++)
            {
                pOffsToInfo[infoIdx] = static_cast<uint32_t>(pBaseForInfo.Distance(pBytptr->Get()));

                auto pInfo = pBytptr->Get<nn::ui2d::ResAnimationInfo>();
                pBytptr->Advance(sizeof(nn::ui2d::ResAnimationInfo));

                // pInfo 書く
                const int TargetCount = featureParamater.m_pFeatureParamaterTypes[infoIdx].m_ParamTypeCount;
                {
                    pInfo->count = static_cast<uint8_t>(TargetCount);
                    pInfo->kind = featureParamater.m_pFeatureParamaterTypes[infoIdx].m_ResAnimationInfoType;
                }

                nn::util::BytePtr pBaseForTarget(pInfo);
                uint32_t* pOffsToTarget = pBytptr->Get<uint32_t>();
                pBytptr->Advance(sizeof(uint32_t) * TargetCount);

                //----------------------------------------------
                // ResAnimationTarget
                for (int targetIdx = 0; targetIdx < TargetCount; targetIdx++)
                {
                    pOffsToTarget[targetIdx] = static_cast<uint32_t>(pBaseForTarget.Distance(pBytptr->Get()));

                    auto pTarget = pBytptr->Get<nn::ui2d::ResAnimationTarget>();
                    nn::util::BytePtr pBaseForKey(pTarget);
                    pBytptr->Advance(sizeof(nn::ui2d::ResAnimationTarget));

                    // pTarget 書く
                    {
                        pTarget->target = featureParamater.m_pFeatureParamaterTypes[infoIdx].m_pParamTypes[targetIdx];

                        pTarget->keyCount = 1;
                        pTarget->curveType = nn::ui2d::AnimCurve_ParameterizedAnim;
                        pTarget->keysOffset = static_cast<uint32_t>(pBaseForKey.Distance(pBytptr->Get()));
                    }

                    //----------------------------------------------
                    // ResParameterizedAnim ResParameterizedAnimParameter
                    {
                        const TransitionTimeLineTrack* pTrack = pNextTransition->m_pTimeLine->GetTrackByFeatureParamaterIndex(contentIdx);
                        if (pTrack == nullptr)
                        {
                            NN_SDK_ASSERT(false);
                            return;
                        }

                        const int animParamaterCount = static_cast<uint16_t>(pTrack->m_TransitionSectionCount) + 1;
                        WriteResParameterizedAnim_(stateLayer, featureParamater, animParamaterCount, pPrevTransition, prevTransitionFrame, pNextTransition, contentIdx, infoIdx, targetIdx, pBytptr);
                    }
                }
            }
        }
    }

public:

    //----------------------------------------------------------
    //! @brief Transition に対応するアニメーションリソースを生成します。（ライブラリ内部用のため利用しないでください）
    //!
    //! @param[in] stateLayer        ステートレイヤーです。
    //! @param[in] pNextTansition       次のトランジションです。
    //! @param[in] pPrevTransition       現在のトランジションです。
    //! @param[in] prevTransitionFrame   現在のトランジションの再生フレームです。
    //!
    void* Build(const StateLayer& stateLayer, Transition* pNextTansition, const Transition* pPrevTransition, float prevTransitionFrame)
    {
        const FeatureParamaterSet& featureParamaterSet = stateLayer.m_FeatureParamaterSet;

        //const char* pDummyFileName = "RuntimeAnim";

        // サイズを計算する！
        const size_t resAnimationBlockSize = CalcAnimationBlockSize_(stateLayer, *pNextTansition);
        const size_t fileSize = resAnimationBlockSize + sizeof(nn::font::detail::BinaryFileHeader);
        void* pBuff = nn::ui2d::Layout::AllocateMemory(fileSize);

        nn::util::BytePtr bytptr(pBuff);

        //----------------------------------------------
        // BinaryFileHeader
        nn::font::detail::BinaryFileHeader* pAnimFileHeader = bytptr.Get<nn::font::detail::BinaryFileHeader>();
        pAnimFileHeader->signature = nn::ui2d::FileSignatureFlan;
        pAnimFileHeader->version = nn::ui2d::BinaryFileFormatVersion;
        pAnimFileHeader->headerSize = sizeof(nn::font::detail::BinaryFileHeader);
        pAnimFileHeader->dataBlocks = 1;
        pAnimFileHeader->byteOrder = nn::font::detail::ByteOrderMark;
        bytptr.Advance(sizeof(nn::font::detail::BinaryFileHeader));

        //----------------------------------------------
        // ResAnimationBlock
        nn::ui2d::ResAnimationBlock* pResAnim = bytptr.Get<nn::ui2d::ResAnimationBlock>();
        pResAnim->animContCount = static_cast<uint16_t>(featureParamaterSet.size());
        pResAnim->fileCount = 0;
        pResAnim->frameSize = static_cast<uint16_t>(pNextTansition->m_pTimeLine->GetTotalLength());
        pResAnim->blockHeader.kind = nn::ui2d::DataBlockKindPaneAnimInfo;
        pResAnim->blockHeader.size = static_cast<uint32_t>(resAnimationBlockSize);
        pResAnim->loop = pNextTansition->m_IsLoop;

        bytptr.Advance(sizeof(nn::ui2d::ResAnimationBlock));

        nn::util::BytePtr pBaseOfOffset(bytptr.Get<void>());

        // オフセット ⇒ ResAnimationConten
        uint32_t* pOffsOfAnimCont = bytptr.Get<uint32_t>();
        bytptr.Advance(sizeof(uint32_t) * pResAnim->animContCount);

        pResAnim->animContOffsetsOffset = static_cast<uint32_t>(nn::util::BytePtr(pResAnim).Distance(pOffsOfAnimCont));

        //----------------------------------------------
        // nn::ui2d::ResAnimationConten
        int count = 0;
        auto endIter = featureParamaterSet.end();
        for (auto iter = featureParamaterSet.begin(); iter != endIter; iter++)
        {
            pOffsOfAnimCont[count] = static_cast<uint32_t>(nn::util::BytePtr(pResAnim).Distance(bytptr.Get()));

            WriteContent_(stateLayer, *iter, pNextTansition, pPrevTransition, prevTransitionFrame, &bytptr);

            count++;
        }

        return pAnimFileHeader;
    }
};

} // namespace ui2d
} // namespace nn

#endif // NN_UI2D_VIEWER_ENABLED
