﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/atk2/atk2_IAudioFrameUpdate.h>
#include <nn/atk2/detail/atk2_Voice.h>
#include <nn/atk2/detail/atk2_CommandManager.h>

namespace nn  { namespace atk2 {

    class AudioEngine;

namespace detail {

    class RendererVoice;

enum VoiceCommandId
{
    VoiceCommandId_AllocVoice,
    VoiceCommandId_FreeVoice,
    VoiceCommandId_SetPriority,
    VoiceCommandId_Play,
    VoiceCommandId_Pause,
    VoiceCommandId_Stop,
    VoiceCommandId_AppendWaveBuffer,
    VoiceCommandId_UpdateParam
};

struct VoiceCommandAlloc : public Command
{
    uint32_t voiceId;
    uint32_t priority;
    void* userId;
};

struct VoiceCommandFree : public Command
{
    uint32_t voiceId;
};

struct VoiceCommandParam : public Command
{
    uint32_t voiceId;
    float volume;
    float pitch;
    bool monoFilterFlag;
    bool biquadFilterFlag;
    BiquadFilterCoefficients biquadFilterCoefficients;
    uint16_t monoFilterCutoff;
    //uint8_t interpolationType;
};

struct VoiceCommandPriority : public Command
{
    uint32_t voiceId;
    uint32_t priority;
};

struct VoiceCommandPlay : public Command
{
    uint32_t            voiceId;
    SampleFormat        sampleFormat;
    uint32_t            sampleRate;
    AdpcmParam          adpcmParam;
    OutputReceiver*     pOutputReceiver;
};

struct VoiceCommandPause : public Command
{
    uint32_t voiceId;
};

struct VoiceCommandAppendWaveBuffer : public Command
{
    uint32_t voiceId;
    void* tag;
    const void* bufferAddress;
    size_t bufferSize;
    size_t sampleLength;
    uint32_t sampleOffset;
    bool adpcmContextEnable;
    AdpcmContext adpcmContext;
    bool loopFlag;
};

// @brief ボイスの変更をレンダラボイスに反映するためのコマンド処理を管理するクラスです
class VoiceCommandManager : public CommandManager, public IAudioFrameUpdate
{
public:
    // @brief 関数の処理結果を表す列挙体です
    enum Result
    {
        Result_Success = 0,        //!< 成功です
        Result_ErrorUnknown        //!< 不明なエラーが発生しました
    };
public:
    // @brief 初期化に使用するパラメータです
    struct InitArg
    {
        int waveBufferPacketCount; //<! 波形バッファパケットの数です
        size_t commandBufferSize;  //<! コマンドバッファのサイズです
        void* workBuffer;          //<! ワークバッファです
        size_t workBufferSize;     //<! ワークバッファのサイズです

        AudioEngine* _pAudioEngine;
    };

    // @brief 初期化に使用するプラットフォーム固有のパラメータです
    struct PlatformInitArg
    {
        void* workBuffer;                   //!< ワークバッファです
        size_t workBufferSize;              //!< ワークバッファのサイズです
        void* workBufferForMemoryPool;      //!< メモリプールにアタッチされたワークバッファです
        size_t workBufferSizeForMemoryPool; //!< メモリプールにアタッチされたワークバッファのサイズです
    };

public:
    // TODO: 波形バッファと ADPCMContext がボイスコマンドにあるバッファを使用しているのはやめた方がいい
    //       ちゃんとボイスかバーチャルボイスの持っている Adpcm コンテキストを参照するようにすること

    // @brief コンストラクタです
    VoiceCommandManager() NN_NOEXCEPT {}

    // @brief 初期化パラメータの初期値を取得します
    void GetDefaultInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;
    size_t GetRequiredBufferSize(const InitArg& arg, const PlatformInitArg& platformArg) const NN_NOEXCEPT;

    void Initialize(InitArg arg, PlatformInitArg PlatformArg);
    void Initialize( void* commandBuffer, size_t commandBufferSize, void* waveBuffer, size_t waveBufferSize, int waveBufferPacketCount ) NN_NOEXCEPT;

    //static size_t GetRequiredWaveBufferMemSize(int waveBufferPacketCount) NN_NOEXCEPT;
    //size_t GetRequiredWaveBufferMemSize(int waveBufferPacketCount) NN_NOEXCEPT;

    bool Update() NN_NOEXCEPT NN_OVERRIDE;
    void Update(uint32_t frameCount) NN_NOEXCEPT;

    // プラットフォーム固有関数
    size_t GetRequiredBufferSizeForMemoryPool(const InitArg& arg, const PlatformInitArg& platformArg) const NN_NOEXCEPT;

private:
    struct WaveBufferPacket
    {
        AdpcmContext adpcmContext;
        WaveBuffer waveBuffer;
    };

    WaveBufferPacket* GetFreeWaveBuffer() NN_NOEXCEPT;

    virtual void ProcessCommandList( Command* commandList ) NN_NOEXCEPT NN_OVERRIDE;
    void VoiceDisposeCallback( RendererVoice* voice, void* arg ) NN_NOEXCEPT;

    // プラットフォーム固有関数
    size_t GetRequiredBufferSizeForWaveBufferPacket(int waveBufferPacketCount) const NN_NOEXCEPT;

private:
    WaveBufferPacket* m_pWaveBufferPacket;
    int m_WaveBufferPacketCount;
    size_t m_CommandBufferSize;
    void* m_CommandBuffer;

    AudioEngine* m_pAudioEngine;
};

}}}
