﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/atk2/detail/atk2_Filter.h>
#include <nn/atk2/detail/atk2_WaveBuffer.h>
#include <nn/atk2/detail/atk2_WaveBufferList.h>

namespace nn  { namespace atk2 {

    class AudioEngine;
    class OutputReceiver;

namespace detail {

    class RendererVoice;

    // @brief ボイスの状態を表す列挙体です
    enum VoiceState
    {
        VoiceState_Play,  //!< 再生指令状態を表します。
        VoiceState_Stop,  //!< 停止状態を表します。
        VoiceState_Pause  //!< 一時停止状態を表します。再生を再開する際は、SetState により再生状態を設定して下さい。
    };

    // @brief ボイスの情報を表す列挙体です
    struct VoiceInfo
    {
        uint32_t playPosition;           //!< サンプル位置です
        VoiceState voiceState;           //!< ボイスの状態です
        WaveBufferState waveBufferState; //!< 波形バッファの状態です
        void* waveBufferTag;             //!< 波形バッファのタグです
        void* pVoice;                    //!< ボイスへのポインタです
    };

    // @brief ボイスを管理するクラスです
    class Voice
    {
    public:
        // @brief コンストラクタです
        Voice() NN_NOEXCEPT;
        // @brief デストラクタです
        ~Voice() NN_NOEXCEPT;

        // @brief ボイスを確保します
        bool AllocVoice( uint32_t priority, AudioEngine* pAudioEngine ) NN_NOEXCEPT;
        // @brief ボイスを解放します
        void Free() NN_NOEXCEPT;
        // @brief 波形バッファを追加します
        void AppendWaveBuffer( WaveBuffer* waveBuffer ) NN_NOEXCEPT;

        // @brief ボイスが有効かどうかを取得します
        bool IsAvailable() const NN_NOEXCEPT;
        // @brief サンプル位置を取得します
        ptrdiff_t GetPlaySamplePosition() const NN_NOEXCEPT;

        // 波形のメタデータ
        // @brief サンプルレートを設定します
        void SetSampleRate(uint32_t sampleRate) NN_NOEXCEPT;
        // @brief サンプルのフォーマットを設定します
        void SetSampleFormat(SampleFormat format) NN_NOEXCEPT;
        // @brief ADPCM パラメータを設定します
        void SetAdpcmParam(const AdpcmParam& param) NN_NOEXCEPT;

        // ボイスのパラメータ
        // @brief 優先度を設定します
        void SetPriority(uint32_t priority) NN_NOEXCEPT;
        // @brief 優先度を取得します
        uint32_t GetPriority() const NN_NOEXCEPT;
        // @brief 音量を設定します
        void SetVolume(float volume) NN_NOEXCEPT;
        // @brief 音量を取得します
        float GetVolume() const NN_NOEXCEPT;
        // @brief ピッチを設定します
        void SetPitch(float pitch) NN_NOEXCEPT;
        // @brief ピッチを取得します
        float GetPitch() const NN_NOEXCEPT;
        //void SetTvMix( const OutputMix& mix ) NN_NOEXCEPT { m_VoiceParam.m_TvMix = mix; }
        //void SetInterpolationType( uint8_t interpolationType ) NN_NOEXCEPT;
        // @brief 単極フィルタを設定します
        void SetMonoFilter( bool enable, uint16_t cutoff = 0 ) NN_NOEXCEPT;
        // @brief Biquad フィルタを設定します
        void SetBiquadFilter( bool enable, const BiquadFilterCoefficients* coef = NULL ) NN_NOEXCEPT;
        // @brief ボイスの出力先を設定します
        void SetOutputReceiver( OutputReceiver* pOutputReceiver ) NN_NOEXCEPT;

        // 内部管理用
        void SetVoiceState(VoiceState state) NN_NOEXCEPT;
        VoiceState GetVoiceState() const NN_NOEXCEPT;

        void UpdateParam() NN_NOEXCEPT;
        void UpdateVoiceStatus() NN_NOEXCEPT;

    private:
        void Initialize( uint32_t priority, AudioEngine* pAudioEngine ) NN_NOEXCEPT;

    private:

        // 波形のメタデータ
        uint32_t     m_SampleRate;
        SampleFormat m_SampleFormat;
        AdpcmParam   m_AdpcmParam;

        // ボイスのパラメータ
        float                    m_Volume;
        float                    m_Pitch;
        //OutputMix                m_TvMix;
        bool                     m_MonoFilterFlag;
        uint16_t                 m_MonoFilterCutoff;     // LPF のカットオフ周波数
        bool                     m_BiquadFilterFlag;
        BiquadFilterCoefficients m_BiquadFilterCoefficients;
        //uint8_t                  m_InterpolationType;    // 補間タイプ
        uint32_t                 m_Priority;
        OutputReceiver*          m_pOutputReceiver;

        // 内部管理用変数
        VoiceState m_VoiceState;

        // 低層ボイスの情報
        // #ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        uint32_t m_VoiceId;
        ptrdiff_t m_PlaySamplePosition;
        uint32_t m_VoiceInfoEnableFlag;
        uint32_t m_CommandTag;
        WaveBufferList m_WaveBufferList;

        // #else
        RendererVoice* m_pRendererVoice;
        // #endif

        AudioEngine* m_pAudioEngine;
    }; // class Voice

}}} // namespace nn::atk::detail
