﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_BitFlagSet.h>
#include <nn/atk2/atk2_IAudioFrameUpdate.h>
#include <nn/atk2/detail/atk2_Voice.h>

namespace nn  { namespace atk2 { namespace detail {

    class RendererVoice;

    // @brief ボイスコマンド処理に使用する仮想ボイスの管理を行います
    class VirtualVoiceManager : public IAudioFrameUpdate
    {
    public:
        static const uint32_t InvalidVoiceId = 0xffffffff;
        static const int VirtualVoiceCount = 256; // NOTE: 32の倍数

    public:
        // @brief コンストラクタです
        VirtualVoiceManager() NN_NOEXCEPT;

        // @brief 初期化を行います
        void Initialize() NN_NOEXCEPT;
        // @brief 仮想ボイスを確保します
        uint32_t AllocVirtualVoice() NN_NOEXCEPT;
        // @brief 仮想ボイスを解放します
        void FreeVirtualVoice(uint32_t id) NN_NOEXCEPT;
        // @brief 仮想ボイスにオーディオレンダラのボイスを紐づけます
        void SetRendererVoice( uint32_t voiceId, RendererVoice* rendererVoice ) NN_NOEXCEPT;
        // @brief 仮想ボイスに紐づけられたオーディオレンダラのボイスを取得します
        RendererVoice* GetRendererVoice( uint32_t voiceId ) NN_NOEXCEPT;
        // @brief ボイスの情報を取得します
        VoiceInfo* GetVoiceInfo( uint32_t voiceId ) NN_NOEXCEPT;
        // @brief ボイスの情報を取得します
        const VoiceInfo* GetVoiceInfo( uint32_t voiceId ) const NN_NOEXCEPT;

        // @brief 更新処理を行います
        bool Update() NN_NOEXCEPT NN_OVERRIDE;

        // @brief ボイスの情報を更新します
        void UpdateVoiceInfo() NN_NOEXCEPT;

        // @brief 確保済みの仮想ボイスの数を取得します。実際に発音中のボイスの数ではないためご注意ください。
        int GetAllocatedVirtualVoiceCount() const NN_NOEXCEPT;

        // @brief 未解放の RendererVoice の数を取得します。
        // GetAllocatedVirtualVoiceCount() が 0 でも VoiceCommand が残っている場合 false が返ってくることがあるため注意してください。
        int GetUnreleasedRendererVoiceCount() const NN_NOEXCEPT;

    private:
        static const int VirtualVoiceElementCount = VirtualVoiceCount / 32;
        // @brief 仮想ボイスの確保状況を表すテーブルです
        uint32_t m_VirtualVoiceAllocationTable[VirtualVoiceElementCount];
        // @brief 現在 Read に用いるためのテーブルのインデックスです
        uint32_t m_VoiceInfoTableRead;
        // @brief レンダラボイスのインスタンスのテーブルです
        RendererVoice* m_RendererVoiceTable[ VirtualVoiceCount ];
        // @brief ボイスとレンダラボイスの情報を同期するためのテーブルです
        VoiceInfo m_VoiceInfoTable[2][ VirtualVoiceCount ];

        // @brief ボイス情報のダーティフラグを管理するテーブルです
        nn::util::BitFlagSet<VirtualVoiceCount> m_VoiceInfoDirtyTable[2];
    };

}}} // namespace nn::atk::detail
