﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn  { namespace atk2 { namespace detail {

    //! @brief  biquad フィルタ係数の構造体です。
    //!
    //!         各係数については、Nintendo SDK リファレンスマニュアルの
    //!         BiquadFilterParameter の説明をご参照ください。
    //!         各係数がどの係数に対応しているかは、それぞれの係数の説明に記載されています。
    struct BiquadFilterCoefficients
    {
        int16_t b0; //!< numerator[0] の係数を表します。
        int16_t b1; //!< numerator[1] の係数を表します。
        int16_t b2; //!< numerator[2] の係数を表します。
        int16_t a1; //!< denominator[0] の係数を表します。
        int16_t a2; //!< denominator[1] の係数を表します。
    };

    //! @brief  biquad フィルタの種類を示す列挙体です。
    //!
    //!         biquad フィルタでは、フィルタの係数を変更することにより
    //!         様々な種類のフィルタを使用することができます。
    //!
    //!         ライブラリで用意されているプリセットの係数を利用するには、
    //!         BiquadFilterType_LowPassFilter ～ BiquadFilterType_BandPassFilter2048 を指定してください。
    //!
    //!         これらプリセットのフィルタ係数はオーディオレンダラのレンダリングサンプルレートを 32kHz に設定した場合の値になります。
    //!         オーディオレンダラのサンプルレートによってはフィルタ係数を変える必要があることにご注意ください。
    //!
    //!         32kHz 以外のレンダリングサンプルレートを使用する場合は、
    //!         プリセットの係数ではなくユーザーで定義したフィルタ係数をご使用ください。
    //!         ユーザー定義の BiquadFilter の使用方法については @ref SoundSystem::SetBiquadFilterCallback を
    //!         レンダリングサンプルレートの変更につきましては @ref SoundSystem::SoundSystemParam::rendererSampleRate を参照ください。
    //!
    //!         biquad フィルタを使用しない場合は BiquadFilterType_None を指定してください。
    //!
    //!         また、ユーザーが設計したフィルタ係数を登録する際のタイプ番号として
    //!         BiquadFilterType_UserMin ～ BiquadFilterType_UserMax
    //!         までの値をご利用ください。
    //!
    //!         biquad フィルタは @ref SoundHandle::SetBiquadFilter
    //!         関数から利用することができます。
    //!
    //! @see SoundHandle::SetBiquadFilter
    //! @see SoundSystem::SetBiquadFilterCallback
    enum BiquadFilterType
    {
        BiquadFilterType_Inherit = -1, //!< バイナリデータの値を使用します。

        BiquadFilterType_None           = 0,  //!< biquad フィルタを使用しません。
        BiquadFilterType_LowPassFilter  = 1,  //!< プリセットのローパスフィルタを使用します。
        BiquadFilterType_HighPassFilter = 2,  //!< プリセットのハイパスフィルタを使用します。

        //! プリセットのバンドパスフィルタ (中心周波数 512Hz) を使用します。
        BiquadFilterType_BandPassFilter512  = 3,

        //! プリセットのバンドパスフィルタ (中心周波数 1024Hz) を使用します。
        BiquadFilterType_BandPassFilter1024 = 4,

        //! プリセットのバンドパスフィルタ (中心周波数 2048Hz) を使用します。
        BiquadFilterType_BandPassFilter2048 = 5,

        BiquadFilterType_LowPassFilterNw4fCompatible48k  = 6,       //!< @briefprivate プリセットのローパスフィルタを使用します。オーディオレンダラのサンプルレートが48kHzの場合に NW4Fと同様のかかり方をするフィルタです。
        BiquadFilterType_HighPassFilterNw4fCompatible48k = 7,       //!< @briefprivate プリセットのハイパスフィルタを使用します。オーディオレンダラのサンプルレートが48kHzの場合に NW4Fと同様のかかり方をするフィルタです。
        BiquadFilterType_BandPassFilter512Nw4fCompatible48k = 8,    //!< @briefprivate プリセットのバンドパスフィルタ (中心周波数 512Hz) を使用します。オーディオレンダラのサンプルレートが48kHzの場合に NW4Fと同様のかかり方をするフィルタです。
        BiquadFilterType_BandPassFilter1024Nw4fCompatible48k = 9,   //!< @briefprivate プリセットのバンドパスフィルタ (中心周波数 1024Hz) を使用します。オーディオレンダラのサンプルレートが48kHzの場合に NW4Fと同様のかかり方をするフィルタです。
        BiquadFilterType_BandPassFilter2048Nw4fCompatible48k  = 10, //!< @briefprivate プリセットのバンドパスフィルタ (中心周波数 2048Hz) を使用します。オーディオレンダラのサンプルレートが48kHzの場合に NW4Fと同様のかかり方をするフィルタです。

        //! ユーザーが登録できるフィルタ係数の種類の最小値です。
        BiquadFilterType_UserMin = 64,

        BiquadFilterType_User0 = BiquadFilterType_UserMin,    //!< ユーザー定義 0 のフィルタを使用します。
        BiquadFilterType_User1,                                  //!< ユーザー定義 1 のフィルタを使用します。
        BiquadFilterType_User2,                                  //!< ユーザー定義 2 のフィルタを使用します。
        BiquadFilterType_User3,                                  //!< ユーザー定義 3 のフィルタを使用します。
        BiquadFilterType_User4,                                  //!< ユーザー定義 4 のフィルタを使用します。
        BiquadFilterType_User5,                                  //!< ユーザー定義 5 のフィルタを使用します。
        BiquadFilterType_User6,                                  //!< ユーザー定義 6 のフィルタを使用します。
        BiquadFilterType_User7,                                  //!< ユーザー定義 7 のフィルタを使用します。
        BiquadFilterType_User8,                                  //!< ユーザー定義 8 のフィルタを使用します。
        BiquadFilterType_User9,                                  //!< ユーザー定義 9 のフィルタを使用します。
        BiquadFilterType_User10,                                 //!< ユーザー定義 10 のフィルタを使用します。
        BiquadFilterType_User11,                                 //!< ユーザー定義 11 のフィルタを使用します。
        BiquadFilterType_User12,                                 //!< ユーザー定義 12 のフィルタを使用します。
        BiquadFilterType_User13,                                 //!< ユーザー定義 13 のフィルタを使用します。
        BiquadFilterType_User14,                                 //!< ユーザー定義 14 のフィルタを使用します。
        BiquadFilterType_User15,                                 //!< ユーザー定義 15 のフィルタを使用します。
        BiquadFilterType_User16,                                 //!< ユーザー定義 16 のフィルタを使用します。
        BiquadFilterType_User17,                                 //!< ユーザー定義 17 のフィルタを使用します。
        BiquadFilterType_User18,                                 //!< ユーザー定義 18 のフィルタを使用します。
        BiquadFilterType_User19,                                 //!< ユーザー定義 19 のフィルタを使用します。
        BiquadFilterType_User20,                                 //!< ユーザー定義 20 のフィルタを使用します。
        BiquadFilterType_User21,                                 //!< ユーザー定義 21 のフィルタを使用します。
        BiquadFilterType_User22,                                 //!< ユーザー定義 22 のフィルタを使用します。
        BiquadFilterType_User23,                                 //!< ユーザー定義 23 のフィルタを使用します。
        BiquadFilterType_User24,                                 //!< ユーザー定義 24 のフィルタを使用します。
        BiquadFilterType_User25,                                 //!< ユーザー定義 25 のフィルタを使用します。
        BiquadFilterType_User26,                                 //!< ユーザー定義 26 のフィルタを使用します。
        BiquadFilterType_User27,                                 //!< ユーザー定義 27 のフィルタを使用します。
        BiquadFilterType_User28,                                 //!< ユーザー定義 28 のフィルタを使用します。
        BiquadFilterType_User29,                                 //!< ユーザー定義 29 のフィルタを使用します。
        BiquadFilterType_User30,                                 //!< ユーザー定義 30 のフィルタを使用します。
        BiquadFilterType_User31,                                 //!< ユーザー定義 31 のフィルタを使用します。
        BiquadFilterType_User32,                                 //!< ユーザー定義 32 のフィルタを使用します。
        BiquadFilterType_User33,                                 //!< ユーザー定義 33 のフィルタを使用します。
        BiquadFilterType_User34,                                 //!< ユーザー定義 34 のフィルタを使用します。
        BiquadFilterType_User35,                                 //!< ユーザー定義 35 のフィルタを使用します。
        BiquadFilterType_User36,                                 //!< ユーザー定義 36 のフィルタを使用します。
        BiquadFilterType_User37,                                 //!< ユーザー定義 37 のフィルタを使用します。
        BiquadFilterType_User38,                                 //!< ユーザー定義 38 のフィルタを使用します。
        BiquadFilterType_User39,                                 //!< ユーザー定義 39 のフィルタを使用します。
        BiquadFilterType_User40,                                 //!< ユーザー定義 40 のフィルタを使用します。
        BiquadFilterType_User41,                                 //!< ユーザー定義 41 のフィルタを使用します。
        BiquadFilterType_User42,                                 //!< ユーザー定義 42 のフィルタを使用します。
        BiquadFilterType_User43,                                 //!< ユーザー定義 43 のフィルタを使用します。
        BiquadFilterType_User44,                                 //!< ユーザー定義 44 のフィルタを使用します。
        BiquadFilterType_User45,                                 //!< ユーザー定義 45 のフィルタを使用します。
        BiquadFilterType_User46,                                 //!< ユーザー定義 46 のフィルタを使用します。
        BiquadFilterType_User47,                                 //!< ユーザー定義 47 のフィルタを使用します。
        BiquadFilterType_User48,                                 //!< ユーザー定義 48 のフィルタを使用します。
        BiquadFilterType_User49,                                 //!< ユーザー定義 49 のフィルタを使用します。
        BiquadFilterType_User50,                                 //!< ユーザー定義 50 のフィルタを使用します。
        BiquadFilterType_User51,                                 //!< ユーザー定義 51 のフィルタを使用します。
        BiquadFilterType_User52,                                 //!< ユーザー定義 52 のフィルタを使用します。
        BiquadFilterType_User53,                                 //!< ユーザー定義 53 のフィルタを使用します。
        BiquadFilterType_User54,                                 //!< ユーザー定義 54 のフィルタを使用します。
        BiquadFilterType_User55,                                 //!< ユーザー定義 55 のフィルタを使用します。
        BiquadFilterType_User56,                                 //!< ユーザー定義 56 のフィルタを使用します。
        BiquadFilterType_User57,                                 //!< ユーザー定義 57 のフィルタを使用します。
        BiquadFilterType_User58,                                 //!< ユーザー定義 58 のフィルタを使用します。
        BiquadFilterType_User59,                                 //!< ユーザー定義 59 のフィルタを使用します。
        BiquadFilterType_User60,                                 //!< ユーザー定義 60 のフィルタを使用します。
        BiquadFilterType_User61,                                 //!< ユーザー定義 61 のフィルタを使用します。
        BiquadFilterType_User62,                                 //!< ユーザー定義 62 のフィルタを使用します。
        BiquadFilterType_User63,                                 //!< ユーザー定義 63 のフィルタを使用します。

        //! ユーザーが登録できるフィルタ係数の種類の最大値です。
        BiquadFilterType_UserMax = BiquadFilterType_User63,

        //! フィルタ係数の種類のデータ上の最小値です。
        BiquadFilterType_DataMin = BiquadFilterType_None,

        //! フィルタ係数の種類の最小値です。
        BiquadFilterType_Min = BiquadFilterType_Inherit,

        //! フィルタ係数の種類の最大値です。
        BiquadFilterType_Max = BiquadFilterType_UserMax
    };

    const BiquadFilterCoefficients BiquadFilterCoefZero = { 0 };

}}}
