﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using LOG = Common.Logger;
using LOG_LEVEL = Common.Logger.Level;

namespace Common
{
    public class Logger
    {
        public enum Level
        {
            LOG_ERROR,
            LOG_WARN,
            LOG_INFO,
            LOG_INFO2,
        }
#if DEBUG
        public const Level DefaultLevel = Level.LOG_INFO;
#else
        public const Level DefaultLevel = Level.LOG_INFO;
#endif

        private static Level __currentLevel = DefaultLevel;
        public static Level CurrentLevel
        {
            get { return __currentLevel; }
            set { __currentLevel = value; }
        }

        private static Dictionary<Level, string> levelHeader =
            new Dictionary<Level, string> { { Level.LOG_ERROR, "[ERROR] " }, { Level.LOG_WARN, "[WARN]  " }, { Level.LOG_INFO, "[INFO]  " }, { Level.LOG_INFO2, "[INFO]  " } };

        private static Logger logger = new Logger();

        private Logger()
        {
#if DEBUG
            isErrorOrWarn = false;
#endif
        }

        ~Logger()
        {
#if DEBUG
            if (isErrorOrWarn == true)
            {
                string toolName = System.IO.Path.GetFileName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                // Console.Error.Write("{0} : this tool has error or warning\n", toolName);
            }
#endif
        }

        public static void SetLevel(Level level)
        {
            __currentLevel = level;
        }

        public static void Log(string format, params object[] args)
        {
            Log(Level.LOG_INFO, format, args);
        }

        public static void Log(Level level, string format, params object[] args)
        {
            if (__currentLevel < level)
            {
                return;
            }
            CheckError(level);
            switch (level)
            {
                case Level.LOG_ERROR:
                    Console.Error.Write(levelHeader[level] + format, args);
                    break;
                default:
                    Console.Write(levelHeader[level] + format, args);
                    break;
            }
        }

        public static void LogLine(string format, params object[] args)
        {
            LogLine(Level.LOG_INFO, format, args);
        }

        public static void LogLine(Level level, string format, params object[] args)
        {
            if (__currentLevel < level)
            {
                return;
            }
            CheckError(level);
            switch (level)
            {
                case Level.LOG_ERROR:
                    Console.Error.WriteLine(levelHeader[level] + format, args);
                    break;
                default:
                    Console.WriteLine(levelHeader[level] + format, args);
                    break;
            }
        }

#if DEBUG
        private static bool isErrorOrWarn = false;
#endif

        private static bool CheckError(Level level)
        {
#if DEBUG
            if (level <= Level.LOG_WARN)
            {
                isErrorOrWarn = true;
                return true;
            }
#endif
            return false;
        }
    }
}
