﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;

using LOG = Common.Logger;
using LOG_LEVEL = Common.Logger.Level;

namespace BackupSaveData
{
    public static class ConvertData
    {
        const string InterMediateDirectoryName0 = "BackupSaveData0";
        const string InterMediateDirectoryName1 = "BackupSaveData1";

        public static bool ConvertAndCopyExportDirectory(string srcDirPath, string destDirPath)
        {
            if (ParseData.CheckUpToDateVersion(srcDirPath))
            {
                if (!FileUtil.CopyDirecotryResursively(srcDirPath, destDirPath))
                {
                    return false;
                }
            }
            else
            {
                // 変換する
                string tempDirectory = FileUtil.GetTempDirectoryPath();
                string interMediateDirectory0 = Path.Combine(tempDirectory, InterMediateDirectoryName0);
                string interMediateDirectory1 = Path.Combine(tempDirectory, InterMediateDirectoryName1);
                FileUtil.DeleteDirectory(interMediateDirectory0);
                FileUtil.DeleteDirectory(interMediateDirectory1);
                FileUtil.CopyDirecotryResursively(srcDirPath, interMediateDirectory0);
                if (!ConvertDirectoryToUpToDateVersion(interMediateDirectory0, interMediateDirectory1))
                {
                    return false;
                }
                FileUtil.CopyDirecotryResursively(interMediateDirectory1, destDirPath);
                FileUtil.DeleteDirectory(interMediateDirectory0);
                FileUtil.DeleteDirectory(interMediateDirectory1);
            }

            return true;
        }

        private static bool ConvertDirectoryToUpToDateVersion(string fromDirPath, string toDirPath)
        {
            int version = ParseData.GetVersion(fromDirPath);
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Convert data ({0} -> {1})", version, ParseData.UpToDateVersion);
            while (version < ParseData.UpToDateVersion)
            {
                if(!Directory.Exists(fromDirPath) && Directory.Exists(toDirPath))
                {
                    Directory.Move(toDirPath, fromDirPath);
                }
                if (!VersionUpDirecotry(version, fromDirPath, toDirPath))
                {
                    return false;
                }
                version++;
                FileUtil.DeleteDirectory(fromDirPath);
            }
            return true;
        }

        private static bool VersionUpDirecotry(int version, string fromDirPath, string toDirPath)
        {
            switch(version)
            {
                case 0:
                    // 変換不要
                    return FileUtil.CopyDirecotryResursively(fromDirPath, toDirPath);
                case 1:
                    return Convert1To2(fromDirPath, toDirPath);
                case 2:
                    LOG.LogLine(LOG_LEVEL.LOG_WARN, "current version ({0})", version);
                    return FileUtil.CopyDirecotryResursively(fromDirPath, toDirPath);
                default:
                    return false;
            }
        }

        private static bool Convert1To2(string fromDirPath, string toDirPath)
        {
            const string UserSaveDataDirectoryName = "user";

            Directory.CreateDirectory(toDirPath);

            // config.xml の version を上げる
            if(ParseData.CreateConfigXml(toDirPath))
            {
                return false;
            }

            // user を作る
            Directory.CreateDirectory(Path.Combine(toDirPath, UserSaveDataDirectoryName));

            IEnumerable<string> titleFolders = Directory.EnumerateDirectories(fromDirPath, "*", SearchOption.TopDirectoryOnly);
            foreach (string titleNameFull in titleFolders)
            {
                string titleName = Path.GetFileName(titleNameFull);

                // タイトルをその下に置く
                // さらにその下に 0001 を作る
                string createDirectoryName = Path.Combine(toDirPath, UserSaveDataDirectoryName, titleName, "0001");
                Directory.CreateDirectory(createDirectoryName);

                // ファイル一式をコピーする
                if(!FileUtil.CopyDirecotryResursively(Path.Combine(fromDirPath, titleName), createDirectoryName))
                {
                    return false;
                }
            }

            return true;
        }
    }
}
