﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;

using LOG = Common.Logger;
using LOG_LEVEL = Common.Logger.Level;

namespace BackupSaveData
{
    public class CommandArgument
    {
        [CommandLineSubCommand("export", Description = "Export all user save data to PC.")]
        public ExportSubCommand ExportCommand { get; set; }

        [CommandLineSubCommand("import", Description = "Import all user save data from PC.")]
        public ImportSubCommand ImportCommand { get; set; }
    }

    public class ExportSubCommand
    {
        [CommandLineValue(0,
            Description = "Set export directory path on PC.",
            ValueName = "Path",
            IsRequired = true)]
        public string Path { get; set; }

        [CommandLineOption("target",
            Description = "Specifies the target devkit.",
            IsRequired = true)]
        public string target { get; set; }

        [CommandLineOption("export-title",
            Description = "Specifies the export title.")]
        public string ExportTitle { get; set; }

        [CommandLineOption("exist-account-only",
            Description = "Export only exist account id of dekit.",
            IsRequired = false)]
        public bool ExistAccountOnly { get; set; }

        [CommandLineOption("verbose",
            Description = "Log verbosely.")]
        public bool Verbose { get; set; }

        public int Run()
        {
            if(!ExportTitleCheck())
            {
                return 1;
            }

            DoExportImport doObj = new DoExportImport();
            return doObj.DoExport(this.target, this.Path, this.ExportTitle, this.ExistAccountOnly, this.Verbose) ? 0 : 1;
        }

        private bool ExportTitleCheck()
        {
            if(ExportTitle == null || ExportTitle.Length == 0)
            {
                return true;
            }
            string checkString = ExportTitle.Replace(" ", "");
            string[] idStrings = checkString.Split(',');
            foreach(string idStr in idStrings)
            {
                UInt64 dummy;
                if (!UInt64.TryParse(idStr, System.Globalization.NumberStyles.HexNumber, null, out dummy))
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Invalid title ID string : {0}", idStr);
                    return false;
                }
            }
            return true;
        }
    }

    public class ImportSubCommand
    {
        [CommandLineValue(0,
            Description = "Set import directory path on PC.",
            ValueName = "Path",
            IsRequired = true)]
        public string Path { get; set; }

        [CommandLineOption("target",
            Description = "Specifies the target devkit.",
            IsRequired = true)]
        public string target { get; set; }

        [CommandLineOption("follow-exist-account",
            Description = "Follow exist account id of dekit.",
            IsRequired = false)]
        public bool FollowExistAccount { get; set; }

        [CommandLineOption("verbose",
            Description = "Log verbosely.")]
        public bool Verbose { get; set; }

        public int Run()
        {
            DoExportImport doObj = new DoExportImport();
            return doObj.DoImport(this.target, this.Path, this.FollowExistAccount, this.Verbose) ? 0 : 1;
        }
    }

}
