﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Xaml;
using SdkEnvironmentCheckerLibrary.Properties;

namespace SdkEnvironmentCheckerLibrary.Serializers
{
    public class SdkEnvironmentCheckerConfig
    {
        public IList<OsVersion> OsVersions { get; } = new NotNullList<OsVersion>();

        public DotNetFrameworkVersion DotNetFrameworkVersion { get; set; }

        public IList<InstalledProgramVersion> InstalledProgramVersions { get; } = new NotNullList<InstalledProgramVersion>();

        public bool NvnGraphicsDriver { get; set; }

        public static SdkEnvironmentCheckerConfig Load(string filepath)
        {
            if (!FilePath.TryCreate(filepath, out var path) ||
                !path.Exists) throw new ArgumentException(string.Format(Resources.FileNotFound, filepath));

            try
            {
                var loaded = XamlServices.Load(path.FullPath);
                if (loaded is SdkEnvironmentCheckerConfig settings) return settings;
            }
            catch { /* ignored : throw error */ }

            throw new InvalidOperationException(string.Format(Resources.FileLoadFailed, filepath));
        }

        public static void Save(string filepath, SdkEnvironmentCheckerConfig settings)
        {
            if (settings == null) throw new ArgumentNullException(nameof(settings));

            if (!FilePath.TryCreate(filepath, out var path))
            {
                throw new ArgumentException(string.Format(Resources.InvalidFilepath, filepath));
            }

            XamlServices.Save(path.FullPath, settings);
        }

        public SdkEnvironmentCheckerInfo ToCheckerInfo()
        {
            var info = new SdkEnvironmentCheckerInfo
            {
                DotNetFrameworkVersion = DotNetFrameworkVersion?.ToCheckerInfo(),
                NvnGraphicsDriver = NvnGraphicsDriver
            };

            OsVersions.Select(t => t.ToCheckerInfo()).ForEach(info.OsVersions.Add);
            InstalledProgramVersions.Select(t => t.ToCheckerInfo()).ForEach(info.InstalledProgramVersions.Add);

            return info;
        }
    }
}
