﻿using System;
using System.Collections.Generic;
using System.Linq;

namespace SdkEnvironmentCheckerLibrary.Checkers
{
    /// <summary>
    /// OS バージョンをチェックします。
    /// https://msdn.microsoft.com/library/windows/desktop/ms724832.aspx
    /// </summary>
    public static class OsVersionChecker
    {
        public static CheckerResult Check(IEnumerable<OsVersionInfo> supportedVersions) => Check(supportedVersions?.ToArray());

        public static CheckerResult Check(params OsVersionInfo[] supportedVersions)
        {
            if (supportedVersions == null) throw new ArgumentNullException(nameof(supportedVersions));

            var name = string.Join(Environment.NewLine, supportedVersions
                .Select(info => info.DisplayName)
                .Where(s => !string.IsNullOrWhiteSpace(s)));

            if (string.IsNullOrWhiteSpace(name))
            {
                return CheckerResult.ConfirmFailed("Unknown OS Version", "'DisplayName' is required");
            }

            if (IsSupported(supportedVersions))
            {
                return CheckerResult.Installed(name, "Available");
            }
            else
            {
                return CheckerResult.NotInstalled(name, "Not available");
            }
        }

        private static bool IsSupported(IEnumerable<OsVersionInfo> supportedVersions)
        {
            if (supportedVersions == null) throw new ArgumentNullException(nameof(supportedVersions));

            // app.manifest に対応 OS を追記しておかないと正しい値が取得できません。
            // app.manifest はこのライブラリでなく、各アプリケーションに追加する必要があります。
            // https://msdn.microsoft.com/ja-jp/library/windows/desktop/dn481241%28v=vs.85%29.aspx?f=255&MSPPError=-2147217396
            var current = Environment.OSVersion.Version;

            foreach (var info in supportedVersions.Where(i => i != null))
            {
                // Major と Minor のみ一致必須
                if (info.Version.Major != current.Major) continue;
                if (info.Version.Minor != current.Minor) continue;
                if (info.Is64BitOnly && !Environment.Is64BitOperatingSystem) continue;

                return true;
            }

            return false;
        }
    }

    public class OsVersionInfo
    {
        public OsVersionInfo(string displayName, Version version, bool is64BitOnly)
        {
            DisplayName = displayName ?? throw new ArgumentNullException(nameof(displayName));
            Version = version ?? throw new ArgumentNullException(nameof(version));
            Is64BitOnly = is64BitOnly;
        }

        public string DisplayName { get; }

        public Version Version { get; }

        public bool Is64BitOnly { get; }
    }
}
